/*
 * gp_engine.cpp
 *
 *  Created on: 7 mars 2013
 *      Author: valot
 */

#include "gp_engine.h"
#include "gp_error.h"
#include "sax/peptide_result_parser.h"
#include "sax/protein_grouping_result.h"
#include "filter/peptide_repro_filter.h"
#include "filter/peptide_by_prot_filter.h"
#include "filter/fasta_filter.h"
#include <QXmlSimpleReader>
#include <QXmlInputSource>
#include <QDebug>

GpEngine::GpEngine(GpParams* params) :
		_p_params(params) {

}

GpEngine::~GpEngine() {
	//delete protein pointer
	qDebug() << "Deleting GpEngine";
	std::map<QString, ProteinMatch*>::iterator proteinIt;
	ProteinMatch * p_protein;
	for (proteinIt = _proteinMatchList.begin();
			proteinIt != _proteinMatchList.end(); ++proteinIt) {
		p_protein = proteinIt->second;
		delete (p_protein);
	}
	_proteinMatchList.clear();

	//delete peptide pointer
	std::map<QString, Peptide*>::iterator peptideIt;
	Peptide * p_peptide;
	for (peptideIt = _peptideList.begin(); peptideIt != _peptideList.end();
			++peptideIt) {
		p_peptide = peptideIt->second;
		delete (p_peptide);
	}
	_peptideList.clear();

	//delete sample pointer
	std::map<QString, Sample*>::iterator SampleIt;
	Sample * p_sample;
	for (SampleIt = _sampleList.begin(); SampleIt != _sampleList.end();
			++SampleIt) {
		p_sample = SampleIt->second;
		delete (p_sample);
	}
	_sampleList.clear();
}

void GpEngine::performedTraitment() {
	this->readInputStream();
	this->filterResult();
	this->performedGrouping();
	this->writeXmlOutputResult();
}

void GpEngine::readInputStream() {
	PeptideResultParser parser(this);
	std::cerr << "Begin reading data" << std::endl;
	QXmlSimpleReader simplereader;
	simplereader.setContentHandler(&parser);
	simplereader.setErrorHandler(&parser);

	QXmlInputSource xmlInputSource(_p_params->getInputStream().device());

	if (simplereader.parse(xmlInputSource)) {
	} else {
		throw GpError(
				QObject::tr("error reading input file :\n").append(
						parser.errorString()));
	}
	std::cerr << "Finish loading data" << std::endl;
}

void GpEngine::filterResult() {
	//remove peptide not reproductibly identified in at least X samples
	PeptideReproFilter peptideFilter(this->_p_params);
	peptideFilter.filterResult(_peptideList, _proteinMatchList);
	//remove protein not identified with at least X reproductible peptides
	PeptideByProtFilter proteinFilter(this->_p_params);
	proteinFilter.filterResult(_peptideList, _proteinMatchList);
	//remove contaminants proteins
	FastaFilter fastaFilter(_p_params);
	fastaFilter.filterResult(_peptideList, _proteinMatchList);
}

void GpEngine::performedGrouping() {
	std::cerr << "Begin Grouping" << std::endl;
	std::map<QString, ProteinMatch*>::iterator proteinIt;
	ProteinMatch * p_protein;
	for (proteinIt = _proteinMatchList.begin();
			proteinIt != _proteinMatchList.end(); ++proteinIt) {
		p_protein = proteinIt->second;
		subgroups.addProteinMatch(p_protein);
	}
	std::cerr << "End subgroup creation : " << subgroups.size() << std::endl;

	groups.groupingSubGroupSet(subgroups);
	std::cerr << "End group creation : " << groups.size() << std::endl;
}

void GpEngine::writeXmlOutputResult() {
	std::cerr << "Begin Export" << std::endl;
	ProteinGroupingResult writer(this->_p_params);
	writer.printGroupingResult(this);
	std::cerr << "Finish Export" << std::endl;
}

ProteinMatch* GpEngine::getProteinMatchInstance(QString access, QString desc) {
	std::map<QString, ProteinMatch*>::iterator it;
	it = this->_proteinMatchList.find(access);
	if (it != _proteinMatchList.end()) {
		//qDebug() << "Protein already exist : " + access;
		return it->second;
	} else {
		ProteinMatch* match = new ProteinMatch(access, desc);
		_proteinMatchList[access] = match;
		qDebug() << "New Protein : " + access;
		return match;
	}
}

Sample* GpEngine::getSampleInstance(QString name) {
	std::map<QString, Sample*>::iterator it;
	it = this->_sampleList.find(name);
	if (it != _sampleList.end())
		return it->second;
	else {
		Sample* samp = new Sample(name);
		_sampleList[name] = samp;
		return samp;
	}
}

Peptide * GpEngine::getPeptideInstance(QString seq, gp_double mh) {
	std::map<QString, Peptide*>::iterator it;
	QString data = seq.replace("L", "I") + QString("|%1").arg((unsigned long) (mh * 100));
	it = this->_peptideList.find(data);
	if (it != _peptideList.end()) {
		//qDebug() << "Peptide already exist : " + data;
		return it->second;
	} else {
		qDebug() << "New Peptide : " + data;
		Peptide* pep = new Peptide(seq, mh);
		_peptideList[data] = pep;
		return pep;
	}
}
