#!/bin/sh

PROJDIR=/data/projects/two-way
CALCDIR=$PROJDIR/calculate
SBINDIR=$PROJDIR/sbin
DATADIR=$PROJDIR/data
GENOMESDIR=/data/databases/genomes
GENOMESCONFIGFILE=$GENOMESDIR/_config

CORES=24
BLOCKPARAMSFILE=input.block.params
GAPPARAMSFILE=input.gap.params
TARGETFILE=input.target
QUERYFILE=input.query
ALIGNFILE=input.align

if [ "$1" = "-h" ]; then
	cat <<EOS
USAGE: `basename $0` [-t targetfile] [-q queryfile]
    [-l minblock] [-i mingap] [-m maxgap] [-o maxover]
    [-n cores] [-a align]

WHERE
    tagetfile - name of the target file
                (default $TARGETFILE)
    queryfile - name of the query file
                (default $QUERYFILE)
    minblock  - minimum length of blocks
    mingap    - minimum length of gaps
    maxgap    - maximum length of gaps
    maxover   - maximum overlap in gaps
    cores     - number of cpu cores
                (default $CORES)
    align     - the alignent method (lastz, last)
                (default $ALIGN)
EOS
	exit 0
fi

# ------------------------------------------------------------------------
# Set parameters
# ------------------------------------------------------------------------

while getopts :t:q:l:i:m:o:n:a: arg; do
	case $arg in
		t) TARGETFILE=$OPTARG ;;
		q) QUERYFILE=$OPTARG ;;
		l) MINBLOCK=$OPTARG ;;
		i) MINGAP=$OPTARG ;;
		m) MAXGAP=$OPTARG ;;
		o) MAXOVER=$OPTARG ;;
		n) CORES=$OPTARG ;;
		a) ALIGN=$OPTARG ;;
	esac
done

BLOCKPARAMS=""
[ "$MINBLOCK" != "" ] && BLOCKPARAMS="-l $MINBLOCK"
[ "$BLOCKPARAMS" = "" -a -f $BLOCKPARAMSFILE ] && BLOCKPARAMS=`cat $BLOCKPARAMSFILE`

GAPPARAMS=""
[ "$MINGAP" != "" ] && GAPPARAMS="$GAPPARAMS -i $MINGAP"
[ "$MAXGAP" != "" ] && GAPPARAMS="$GAPPARAMS -m $MAXGAP"
[ "$MAXOVER" != "" ] && GAPPARAMS="$GAPPARAMS -o $MAXOVER"
[ "$GAPPARAMS" = "" -a -f $GAPPARAMSFILE ] && GAPPARAMS=`cat $GAPPARAMSFILE`

MAINPGM="$CALCDIR/align/twoway-main.sh"
[ "$ALIGN" = "" ] && ALIGN=`cat $ALIGNFILE`
[ "$ALIGN" != "last" -a "$ALIGN" != "lastz" ] && echo "Please set a valid align method!" && exit 1
[ "$ALIGN" = "last" ] && MAINPGM="$MAINPGM -L"

# ------------------------------------------------------------------------
# Set parameters from param files!  Otherwise they are missing
# ------------------------------------------------------------------------

setparams () {
	while getopts :t:q:l:i:m:o:n: arg; do
		case $arg in
			l) MINBLOCK=$OPTARG ;;
			i) MINGAP=$OPTARG ;;
			m) MAXGAP=$OPTARG ;;
			o) MAXOVER=$OPTARG ;;
		esac
	done
}

setparams $GAPPARAMS
setparams $BLOCKPARAMS

#
# ------------------------------------------------------------------------
# Execute a command
# ------------------------------------------------------------------------
#
execute () {
	local COMMAND=$1
	echo "Execute: $COMMAND"
	$COMMAND 2>&1
	RES=$?
	[ "$RES" != "0" ] && echo "Command failed with exit code $RES" && exit 1
}

# ------------------------------------------------------------------------
# Calculate data
# ------------------------------------------------------------------------

execute "$MAINPGM -n $CORES -t $TARGETFILE -q $QUERYFILE"
ERROR="N"
for FILE in calc.result.axt input.query.sizes; do
	[ ! -f $FILE ] && echo "Missing file $FILE!" && ERROR="Y"
done
[ "$ERROR" = "Y" ] && exit 1

execute "$CALCDIR/createblock.pl -fa calc.result.axt -fb calc.block.bed -fs input.query.sizes $BLOCKPARAMS"
execute "$CALCDIR/creategap.pl -fb calc.block.bed -fg calc.gap.bed $GAPPARAMS"
execute "$CALCDIR/createfasta.pl -fb calc.block.bed -ft input.target -fto calc.target.block.fasta -fq input.query -fqo calc.query.block.fasta"
execute "$CALCDIR/createfasta.pl -fb calc.gap.bed -ft input.target -fto calc.target.gap.fasta -fq input.query -fqo calc.query.gap.fasta"
execute "$CALCDIR/preparedistr.pl`if [ "$MINGAP" != "" ]; then echo \" -m $MINGAP\"; else echo \"\"; fi`"

# ------------------------------------------------------------------------
# Zip big output files
# ------------------------------------------------------------------------

for FILE in `ls *.axt *.fasta 2>/dev/null`; do
	execute "gzip $FILE"
done

# ------------------------------------------------------------------------
# Clean temporary folders
# ------------------------------------------------------------------------

for DIR in `ls calc.* 2>/dev/null`; do
	[ -d $DIR ] && execute "rm -rf $DIR"
done

exit 0