#!/usr/bin/env bash
# Created by Fengjun Zhang

set -eo pipefail

__base="$(basename "${BASH_SOURCE[0]}")"
__sd="$(dirname "${BASH_SOURCE[0]}")"
__wd=$PWD

#
# ------------------------------------------------------------------------
# Document
#
# Input files:
# $input_target $input_query [$file_config]
#
# Output files:
# $file_log [$input_target|query link] ${name_axt}.axt (default: "calc.result.axt")
# ${name_target|query}.{sizes|2bit} ${name_axt}.tmp.tar.gz
# (when split mode) ${name_target|query}_preSplit.tar.gz
# (when LAST mode) ${name_axt}_LAST.tar.gz
# ------------------------------------------------------------------------
#


#
# ------------------------------------------------------------------------
# Usage document
# ------------------------------------------------------------------------
#

function usage_doc () {
  e=$(printf "\e")
  fBold="$e[1m"
  fUnL="$e[4m"
  unF="$e[0m"
  cat << EOF

$__base document
${fBold}NAME$unF
    $fBold$__base$unF

${fBold}USAGE$unF
    bash $fBold$__base$unF [-d][-cLlnox ...] -t TARGET_FA -q QUERY_FA

${fBold}DESCRIPTION$unF
    $fBold$__base$unF is a bash script to run 2-way alignment

${fBold}OPTIONS$unF

    [optional]
    -d        debug mode
    -L        LAST mode
    -c ${fUnL}string$unF config file
    -l ${fUnL}string$unF log file
    -n ${fUnL}number$unF number of threads (default: 20)
    -o ${fUnL}string$unF name of output
    -x ${fUnL}number$unF split target and query sequence less than ${fUnL}number$unF bp

    [obligatory]
    -t ${fUnL}string$unF target genome
    -q ${fUnL}string$unF query genome

EOF
  exit
}

#
# ------------------------------------------------------------------------
# Check options and arguments
# ------------------------------------------------------------------------
#

function opt_check () {

  local OPTIND

  while getopts ":dLc:l:n:o:x:t:q:" opt; do
    case $opt in
      d)
        isDebugging=1 ;;
      L)
        isLASTmode=1 ;;
      c)
        file_config=$OPTARG ;;
      l)
        file_log=$OPTARG ;;
      n)
        num_threads=$OPTARG ;;
      o)
        file_out=$OPTARG ;;
      x)
        limit_len=$OPTARG ;;
      t)
        input_target=$OPTARG ;;
      q)
        input_query=$OPTARG ;;
      \?)
        echo "Invalid option: -$OPTARG"
        usage_doc ;;
      :)
        echo "Option -$OPTARG requires an argument."
        usage_doc ;;
    esac
  done

  # obligatory
  [ ! -f "$input_target" ] && echo "[Err] Cannot open target genome: $input_target" && usage_doc
  [ ! -f "$input_query" ] && echo "[Err] Cannot open query genome: $input_query" && usage_doc

  # optional default
  [ -z $file_log ] && file_log="calc.2way.log"
  [ -z $num_threads ] && num_threads=20
  [ -z $file_out ] && file_out="calc.result.axt"

}

#
# ------------------------------------------------------------------------
# run_last function
# ------------------------------------------------------------------------
#

function run_last () {

  if [ $isDebugging ]; then
    exec >>$file_log 2>&1
    set -o xtrace
    echo "[Log] debug mode set"
  fi

  lastdb -cR10 calc.targetDB $input_target
  lastal calc.targetDB $input_query > ${file_out%.*}.maf

  maf-convert axt ${file_out%.*}.maf > $file_out
  tar -zcf ${file_out}_LAST.tar.gz ${file_out%.*}.maf ./*calc.targetDB*
  rm -r ${file_out%.*}.maf ./*calc.targetDB*

}

#
# ------------------------------------------------------------------------
# init_lastz function
# ------------------------------------------------------------------------
#

function init_lastz () {

  str_exec=$__sd"/init2way.sh -t "$input_target" -q "$input_query

  [ $isDebugging ] && str_exec=$str_exec" -d"
  [ $file_log ] && str_exec=$str_exec" -l "$file_log
  [ $limit_len ] && str_exec=$str_exec" -x "$limit_len

  bash $str_exec

}

#
# ------------------------------------------------------------------------
# assign_lastz function
# ------------------------------------------------------------------------
#

function assign_lastz () {

  str_exec=$__sd"/lastz_assign.py"

  [ $isDebugging ] && str_exec=$str_exec" -d"
  [ $file_log ] && str_exec=$str_exec" -l "$file_log
  [ $file_config ] && str_exec=$str_exec" -c "$file_config
  [ $num_threads ] && str_exec=$str_exec" -n "$num_threads
  [ $file_out ] && str_exec=$str_exec" -o "$file_out

  str_exec=$str_exec" "$input_target" "$input_query
  python $str_exec

}

#
# ------------------------------------------------------------------------
# chain_net function
# ------------------------------------------------------------------------
#

function chain_net () {

  str_exec=$__sd"/mergeChainNet.sh -a "$file_out" -t "$input_target" -q "$input_query

  [ $isDebugging ] && str_exec=$str_exec" -d"
  [ $file_log ] && str_exec=$str_exec" -l "$file_log

  bash $str_exec

}

#
# ------------------------------------------------------------------------
# Main function
# ------------------------------------------------------------------------
#

opt_check "$@"
name_target=$(basename ${input_target%.fa})
name_query=$(basename ${input_query%.fa})

dir=$(dirname $input_target)
if [[ "$dir" != "." ]] || [[ "$input_target" != *".fa" ]]; then
  [[ ! -a $name_target.fa ]] && ln -s $input_target ${name_target}.fa || true
  input_target=${name_target}.fa
fi
dir=$(dirname $input_query)
if [[ "$dir" != "." ]] || [[ "$input_query" != *".fa" ]]; then
  [[ ! -a $name_query.fa ]] && ln -s $input_query ${name_query}.fa || true
  input_query=${name_query}.fa
fi

if [ $isLASTmode ]; then

  run_last

else

  init_lastz
  assign_lastz

fi

chain_net
