#!/usr/bin/env bash
# Created by Fengjun Zhang

__base="$(basename "${BASH_SOURCE[0]}")"
__wd=$PWD

#
# ------------------------------------------------------------------------
# Document
#
# Input files:
# $input_target $input_query [$file_config]
#
# Output files:
# $file_log $file_out $name_tempIndicator
# ------------------------------------------------------------------------
#


#
# ------------------------------------------------------------------------
# Usage document
# ------------------------------------------------------------------------
#
function usage_doc () {
  e=$(printf "\e")
  fBold="$e[1m"
  fUnL="$e[4m"
  unF="$e[0m"
  cat <<EOF

$__base document
${fBold}NAME$unF
    $fBold$__base$unF

${fBold}USAGE$unF
    bash $fBold$__base$unF [-d][-l LOG_FILE][-c CONFIG_FILE] -t TARGET_FA
    -q QUERY_FA -o FILE_OUT -i FILE_INDICATOR

${fBold}DESCRIPTION$unF
    $fBold$__base$unF is a bash script to run lastz for 2-way project

EOF
  exit
}



#
# ------------------------------------------------------------------------
# Check options and arguments
# ------------------------------------------------------------------------
#
function opt_check () {

  local OPTIND

  while getopts ":dl:c:i:t:q:o:" opt; do
    case $opt in
      c)
        file_config=$OPTARG ;;
      d)
        is_debug=1 ;;
      i)
        name_tempIndicator=$OPTARG ;;
      l)
        file_log=$OPTARG ;;
      t)
        input_target=$OPTARG ;;
      q)
        input_query=$OPTARG ;;
      o)
        file_out=$OPTARG ;;
      \?)
        echo "[Err] Invalid option: -$OPTARG"
        usage_doc ;;
      :)
        echo "[Err] Option -$OPTARG requires an argument."
        usage_doc ;;
    esac
  done

  # log files
  if [[ -z "$file_log" ]]; then
    file_log="runLASTZ.log"
  fi
  exec 2>>$file_log
  if [[ ! -z "$is_debug" ]]; then
    exec >>$file_log 2>&1
    set -o xtrace
    echo "[Log] debug mode set"
  fi

  # obligatory arguments
  if [[ -z "$input_target" ]]; then
    echo "[Err] target genome should be provided." >&2
    usage_doc
  fi
  if [[ -z "$input_query" ]]; then
    echo "[Err] query genome should be provided." >&2
    usage_doc
  fi
  if [[ -z "$file_out" ]]; then
    echo "[Err] output file should be specified." >&2
    usage_doc
  fi
  if [[ -z "$name_tempIndicator" ]]; then
    echo "[Err] name of progress indicator file should be specified." >&2
    usage_doc
  fi

  # optional arguments
  if [[ -z "$file_config" ]]; then
    str_config="--notransition --traceback=1500M"
  else
    str_config=$(echo -n $(head -1 $file_config))
  fi


}


#
# ------------------------------------------------------------------------
# Main function
# ------------------------------------------------------------------------
#

opt_check "$@"

lastz $input_target $input_query $str_config --output=$file_out --format=axt --markend

touch $name_tempIndicator
