#!/usr/bin/env bash
# Created by Fengjun Zhang

set -eo pipefail

__base="$(basename "${BASH_SOURCE[0]}")"
__wd=$PWD

#
# ------------------------------------------------------------------------
# Document
#
# Input files:
# $input_target $input_query $input_axt
#
# Output files:
# $file_log [$input_target/query link] ${name_axt}.tmp.tar.gz
# ${name_target/query}.sizes|2bit ${name_axt}.axt
# ------------------------------------------------------------------------
#


#
# ------------------------------------------------------------------------
# Usage document
# ------------------------------------------------------------------------
#
function usage_doc () {
  e=$(printf "\e")
  fBold="$e[1m"
  fUnL="$e[4m"
  unF="$e[0m"
  cat <<EOF

$__base document
${fBold}NAME$unF
    $fBold$__base$unF

${fBold}USAGE$unF
    bash $fBold$__base$unF [-d][-l LOG_FILE] -a AXT -t TARGET_FA -q QUERY_FA

${fBold}DESCRIPTION$unF
    $fBold$__base$unF is a bash script to create final result for 2-way project

EOF
  exit
}


#
# ------------------------------------------------------------------------
# Check options and arguments
# ------------------------------------------------------------------------
#
function opt_check () {

  local OPTIND

  while getopts ":a:dl:t:q:" opt; do
    case $opt in
      a)
        input_axt=$OPTARG ;;
      d)
        is_debug=1 ;;
      l)
        file_log=$OPTARG ;;
      t)
        input_target=$OPTARG ;;
      q)
        input_query=$OPTARG ;;
      \?)
        echo "[Err] Invalid option: -$OPTARG"
        usage_doc ;;
      :)
        echo "[Err] Option -$OPTARG requires an argument."
        usage_doc ;;
    esac
  done

  # log files
  if [[ -z "$file_log" ]]; then
    file_log="chainMergeSort.log"
  fi
  exec 2>>$file_log
  if [[ ! -z "$is_debug" ]]; then
    exec >>$file_log 2>&1
    set -o xtrace
    echo "[Log] debug mode set"
  fi

  # obligatory arguments
  if [[ -z "$input_axt" ]]; then
    echo "[Err] axt file should be provided." >&2
    usage_doc
  fi
  if [[ -z "$input_target" ]]; then
    echo "[Err] target genome should be provided." >&2
    usage_doc
  fi
  if [[ -z "$input_query" ]]; then
    echo "[Err] query genome should be provided." >&2
    usage_doc
  fi


}

opt_check "$@"
name_target=$(basename ${input_target%.fa})
name_query=$(basename ${input_query%.fa})
name_axt=$(basename ${input_axt%.axt})

# check if genomes are in working dir, if no create link
dir=$(dirname $input_target)
if [[ "$dir" != "." ]] || [[ "$input_target" != *".fa" ]]; then
  [[ ! -a $name_target.fa ]] && ln -s $input_target ${name_target}.fa || true
fi
dir=$(dirname $input_query)
if [[ "$dir" != "." ]] || [[ "$input_query" != *".fa" ]]; then
  [[ ! -a $name_query.fa ]] && ln -s $input_query ${name_query}.fa || true
fi
dir=$(dirname $input_axt)
if [[ "$dir" != "." ]]; then
  [[ ! -a ${name_axt}_tmp.axt ]] && ln -s $input_axt ${name_axt}_tmp.axt || true
else
  [[ ! -a ${name_axt}_tmp.axt ]] && mv $input_axt ${name_axt}_tmp.axt || true
fi

# create 2bit and sizes files
[[ ! -a $name_target.2bit ]] && faToTwoBit $name_target.fa $name_target.2bit || true
[[ ! -a $name_target.sizes ]] && faSize -detailed ${name_target}.fa > $name_target.sizes || true
[[ ! -a $name_query.2bit ]] && faToTwoBit $name_query.fa $name_query.2bit || true
[[ ! -a $name_query.sizes ]] && faSize -detailed ${name_query}.fa > $name_query.sizes || true

[[ ! -a $name_axt.chain ]] && axtChain -minScore=5000 -linearGap=medium ${name_axt}_tmp.axt $name_target.2bit $name_query.2bit $name_axt.chain || true
[[ ! -a $name_axt.sortedChain ]] && chainSort $name_axt.chain $name_axt.sortedChain || true

[[ ! -a $name_axt.preNet ]] && chainPreNet $name_axt.sortedChain $name_target.sizes $name_query.sizes $name_axt.preNet || true
[[ ! -a $name_target.net ]] && chainNet $name_axt.preNet $name_target.sizes $name_query.sizes $name_target.net $name_query.net || true

netToAxt $name_target.net $name_axt.preNet $name_target.2bit $name_query.2bit ${name_axt}.axt
tar -zcf ${name_axt}.tmp.tar.gz ${name_axt}_tmp.axt $name_axt.chain $name_axt.sortedChain $name_axt.preNet *.net
rm ${name_axt}_tmp.axt $name_axt.chain $name_axt.sortedChain $name_axt.preNet *.net
