#ifndef _NWAYRESULT_HH
#define _NWAYRESULT_HH

#include <iostream>
#include <fstream>
#include <string.h>

#include "bedfile.hh"
#include "cparams.hh"
#include "params.hh"

using namespace std;

/**
 * A range (start - end)
 */
struct Range {
	int start;
	int end;

public:
	Range(int start, int end) {
		this->start = start;
		this->end = end;
	}
};

/**
 * Preliminary data
 */
struct Preliminary {
	char chr[64];
	int start = 0;
	int end = 0;
	int len = 0;
	char strand;

public:
	Preliminary(char* chr, int start, int end, int len, char strand) {
		strcpy(this->chr, chr);
		this->start = start;
		this->end = end;
		this->len = len;
		this->strand = strand;
	}

	bool isValid() {
		return start < end;
	}

	void debug() {
		cout << "Preliminary(chr=" << chr << ", start=" << start << ",end=" << end << ",len=" << len << ")\n";
	}
};

/**
 * Alignment of target and query
 */
struct Alignment {
	char* tchr;
	int tstart;
	int tend;
	char* qchr;
	int qstart;
	int qend;
	char strand;

public:
	Alignment() {
	}

	Alignment(BedFile* bed, char* entry) {
		set(bed, entry);
	}

	void set(BedFile* bed, char* entry) {
		this->tchr = bed->getChr1(entry);
		this->tstart = bed->getStart1(entry);
		this->tend = bed->getEnd1(entry);
		this->qchr = bed->getChr2(entry);
		this->qstart = bed->getStart2(entry);
		this->qend = bed->getEnd2(entry);
		this->strand = bed->getStrand(entry);
	}
};

/**
 * The data for every coordinate line
 */
class Result {
public:
	char* spec;
	char* chr;
	int start;
	int end;

	char buf[256];
	ofstream* out = NULL;

	Result(char* outfile) {
		out = new ofstream(outfile);
	}

	~Result() {
		if (out != NULL) {
			out->close();
		}
	}

	void writeHeader(char** specs, int specn) {
		if (specn > 0) {
			out->write(specs[0], strlen(specs[0]));
		}
		for (int i = 1; i < specn; i++) {
			out->write("\t", 1);
			out->write(specs[i], strlen(specs[i]));
		}
		out->write("\n", 1);
	}

	void write(bool tab, char* result, char* info) {
		if (tab) {
			out->write("\t", 1);
		}
		snprintf(buf, sizeof(buf) - 1, "%s/%s", result, info);
		out->write(buf, strlen(buf));
	}

	void write(bool tab, char* symbol, char* info, char* chr, int start, int end, char strand) {
		if (tab) {
			out->write("\t", 1);
		}
		snprintf(buf, sizeof(buf) - 1, "%s/%s/%s:%d-%d/%c", symbol, info, chr, start, end, strand);
		out->write(buf, strlen(buf));
	}

	void tab() {
		out->write("\t", 1);
	}

	void newline() {
		out->write("\n", 1);
	}
};

class NwayResult {
	NwayParams* params = NULL;

	/**
	 * Adjust the range
	 */
	struct Range adjustRange(Alignment a, struct Range r);

	int getPoint(Alignment a, int tpoint);

	/**
	 * Returns the coverage
	 *
	 * @Params
	 * cstart,cend - coordinate start/end
	 * gstart,gend - gap start/end
	 */
	float getCoverage(int cstart, int cend, int gstart, int gend);

	/**
	 * Analyze in forward direction
	 */
	void analyzeForeward(Result* result, BedFile* gaps, BedFile* blocks);

	/**
	 * Further analyze in forward direction
	 */
	void reanalyzeForward(Result* result, BedFile* gaps, BedFile* blocks);

	/**
	 * Get preliminary data
	 */
	struct Preliminary getPreliminary(Result* result, BedFile* gaps);

	/**
	 * Get strand from preliminary data and gap/block
	 */
	char getStrand(char pstrand, char strand);

	/**
	 * Analyse in reverse direction
	 */
	void analyzeReverse(Result* result, BedFile* gaps, BedFile* blocks, Preliminary* prelim);

public:
	NwayResult(CommandParams* cparams, NwayParams* params);
};

#endif
