#include <iostream>
#include <fstream>
#include <string.h>
#include <math.h>

#include "exception.hh"
#include "bedfiles.hh"
#include "result.hh"

using namespace std;

NwayResult::NwayResult(CommandParams* cparams, NwayParams* params) {
	this->params = params;

	ifstream in(cparams->coordfile);

	if (!in.good()) {
		char message[128];
		sprintf(message, "coordfile %s cannot be read!", cparams->coordfile);
		throw NwayException(message);
	}

	BedFiles files(cparams->dbpath);
	Result result(cparams->nwayfile);
	result.writeHeader(cparams->specs, cparams->specn);

	char* target = cparams->specs[0];
	char line[256];
	int count = 0;

	// Loop over all coordinates
	while (!in.eof()) {
		in.getline(line, sizeof(line) - 1);
		if (strlen(line) >= 10) {
			if ((++count % 1000) == 0) {
				cout << "Analyze line " << count << "\n" << flush;
			}

			result.spec = strtok(line, " \t");
			result.chr = strtok(NULL, " \t");
			result.start = atoi(strtok(NULL, " \t"));
			result.end = atoi(strtok(NULL, " \t"));

			// Forward search
			if (strcmp(result.spec, target) == 0) {
				// First column
				result.write(false, (char*) "+", (char*) "x", result.chr, result.start, result.end, '+');

				// Then all other species
				for (int i = 1; i < cparams->specn; i++) {
					BedFile* gaps = files.get(target, cparams->specs[i], 'g');
					BedFile* blocks = files.get(target, cparams->specs[i], 'b');
					analyzeForeward(&result, gaps, blocks);
				}
			}

			// Reverse search
			else {
				Preliminary prelim = getPreliminary(&result, files.get(target, result.spec, 'g'));
				if (!prelim.isValid()) {
					continue;
				}

				// First column
				result.write(false, (char*) "-", (char*) "x", prelim.chr, prelim.start, prelim.end, prelim.strand);

				// All other species
				for (int i = 1; i < cparams->specn; i++) {
					char* spec = cparams->specs[i];

					// If the preliminary species is the same as in coordinate
					if (strcmp(result.spec, spec) == 0) {
						result.write(true, (char*) "+", (char*) "rp", result.chr, result.start, result.end, '+');
						continue;
					}

					BedFile* gaps = files.get(target, spec, 'g');
					BedFile* blocks = files.get(target, spec, 'b');
					analyzeReverse(&result, gaps, blocks, &prelim);
				}
			}

			result.newline();
		}
	}

	in.close();
}

float NwayResult::getCoverage(int cstart, int cend, int gstart, int gend) {
	if (gend - gstart == 1) {
		return 1.0;
	} else {
		return min(float(min(cend, gend)) - float(max(cstart, gstart)) / float(gend - gstart - 1), (float) 1.0);
	}
}

struct Range NwayResult::adjustRange(Alignment a, struct Range r) {
	int targetSize = a.tend - a.tstart + 1;
	int querySize = a.qend - a.qstart + 1;
	int flankLeft = r.start - a.tstart + 1;
	int flankRight = a.tend - r.end + 1;
	int coordSize = r.end - r.start + 1;

	// The block start and end
	int start;
	int end;

	if (min(flankLeft, flankRight) < params->minblockflanks) {
		if (flankLeft < flankRight) {
			if (a.strand == '+') {
				start = a.qstart + flankLeft;
				end = start + coordSize;
			} else {
				end = a.qend - flankLeft;
				start = end - coordSize;
			}
		} else {
			if (a.strand == '+') {
				end = a.qend - flankRight;
				start = end - coordSize;
			} else {
				start = a.qstart + flankRight;
				end = start + coordSize;
			}
		}
	} else {
		// If flanks are long, we should take in account coefficient of stains similarity (css)
		double css = double(querySize) / double(targetSize);

		// Than we have to find "middle point" from both flanks (according strand value).

		// The midpoint's left and right
		double left;
		double right;

		if (a.strand == '+') {
			left = a.qstart + (flankLeft * css);
			right = a.qend - (flankRight * css);
		} else {
			left = a.qstart + (flankRight * css);
			right = a.qend - (flankLeft * css);
		}

		double center = (left + right) / 2.0;

		// Now we can calculate start and end of sequence inside block.
		start = (int) round(center - coordSize / 2.0);
		end = (int) round(center + coordSize / 2.0);
	}

	return Range(start, end);
}

int NwayResult::getPoint(Alignment a, int tpoint) {
	int targetSize = a.tend - a.tstart + 1;
	int querySize = a.qend - a.qstart + 1;
	int flankLeft = tpoint - a.tstart + 1;
	int flankRight = a.tend - tpoint + 1;

	// The point in the query block
	int point;

	if (min(flankLeft, flankRight) < params->minblockflanks) {
		if (flankLeft < flankRight) {
			if (a.strand == '+') {
				point = a.qstart + flankLeft;
			} else {
				point = a.qend - flankLeft;
			}
		} else {
			if (a.strand == '+') {
				point = a.qend - flankRight;
			} else {
				point = a.qstart + flankRight;
			}
		}
	} else {
		// If flanks are long, we should take in account coefficient of stains similarity (css)
		double css = querySize / targetSize;

		// Than we have to find "middle point" from both flanks (according strand value).

		// The midpoint's left and right
		double left;
		double right;

		if (a.strand == '+') {
			left = a.qstart + (flankLeft * css);
			right = a.qend - (flankRight * css);
		} else {
			left = a.qstart + (flankRight * css);
			right = a.qend - (flankLeft * css);
		}

		point = (int) round((left + right) / 2.0);
	}

	return point;
}

void NwayResult::analyzeForeward(Result* result, BedFile* gaps, BedFile* blocks) {
	for (int i = gaps->getFirstIndex1(result->chr); i != ENDOFLIST; i = gaps->getNextIndex1(i)) {
		char* entry = gaps->getEntry(i);

		// Check chromosome in result and target
		if (strcmp(result->chr, gaps->getChr1(entry)) != 0)
			continue;

		// The target and query widths
		int tw = abs(gaps->getEnd1(entry) - gaps->getStart1(entry) - 1);
		int qw = abs(gaps->getEnd2(entry) - gaps->getStart2(entry) - 1);

		if (tw <= params->mingapsize || qw > params->mingapsize)
			continue;

		// Assume insert is in target
		if (tw > qw) {
			if (params->gapsearch == 1) {
				// Check overlap (it should be smaller or equal than maxover)
				if (max(gaps->getStart1(entry) - result->start, result->end - gaps->getEnd1(entry)) > params->maxover)
					continue;

				// Check coverage, it should be bigger or equal than mincov)
				if (getCoverage(result->start, result->end, gaps->getStart1(entry), gaps->getEnd1(entry))
						< params->mincov)
					continue;
			} else {
				// Check distance
				if (max(abs(result->start - gaps->getStart1(entry)), abs(gaps->getEnd1(entry) - result->end))
						> params->maxdist)
					continue;
			}

			result->write(true, (char*) "-", (char*) "fgq", gaps->getChr2(entry), gaps->getStart2(entry),
					gaps->getEnd2(entry), gaps->getStrand(entry));
			return;
		}

		// Else insert is in query
		else {
			// Check that coordinates are around the gap in target.
			if (result->start >= gaps->getStart1(entry) && result->end <= gaps->getEnd1(entry))
				continue;

			result->write(true, (char*) "++", (char*) "fgt", gaps->getChr2(entry), gaps->getStart2(entry),
					gaps->getEnd2(entry), gaps->getStrand(entry));
			return;
		}
	}

	for (int i = blocks->getFirstIndex1(result->chr); i != ENDOFLIST; i = blocks->getNextIndex1(i)) {
		char* entry = blocks->getEntry(i);

		// Check chromosome in row and target
		if (strcmp(result->chr, blocks->getChr1(entry)) != 0)
			continue;

		if (min(result->start - blocks->getStart1(entry), blocks->getEnd1(entry) - result->end) < params->minflanks)
			continue;

		// The target and query widths
		int tw = abs(blocks->getEnd1(entry) - blocks->getStart1(entry) - 1);
		int qw = abs(blocks->getEnd2(entry) - blocks->getStart2(entry) - 1);

		char* symbol = tw >= params->mingapsize && qw >= params->mingapsize ? (char*) "+" : (char*) "-";
		Range r = adjustRange(Alignment(blocks, entry), Range(result->start, result->end));

		result->write(true, symbol, (char*) "fb", blocks->getChr2(entry), r.start, r.end, blocks->getStrand(entry));
		return;
	}

	reanalyzeForward(result, gaps, blocks);

	// Already done in reanalyze!
	// result->write(true, (char*) "N", (char*) "f");
}

void NwayResult::reanalyzeForward(Result* result, BedFile* gaps, BedFile* blocks) {
	int counter = 0;

	// Left block counter
	int lbcounter = 0;
	// Right block counter
	int rbcounter = 0;
	// Left gap counter
	int lgcounter = 0;
	// Right gap counter
	int rgcounter = 0;

	// First step

	struct Alignment lblock;
	struct Alignment rblock;
	struct Alignment lgap;
	struct Alignment rgap;

	for (int i = blocks->getFirstIndex1(result->chr); i != ENDOFLIST; i = blocks->getNextIndex1(i)) {
		char* entry = blocks->getEntry(i);

		// First block is 1, not 0!
		counter++;

		// Everything found in blocks?
		if (min(lbcounter, rbcounter) > 0)
			continue;

		// Check chromosome in row and target
		if (strcmp(result->chr, blocks->getChr1(entry)) != 0)
			continue;

		// Check left row border (bigger or equal then start AND smaller or equal then end)
		if (blocks->getStart1(entry) <= result->start && result->start <= blocks->getEnd1(entry)) {
			lbcounter = counter;
			lblock.set(blocks, entry);
		}

		// Check right row border (bigger or equal then start AND smaller or equal then end)
		if ((blocks->getEnd1(entry) <= result->end && result->end <= blocks->getEnd1(entry))) {
			rbcounter = counter;
			rblock.set(blocks, entry);
		}
	}

	// If one of row border is not found in block, test gaps
	if (lbcounter == 0 || rbcounter == 0) {
		counter = 0;

		for (int i = gaps->getFirstIndex1(result->chr); i != ENDOFLIST; i = gaps->getNextIndex1(i)) {
			char* entry = gaps->getEntry(i);

			counter++;

			// Check chromosome in row and target
			if (strcmp(result->chr, gaps->getChr1(entry)) != 0)
				continue;

			// If left row border is not found:
			if (lbcounter == 0) {
				// Check left row border (bigger then start AND smaller then end).
				if (gaps->getStart1(entry) < result->start && result->start < gaps->getEnd1(entry)) {
					lgcounter = counter;
					lgap.set(gaps, entry);
				}
			}

			// If right row border is not found
			if (rbcounter == 0) {
				// Check right row border in gaps
				if (gaps->getStart1(entry) < result->end && result->end < gaps->getEnd1(entry)) {
					rgcounter = counter;
					rgap.set(gaps, entry);
				}
			}
		}
	}

	// Second step

	if (lbcounter > 0 && rbcounter > 0) {
		// Both row borders are in same block, but it is not "clean plus"
		if (lbcounter == rbcounter) {
			Range r = Range(0, 0);
			if (lblock.strand == '+') {
				r.start = getPoint(lblock, result->start);
				r.end = getPoint(lblock, result->end);
			} else {
				r.start = getPoint(lblock, result->end);
				r.end = getPoint(lblock, result->start);
			}
			// Range r = adjustRange(lblock, Range(lblock.qstart, lblock.qend));
			result->write(true, (char*) "+?", (char*) "f+", lblock.qchr, r.start, r.end, lblock.strand);
		} else {
			// Blocks are different, but are query chromosomes the same?
			if (strcmp(lblock.qchr, rblock.qchr) == 0) {
				Range r = Range(0, 0);
				if (lblock.strand == '+') {
					r.start = getPoint(lblock, result->start);
					r.end = getPoint(rblock, result->end);
				} else {
					r.start = getPoint(rblock, result->start);
					r.end = getPoint(lblock, result->end);
				}
				result->write(true, (char*) "++", (char*) "f+", lblock.qchr, r.start, r.end, rblock.strand);
			} else {
				// Broken connection between blocks
				result->write(true, (char*) "+N", (char*) "f+");
			}
		}
	} else if (lbcounter > 0 && rgcounter > 0) {
		// Left row border is in block, and right one is in gap, same chromosome
		if (strcmp(lblock.qchr, rgap.qchr) == 0) {
			// Calculating overlaps with block and gap.
			int lbs = lblock.tend - result->start;
			int rgs = result->end - rgap.tstart;

			// If overlap with block smaller than
			if (lbs < params->maxover) {
				result->write(true, (char*) "-?", (char*) "f+", rgap.qchr, rgap.qstart, rgap.qend, rgap.strand);
			} else {
				Range r = Range(getPoint(lblock, result->start), lblock.strand == '+' ? rgap.qend : rgap.qstart);
				// If block overlap is smaller
				if (lbs < rgs) {
					result->write(true, (char*) "-+", (char*) "f+", lblock.qchr, r.start, r.end, rgap.strand);
				} else {
					result->write(true, (char*) "+-", (char*) "f+", lblock.qchr, r.start, r.end, rgap.strand);
				}
			}
		} else {
			// Connection between block and gap is brocken
			result->write(true, (char*) "+N", (char*) "f+");
		}
	} else if (lgcounter > 0 && rbcounter > 0) {
		// Right row border is in block, and right one is in gap.
		if (strcmp(rblock.qchr, lgap.qchr) == 0) {
			// Calculating overlaps with block and gap.
			int rbs = result->end - rblock.tstart;
			int lgs = lgap.tend - result->start;

			if (rbs < params->maxover) {
				result->write(true, (char*) "-?", (char*) "f+", lgap.qchr, lgap.qstart, rblock.qend, lgap.strand);
			} else {
				Range r = Range(getPoint(rblock, result->start), rblock.strand == '+' ? lgap.qstart : rgap.qend);
				// If block overlap is smaller
				if (rbs < lgs) {
					result->write(true, (char*) "-+", (char*) "f+", lgap.qchr, r.start, r.end, lgap.strand);
				} else {
					result->write(true, (char*) "+-", (char*) "f+", lgap.qchr, r.start, r.end, lgap.strand);
				}
			}
		} else {
			// Connection between block and gap is broken
			result->write(true, (char*) "+N", (char*) "f+");
		}
	} else if (lbcounter > 0 && rgcounter == 0) {
		// Only one side found
		result->write(true, (char*) "+N", (char*) "f+");
	} else if (lgcounter == 0 && rbcounter > 0) {
		// Only one side found
		result->write(true, (char*) "+N", (char*) "f+");
	} else if (lgcounter > 0 && rgcounter > 0) {
		if (lgcounter == rgcounter) {
			// Both row borders are in same gap, but it is not "clean minus".
			result->write(true, (char*) "-?", (char*) "f+", lgap.qchr, lgap.qstart, rgap.qend, rgap.strand);
		} else {
			// Gaps are different.
			if (strcmp(lgap.qchr, rgap.qchr) == 0) {
				// Two gaps are detected
				if (lgap.strand == '+') {
					result->write(true, (char*) "--", (char*) "f+", lgap.qchr, lgap.qstart, rgap.qend, lgap.strand);
				} else {
					result->write(true, (char*) "--", (char*) "f+", lgap.qchr, rgap.qstart, lgap.qend, lgap.strand);
				}
			} else {
				// Broken connection between gaps
				result->write(true, (char*) "-N", (char*) "f+");
			}
		}
	} else {
		// Nothing significant found
		result->write(true, (char*) "N", (char*) "f+");
	}
}

struct Preliminary NwayResult::getPreliminary(Result* result, BedFile* gaps) {
	for (int i = gaps->getFirstIndex2(result->chr); i != ENDOFLIST; i = gaps->getNextIndex2(i)) {
		char* entry = gaps->getEntry(i);

		// Check chromosome in row and target
		if (strcmp(result->chr, gaps->getChr2(entry)) != 0)
			continue;

		// The target width
		int tw = abs(gaps->getEnd1(entry) - gaps->getStart1(entry) - 1);

		// Next if target gap is small
		if (tw > params->mingapsize)
			continue;

		if (params->gapsearch == 1) {
			// Check overlap (it should be smaller or equal than maxover)
			if (max(abs(result->start - gaps->getStart2(entry)), abs(gaps->getEnd2(entry) - result->end))
					> params->maxover)
				continue;

			// Check coverage, it should be bigger or equal than mincov)
			if (getCoverage(result->start, result->end, gaps->getStart2(entry), gaps->getEnd2(entry)) < params->mincov)
				continue;
		} else {
			// Check distance
			if (max(abs(result->start - gaps->getStart2(entry)), abs(gaps->getEnd2(entry) - result->end))
					> params->maxdist)
				continue;
		}

		return Preliminary(gaps->getChr1(entry), gaps->getStart1(entry), gaps->getEnd1(entry),
				result->end - result->start + 1, gaps->getStrand(entry));
	}

	return Preliminary((char*) "", 0, 0, 0, ' ');
}

char NwayResult::getStrand(char pstrand, char strand) {
	if (pstrand == '+') {
		return strand;
	} else {
		return strand == '+' ? '-' : '+';
	}
}

void NwayResult::analyzeReverse(Result* result, BedFile* gaps, BedFile* blocks, Preliminary* prelim) {
	for (int i = blocks->getFirstIndex1(prelim->chr); i != ENDOFLIST; i = blocks->getNextIndex1(i)) {
		char* entry = blocks->getEntry(i);

		// Check chromosome in row and target
		if (strcmp(prelim->chr, blocks->getChr1(entry)) != 0)
			continue;

		if (prelim->end < blocks->getStart1(entry))
			continue;

		if (prelim->start > blocks->getEnd1(entry))
			continue;

		// Check flanks
		if (min(prelim->start - blocks->getStart1(entry), blocks->getEnd1(entry) - prelim->end) < params->minflanks)
			continue;

		Range r = adjustRange(Alignment(blocks, entry), Range(prelim->start, prelim->end));

		result->write(true, (char*) "-", (char*) "rb", blocks->getChr2(entry), r.start, r.end,
				this->getStrand(prelim->strand, blocks->getStrand(entry)));
		return;
	}

	for (int i = gaps->getFirstIndex1(prelim->chr); i != ENDOFLIST; i = gaps->getNextIndex1(i)) {
		char* entry = gaps->getEntry(i);

		// Check chromosome in result and target
		if (strcmp(prelim->chr, gaps->getChr1(entry)) != 0)
			continue;

		// The target and query widths
		int tw = abs(gaps->getEnd1(entry) - gaps->getStart1(entry) - 1);
		int qw = abs(gaps->getEnd2(entry) - gaps->getStart2(entry) - 1);

		// Check minimum gap size
		if (tw > params->mingapsize && qw <= params->mingapsize)
			continue;

		// Check maximum distance
		if (max(abs(prelim->start - gaps->getStart1(entry)), abs(prelim->end - gaps->getEnd1(entry))) > params->maxdist)
			continue;

		// Check maximum diffratio
		if ((abs(prelim->len - qw) / prelim->len) >= params->maxdiffratio)
			continue;

		result->write(true, (char*) "+", (char*) "rgt", gaps->getChr2(entry), gaps->getStart2(entry),
				gaps->getEnd2(entry), this->getStrand(prelim->strand, gaps->getStrand(entry)));
		return;
	}

	result->write(true, (char*) "N", (char*) "r");
}
