#ifndef _BEDFILE_HH
#define _BEDFILE_HH

#include <iostream>
#include <fstream>
#include <string.h>

#include "exception.hh"

// The end-of-list
#define ENDOFLIST -1

using namespace std;

/**
 * Class to manage bed files effectiv, meaning to create a binary
 * version from a text bed file which can be put into memory and used
 * without any changes.
 */
class BedFile {
	//
	// The header of the binfile
	//

	// The chromosome hash size
	int chrhashsize = 0;
	// Max size of chromosome 1
	int chr1size = 0;
	// Max size of chromosome 2
	int chr2size = 0;
	// Number of entries
	int entrycount = 0;

	// The indices of first elements of a specific hash value
	int* chr1hash = NULL;
	// The indices of first elements of a specific hash value
	int* chr2hash = NULL;

	//
	// The data/entries of the binfile
	//

	char* entries = NULL;

	//
	// Additional (class) info
	//

	// Size of a single entry data
	int entrysize = 0;

	// The offsets of the fields
	int chr1offset = 0;
	int start1offset = 0;
	int end1offset = 0;
	int next1offset = 0;
	int chr2offset = 0;
	int start2offset = 0;
	int end2offset = 0;
	int next2offset = 0;
	int strandoffset = 0;

	void freemem(char* p);

	int* createChrHash();

	/**
	 * Initialize values and create memory arrays based on chrhashsize,
	 * chr1size and chr2size
	 */
	void init();

	/**
	 * Align chromosome size to a value of {4,8,12,...}
	 */
	int alignchrsize(int chrsize);

	void createHashChains();

public:
	BedFile(char* binfile);

	BedFile(char* bedfile, int chrhashsize);

	~BedFile();

	/**
	 * Return a hash (index) for a chromosome
	 */
	inline int chrhash(char* chr) {
		int hash = 0;
		int index = 1;
		char c;
		while ((c = *chr++) != 0) {
			hash += (c - 32) * index++;
		}
		return hash % chrhashsize;
	}

	inline char* getEntry(int index) {
		return &entries[entrysize * index];
	}

	// Moving

	inline int getFirstIndex1(char* chr) {
		return chr1hash[chrhash(chr)];
	}

	inline int getNextIndex1(int index) {
		return getNext1(getEntry(index));
	}

	inline int getFirstIndex2(char* chr) {
		return chr2hash[chrhash(chr)];
	}

	inline int getNextIndex2(int index) {
		return getNext2(getEntry(index));
	}

	// Setter

	inline void setChr1(char* entry, char* chr) {
		strcpy(entry, chr);
	}

	inline void setStart1(char* entry, int value) {
		*(int*) (entry + start1offset) = value;
	}

	inline void setEnd1(char* entry, int value) {
		*(int*) (entry + end1offset) = value;
	}

	inline void setNext1(char* entry, int index) {
		*(int*) (entry + next1offset) = index;
	}

	inline void setChr2(char* entry, char* chr) {
		strcpy(entry + chr2offset, chr);
	}

	inline void setStart2(char* entry, int value) {
		*(int*) (entry + start2offset) = value;
	}

	inline void setEnd2(char* entry, int value) {
		*(int*) (entry + end2offset) = value;
	}

	inline void setNext2(char* entry, int index) {
		*(int*) (entry + next2offset) = index;
	}

	inline void setStrand(char* entry, char strand) {
		*(char*) (entry + strandoffset) = strand;
	}

	// Getter

	inline char* getChr1(char* entry) {
		return entry;
	}

	inline int getStart1(char* entry) {
		return (int) *(int*) (entry + start1offset);
	}

	inline int getEnd1(char* entry) {
		return (int) *(int*) (entry + end1offset);
	}

	inline int getNext1(char* entry) {
		return (int) *(int*) (entry + next1offset);
	}

	inline char* getChr2(char* entry) {
		return &entry[chr2offset];
	}

	inline int getStart2(char* entry) {
		return (int) *(int*) (entry + start2offset);
	}

	inline int getEnd2(char* entry) {
		return (int) *(int*) (entry + end2offset);
	}

	inline int getNext2(char* entry) {
		return (int) *(int*) (entry + next2offset);
	}

	inline char getStrand(char* entry) {
		return *(entry + strandoffset);
	}

	/**
	 * Load a binary bed file into memory
	 */
	void load(char* binfile);
	/**
	 * Load a bed file into memory
	 */
	void loadBed(char* bedfile);

	/**
	 * Write the internal data into a file
	 */
	void write(char* binfile);
};

#endif
