library(ggplot2)
library(cowplot)
library(mgcv)
library(qqplotr)




#----------------- Prepare human gseq data ---------------------


RH_human <- read.table("RH_human_gseq.txt",header=TRUE,sep="\t",stringsAsFactors=FALSE)

# Get rows at beginning of each chromosome:
RH_human_start <- RH_human[RH_human$posS == 0 & RH_human$posE == 1e6,]

# Get rid of ramp ups and ramp downs:
RH_human <- RH_human[c(0,diff(RH_human$pos)) == 1e4,]

# combine RH_human without ramps and RH_human_start:
RH_human <- rbind(RH_human_start,RH_human)


# Sort:
chrOrder<-paste("chr",c(1:22,"X","Y"),sep="")
RH_human$Chromosome <-factor(RH_human$Chromosome, levels=chrOrder)
RH_human <- RH_human[order(RH_human$Chromosome, RH_human$pos), ]
RH_human$Chromosome <- as.character(RH_human$Chromosome)


# # Transform chr1 etc. to numbers
RH_human$Chromosome <- gsub('chr', '', RH_human$Chromosome)
RH_human[RH_human$Chromosome == "X","Chromosome"] <- 23
RH_human[RH_human$Chromosome == "Y","Chromosome"] <- 24
chrOrder<-c(1:24)
RH_human$Chromosome <-factor(RH_human$Chromosome, levels=chrOrder)
RH_human <- RH_human[order(RH_human$Chromosome, RH_human$pos), ]
RH_human$Chromosome <- as.numeric(RH_human$Chromosome)

# Compute chromosome size
gen_coord <- aggregate(pos~Chromosome,FUN=max,data=RH_human)
colnames(gen_coord)[2] <- "chr_size"
gen_coord$Chromosome <-factor(gen_coord$Chromosome, levels=chrOrder)
gen_coord <- gen_coord[order(gen_coord$Chromosome), ]
gen_coord$Chromosome <- as.numeric(gen_coord$Chromosome)

# Use cumsum to make genome coordinates
gen_coord$coord <- c(0,cumsum(gen_coord$chr_size)[-24])

# merge genome coordinates with RH_human
RH_human <- merge(RH_human,gen_coord[,c("Chromosome","coord")])
RH_human$Chromosome <-factor(RH_human$Chromosome, levels=chrOrder)
RH_human <- RH_human[order(RH_human$Chromosome, RH_human$pos), ]
RH_human$Chromosome <- as.numeric(RH_human$Chromosome)

RH_human$coord <- RH_human$pos + RH_human$coord

# get rid of chrY, because no chrY seq in hamster genome
RH_human <- RH_human[RH_human$Chromosome != "chrY",]

# Get rid of unneeded coord column at end of RH_human
RH_human <- RH_human[,-ncol(RH_human)]


# ------------------ Read in and prepare ancillary tables -------------------------------

cell <- read.table("cell_label_info.txt",sep="\t",stringsAsFactors=FALSE,header=TRUE)
sum_reads <- colSums(RH_human[,5:ncol(RH_human)])
reads <- read.table("RH_pool_human_total_align.txt",sep="\t",stringsAsFactors=FALSE,header=TRUE) # uses mapped human reads, cf human_AIC_1.R



# --------- Down-sample RH_human to resid tables and prepare for gam ------------------------------


resid <- rbind(
RH_human[RH_human$Chromosome == 1 & RH_human$posS == 20980000,],
RH_human[RH_human$Chromosome == 3 & RH_human$posS == 97720000,],
RH_human[RH_human$Chromosome == 4 & RH_human$posS == 101710000,],
RH_human[RH_human$Chromosome == 6 & RH_human$posS == 108360000,],
RH_human[RH_human$Chromosome == 10 & RH_human$posS == 59240000,],
RH_human[RH_human$Chromosome == 11 & RH_human$posS == 41580000,],
RH_human[RH_human$Chromosome == 12 & RH_human$posS == 75090000,],
RH_human[RH_human$Chromosome == 15 & RH_human$posS == 87900000,]
)


dim(resid)
# [1]   8 119


resid_l <- reshape(resid, 
  varying = c(colnames(resid[c(5:ncol(resid))])), 
  v.names = "copy",
  timevar = "RH_ID", 
  times = c(colnames(resid[c(5:ncol(resid))])), 
  new.row.names = 1:1e6,
  direction = "long")
  
resid_l$week <- 0
resid_l[grepl("_w0_",resid_l$RH_ID),]$week <- 0
resid_l[grepl("_w1_",resid_l$RH_ID),]$week <- 1
resid_l[grepl("_w2_",resid_l$RH_ID),]$week <- 2
resid_l[grepl("_w3_",resid_l$RH_ID),]$week <- 3
resid_l[grepl("_w4_",resid_l$RH_ID),]$week <- 4
resid_l[grepl("_w6_",resid_l$RH_ID),]$week <- 6

resid_l$conc <- 0
resid_l[grepl("_d0",resid_l$RH_ID),]$conc <- 0
resid_l[grepl("_d8",resid_l$RH_ID),]$conc <- 8
resid_l[grepl("_d25",resid_l$RH_ID),]$conc <- 25
resid_l[grepl("_d75",resid_l$RH_ID),]$conc <- 75

resid_l$pool <- 0
resid_l[grepl("RH1_",resid_l$RH_ID),]$pool <- 1
resid_l[grepl("RH2_",resid_l$RH_ID),]$pool <- 2
resid_l[grepl("RH3_",resid_l$RH_ID),]$pool <- 3
resid_l[grepl("RH4_",resid_l$RH_ID),]$pool <- 4
resid_l[grepl("RH5_",resid_l$RH_ID),]$pool <- 5
resid_l[grepl("RH6_",resid_l$RH_ID),]$pool <- 6


resid_l <- merge(resid_l,cell)
resid_l$sum_reads <- sum_reads[resid_l$RH_ID]
resid_l  <- merge(resid_l,reads[,c(1:5,9)])
colnames(resid_l)[13] <- "total_reads"


resid_l$pool <- as.factor(resid_l$pool)
resid_l$cell <- as.factor(resid_l$cell)

dim(resid_l)
# [1] 920  13

head(resid_l)
  # week conc pool     RH_ID cell Chromosome      posS      posE       pos copy id sum_reads total_reads
# 1    0    0    1 RH1_w0_d0    1          1  20980000  21980000  21480000   53  1  28834954      298433
# 2    0    0    1 RH1_w0_d0    1          3  97720000  98720000  98220000  108  2  28834954      298433
# 3    0    0    1 RH1_w0_d0    1          4 101710000 102710000 102210000   40  3  28834954      298433
# 4    0    0    1 RH1_w0_d0    1          6 108360000 109360000 108860000   75  4  28834954      298433
# 5    0    0    1 RH1_w0_d0    1         10  59240000  60240000  59740000  102  5  28834954      298433
# 6    0    0    1 RH1_w0_d0    1         11  41580000  42580000  42080000   85  6  28834954      298433




# ----------------------------- ggplot ---------------------------------



# from https://stackoverflow.com/questions/8197559/emulate-ggplot2-default-color-palette
# function for default ggplot2 colors

gg_color_hue <- function(n) {
  hues = seq(15, 375, length = n + 1)
  hcl(h = hues, l = 65, c = 100)[1:n]
}

theme2 <- theme(
	plot.margin = unit(c(0.5,0.5,0.5,0.5), "cm"),
	panel.grid.major = element_blank(), 
	panel.grid.minor = element_blank(), 
	panel.background = element_blank(),  
	axis.line.x = element_line(colour = "black", size = 0.1), 
	axis.line.y = element_line(colour = "black", size = 0.1), 
	axis.ticks = element_line(colour = "black", size = 0.1),
	axis.text=element_text(size=12), #numbers on tick marks of x and y axes
	axis.title=element_text(size=14), #titles of x and y axes
	axis.title.y=element_text(margin=margin(t=0,r=13,b=0,l=0)), #moves y axis title by adding margin space to bottom, trbl
	axis.title.x=element_text(margin=margin(t=10,r=0,b=0,l=0)),  #moves x axis title by adding margin space to top,trbl
	plot.subtitle = element_text(size=14, face="plain", hjust = 0.5), #hjust shifts right
	legend.margin=margin(t=0,r=0,b=0,l=0,unit = "pt"),
 	legend.box.margin=margin(t=0,r=0,b=0,l=0,unit = "pt"),
 	legend.key.height = unit(0.05, "cm"),
	legend.key.width = unit(0.2, "cm"),
 	legend.spacing.y = unit(0.05, 'cm'),
 	legend.spacing.x = unit(0.1, 'cm'),
 	legend.title = element_text(size = 9), 
 	legend.text = element_text(size = 8),
 	legend.key = element_blank()
 	# legend.title.align=0.0
	)
	
	
size_point <- 0.3
size_hline <- 0.1




# ---------------- histograms --------------------------


titles <- c(
			"Chr1:20,980,000-21,980,000",
			"Chr3:97,720,000-98,720,000",
			"Chr4:101,710,000-102,710,000",
			"Chr6:108,360,000-109,360,000",
			"Chr10:59,240,000-60,240,000",
			"Chr11:41,580,000-42,580,000",
			"Chr12:75,090,000-76,090,000",
			"Chr15:87,900,000-88,900,000"
			)



# # -------- Loop to make 8 pictures --------------------------------------------------------
# # -------- Neat and efficient, but could not solve lazy evaluation problem here ---------------


# pics <- list()

# # adjusts density scales, but does not work in loop because of .env problems in ggplot which I could not solve. Only last value is relevant.
# density_scale <- c(55,75,75,75,75,75,75,75)


# R_y_axis_scale <- 0.01363636

# for(i in c(1:max(resid_l$id))) {
	
# resids.df <- subset(resid_l,resid_l$id == i)

# m1_nb <- gam(copy ~ week * conc + s(pool, bs = "re") + s(cell, bs = "re") + offset(log(total_reads)), data = resids.df, family = nb, method = "REML")


# resids.df$res <- residuals(m1_nb)

# n = length(unique(resids.df$pool))
# colores_1 = gg_color_hue(n)


# avg <- mean(resids.df$res)
# standev <- sd(resids.df$res)


# funcx <- function(x) density_scale[i]*dnorm(x, avg, standev)


# pics[[i]] <- ggplot() + 
		# theme2 + 
		# theme(legend.key=element_blank()) +
		# geom_histogram(
			# data=resids.df,
			# bins=10,
			# alpha=0.4,
			# aes(res,fill=pool),
			# ) + 
		# geom_line(data=resids.df, lwd =0.2, aes(x=res,y = ..density..* density_scale[i],colour = 'Empirical'),stat = 'density') +
		# stat_function(
			# fun = funcx, 
			# aes(colour = "Normal"),
			# lwd =0.2
			# )  +
		# scale_colour_manual(
			# name = '', 
			# values = c('red', 'blue')
			# ) + 
		# scale_fill_manual(
			# values=colores_1,
			# name ="Pool", 
			# labels=c(1,2,3,4,5,6)
			# ) +
 		# # guides(
 			# # colour = guide_legend(override.aes = list(fill=NA,shape=1,size=1),ncol=2,byrow=TRUE)
 			# # ) +
		# guides(fill=guide_legend(ncol=2)) +
	 	# theme(
	 		# legend.position = c(0.85,0.75), 
	 		# legend.title = element_text(size = 6), 
	 		# legend.text = element_text(size = 5),
	 		# legend.title.align=0.6
	 		# ) +
		# scale_y_continuous(
		    # "Counts", 
		    # sec.axis = sec_axis(~ . * R_y_axis_scale, name = "Density\n")) + # approx y axis scale factor for densities so sum == 1
		# # scale_x_continuous(breaks = c(0,1,2,3,4,6), labels = c(0,1,2,3,4,6)) +
		# # ggtitle("") + 
		# xlab("Residuals") + 
		# # ylab(expression("Counts")) + 
		# labs(subtitle=titles[i])
# print(pics[[i]])

# }




# #------------------Make file --------------------------


# pdf("resid_hist_1.pdf",width=7.5,height= 10,useDingbats=FALSE)


# plot_grid(
		# pics[[1]], 
		# pics[[2]],
		# pics[[3]],
		# pics[[4]],
		# pics[[5]],
		# pics[[6]],
		# pics[[7]],
		# pics[[8]],
		# labels=c("A", "B","C","D","E","F","G","H"), 
		# ncol = 2, 
		# nrow = 4, 
		# label_size = 12
		# )
# dev.off()

# # ----------------------------------------------------



# -------- Brute force to make 8 pictures ----------------


pics <- list()

# adjusts density scales

density_scale <- c(55,75,75,75,75,75,75,75)





# ------------------ pic 1 ---------------------
	
resids.df <- subset(resid_l,resid_l$id == 1)

m1_nb <- gam(copy ~ week * conc + s(pool, bs = "re") + s(cell, bs = "re") + offset(log(total_reads)), data = resids.df, family = nb, method = "REML")


resids.df$res <- residuals(m1_nb)

n = length(unique(resids.df$pool))
colores_1 = gg_color_hue(n)


avg <- mean(resids.df$res)
standev <- sd(resids.df$res)


funcx <- function(x) density_scale[1]*dnorm(x, avg, standev)

# Y scale empirically gives normal peak of ~ 1/sqrt(2*pi) = 0.43, which is correct peak if standev ~ 0.9, as it does for these 8 graphs
# Not sure where I got figure of 0.01363636 from, I think by comparing graph heights, but it certainly seems correct for last 7/8 graphs. Needed slight correction for first graph.

R_y_axis_scale <- 0.01363636 * 75/density_scale[1]


pics[[1]] <- ggplot() + 
		theme2 + 
		geom_histogram(
			data=resids.df,
			bins=10,
			alpha=0.4,
			aes(res,fill=pool),
			) + 
		geom_line(data=resids.df, lwd =0.2, aes(x=res,y = ..density..* density_scale[1],colour = 'Empirical'),stat = 'density') +
		stat_function(
			fun = funcx, 
			aes(colour = "Normal"),
			lwd =0.2
			)  +
		scale_colour_manual(
			name = '', 
			values = c('red', 'blue')
			) + 
		scale_fill_manual(
			values=colores_1,
			name ="Pool", 
			labels=c(1,2,3,4,5,6)
			) +
 		# guides(
 			# colour = guide_legend(override.aes = list(fill=NA,shape=1,size=1),ncol=2,byrow=TRUE)
 			# ) +
		guides(fill=guide_legend(ncol=2)) +
	 	theme(
	 		legend.position = c(0.85,0.75), 
	 		# legend.title = element_text(size = 6), 
	 		# legend.text = element_text(size = 5),
	 		legend.title.align=0.6
	 		) +
		scale_y_continuous(
		    "Counts", 
		    sec.axis = sec_axis(~ . * R_y_axis_scale, name = "Density\n")) + # approx y axis scale factor for densities so sum == 1
		# scale_x_continuous(breaks = c(0,1,2,3,4,6), labels = c(0,1,2,3,4,6)) +
		# ggtitle("") + 
		xlab("Residuals") + 
		# ylab(expression("Counts")) + 
		labs(subtitle=titles[1])
print(pics[[1]])

# ------------------ pic 2 ---------------------
	
resids.df <- subset(resid_l,resid_l$id == 2)

m1_nb <- gam(copy ~ week * conc + s(pool, bs = "re") + s(cell, bs = "re") + offset(log(total_reads)), data = resids.df, family = nb, method = "REML")


resids.df$res <- residuals(m1_nb)

n = length(unique(resids.df$pool))
colores_1 = gg_color_hue(n)


avg <- mean(resids.df$res)
standev <- sd(resids.df$res)


funcx <- function(x) density_scale[2]*dnorm(x, avg, standev)

R_y_axis_scale <- 0.01363636 * 75/density_scale[2]


pics[[2]] <- ggplot() + 
		theme2 + 
		theme(legend.key=element_blank()) +
		geom_histogram(
			data=resids.df,
			bins=10,
			alpha=0.4,
			aes(res,fill=pool),
			) + 
		geom_line(data=resids.df, lwd =0.2, aes(x=res,y = ..density..* density_scale[2],colour = 'Empirical'),stat = 'density') +
		stat_function(
			fun = funcx, 
			aes(colour = "Normal"),
			lwd =0.2
			)  +
		scale_colour_manual(
			name = '', 
			values = c('red', 'blue')
			) + 
		scale_fill_manual(
			values=colores_1,
			name ="Pool", 
			labels=c(1,2,3,4,5,6)
			) +
 		# guides(
 			# colour = guide_legend(override.aes = list(fill=NA,shape=1,size=1),ncol=2,byrow=TRUE)
 			# ) +
		guides(fill=guide_legend(ncol=2)) +
	 	theme(
	 		legend.position = c(0.85,0.75), 
	 		# legend.title = element_text(size = 6), 
	 		# legend.text = element_text(size = 5),
	 		legend.title.align=0.6
	 		) +
		scale_y_continuous(
		    "Counts", 
		    sec.axis = sec_axis(~ . * R_y_axis_scale, name = "Density\n")) + # approx y axis scale factor for densities so sum == 1
		# scale_x_continuous(breaks = c(0,1,2,3,4,6), labels = c(0,1,2,3,4,6)) +
		# ggtitle("") + 
		xlab("Residuals") + 
		# ylab(expression("Counts")) + 
		labs(subtitle=titles[2])
print(pics[[2]])

# ------------------ pic 3 ---------------------
	
resids.df <- subset(resid_l,resid_l$id == 3)

m1_nb <- gam(copy ~ week * conc + s(pool, bs = "re") + s(cell, bs = "re") + offset(log(total_reads)), data = resids.df, family = nb, method = "REML")


resids.df$res <- residuals(m1_nb)

n = length(unique(resids.df$pool))
colores_1 = gg_color_hue(n)


avg <- mean(resids.df$res)
standev <- sd(resids.df$res)


funcx <- function(x) density_scale[3]*dnorm(x, avg, standev)

R_y_axis_scale <- 0.01363636 * 75/density_scale[3]


pics[[3]] <- ggplot() + 
		theme2 + 
		theme(legend.key=element_blank()) +
		geom_histogram(
			data=resids.df,
			bins=10,
			alpha=0.4,
			aes(res,fill=pool),
			) + 
		geom_line(data=resids.df, lwd =0.2, aes(x=res,y = ..density..* density_scale[3],colour = 'Empirical'),stat = 'density') +
		stat_function(
			fun = funcx, 
			aes(colour = "Normal"),
			lwd =0.2
			)  +
		scale_colour_manual(
			name = '', 
			values = c('red', 'blue')
			) + 
		scale_fill_manual(
			values=colores_1,
			name ="Pool", 
			labels=c(1,2,3,4,5,6)
			) +
 		# guides(
 			# colour = guide_legend(override.aes = list(fill=NA,shape=1,size=1),ncol=2,byrow=TRUE)
 			# ) +
		guides(fill=guide_legend(ncol=2)) +
	 	theme(
	 		legend.position = c(0.85,0.75), 
	 		# legend.title = element_text(size = 6), 
	 		# legend.text = element_text(size = 5),
	 		legend.title.align=0.6
	 		) +
		scale_y_continuous(
		    "Counts", 
		    sec.axis = sec_axis(~ . * R_y_axis_scale, name = "Density\n")) + # approx y axis scale factor for densities so sum == 1
		# scale_x_continuous(breaks = c(0,1,2,3,4,6), labels = c(0,1,2,3,4,6)) +
		# ggtitle("") + 
		xlab("Residuals") + 
		# ylab(expression("Counts")) + 
		labs(subtitle=titles[3])
print(pics[[3]])

# ------------------ pic 4 ---------------------
	
resids.df <- subset(resid_l,resid_l$id == 4)

m1_nb <- gam(copy ~ week * conc + s(pool, bs = "re") + s(cell, bs = "re") + offset(log(total_reads)), data = resids.df, family = nb, method = "REML")


resids.df$res <- residuals(m1_nb)

n = length(unique(resids.df$pool))
colores_1 = gg_color_hue(n)


avg <- mean(resids.df$res)
standev <- sd(resids.df$res)


funcx <- function(x) density_scale[4]*dnorm(x, avg, standev)

R_y_axis_scale <- 0.01363636 * 75/density_scale[4]


pics[[4]] <- ggplot() + 
		theme2 + 
		theme(legend.key=element_blank()) +
		geom_histogram(
			data=resids.df,
			bins=10,
			alpha=0.4,
			aes(res,fill=pool),
			) + 
		geom_line(data=resids.df, lwd =0.2, aes(x=res,y = ..density..* density_scale[4],colour = 'Empirical'),stat = 'density') +
		stat_function(
			fun = funcx, 
			aes(colour = "Normal"),
			lwd =0.2
			)  +
		scale_colour_manual(
			name = '', 
			values = c('red', 'blue')
			) + 
		scale_fill_manual(
			values=colores_1,
			name ="Pool", 
			labels=c(1,2,3,4,5,6)
			) +
 		# guides(
 			# colour = guide_legend(override.aes = list(fill=NA,shape=1,size=1),ncol=2,byrow=TRUE)
 			# ) +
		guides(fill=guide_legend(ncol=2)) +
	 	theme(
	 		legend.position = c(0.85,0.75), 
	 		# legend.title = element_text(size = 6), 
	 		# legend.text = element_text(size = 5),
	 		legend.title.align=0.6
	 		) +
		scale_y_continuous(
		    "Counts", 
		    sec.axis = sec_axis(~ . * R_y_axis_scale, name = "Density\n")) + # approx y axis scale factor for densities so sum == 1
		# scale_x_continuous(breaks = c(0,1,2,3,4,6), labels = c(0,1,2,3,4,6)) +
		# ggtitle("") + 
		xlab("Residuals") + 
		# ylab(expression("Counts")) + 
		labs(subtitle=titles[4])
print(pics[[4]])

# ------------------ pic 5 ---------------------
	
resids.df <- subset(resid_l,resid_l$id == 5)

m1_nb <- gam(copy ~ week * conc + s(pool, bs = "re") + s(cell, bs = "re") + offset(log(total_reads)), data = resids.df, family = nb, method = "REML")


resids.df$res <- residuals(m1_nb)

n = length(unique(resids.df$pool))
colores_1 = gg_color_hue(n)


avg <- mean(resids.df$res)
standev <- sd(resids.df$res)


funcx <- function(x) density_scale[5]*dnorm(x, avg, standev)

R_y_axis_scale <- 0.01363636 * 75/density_scale[5]


pics[[5]] <- ggplot() + 
		theme2 + 
		theme(legend.key=element_blank()) +
		geom_histogram(
			data=resids.df,
			bins=10,
			alpha=0.4,
			aes(res,fill=pool),
			) + 
		geom_line(data=resids.df, lwd =0.2, aes(x=res,y = ..density..* density_scale[5],colour = 'Empirical'),stat = 'density') +
		stat_function(
			fun = funcx, 
			aes(colour = "Normal"),
			lwd =0.2
			)  +
		scale_colour_manual(
			name = '', 
			values = c('red', 'blue')
			) + 
		scale_fill_manual(
			values=colores_1,
			name ="Pool", 
			labels=c(1,2,3,4,5,6)
			) +
 		# guides(
 			# colour = guide_legend(override.aes = list(fill=NA,shape=1,size=1),ncol=2,byrow=TRUE)
 			# ) +
		guides(fill=guide_legend(ncol=2)) +
	 	theme(
	 		legend.position = c(0.85,0.75), 
	 		# legend.title = element_text(size = 6), 
	 		# legend.text = element_text(size = 5),
	 		legend.title.align=0.6
	 		) +
		scale_y_continuous(
		    "Counts", 
		    sec.axis = sec_axis(~ . * R_y_axis_scale, name = "Density\n")) + # approx y axis scale factor for densities so sum == 1
		# scale_x_continuous(breaks = c(0,1,2,3,4,6), labels = c(0,1,2,3,4,6)) +
		# ggtitle("") + 
		xlab("Residuals") + 
		# ylab(expression("Counts")) + 
		labs(subtitle=titles[5])
print(pics[[5]])

# ------------------ pic 6 ---------------------
	
resids.df <- subset(resid_l,resid_l$id == 6)

m1_nb <- gam(copy ~ week * conc + s(pool, bs = "re") + s(cell, bs = "re") + offset(log(total_reads)), data = resids.df, family = nb, method = "REML")


resids.df$res <- residuals(m1_nb)

n = length(unique(resids.df$pool))
colores_1 = gg_color_hue(n)


avg <- mean(resids.df$res)
standev <- sd(resids.df$res)


funcx <- function(x) density_scale[6]*dnorm(x, avg, standev)

R_y_axis_scale <- 0.01363636 * 75/density_scale[6]


pics[[6]] <- ggplot() + 
		theme2 + 
		theme(legend.key=element_blank()) +
		geom_histogram(
			data=resids.df,
			bins=10,
			alpha=0.4,
			aes(res,fill=pool),
			) + 
		geom_line(data=resids.df, lwd =0.2, aes(x=res,y = ..density..* density_scale[6],colour = 'Empirical'),stat = 'density') +
		stat_function(
			fun = funcx, 
			aes(colour = "Normal"),
			lwd =0.2
			)  +
		scale_colour_manual(
			name = '', 
			values = c('red', 'blue')
			) + 
		scale_fill_manual(
			values=colores_1,
			name ="Pool", 
			labels=c(1,2,3,4,5,6)
			) +
 		# guides(
 			# colour = guide_legend(override.aes = list(fill=NA,shape=1,size=1),ncol=2,byrow=TRUE)
 			# ) +
		guides(fill=guide_legend(ncol=2)) +
	 	theme(
	 		legend.position = c(0.85,0.75), 
	 		# legend.title = element_text(size = 6), 
	 		# legend.text = element_text(size = 5),
	 		legend.title.align=0.6
	 		) +
		scale_y_continuous(
		    "Counts", 
		    sec.axis = sec_axis(~ . * R_y_axis_scale, name = "Density\n")) + # approx y axis scale factor for densities so sum == 1
		# scale_x_continuous(breaks = c(0,1,2,3,4,6), labels = c(0,1,2,3,4,6)) +
		# ggtitle("") + 
		xlab("Residuals") + 
		# ylab(expression("Counts")) + 
		labs(subtitle=titles[6])
print(pics[[6]])


# ------------------ pic 7 ---------------------
	
resids.df <- subset(resid_l,resid_l$id == 7)

m1_nb <- gam(copy ~ week * conc + s(pool, bs = "re") + s(cell, bs = "re") + offset(log(total_reads)), data = resids.df, family = nb, method = "REML")


resids.df$res <- residuals(m1_nb)

n = length(unique(resids.df$pool))
colores_1 = gg_color_hue(n)


avg <- mean(resids.df$res)
standev <- sd(resids.df$res)


funcx <- function(x) density_scale[7]*dnorm(x, avg, standev)

R_y_axis_scale <- 0.01363636 * 75/density_scale[7]


pics[[7]] <- ggplot() + 
		theme2 + 
		theme(legend.key=element_blank()) +
		geom_histogram(
			data=resids.df,
			bins=10,
			alpha=0.4,
			aes(res,fill=pool),
			) + 
		geom_line(data=resids.df, lwd =0.2, aes(x=res,y = ..density..* density_scale[7],colour = 'Empirical'),stat = 'density') +
		stat_function(
			fun = funcx, 
			aes(colour = "Normal"),
			lwd =0.2
			)  +
		scale_colour_manual(
			name = '', 
			values = c('red', 'blue')
			) + 
		scale_fill_manual(
			values=colores_1,
			name ="Pool", 
			labels=c(1,2,3,4,5,6)
			) +
 		# guides(
 			# colour = guide_legend(override.aes = list(fill=NA,shape=1,size=1),ncol=2,byrow=TRUE)
 			# ) +
		guides(fill=guide_legend(ncol=2)) +
	 	theme(
	 		legend.position = c(0.85,0.75), 
	 		# legend.title = element_text(size = 6), 
	 		# legend.text = element_text(size = 5),
	 		legend.title.align=0.6
	 		) +
		scale_y_continuous(
		    "Counts", 
		    sec.axis = sec_axis(~ . * R_y_axis_scale, name = "Density\n")) + # approx y axis scale factor for densities so sum == 1
		# scale_x_continuous(breaks = c(0,1,2,3,4,6), labels = c(0,1,2,3,4,6)) +
		# ggtitle("") + 
		xlab("Residuals") + 
		# ylab(expression("Counts")) + 
		labs(subtitle=titles[7])
print(pics[[7]])


# ------------------ pic 8 ---------------------
	
resids.df <- subset(resid_l,resid_l$id == 8)

m1_nb <- gam(copy ~ week * conc + s(pool, bs = "re") + s(cell, bs = "re") + offset(log(total_reads)), data = resids.df, family = nb, method = "REML")


resids.df$res <- residuals(m1_nb)

n = length(unique(resids.df$pool))
colores_1 = gg_color_hue(n)


avg <- mean(resids.df$res)
standev <- sd(resids.df$res)


funcx <- function(x) density_scale[8]*dnorm(x, avg, standev)

R_y_axis_scale <- 0.01363636 * 75/density_scale[8]


pics[[8]] <- ggplot() + 
		theme2 + 
		theme(legend.key=element_blank()) +
		geom_histogram(
			data=resids.df,
			bins=10,
			alpha=0.4,
			aes(res,fill=pool),
			) + 
		geom_line(data=resids.df, lwd =0.2, aes(x=res,y = ..density..* density_scale[8],colour = 'Empirical'),stat = 'density') +
		stat_function(
			fun = funcx, 
			aes(colour = "Normal"),
			lwd =0.2
			)  +
		scale_colour_manual(
			name = '', 
			values = c('red', 'blue')
			) + 
		scale_fill_manual(
			values=colores_1,
			name ="Pool", 
			labels=c(1,2,3,4,5,6)
			) +
 		# guides(
 			# colour = guide_legend(override.aes = list(fill=NA,shape=1,size=1),ncol=2,byrow=TRUE)
 			# ) +
		guides(fill=guide_legend(ncol=2)) +
	 	theme(
	 		legend.position = c(0.85,0.75), 
	 		# legend.title = element_text(size = 6), 
	 		# legend.text = element_text(size = 5),
	 		legend.title.align=0.6
	 		) +
		scale_y_continuous(
		    "Counts", 
		    sec.axis = sec_axis(~ . * R_y_axis_scale, name = "Density\n")) + # approx y axis scale factor for densities so sum == 1
		# scale_x_continuous(breaks = c(0,1,2,3,4,6), labels = c(0,1,2,3,4,6)) +
		# ggtitle("") + 
		xlab("Residuals") + 
		# ylab(expression("Counts")) + 
		labs(subtitle=titles[8])
print(pics[[8]])


#------------------Make file --------------------------


pdf("resid_hist_1.pdf",width=7.5,height= 10,useDingbats=FALSE)


plot_grid(
		plotlist=pics[1:8], 
		labels=LETTERS[1:8], 
		ncol = 2, 
		nrow = 4, 
		label_size = 16
		)
dev.off()


# ----------------------------------------------------

















