library(lme4)
library(multcomp)
library(sjstats)
library(performance)
library(insight)
library(emmeans)


# -------------------- fxn ---------------------------------


sem <- function(x) {sqrt(var(x,na.rm=TRUE)/sum(!is.na(x)))}

compare <- function(a,b) {
	print(t.test(a,b))
	
	print(paste0("exact P value = ", t.test(a,b)$p.value))
	
	print(paste0("mean of a = ", mean(a, na.rm = TRUE)))
	print(paste0("sem of a = ", sem(a)))
	print(paste0("sd of a = ", sd(a, na.rm = TRUE)))
	print(paste0("number in a = ", sum(!is.na(a))))
	
	print(paste0("mean of b = ", mean(b, na.rm = TRUE)))
	print(paste0("sem of b = ", sem(b)))
	print(paste0("sd of b = ", sd(b, na.rm = TRUE)))
	print(paste0("number in b = ", sum(!is.na(b))))
	
}




# ---------- Analyze human myco ferm qtls -----------------


# log10P_human.txt constructed on Hoffman, cf folder "log10P_human"
# log10P_human_plus_myco_ferm.txt constructed on Hoffman, cf folder "log10P_human_plus_myco_ferm"

logP <- read.table("log10P_human.txt",header=TRUE,sep="\t",stringsAsFactors=FALSE)
myco_ferm <- read.table("log10P_human_plus_myco_ferm.txt",header=TRUE,sep="\t",stringsAsFactors=FALSE)




# ------------ Correlations of logP vals before and after addition of human mito --------------

human_cor_logP <- data.frame(
								test = character(),
								R = numeric(),
								t = numeric(),
								df = numeric(),
								P = numeric(),
								stringsAsFactors=FALSE
								)

for(i in c(5:16)) {
	
	cor_ans <- cor.test(logP[seq(1,nrow(logP),1e2),i], myco_ferm[seq(1,nrow(myco_ferm),1e2),i])
	
	human_cor_logP[i-4,"test"] <- colnames(logP)[i]
	human_cor_logP[i-4,"R"] <- cor_ans$estimate
	human_cor_logP[i-4,"t"] <- cor_ans$statistic
	human_cor_logP[i-4,"df"] <- cor_ans$parameter
	human_cor_logP[i-4,"P"] <- cor_ans$p.value
	
	}


human_cor_logP
            # test         R         t   df P
# 1   log10p_g_0nM 0.9994216 1623.5558 3052 0
# 2   log10p_g_8nM 0.9990912 1294.9014 3052 0
# 3  log10p_g_25nM 0.9977948  830.4860 3052 0
# 4  log10p_g_75nM 0.9958082  601.4617 3052 0
# 5   log10p_g_avg 0.9976037  796.5754 3052 0
# 6    log10p_d_w1 0.9704009  221.9865 3052 0       <<<<<<<<<<<<<< use in paper
# 7    log10p_d_w2 0.9899464  386.6539 3052 0
# 8    log10p_d_w3 0.9981598  909.3890 3052 0       
# 9    log10p_d_w4 0.9989006 1177.1643 3052 0
# 10   log10p_d_w6 0.9993360 1515.2520 3052 0
# 11  log10p_d_avg 0.9984696  997.4104 3052 0
# 12 log10p_g_d_Ix 0.9977254  817.6727 3052 0





mean(human_cor_logP$R)
# [1] 0.9952215  <<<<<<<<<<<<<< use in paper

sem(human_cor_logP$R)
# [1] 0.002375801  <<<<<<<<<<<<<< use in paper






# --------- mycoplasma qtls ------------------------




# no myco qtls exceed permutation threshold. <<<<<<<<<<<<<<<< use in paper

min(10^-myco_ferm$log10p_myco_ferm)
# [1] 1.137319e-05


max(myco_ferm$log10p_myco_ferm)
# [1] 4.944118

myco_thresh <- read.table("human_plus_myco_ferm_thresh_95.txt",sep="\t",header=FALSE,stringsAsFactors=FALSE,row.names="V1")

myco_thresh[c("log10p_myco_ferm"),]
# [1] 13.0384




# no myco qtls exceed FDR threshold. <<<<<<<<<<<<<< use in paper

myco_ferm$log10q_myco_ferm <- -log10(p.adjust(10^-myco_ferm$log10p_myco_ferm))


max( myco_ferm$log10q_myco_ferm)
# [1] 0

max(myco_ferm$log10q_myco_ferm)
# [1] 0

min(10^-myco_ferm$log10q_myco_ferm)
# [1] 1

# All log10q_myco_ferm are zero, ie all myco FDRs are == 1. <<<<<<<<<<<<<<<<< use in paper

unique(myco_ferm$log10q_myco_ferm)
# [1] 0

unique(10^-myco_ferm$log10q_myco_ferm)
# [1] 1










plot(myco_ferm$log10p_myco_ferm,cex=0.1)
# identify max peak
# identify(myco_ferm$log10p_myco_ferm,cex=0.1)
# [1] 50888



# Interestingly max myco peak, even though far from significant, is same as for human mitochondria (cf human_mito_1.R)
myco_ferm[c(50888),c("Chromosome","posS","posE","pos","log10p_myco_ferm","coef_myco_ferm","log10q_myco_ferm")]
      # Chromosome     posS     posE      pos log10p_myco_ferm coef_myco_ferm log10q_myco_ferm
# 50888       chr3 14740000 15740000 15240000         4.944118  -0.0002492764                0    CAPN7/SH3BP5 <<<<<<<<<<<< do not use in paper. FDR = 10^-0 = 1

# compare with (from human_mito_1.R)

# hum_mito[c(50888),c("Chromosome","posS","posE","pos","log10p_human_mito","coef_human_mito","log10q_human_mito")]
      # Chromosome     posS     posE      pos log10p_human_mito coef_human_mito log10q_human_mito
# 50888       chr3 14740000 15740000 15240000          8.574216      0.00162057           3.08936  CAPN7/SH3BP5 <<<<<<<<<<<< use in paper. FDR = 10^-3.08936 = 0.0008140292


# However, note that coef for myco is negative, unlike coef for mito which is positive. Thus, increased CAPN7 decreases human myco. Makes sense, eg DOI: 10.1111/j.1574-6968.2010.01893.x
# However, above paper says inhibited calpain is cell protective. In contrast, our data suggests increased calpain 7 decreases mycoplasma.
# In any event, FDR is far too weak to report, so ignore for paper.


# ------------------------ myco spp in HEK293 and A23 ---------------------------


# Myco_HEK293_A23.txt is taken from Arshad file "myco_mapped_read.xlsx"

myco_hek_a23 <- read.table("Myco_HEK293_A23.txt",header=TRUE,sep="\t",stringsAsFactors=FALSE)
colnames(myco_hek_a23)[1] <- "myco_spp"

# Reads aligned to human genome in HEK293 is 40128368  taken from "A23_HEK_mito_copy_num_1.R"
# Reads alignted to human mitochondria in HEK293 is 153367 taken from "A23_HEK_mito_copy_num_1.R"

# Do not remove mito reads as below, because myco read rate represents total aligned human sequence
# HEK293_reads <- 40128368 - 153367

# instead do this
HEK293_reads <- 40128368

# Reads aligned to hamster genome in A23 is 33234329  taken from "A23_HEK_mito_copy_num_1.R"
# Reads alignted to hamster mitochondria in A23 is 95975 taken from "A23_HEK_mito_copy_num_1.R"


# Do not remove mito reads as below, because myco read rate represents total aligned hamster sequence
# A23_reads <- 33234329 - 95975

# instead do this
A23_reads <- 33234329


myco_hek_a23$hek_read_rate <- (myco_hek_a23$HEK293.1_mismatch/HEK293_reads)*1e6
myco_hek_a23$a23_read_rate <- (myco_hek_a23$A23.1_mismatch/A23_reads)*1e6



# Hamster genome excluding mitochondria is 2368906908 bp from "A23_HEK_mito_copy_num_1.R"
# Human genome excluding mitochondrion is 3088269832 bp from "A23_HEK_mito_copy_num_1.R"

Hamster_genome_length <- 2368906908
Human_genome_length <- 3088269832


# Recalx species specific aligments to HEK293 and A23 this time removing mitochondria, because using genome lengths without the mitochondria for copy number
HEK293_reads <- 40128368 - 153367
A23_reads <- 33234329 - 95975


# M. hominis ATCC 23114 genome is 665,445 bp (NC_013511.1) (https://www.ncbi.nlm.nih.gov/search/all/?term=NC%20013511.1)
# M. hyorhinis MCLD genome is 829,709 bp (NC_017519.1) (https://www.ncbi.nlm.nih.gov/search/all/?term=NC+017519.1)
# M. fermentans M64 genome is 1,118,751 bp (NC_014921.1) (https://www.ncbi.nlm.nih.gov/search/all/?term=NC+014921.1)
# Acholeplasma laidlawii PG-8A genome is 1,496,992 bp (NC_010163.1) (https://www.ncbi.nlm.nih.gov/search/all/?term=NC+010163.1)
# Mycoplasma pneumoniae M129 genome is 816,394 bp (NC_000912.1) (https://www.ncbi.nlm.nih.gov/search/all/?term=NC+000912.1)

myco_genome_length <- c(665445, 829709, 1118751, 1496992, 816394)

myco_hek_a23$myco_genome_length <- myco_genome_length


myco_hek_a23$hek_myc_copy_num <- 2*(myco_hek_a23$HEK293.1_mismatch/myco_hek_a23$myco_genome_length)*(Human_genome_length/HEK293_reads)
myco_hek_a23$a23_myc_copy_num <- 2*(myco_hek_a23$A23.1_mismatch/myco_hek_a23$myco_genome_length)*(Hamster_genome_length/A23_reads)


 myco_hek_a23
                                     # myco_spp HEK293.1_mismatch HEK293.4_mismatch A23.1_mismatch A23.4_mismatch hek_read_rate a23_read_rate myco_genome_length hek_myc_copy_num a23_myc_copy_num
# 1 Mycoplasma hominis ATCC 23114 (NC 013511.1)                 0                 0            262            262      0.000000      7.883415             665445      0.000000000       0.05629064
# 2               M. hyorhinisMCLD(NC 017519.1)                47                47            195            195      1.171241      5.867427             829709      0.008752433       0.03360128
# 3     Mycoplasma fermentans M64 (NC 014921.1)                 0                 0           2060           2060      0.000000     61.984101            1118751      0.000000000       0.26325755
# 4  Acholeplasma laidlawii PG-8A (NC 010163.1)                 0                 0              0              0      0.000000      0.000000            1496992      0.000000000       0.00000000
# 5                  Mycoplasma pneumoniae M129                 0                 0              0              0      0.000000      0.000000             816394      0.000000000       0.00000000



# ------------------- myco ferm copy number in RH pools and all RH samples ------------------------


# myco_ferm_gseq.txt is a file derived from Arshad file "Myco_raw_reads_des.txt" (cf "Create_myco_ferm_gseq.R")

myco_ferm_RH <- read.table("myco_ferm_gseq.txt", header=TRUE, sep="\t", stringsAsFactors=FALSE)

RH_hamster_reads <- read.table("RH_pool_hamster_total_align.txt",sep="\t",stringsAsFactors=FALSE,header=TRUE) # uses mapped hamster reads, cf hamster_AIC_1.R
RH_hamster_reads <- RH_hamster_reads[,c("RH_ID", "pool", "week", "conc", "cell", "hamster_aligned_and_human_unaligned")]
colnames(RH_hamster_reads)[6] <- "hamster_reads"

# "hamster_mito_gseq.txt" made as described in "Create_RH_hamster_mito_gseq.R"

RH_hamster_mito <- read.table("hamster_mito_gseq.txt",sep="\t",stringsAsFactors=FALSE,header=TRUE)

myco_ferm_RH <- merge(myco_ferm_RH, RH_hamster_reads)

myco_ferm_RH <- merge(myco_ferm_RH, RH_hamster_mito[,c("RH_ID", "pool", "conc", "week", "cell", "hamster_mito_reads")])

head(myco_ferm_RH)
      # RH_ID pool conc week cell myco_ferm_reads hamster_reads hamster_mito_reads
# 1  RH1_w0_d0    1    0    0    1              19      37595178             103116
# 2  RH1_w1_d0    1    0    1    2              19      42886773             132305
# 3 RH1_w1_d25    1   25    1    4              24      49651729             100796
# 4 RH1_w1_d75    1   75    1    5              32      59267597             128641
# 5  RH1_w1_d8    1    8    1    3              44      55742797             138515
# 6  RH1_w2_d0    1    0    2    2              23      34210971              66957



# Following logic above to get myco ferm copy number in all RH samples

myco_ferm_RH$myco_ferm_copy_num <- 2*(myco_ferm_RH$myco_ferm_reads/myco_genome_length[3])*(Hamster_genome_length/myco_ferm_RH$hamster_reads)

head(myco_ferm_RH)
       # RH_ID pool conc week cell myco_ferm_reads hamster_reads hamster_mito_reads myco_ferm_copy_num
# 1  RH1_w0_d0    1    0    0    1              19      37595178             103116        0.002140257
# 2  RH1_w1_d0    1    0    1    2              19      42886773             132305        0.001876181
# 3 RH1_w1_d25    1   25    1    4              24      49651729             100796        0.002047017
# 4 RH1_w1_d75    1   75    1    5              32      59267597             128641        0.002286532
# 5  RH1_w1_d8    1    8    1    3              44      55742797             138515        0.003342785
# 6  RH1_w2_d0    1    0    2    2              23      34210971              66957        0.002847128


myco_ferm_RH[myco_ferm_RH$week==0 & myco_ferm_RH$conc==0,]

       # RH_ID pool conc week cell myco_ferm_reads hamster_reads hamster_mito_reads myco_ferm_copy_num
# 1  RH1_w0_d0    1    0    0    1              19      37595178             103116        0.002140257
# 21 RH2_w0_d0    2    0    0    6            1236      30754890             109494        0.170195803
# 40 RH3_w0_d0    3    0    0   11             654      45668817              96305        0.060646053
# 60 RH4_w0_d0    4    0    0   16             416      28092611              61749        0.062711296
# 80 RH5_w0_d0    5    0    0   21            1814      38167486              89994        0.201274280
# 98 RH6_w0_d0    6    0    0   26            1354      37193349              96898        0.154169311



# --------------- myco ferm copy number in RH pools and all samples ---------------------------


# test whether myco_ferm copy num is sig diff from zero in RH pools


mean(myco_ferm_RH[myco_ferm_RH$week==0 & myco_ferm_RH$conc==0,"myco_ferm_copy_num"])
# [1] 0.1085228  <<<<<<<<<<<<<<< use in paper


sem(myco_ferm_RH[myco_ferm_RH$week==0 & myco_ferm_RH$conc==0,"myco_ferm_copy_num"])
# [1] 0.03172702  <<<<<<<<<<<<<<< use in paper


t.test(myco_ferm_RH[myco_ferm_RH$week==0 & myco_ferm_RH$conc==0,"myco_ferm_copy_num"])

	# One Sample t-test <<<<<<<<<<<<<<< use in paper

# data:  myco_ferm_RH[myco_ferm_RH$week == 0 & myco_ferm_RH$conc == 0,     "myco_ferm_copy_num"]
# t = 3.4205, df = 5, p-value = 0.01883
# alternative hypothesis: true mean is not equal to 0
# 95 percent confidence interval:
 # 0.02696592 0.19007975
# sample estimates:
# mean of x 
# 0.1085228 


# test whether myco_ferm copy num is sig diff from zero in all samples
# Not apt test because samples are not independent. Use LMM instead.

mean(myco_ferm_RH[,"myco_ferm_copy_num"])
# [1] 0.1364919  <<<<<<<<<<<<<<<< do not use in paper


sem(myco_ferm_RH[,"myco_ferm_copy_num"])
# [1] 0.01096928  <<<<<<<<<<<<<<<< do not use in paper



t.test(myco_ferm_RH[,"myco_ferm_copy_num"])

	# One Sample t-test

# data:  myco_ferm_RH[, "myco_ferm_copy_num"]
# t = 12.443, df = 114, p-value < 2.2e-16
# alternative hypothesis: true mean is not equal to 0
# 95 percent confidence interval:
 # 0.1147618 0.1582219
# sample estimates:
# mean of x 
# 0.1364919 


t.test(myco_ferm_RH[,"myco_ferm_copy_num"])$p.value
# [1] 5.661018e-23



# ------------- Mixed mod calx: myco ferm copy number based on hamster reads ---------------------


human_myco_copy_hamster_reads <- data.frame(
									myco_spp = as.character(),
									coef_g_0nM = numeric(),
									coef_g_8nM = numeric(),
									coef_g_25nM = numeric(),
									coef_g_75nM = numeric(),
									coef_g_avg = numeric(),
									coef_d_w1 = numeric(),
									coef_d_w2 = numeric(),
									coef_d_w3 = numeric(),
									coef_d_w4 = numeric(),
									coef_d_w6 = numeric(),
									coef_d_avg = numeric(),
									coef_g_d_Ix = numeric(),
									Z_g_0nM = numeric(),
									Z_g_8nM = numeric(),
									Z_g_25nM = numeric(),
									Z_g_75nM = numeric(),
									Z_g_avg = numeric(),
									Z_d_w1 = numeric(),
									Z_d_w2 = numeric(),
									Z_d_w3 = numeric(),
									Z_d_w4 = numeric(),
									Z_d_w6 = numeric(),
									Z_d_avg = numeric(),
									Z_g_d_Ix = numeric(),
									log10p_g_0nM = numeric(),
									log10p_g_8nM = numeric(),
									log10p_g_25nM = numeric(),
									log10p_g_75nM = numeric(),
									log10p_g_avg = numeric(),
									log10p_d_w1 = numeric(),
									log10p_d_w2 = numeric(),
									log10p_d_w3 = numeric(),
									log10p_d_w4 = numeric(),
									log10p_d_w6 = numeric(),
									log10p_d_avg = numeric(),
									log10p_g_d_Ix = numeric(),
									stringsAsFactors=FALSE
									)



				
		m1 <- lmer(myco_ferm_copy_num ~ week * conc + (1|pool/cell), data = myco_ferm_RH, REML=TRUE)
	
	
# Human myco copy number signficantly different from zero at 0 nM paclitaxel and 0 weeks:
		
		
		summary(glht(m1),test=adjusted("none"))
		
		
	 # Simultaneous Tests for General Linear Hypotheses

# Fit: lmer(formula = myco_ferm_copy_num ~ week * conc + (1 | pool/cell), 
    # data = myco_ferm_RH, REML = TRUE)

# Linear Hypotheses:
                   # Estimate Std. Error z value Pr(>|z|)    
# (Intercept) == 0  0.1408057  0.0328093   4.292 1.77e-05 *** 
# week == 0         0.0008013  0.0057843   0.139    0.890    
# conc == 0        -0.0009381  0.0007757  -1.209    0.227    
# week:conc == 0    0.0001646  0.0001715   0.960    0.337    
# ---
# Signif. codes:  0 ‘***’ 0.001 ‘**’ 0.01 ‘*’ 0.05 ‘.’ 0.1 ‘ ’ 1
# (Adjusted p values reported -- none method)






# use lmer.df = "asymptotic" to agree exactly with above

test(emmeans(m1,lmer.df = "kenward-roger",specs=c("week","conc"),at=list(week=c(0),conc=c(0))))
 # week conc emmean     SE   df t.ratio p.value
    # 0    0  0.141 0.0328 11.3 4.291   0.0012 

summary(contrast(emmeans(m1, lmer.df = "kenward-roger", specs="week", at=list(week=c(0,1), conc=0)),list(c(-1,1))))
# NOTE: Results may be misleading due to involvement in interactions
 # contrast estimate      SE   df t.ratio p.value
 # c(-1, 1) 0.000801 0.00582 99.1 0.138   0.8907 

# # Alternatively can use emtrends
## test(emtrends(m1, var="week", specs = "conc", lmer.df = "kenward-roger",at=list(conc=c(0))))
 # conc week.trend      SE   df t.ratio p.value
    # 0   0.000801 0.00582 99.1 0.138   0.8907 
 

 summary(contrast(emmeans(m1, lmer.df = "kenward-roger", specs="conc", at=list(conc=c(0,1), week=0)),list(c(-1,1))))
# NOTE: Results may be misleading due to involvement in interactions
 # contrast  estimate       SE   df t.ratio p.value
 # c(-1, 1) -0.000938 0.000778 83.4 -1.205  0.2315 


summary(contrast(emmeans(m1, lmer.df = "kenward-roger", specs=c("week","conc"), at=list(week=c(0,1),conc=c(0,1))),list(c(1,-1,-1,1))))
 # contrast        estimate       SE   df t.ratio p.value
 # c(1, -1, -1, 1) 0.000165 0.000172 93.6 0.956   0.3415 
 
 
# not much diff at av conditional values. However, these results are superior, cf https://stackoverflow.com/questions/52866203/hypothesis-test-for-intercepts-in-general-mixed-linear-models-with-r
test(emmeans(m1,lmer.df = "kenward-roger",specs=c("week","conc"),at=list(week=c(3.2),conc=c(27))))
 # week conc emmean     SE   df t.ratio p.value
  # 3.2   27  0.132 0.0269 5.16 4.908   0.0041  <<<<<<<<<<<< use in paper



		
# Alternatively, let's look at RH pools, human myco copy number still sig diff from zero.
# Decided to use this human myco copy number (in addition to that from mixed model at 27 nM paclitaxel and 3.2 wks), as empirical (rather than model based estimate) of human myco copy number.
# This test is duplicated above.

t.test(myco_ferm_RH[myco_ferm_RH$week==0 & myco_ferm_RH$conc==0,"myco_ferm_copy_num"])

	# One Sample t-test 

# data:  myco_ferm_RH[myco_ferm_RH$week == 0 & myco_ferm_RH$conc == 0,     "myco_ferm_copy_num"]
# t = 3.4205, df = 5, p-value = 0.01883
# alternative hypothesis: true mean is not equal to 0
# 95 percent confidence interval:
 # 0.02696592 0.19007975
# sample estimates:
# mean of x 
# 0.1085228 
		
		
		

		 
		glht_growth <- glht(m1, linfct = c(
					"week == 0",
 					"week + 8*week:conc == 0",
					"week + 25*week:conc == 0",
 					"week + 75*week:conc == 0",
 					"week + (27)*week:conc == 0"))
 					
		glht_drug <- glht(m1, linfct = c(
					"conc + 1*week:conc == 0",
  					"conc + 2*week:conc == 0",
 					"conc + 3*week:conc == 0",
  					"conc + 4*week:conc == 0",
 					"conc + 6*week:conc == 0",
 					"conc + (3.2)*week:conc == 0"))
 					
 		glht_omni <- glht(m1)
 		 					
 		
 	
 	growth_coef <- summary(glht_growth,test = adjusted("none"))$test$coefficients
 	drug_coef <- summary(glht_drug,test = adjusted("none"))$test$coefficients
 	Ix_coef <- summary(glht_omni,test = adjusted("none"))$test$coefficients["week:conc"]				
 	growth_stat <- summary(glht_growth,test = adjusted("none"))$test$tstat
 	drug_stat <- summary(glht_drug,test = adjusted("none"))$test$tstat
 	Ix_stat <- summary(glht_omni,test = adjusted("none"))$test$tstat["week:conc"]
 	
 		
 		
 		
 	human_myco_copy_hamster_reads[1,] <- c(
				 							"myco_ferm",
				 							growth_coef,
				 							drug_coef,
				 							Ix_coef,
				 							growth_stat,
				 							drug_stat,
				 							Ix_stat,
				 							-log10(2*pnorm(-abs(growth_stat))),
				 							-log10(2*pnorm(-abs(drug_stat))),
				 							-log10(2*pnorm(-abs(Ix_stat)))
				 							)
				 							
				 							
human_myco_copy_hamster_reads
	   # myco_spp           coef_g_0nM         coef_g_8nM         coef_g_25nM        coef_g_75nM          coef_g_avg             coef_d_w1             coef_d_w2
# 1 myco_ferm 0.000801343083603291 0.0021180562834938 0.00491607183326113 0.0131455293325768 0.00524525013323376 -0.000773499154003057 -0.000608910004016744
             # coef_d_w3             coef_d_w4            coef_d_w6            coef_d_avg          coef_g_d_Ix           Z_g_0nM           Z_g_8nM         Z_g_25nM
# 1 -0.00044432085403043 -0.000279731704044116 4.94465959285109e-05 -0.000411403024033167 0.000164589149986314 0.138537958923914 0.419210247974245 1.06848126744191
          # Z_g_75nM          Z_g_avg            Z_d_w1            Z_d_w2             Z_d_w3             Z_d_w4             Z_d_w6            Z_d_avg
# 1 1.27185866673154 1.12409551377738 -1.17789163845814 -1.07705758545079 -0.860218729047774 -0.535609841309023 0.0728196713276734 -0.801779556677533
           # Z_g_d_Ix       log10p_g_0nM      log10p_g_8nM     log10p_g_25nM     log10p_g_75nM     log10p_g_avg       log10p_d_w1       log10p_d_w2       log10p_d_w3
# 1 0.959968957045902 0.0507001447022031 0.170656027385565 0.544692937225696 0.691599219371767 0.58340524025122 0.621893284866455 0.550591741281333 0.409304719864826
        # log10p_d_w4        log10p_d_w6      log10p_d_avg     log10p_g_d_Ix
# 1 0.227510897847675 0.0259723261020858 0.373987801679567 0.472278818302972	 		


# recast as normal P vals not -log10P


	human_myco_copy_hamster_reads <- data.frame(
									myco_spp = as.character(),
									coef_g_0nM = numeric(),
									coef_g_8nM = numeric(),
									coef_g_25nM = numeric(),
									coef_g_75nM = numeric(),
									coef_g_avg = numeric(),
									coef_d_w1 = numeric(),
									coef_d_w2 = numeric(),
									coef_d_w3 = numeric(),
									coef_d_w4 = numeric(),
									coef_d_w6 = numeric(),
									coef_d_avg = numeric(),
									coef_g_d_Ix = numeric(),
									Z_g_0nM = numeric(),
									Z_g_8nM = numeric(),
									Z_g_25nM = numeric(),
									Z_g_75nM = numeric(),
									Z_g_avg = numeric(),
									Z_d_w1 = numeric(),
									Z_d_w2 = numeric(),
									Z_d_w3 = numeric(),
									Z_d_w4 = numeric(),
									Z_d_w6 = numeric(),
									Z_d_avg = numeric(),
									Z_g_d_Ix = numeric(),
									P_g_0nM = numeric(),
									P_g_8nM = numeric(),
									P_g_25nM = numeric(),
									P_g_75nM = numeric(),
									P_g_avg = numeric(),
									P_d_w1 = numeric(),
									P_d_w2 = numeric(),
									P_d_w3 = numeric(),
									P_d_w4 = numeric(),
									P_d_w6 = numeric(),
									P_d_avg = numeric(),
									P_g_d_Ix = numeric(),
									stringsAsFactors=FALSE
									)

	
	human_myco_copy_hamster_reads[1,] <- c(
				 							"myco_ferm",
				 							growth_coef,
				 							drug_coef,
				 							Ix_coef,
				 							growth_stat,
				 							drug_stat,
				 							Ix_stat,
				 							2*pnorm(-abs(growth_stat)),
				 							2*pnorm(-abs(drug_stat)),
				 							2*pnorm(-abs(Ix_stat))
				 							)
	
human_myco_copy_hamster_reads
   # myco_spp           coef_g_0nM         coef_g_8nM         coef_g_25nM        coef_g_75nM          coef_g_avg             coef_d_w1             coef_d_w2
# 1 myco_ferm 0.000801343083603291 0.0021180562834938 0.00491607183326113 0.0131455293325768 0.00524525013323376 -0.000773499154003057 -0.000608910004016744
             # coef_d_w3             coef_d_w4            coef_d_w6            coef_d_avg          coef_g_d_Ix           Z_g_0nM           Z_g_8nM         Z_g_25nM
# 1 -0.00044432085403043 -0.000279731704044116 4.94465959285109e-05 -0.000411403024033167 0.000164589149986314 0.138537958923914 0.419210247974245 1.06848126744191
          # Z_g_75nM          Z_g_avg            Z_d_w1            Z_d_w2             Z_d_w3             Z_d_w4             Z_d_w6            Z_d_avg
# 1 1.27185866673154 1.12409551377738 -1.17789163845814 -1.07705758545079 -0.860218729047774 -0.535609841309023 0.0728196713276734 -0.801779556677533
           # Z_g_d_Ix           P_g_0nM           P_g_8nM         P_g_25nM          P_g_75nM           P_g_avg            P_d_w1            P_d_w2            P_d_w3
# 1 0.959968957045902 0.889815271951989 0.675062483201649 0.28530347590173 0.203423339966758 0.260972508303681 0.238839808960737 0.281454540183144 0.389668482410533
             # P_d_w4            P_d_w6          P_d_avg          P_g_d_Ix
# 1 0.592228225775567 0.941949617057568 0.42268048620922 0.337070838733936 






# emmeans
# Provides kenward-roger
# lmer.df="asymptotic" agrees with glht exactly

# growth

summary(contrast(emmeans(m1, lmer.df = "kenward-roger", specs="week", at=list(week=c(0,1), conc=27)),list(c(-1,1))))
# NOTE: Results may be misleading due to involvement in interactions
 # contrast estimate      SE   df t.ratio p.value
 # c(-1, 1)  0.00525 0.00468 91.5 1.121   0.2654  <<<<<<<<<<< use in paper
 
 
summary(contrast(emmeans(m1, lmer.df = "kenward-roger", specs="week", at=list(week=c(0,1), conc=27)),list(c(-1,1))))$p.value
# NOTE: Results may be misleading due to involvement in interactions
# [1] 0.2653911 <<<<<<<<<<< use in paper
 
 

# drug
summary(contrast(emmeans(m1, lmer.df = "kenward-roger", specs="conc", at=list(conc=c(0,1), week=3.2)),list(c(-1,1))))
# NOTE: Results may be misleading due to involvement in interactions
 # contrast  estimate       SE   df t.ratio p.value
 # c(-1, 1) -0.000411 0.000515 24.7 -0.799  0.4322    <<<<<<<<<<< use in paper



# Ix
summary(contrast(emmeans(m1, lmer.df = "kenward-roger", specs=c("week","conc"), at=list(week=c(0,1),conc=c(0,1))),list(c(1,-1,-1,1))))
 # contrast        estimate       SE   df t.ratio p.value
 # c(1, -1, -1, 1) 0.000165 0.000172 93.6 0.956   0.3415   <<<<<<<<<<< use in paper



	

# pool 
# (1|pool/cell) == (1|pool) + (1|pool:cell), which, because of cell indexing == (1|pool) + (1|cell)
anova(lmer(myco_ferm_copy_num ~ week * conc + (1|pool) + (1|cell), data = myco_ferm_RH, REML=TRUE),lmer(myco_ferm_copy_num ~ week * conc + (1|cell), data = myco_ferm_RH, REML=TRUE))
# refitting model(s) with ML (instead of REML)
# Data: myco_ferm_RH
# Models:
# lmer(myco_ferm_copy_num ~ week * conc + (1 | cell), data = myco_ferm_RH, REML = TRUE): myco_ferm_copy_num ~ week * conc + (1 | cell)
# lmer(myco_ferm_copy_num ~ week * conc + (1 | pool) + (1 | cell), data = myco_ferm_RH, REML = TRUE): myco_ferm_copy_num ~ week * conc + (1 | pool) + (1 | cell)
                                                                                                   # Df     AIC     BIC  logLik deviance  Chisq Chi Df Pr(>Chisq)  
# lmer(myco_ferm_copy_num ~ week * conc + (1 | cell), data = myco_ferm_RH, REML = TRUE)               6 -187.78 -171.31  99.889  -199.78                           
# lmer(myco_ferm_copy_num ~ week * conc + (1 | pool) + (1 | cell), data = myco_ferm_RH, REML = TRUE)  7 -189.72 -170.50 101.860  -203.72 3.9402      1    0.04714 * <<<<<<<<<< use in paper
# ---
# Signif. codes:  0 ‘***’ 0.001 ‘**’ 0.01 ‘*’ 0.05 ‘.’ 0.1 ‘ ’ 1






# cell 
anova(lmer(myco_ferm_copy_num ~ week * conc + (1|pool) + (1|cell), data = myco_ferm_RH, REML=TRUE),lmer(myco_ferm_copy_num ~ week * conc + (1|pool), data = myco_ferm_RH, REML=TRUE))
# refitting model(s) with ML (instead of REML)
# Data: myco_ferm_RH
# Models:
# lmer(myco_ferm_copy_num ~ week * conc + (1 | pool), data = myco_ferm_RH, REML = TRUE): myco_ferm_copy_num ~ week * conc + (1 | pool)
# lmer(myco_ferm_copy_num ~ week * conc + (1 | pool) + (1 | cell), data = myco_ferm_RH, REML = TRUE): myco_ferm_copy_num ~ week * conc + (1 | pool) + (1 | cell)
                                                                                                   # Df     AIC    BIC  logLik deviance  Chisq Chi Df Pr(>Chisq)    
# lmer(myco_ferm_copy_num ~ week * conc + (1 | pool), data = myco_ferm_RH, REML = TRUE)               6 -178.77 -162.3  95.383  -190.77                             
# lmer(myco_ferm_copy_num ~ week * conc + (1 | pool) + (1 | cell), data = myco_ferm_RH, REML = TRUE)  7 -189.72 -170.5 101.860  -203.72 12.953      1  0.0003194 *** <<<<<<<<<< use in paper
# ---
# Signif. codes:  0 ‘***’ 0.001 ‘**’ 0.01 ‘*’ 0.05 ‘.’ 0.1 ‘ ’ 1



# pool and cell together
anova(lmer(myco_ferm_copy_num ~ week * conc + (1|pool/cell), data = myco_ferm_RH, REML=TRUE),lm(myco_ferm_copy_num ~ week * conc, data = myco_ferm_RH, REML=TRUE))
# refitting model(s) with ML (instead of REML)
# lm(myco_ferm_copy_num ~ week * conc, data = myco_ferm_RH, REML = TRUE): myco_ferm_copy_num ~ week * conc
# lmer(myco_ferm_copy_num ~ week * conc + (1 | pool/cell), data = myco_ferm_RH, REML = TRUE): myco_ferm_copy_num ~ week * conc + (1 | pool/cell)
                                                                                           # Df     AIC     BIC  logLik deviance  Chisq Chi Df Pr(>Chisq)    
# lm(myco_ferm_copy_num ~ week * conc, data = myco_ferm_RH, REML = TRUE)                      5 -159.49 -145.77  84.746  -169.49                             
# lmer(myco_ferm_copy_num ~ week * conc + (1 | pool/cell), data = myco_ferm_RH, REML = TRUE)  7 -189.72 -170.50 101.860  -203.72 34.227      2  3.695e-08 ***
# ---
# Signif. codes:  0 ‘***’ 0.001 ‘**’ 0.01 ‘*’ 0.05 ‘.’ 0.1 ‘ ’ 1







# icc


performance::icc(m1)
# # Intraclass Correlation Coefficient

     # Adjusted ICC: 0.469
  # Conditional ICC: 0.463





# pool
(get_variance(m1)$var.intercept["pool"])/(get_variance(m1)$var.intercept["pool"] + get_variance(m1)$var.intercept["cell:pool"] + get_variance(m1)$var.residual)
     # pool 
# 0.2211725 <<<<<<<<<< use in paper




# cell
(get_variance(m1)$var.intercept["cell:pool"])/(get_variance(m1)$var.intercept["pool"] + get_variance(m1)$var.intercept["cell:pool"] + get_variance(m1)$var.residual)
# cell:pool 
# 0.2477078 <<<<<<<<<< use in paper


	

# pool and cell together
(get_variance(m1)$var.intercept["pool"] + get_variance(m1)$var.intercept["cell:pool"])/(get_variance(m1)$var.intercept["pool"] + get_variance(m1)$var.intercept["cell:pool"] + get_variance(m1)$var.residual)
     # pool 
# 0.4688802




























