# Chromosome level Manhattan plot for growth. Plot of copy number changes

#install.packages("ggplot2")
library(ggplot2)
library(cowplot) #used with plot_grid 

# for summarySE()
library(Rmisc)
library(mgcv)

library(pdftools)
library(magick)


#----------------Aesthetics ---------------------------


theme2 <- theme(
	plot.margin = unit(c(t=1.2,r=0.4,b=1.2,l=0.4), "cm"),
	panel.grid.major = element_blank(), 
	panel.grid.minor = element_blank(), 
	panel.background = element_blank(), 
	legend.position="none", 
	axis.line.x = element_line(colour = "black", size = 0.1), 
	axis.line.y = element_line(colour = "black", size = 0.1), 
	axis.ticks = element_line(colour = "black", size = 0.1),
	axis.text=element_text(size=12), #numbers on tick marks of x and y axes
	axis.title=element_text(size=14), #titles of x and y axes
	axis.title.y=element_text(margin=margin(0,13,0,0)), #moves y axis title by adding margin space to bottom
	axis.title.x=element_text(margin=margin(10,0,0,0)),  #moves x axis title by adding margin space to top
	plot.title = element_text(size=32, face="bold", hjust = -0.14), #can provide "A","B", by ggtitle, but used plot_grid wch can shift more left
	plot.subtitle = element_text(size=14, face="plain", hjust = 0.5) #hjust shifts right
	)




# size_point <- 0.3
size_hline <- 0.2


# darkest two hues from 3-class PuBuGn in color brewer
# cb1<-rep(c("#1c9099", "#a6bddb"), 12)

# # darkest two hues from 3-class PuBu in color brewer
# cb1<-rep(c("#2b8cbe", "#a6bddb"), 12)


# #attractive pinks, greys
# cb1<-c("#999999", "#E69F00", "#56B4E9", "#009E73", "#F0E442", "#0072B2", "#D55E00", "#CC79A7","#999999", "#E69F00", "#56B4E9", "#E69F00", "#009E73", "#F0E442", "#0072B2", "#56B4E9", "#009E73", "#F0E442", "#0072B2", "#D55E00", "#CC79A7","#999999", "#D55E00", "#CC79A7")

# cb1_rev <- c("#CC79A7", "#D55E00", "#0072B2", "#F0E442", "#009E73", "#56B4E9", "#E69F00","#999999", "#CC79A7", "#D55E00", "#0072B2", "#D55E00", "#F0E442", "#009E73", "#56B4E9", "#0072B2", "#F0E442", "#009E73", "#56B4E9", "#E69F00","#999999", "#CC79A7", "#E69F00","#999999")

# #'4-class RdBu'
# cb2 <- c('#ca0020','#f4a582','#92c5de','#0571b0','#ca0020','#f4a582','#92c5de','#0571b0','#ca0020','#f4a582','#92c5de','#f4a582','#0571b0','#ca0020','#f4a582','#92c5de','#0571b0','#ca0020','#f4a582','#92c5de','#0571b0','#ca0020','#92c5de','#0571b0')

# #'4-class RdYlBu'
# cb3 <- c('#d7191c','#fdae61','#abd9e9','#2c7bb6','#d7191c','#fdae61','#abd9e9','#2c7bb6','#d7191c','#fdae61','#abd9e9','#fdae61','#2c7bb6','#d7191c','#fdae61','#abd9e9','#2c7bb6','#d7191c','#fdae61','#abd9e9','#2c7bb6','#d7191c','#abd9e9','#2c7bb6')
	
	




#----------------- Prepare human logP ---------------------


logP <- read.table("log10P_human.txt",header=TRUE,sep="\t",stringsAsFactors=FALSE)
human_thresh_95 <- read.table("human_thresh_95.txt",header=FALSE,sep="\t",stringsAsFactors=FALSE,row.names=1,col.names=c("","thresh"))



# Sort:
chrOrder<-paste("chr",c(1:22,"X"),sep="")
logP$Chromosome <- factor(logP$Chromosome, levels=chrOrder)
logP <- logP[order(logP$Chromosome, logP$pos), ]
logP$Chromosome <- as.character(logP$Chromosome)



# # Transform chr1 etc. to numbers
# logP$Chromosome <- gsub('chr', '', logP$Chromosome)
# logP[logP$Chromosome == "X","Chromosome"] <- 23
# chrOrder<-c(1:23)
# logP$Chromosome <- factor(logP$Chromosome, levels=chrOrder)
# logP <- logP[order(logP$Chromosome, logP$pos), ]
# logP$Chromosome <- as.numeric(logP$Chromosome)

# # Compute chromosome size
# gen_coord <- aggregate(pos~Chromosome,FUN=max,data=logP)
# colnames(gen_coord)[2] <- "chr_size"
# gen_coord$Chromosome <-factor(gen_coord$Chromosome, levels=chrOrder)
# gen_coord <- gen_coord[order(gen_coord$Chromosome), ]
# gen_coord$Chromosome <- as.numeric(gen_coord$Chromosome)

# # Use cumsum to make genome coordinates
# gen_coord$coord <- c(0,cumsum(gen_coord$chr_size)[-23])

# # merge genome coordinates with logP
# logP <- merge(logP,gen_coord[,c("Chromosome","coord")])
# logP$Chromosome <-factor(logP$Chromosome, levels=chrOrder)
# logP <- logP[order(logP$Chromosome, logP$pos), ]
# logP$Chromosome <- as.numeric(logP$Chromosome)

# logP$coord <- logP$pos + logP$coord


# # find midpoints of chromosomes for breaks in ggplot
# mid <- function(x) {(max(x)+min(x))/2}
# chr_mid <- aggregate(coord~Chromosome,FUN = mid,data=logP)
# colnames(chr_mid)[2] <- "mid"
# chr_mid$Chromosome <-factor(chr_mid$Chromosome, levels=chrOrder)
# chr_mid <- chr_mid[order(chr_mid$Chromosome), ]
# chr_mid$Chromosome <- as.numeric(chr_mid$Chromosome)

# # Define breaks as mid-points chromosomes
# breaks <- chr_mid$mid


# # attractive grey and skyblue color scheme
# cb1<-rep(c("grey", "skyblue"), 12)


# standard black color scheme
cb1<-rep(c("black", "black"), 12)



# labels <- as.character(c(1:9,"",11,"",13,"","",16,"","","",20,"","","X"))


# --------- (1) chr1, 0nM ----------------------------

# choose chr number
i <- 1


# id points by hand

g <- read.delim("growth_loci.txt",header=TRUE,sep="\t",stringsAsFactors=FALSE,check.names=FALSE)

# # id points by hand
# plot(logP[logP=="chr1","log10p_g_0nM"],cex=0.05)
# points_p1 <- identify(logP[logP=="chr1","log10p_g_0nM"],cex=0.5)

# # points_p1
# [1]  1949  5221 10376 16009 22604

# START HERE if do not wish to do hand picking again
points_p1 <- c(1949, 5221, 10376, 16009, 22604)

# takes few secs
labels_p1 <- merge(logP[logP$Chromosome=="chr1",][points_p1,c("Chromosome","pos")],g[g$conc==0,c("Chromosome","pos","conc","log10P","geneSymbol")])
labels_p1 <- labels_p1[labels_p1$conc==0,]
labels_p1 <- labels_p1[order(labels_p1$pos),]
labels_p1$nudge_x <- 0
labels_p1$nudge_y <- 3
labels_p1[labels_p1$geneSymbol=="IFFO2","nudge_x"] <- 1e-12 # cannot use 0 as 1st vector member because bug in ggplot2, cf https://github.com/tidyverse/ggplot2/issues/2977
labels_p1[labels_p1$geneSymbol=="OSBPL9","nudge_x"] <- -7
labels_p1[labels_p1$geneSymbol=="COL11A1","nudge_x"] <- 20
labels_p1[labels_p1$geneSymbol=="APCS","nudge_x"] <- 0
labels_p1[labels_p1$geneSymbol=="ENAH","nudge_x"] <- 0



balloon_scale <- 0.5 # inflation factor for significant points	
size_point <- 0.1*(1 + balloon_scale*(logP[logP$Chromosome==paste0("chr",i),"log10p_g_0nM"]/max(logP[logP$Chromosome==paste0("chr",i),"log10p_g_0nM"]))) # scale significant points



p1 <- ggplot() + 
		geom_point(
			data = logP[logP$Chromosome==paste0("chr",i),], 
			size= size_point,
			stroke=0, 
				aes(
					x = pos/1e6, 
					y = log10p_g_0nM, 
					color="as.factor(Chromosome)"
					)
				) +
		geom_text(data = labels_p1, aes(x = pos/1e6, y = log10P,label=geneSymbol, fontface = "italic"), nudge_x=labels_p1$nudge_x, nudge_y=labels_p1$nudge_y,  colour = "black", size = 2.5) + # nudge_x and nudge_y gives warning, but seems to work
		# geom_text( aes(x = 18990000/1e6, y = 44.74028), label="IFFO2", colour = "black", size = 3, nudge_x=0, nudge_y=2) +
		# geom_text( aes(x = 51710000/1e6, y = 23.24700), label="OSBPL9", colour = "black", size = 3, nudge_x=0, nudge_y=2) +
		# geom_text( aes(x = 103260000/1e6, y = 54.41426), label="COL11A1", colour = "black", size = 3, nudge_x=12, nudge_y=2) +
		# geom_text( aes(x = 159590000/1e6, y = 20.71389), label="APCS", colour = "black", size = 3, nudge_x=0, nudge_y=2) +
		# geom_text( aes(x = 225540000/1e6, y = 18.84761), label="ENAH", colour = "black", size = 3, nudge_x=0, nudge_y=2) +
		scale_color_manual(values=cb1) +
		theme2 +
		# scale_x_continuous(breaks = breaks, labels = labels) +
		scale_x_continuous() +
		xlab(paste0("Chromosome ", i, " (Mb)")) + 
		ylab(expression('-log'[10]*italic('P'))) +
		geom_hline(yintercept= human_thresh_95["log10p_g_0nM",], linetype="solid", color = "red", size=size_hline) +
		labs(subtitle="Growth (0 nM)") #+
		#scale_y_continuous(breaks=seq(0,120,20),limit = c(0, 120))
print(p1)


# --------- (2) chr3, 8nM ----------------------------


# choose chr number
i <- 3

# id points by hand

g <- read.delim("growth_loci.txt",header=TRUE,sep="\t",stringsAsFactors=FALSE,check.names=FALSE)

# # id points by hand
# plot(logP[logP=="chr3","log10p_g_8nM"],cex=0.05)
# points_p2 <- identify(logP[logP=="chr3","log10p_g_8nM"],cex=0.5)

# # points_p2
#[1]  5068 11203 12727 14192 15583 17585

# START HERE if do not wish to do hand picking again
points_p2 <- c(5068, 11203, 12727, 14192, 17585)

# takes few secs
labels_p2 <- merge(logP[logP$Chromosome=="chr3",][points_p2,c("Chromosome","pos")],g[g$conc==8,c("Chromosome","pos","conc","log10P","geneSymbol")])
labels_p2 <- labels_p2[labels_p2$conc==8,]
labels_p2 <- labels_p2[order(labels_p2$pos),]
labels_p2$nudge_x <- 0
labels_p2$nudge_y <- 5
labels_p2[labels_p2$geneSymbol=="SEMA3F","nudge_x"] <- 8 # cannot use 0 as 1st vector member because bug in ggplot2, cf https://github.com/tidyverse/ggplot2/issues/2977
labels_p2[labels_p2$geneSymbol=="CD96","nudge_x"] <- 0
labels_p2[labels_p2$geneSymbol=="CHCHD6","nudge_x"] <- 0
labels_p2[labels_p2$geneSymbol=="ZBTB38","nudge_x"] <- 5
labels_p2[labels_p2$geneSymbol=="NAALADL2","nudge_x"] <- -3
labels_p2[labels_p2$geneSymbol=="NAALADL2","nudge_y"] <- 7





balloon_scale <- 0.5 # inflation factor for significant points	
size_point <- 0.1*(1 + balloon_scale*(logP[logP$Chromosome==paste0("chr",i),"log10p_g_8nM"]/max(logP[logP$Chromosome==paste0("chr",i),"log10p_g_8nM"]))) # scale significant points



p2 <- ggplot() + 
		geom_point(
			data = logP[logP$Chromosome==paste0("chr",i),], 
			size= size_point,
			stroke=0, 
				aes(
					x = pos/1e6, 
					y = log10p_g_8nM, 
					color="as.factor(Chromosome)"
					)
				) +
		geom_text(data = labels_p2, aes(x = pos/1e6, y = log10P,label=geneSymbol, fontface = "italic"), nudge_x=labels_p2$nudge_x, nudge_y=labels_p2$nudge_y,  colour = "black", size = 2.5) + # nudge_x and nudge_y gives warning, but seems to work
		scale_color_manual(values=cb1) +
		theme2 +
		# scale_x_continuous(breaks = breaks, labels = labels) +
		scale_x_continuous() +
		xlab(paste0("Chromosome ", i, " (Mb)")) + 
		ylab(expression('-log'[10]*italic('P'))) +
		geom_hline(yintercept= human_thresh_95["log10p_g_8nM",], linetype="solid", color = "red", size=size_hline) +
		labs(subtitle="Growth (8 nM)") #+
		#scale_y_continuous(breaks=seq(0,120,20),limit = c(0, 120))
print(p2)



# --------- (3) chr16, 25nM ----------------------------


# choose chr number
i <- 16

# id points by hand

g <- read.delim("growth_loci.txt",header=TRUE,sep="\t",stringsAsFactors=FALSE,check.names=FALSE)

# # id points by hand
# plot(logP[logP=="chr16","log10p_g_25nM"],cex=0.05)
# points_p3 <- identify(logP[logP=="chr16","log10p_g_25nM"],cex=0.5)

# # points_p3
#[1] 2019 3150 5438 7006 8390

# START HERE if do not wish to do hand picking again
points_p3 <- c(2019, 3150, 5438, 7006, 8390)

# takes few secs
labels_p3 <- merge(logP[logP$Chromosome == "chr16",][points_p3,c("Chromosome","pos")],g[g$conc==25,c("Chromosome","pos","conc","log10P","geneSymbol")])
labels_p3 <- labels_p3[labels_p3$conc==25,]
labels_p3 <- labels_p3[order(labels_p3$pos),]
labels_p3$nudge_x <- 0
labels_p3$nudge_y <- 6
labels_p3[labels_p3$geneSymbol=="VPS35L","nudge_x"] <- 1e-12 # cannot use 0 as 1st vector member because bug in ggplot2, cf https://github.com/tidyverse/ggplot2/issues/2977
labels_p3[labels_p3$geneSymbol=="STX1B","nudge_x"] <- 2
labels_p3[labels_p3$geneSymbol=="FTO","nudge_x"] <- 0
labels_p3[labels_p3$geneSymbol=="NFAT5","nudge_x"] <- 0
labels_p3[labels_p3$geneSymbol=="CDH13","nudge_x"] <- 0



balloon_scale <- 0.5 # inflation factor for significant points	
size_point <- 0.1*(1 + balloon_scale*(logP[logP$Chromosome==paste0("chr",i),"log10p_g_8nM"]/max(logP[logP$Chromosome==paste0("chr",i),"log10p_g_8nM"]))) # scale significant points



p3 <- ggplot() + 
		geom_point(
			data = logP[logP$Chromosome==paste0("chr",i),], 
			size= size_point,
			stroke=0, 
				aes(
					x = pos/1e6, 
					y = log10p_g_25nM, 
					color="as.factor(Chromosome)"
					)
				) +
		geom_text(data = labels_p3, aes(x = pos/1e6, y = log10P,label=geneSymbol, fontface = "italic"), nudge_x=labels_p3$nudge_x, nudge_y=labels_p3$nudge_y,  colour = "black", size = 2.5) + # nudge_x and nudge_y gives warning, but seems to work
		scale_color_manual(values=cb1) +
		theme2 +
		# scale_x_continuous(breaks = breaks, labels = labels) +
		scale_x_continuous() +
		xlab(paste0("Chromosome ", i, " (Mb)")) + 
		ylab(expression('-log'[10]*italic('P'))) +
		geom_hline(yintercept= human_thresh_95["log10p_g_25nM",], linetype="solid", color = "red", size=size_hline) +
		labs(subtitle="Growth (25 nM)") #+
		#scale_y_continuous(breaks=seq(0,120,20),limit = c(0, 120))
print(p3)



# ------------- (4) line plot copy number change 0 nM -----------------------

# cf plot_nb_graph_1.R


gg_color_hue <- function(n) {
  hues = seq(15, 375, length = n + 1)
  hcl(h = hues, l = 65, c = 100)[1:n]
}


# g_unique
g_unique <- read.delim("growth_loci_unique.txt",header=TRUE,sep="\t",stringsAsFactors=FALSE,check.names=FALSE)

# copy number data
RH_human <- read.table("RH_human_gseq.txt",header=TRUE,sep="\t",stringsAsFactors=FALSE)

# Read in and prepare ancillary tables 

cell <- read.table("cell_label_info.txt",sep="\t",stringsAsFactors=FALSE,header=TRUE)
sum_reads <- colSums(RH_human[,5:ncol(RH_human)])
reads <- read.table("RH_pool_human_total_align.txt",sep="\t",stringsAsFactors=FALSE,header=TRUE) # uses mapped human reads, cf human_AIC_1.R


# select growth genes at 0 nM
g_unique[g_unique$geneSymbol %in% c("INPP4B","ENTPD3","LDAH","MAPKAPK3","FMNL2","WWC2"),c(1,4,5,6,7,8,11)]
    # Chromosome       pos conc   log10P   sig_coef dist geneSymbol
# 244       chr4 142820000    0 53.21576 -0.3839872    0     INPP4B
# 161       chr3  40390000    0 41.86878 -0.2529405    0     ENTPD3
# 82        chr2  20820000    0 36.33119 -0.3299637    0       LDAH
# 164       chr3  50620000    0 35.30138 -0.3384411    0   MAPKAPK3
# 128       chr2 152490000    0 32.35739 -0.3210492    0      FMNL2
# 260       chr4 183100000    0 26.53048 -0.2712388    0       WWC2
 
genes <- merge(g_unique[g_unique$geneSymbol %in% c("INPP4B","ENTPD3","LDAH","MAPKAPK3","FMNL2","WWC2"),c(1,4,5,6,8,11)],RH_human)


# Prepare for mgcv::gam

genes_l <- reshape(genes[,c(6,9:ncol(genes))], 
  varying = c(colnames(genes[c(9:ncol(genes))])), 
  v.names = "copy",
  timevar = "RH_ID", 
  times = c(colnames(genes[c(9:ncol(genes))])), 
  new.row.names = 1:1e6,
  direction = "long")
  
genes_l$week <- 0
genes_l[grepl("_w0_",genes_l$RH_ID),]$week <- 0
genes_l[grepl("_w1_",genes_l$RH_ID),]$week <- 1
genes_l[grepl("_w2_",genes_l$RH_ID),]$week <- 2
genes_l[grepl("_w3_",genes_l$RH_ID),]$week <- 3
genes_l[grepl("_w4_",genes_l$RH_ID),]$week <- 4
genes_l[grepl("_w6_",genes_l$RH_ID),]$week <- 6

genes_l$conc <- 0
genes_l[grepl("_d0",genes_l$RH_ID),]$conc <- 0
genes_l[grepl("_d8",genes_l$RH_ID),]$conc <- 8
genes_l[grepl("_d25",genes_l$RH_ID),]$conc <- 25
genes_l[grepl("_d75",genes_l$RH_ID),]$conc <- 75

genes_l$pool <- 0
genes_l[grepl("RH1_",genes_l$RH_ID),]$pool <- 1
genes_l[grepl("RH2_",genes_l$RH_ID),]$pool <- 2
genes_l[grepl("RH3_",genes_l$RH_ID),]$pool <- 3
genes_l[grepl("RH4_",genes_l$RH_ID),]$pool <- 4
genes_l[grepl("RH5_",genes_l$RH_ID),]$pool <- 5
genes_l[grepl("RH6_",genes_l$RH_ID),]$pool <- 6



genes_l <- merge(genes_l,cell)
genes_l$sum_reads <- sum_reads[genes_l$RH_ID]
genes_l  <- merge(genes_l,reads[,c(1:5,9)])
colnames(genes_l)[10] <- "total_reads"


genes_l$pool <- as.factor(genes_l$pool)
genes_l$cell <- as.factor(genes_l$cell)
gene_levels <- g_unique[g_unique$geneSymbol %in% unique(genes_l$geneSymbol),c(6,7,11)]
gene_levels <- gene_levels[order(-gene_levels$sig_coef),]
genes_l$geneSymbol <- factor(genes_l$geneSymbol,levels=gene_levels$geneSymbol)



genes_l$phat = as.numeric(NA)
genes_l$phat_se = as.numeric(NA)
genes_l$phat_center = as.numeric(NA)
genes_l$log_copy = as.numeric(NA)

for (i in c(1:6)) {
m1_nb <- gam(copy ~ week * conc + s(pool, bs = "re") + s(cell, bs = "re") + offset(log(total_reads)), data = genes_l[genes_l$id == i,], family = nb, method = "REML")

genes_l[genes_l$id == i,]$phat <- predict(m1_nb, se.fit=TRUE,type="link")$fit
genes_l[genes_l$id == i,]$phat_se <- predict(m1_nb, se.fit=TRUE,type="link")$se.fit

# omnibus center for growth since all lines originate from week == 0, conc == 0
genes_l[genes_l$id == i,]$phat_center <- genes_l[genes_l$id == i,"phat"] - mean(genes_l[genes_l$id == i & genes_l$week == 0 & genes_l$conc == 0,"phat"])
genes_l[genes_l$id == i,]$log_copy <- log(genes_l[genes_l$id == i,]$copy) - mean(log(genes_l[genes_l$id == i & genes_l$week==0 & genes_l$conc==0,"copy"]))
}




n = length(unique(genes_l$geneSymbol))
colores_1 = gg_color_hue(n)
names(colores_1) <- gene_levels$geneSymbol


# chose jitter of zero, as made for a tidier plot
jitter_factor <- 0
jitter <- data.frame(week=rep(unique(genes_l$week),n),geneSymbol=rep(gene_levels$geneSymbol,each=length(unique(genes_l$week))),jitter=jitter(rep(unique(genes_l$week),n),jitter_factor))
genes_l <- merge(jitter,genes_l)


# Have to redo factor levels of genes_l$geneSymbol after merge, for some reason
genes_l$geneSymbol <- factor(genes_l$geneSymbol,levels=gene_levels$geneSymbol)


# provide summary table if wish to adjust CI to sem using group number, N
summary_genes <- summarySE(genes_l, measurevar="phat_center", groupvars=c("week","geneSymbol","conc"))








# use lm to provide line and CI

p4 <- ggplot() + 
 	theme2 + 
	theme(legend.key=element_blank()) +
	geom_point(
		data=genes_l[genes_l$conc==0,],
		shape=1,
		stroke=0.2,
		size=1.0,
		aes(
			x=jitter, 
			y=log2(exp(1))*log_copy, 
			colour=geneSymbol
			)
		) +
    geom_smooth(
	    data=genes_l[genes_l$conc==0,], 
	    method = "lm",  
	    	formula=	 y~0+x,
	    aes(
		    	y=log2(exp(1))*phat_center,
		    	x=week,
		    group=geneSymbol,
		    colour=geneSymbol#,
		    # fill=geneSymbol
		    ),
	    se=TRUE,
	    level=0.95,
	    size=0.3,
	    fill="grey",
	    alpha=0.1
	    ) +
	scale_color_manual(
		values=colores_1,
		name =NULL, 
		labels=levels(genes_l[genes_l$conc==0,"geneSymbol"])
		) +
	guides(
		shape=FALSE,
		fill=FALSE,
 		colour = guide_legend(
			 		override.aes = list(
			 		fill=NA,
			 		shape=NA,
			 		size=0.3
			 		),
		 		ncol=1,
		 		byrow=TRUE
		 		)
 		) +
	theme(
		legend.position = "right", 
 		legend.title = element_text(size = 9), 
 		legend.text = element_text(size = 8, face = "italic"),
 		legend.title.align=0.2,
 		legend.key.height = unit(0.1, 'lines'), 
 		legend.margin=margin(t = 0, r = -0.2, b = 0, l = -0.4, unit = "cm")
 		) +
	scale_x_continuous(breaks = c(0,1,2,3,4,6), labels = c(0,1,2,3,4,6)) +
	# ggtitle("") + 
	xlab("Weeks") + 
	ylab(expression(Delta*log[2]~(Reads))) + 
	labs(subtitle="Growth (0 nM)")
print(p4)






# --------------- Provide 0 nM growth to UCSC genome browser ---------------------



# ############## DO NOT DELETE #######################
# Important if want to reconstruct 0 nM growth data for UCSC genome browser


# g_0nM <- logP[,c("Chromosome","pos","log10p_g_0nM")]
# g_0nM$posS <- g_0nM$pos + 1
# g_0nM <- g_0nM[,c("Chromosome","pos","posS","log10p_g_0nM")]
# g_0nM$pos <- format(g_0nM$pos,scientific=FALSE)
# g_0nM$posS <- format(g_0nM$posS,scientific=FALSE)


# head(g_0nM)
  # # Chromosome       pos      posS log10p_g_0nM
# # 1       chr1      5000      5001 6.982125e-09
# # 2       chr1     10000     10001 3.158713e-01
# # 3       chr1     15000     15001 3.842510e-01
# # 4       chr1     20000     20001 5.904982e-01
# # 5       chr1     25000     25001 6.445060e-01
# # 6       chr1     30000     30001 3.869687e-01

# write.table(g_0nM, "growth_0nM_log10P.txt",quote=FALSE,sep="\t",row.names=FALSE,col.names=FALSE)


# # # Place following header at top of paclitaxel_logP_wk6.txt and use in bedGraph format on ucsc genome browser
# # track type=bedGraph name="-log10P" description="use name of gene zoomed in on" visibility=full color=0,0,255 altColor=255,0,0 priority=20

# # Custom track settings
# # Display mode: full
# # Type of graph: points
# # Track height: 128 pixels
# # Data view scaling: auto-scale to data view
# # Always include zero: ON
# # Vertical viewing range:  min: 0; max: 1000  (range: 0 to 1000) (greyed out)
# # Transform function: Transform data points by: NONE
# # Windowing function: mean
# # Smoothing window: OFF
# # Negate values: not selected
# # Draw y indicator lines: 
# # at y = 0.0: ON at y = 6.875376 ON (corresponds to human_thresh_95.txt, for log10p_g_0nM)


# # Configure Image page on ucsc genome browser:
# # image width:	400	pixels
# # label area width:	8	characters	
# # text size: 12


# # Remember to get rid of splicoforms and nc genes by clicking GENCODE v31 bar on left of diagram, if desired. Then click on wrench and select "Transcript class": coding, "Transcript Annotation Method": All, "Transcript Biotype": protein_coding, "Tag":appris_principal_1, "Support Level": All. Leave "Highlight items" unselected.




# ------------------ (5) CTTNBP2 logP vs Gencode, UCSC genome browser ----------------



p5 <- ggdraw() + draw_image(magick::image_read_pdf("CTTNBP2_0nM.pdf", density = 300),scale=1.3) + coord_cartesian(clip = "off") # + draw_label("Paclitaxel", fontface='plain', size=12, x=0.55,y=0.92) + draw_label("D", fontface='bold',x=0.05,y=0.98)

# CTTNBP2: chr7:117,000000-118500000

# ------------------ (6) MCTP2 logP vs Gencode, UCSC genome browser ----------------



p6 <- ggdraw() + draw_image(magick::image_read_pdf("MCTP2_0nM.pdf", density = 300),scale=1.3) + coord_cartesian(clip = "off") # + draw_label("Paclitaxel", fontface='plain', size=12, x=0.55,y=0.92) + draw_label("D", fontface='bold',x=0.05,y=0.98)


# MCTP2: chr15:93000000-96000000



# --------------- Combine panels ---------------------

# warning msgs because of nudge_x, nudge_y in p1, p2, p3, but seem to be inconsequential
# File size ~ 2.6 Mb

pdf("g_loci_chr_1.pdf", width=7.5, height=10, useDingbats = FALSE)
plot_grid(p1, p2, p3,  p4, p5, p6, ncol = 2, nrow = 3, labels=c("A", "B", "C", "D", "E","F"), label_size = 16, align="h")
dev.off()




tiff("g_loci_chr_1.tif",width=7.5,height= 10,units="in",res=300)
plot_grid(p1, p2, p3,  p4, p5, p6, ncol = 2, nrow = 3, labels=c("A", "B", "C", "D", "E","F"), label_size = 16, align="h")
dev.off()




# if smaller file size required:
# File size ~0.76 Mb

png("g_loci_chr_1.png",width=7.5,height=10,units="in",res=300)
plot_grid(p1, p2, p3,  p4, p5, p6, ncol = 2, nrow = 3, labels=c("A", "B", "C", "D", "E","F"), label_size = 16, align="h")
dev.off()




p5 <- ggdraw() + draw_image(magick::image_read_pdf("CTTNBP2_0nM.pdf", density = 1200),scale=1.3) + coord_cartesian(clip = "off") # + draw_label("Paclitaxel", fontface='plain', size=12, x=0.55,y=0.92) + draw_label("D", fontface='bold',x=0.05,y=0.98)
p6 <- ggdraw() + draw_image(magick::image_read_pdf("MCTP2_0nM.pdf", density = 1200),scale=1.3) + coord_cartesian(clip = "off") # + draw_label("Paclitaxel", fontface='plain', size=12, x=0.55,y=0.92) + draw_label("D", fontface='bold',x=0.05,y=0.98)


pdf("g_loci_chr_hi_res_1.pdf", width=7.5, height=10, useDingbats = FALSE)
plot_grid(p1, p2, p3,  p4, p5, p6, ncol = 2, nrow = 3, labels=c("A", "B", "C", "D", "E","F"), label_size = 16, align="h")
dev.off()


png("g_loci_chr_hi_res_1.png",width=7.5,height=10,units="in",res=1200)
plot_grid(p1, p2, p3,  p4, p5, p6, ncol = 2, nrow = 3, labels=c("A", "B", "C", "D", "E","F"), label_size = 16, align="h")
dev.off()











