# Plot coefs for growth, drug and Ix

library(ggplot2)
library(cowplot) #used with plot_grid 
library(qqplotr)
library(scales)

# --------------- get_legend ---------------------------


get_legend<-function(myggplot){
  tmp <- ggplot_gtable(ggplot_build(myggplot))
   leg <- which(sapply(tmp$grobs, function(x) x$name) == "guide-box")
   legend <- tmp$grobs[[leg]]
   return(legend)
 }



# --------------------- sem ------------------------

sem <- function(x) {sqrt(var(x,na.rm=TRUE)/sum(!is.na(x)))}

compare <- function(a,b) {
	print(t.test(a,b))
	
	print(paste0("exact P value = ", t.test(a,b)$p.value))
	
	print(paste0("mean of a = ", mean(a, na.rm = TRUE)))
	print(paste0("sem of a = ", sem(a)))
	print(paste0("sd of a = ", sd(a, na.rm = TRUE)))
	print(paste0("number in a = ", sum(!is.na(a))))
	
	print(paste0("mean of b = ", mean(b, na.rm = TRUE)))
	print(paste0("sem of b = ", sem(b)))
	print(paste0("sd of b = ", sd(b, na.rm = TRUE)))
	print(paste0("number in b = ", sum(!is.na(b))))
	
}


#--------------------- Aesthetics ---------------------------


theme2 <- theme(
	plot.margin = unit(c(0.5,0.5,0.5,0.5), "cm"),
	panel.grid.major = element_blank(), 
	panel.grid.minor = element_blank(), 
	panel.background = element_blank(), 
	axis.line.x = element_line(colour = "black", size = 0.1), 
	axis.line.y = element_line(colour = "black", size = 0.1), 
	axis.ticks = element_line(colour = "black", size = 0.1),
	axis.text=element_text(size=12), #numbers on tick marks of x and y axes
	axis.title=element_text(size=14), #titles of x and y axes
	axis.title.y=element_text(margin=margin(t=0,r=7,b=0,l=0)), # moves y axis title by adding margin space to bottom, trbl
	axis.title.x=element_text(margin=margin(t=7,r=0,b=0,l=0)),  # moves x axis title by adding margin space to top,trbl
	plot.title = element_text(size=32, face="bold", hjust = -0.14), #can provide "A","B", by ggtitle, but used plot_grid wch can shift more left
	plot.subtitle = element_text(size=14, face="plain", hjust = 0.5), #hjust shifts right
	legend.position=c(0.7,0.9), 
	legend.key = element_rect(fill = NA, colour = NA),
	legend.margin=margin(t=0,r=0,b=0,l=0,unit = "pt"),
 	legend.box.margin=margin(t=0,r=0,b=0,l=0,unit = "pt"),
 	legend.key.height = unit(0.3, "cm"),
	legend.key.width = unit(0.3, "cm"),
 	legend.spacing.y = unit(0.3, 'cm'),
 	legend.spacing.x = unit(0.3, 'cm'),
 	legend.title = element_blank(),  
 	# legend.title = element_text(size = 11), 
 	legend.text = element_text(size = 10)
	)
	
	
size_point <- 0.3
size_hline <- 0.1
	

# -------------- gg_color_hue ------------------

gg_color_hue <- function(n) {
  hues = seq(15, 375, length = n + 1)
  hcl(h = hues, l = 65, c = 100)[1:n]
}

	


#----------------- Prepare human logP ---------------------


logP <- read.table("log10P_human.txt",header=TRUE,sep="\t",stringsAsFactors=FALSE)
human_thresh_95 <- read.table("human_thresh_95.txt",header=FALSE,sep="\t",stringsAsFactors=FALSE,row.names=1,col.names=c("","thresh"))



# Sort:
chrOrder<-paste("chr",c(1:22,"X"),sep="")
logP$Chromosome <- factor(logP$Chromosome, levels=chrOrder)
logP <- logP[order(logP$Chromosome, logP$pos), ]
logP$Chromosome <- as.character(logP$Chromosome)



# -------- read in g_unique, d_unique, Ix -------------

g_unique <- read.delim("growth_loci_unique.txt",header=TRUE,sep="\t",stringsAsFactors=FALSE,check.names=FALSE)
d_unique <- read.delim("paclitaxel_loci_unique.txt",header=TRUE,sep="\t",stringsAsFactors=FALSE,check.names=FALSE)
Ix <- read.delim("Ix_loci.txt",header=TRUE,sep="\t",stringsAsFactors=FALSE,check.names=FALSE)




# ------------- growth ---------------------------------


growth_coef_neg <- data.frame(coef=g_unique[g_unique$sig_coef < 0,"sig_coef"],type="Loci")
growth_coef_pos <- data.frame(coef=g_unique[g_unique$sig_coef >= 0,"sig_coef"],type="Loci")
growth_coef <- rbind(growth_coef_neg, growth_coef_pos)
growth_coef_null <- data.frame(coef=unlist(logP[seq(1,nrow(logP),100),c(17:21)]),type="Null")

growth_coef_neg_mean <- mean(g_unique[g_unique$sig_coef < 0,"sig_coef"])
growth_coef_pos_mean <- mean(g_unique[g_unique$sig_coef >= 0,"sig_coef"])
growth_coef_null_mean <- mean(unlist(logP[seq(1,nrow(logP),100),c(17:21)]))

growth_coef_neg_sd <- sd(g_unique[g_unique$sig_coef < 0,"sig_coef"])
growth_coef_pos_sd <- sd(g_unique[g_unique$sig_coef >= 0,"sig_coef"])
growth_coef_null_sd <- sd(unlist(logP[seq(1,nrow(logP),100),c(17:21)]))

density_scale_growth <- 0.18

bimodal <- function(x,rel_density1,mean1,sd1,mean2,sd2) {rel_density1 * dnorm(x, mean=mean1, sd=sd1)   +   dnorm(x, mean=mean2, sd=sd2) / (rel_density1+1)}
	
funcx_growth <- function(x) {
	density_scale_growth * bimodal(x,sum(g_unique$sig_coef >= 0)/length(g_unique$sig_coef),growth_coef_pos_mean, growth_coef_pos_sd,growth_coef_neg_mean, growth_coef_neg_sd)
	}
	
funcx_growth_null <- function(x) {
	density_scale_growth * bimodal(x,0,0,0,growth_coef_null_mean, growth_coef_null_sd)
	}


colores_growth <- gg_color_hue(length(unique(c("Loci","Null"))))
names(colores_growth) <- c("Loci","Null")



p1 <- ggplot() +
	  geom_histogram(
		  data = growth_coef, 
		  aes(
			  x = coef, 
			  y = density_scale_growth * ..density..,
			  fill = type
			  ),
		  show.legend=FALSE,
		  binwidth=0.05,
		  alpha=0.1
		  ) +
	  geom_histogram(
		  data = growth_coef_null, 
		  aes(
			  x = coef, 
			  y = density_scale_growth * ..density..,
			  fill = type
			  ),
		  show.legend=FALSE,
		  binwidth=0.05,
		  alpha=0.1
		  ) +
	stat_function(
		fun = funcx_growth, 
		aes(colour = colores_growth[2]),
		lwd =0.2
		)  +
	stat_function(
		fun = funcx_growth_null, 
		aes(colour = colores_growth[1]),
		lwd =0.2
		)  +
	scale_colour_manual(
		name = '', 
		values = as.vector(colores_growth),
		labels = names(colores_growth)
		) + 
	  # geom_line(
		  # data = growth_coef, 
		  # lwd =0.2, 
		  # aes(
			  # x = coef,
			  # y = ..density..,
			  # colour = type,
			  # group = type
			  # ),
		  # show.legend=TRUE,
		  # stat = 'density'
	  	  # ) + 
	  	# scale_linetype_manual(
			# values=as.vector(colores_growth),
			# labels=names(colores_growth),
			# name=NULL		
			# ) +
		# guides(
			# shape=FALSE,
		 		# fill = guide_legend(
				 		# override.aes = list(
				 		# #fill=NA,
				 		# shape=NA,
				 		# size=0.1
				 		# ),
			 		# ncol=1,
			 		# byrow=TRUE
			 		# )
		 		 # ) + 
	  theme2 + 
	  theme(
		  legend.position=c(0.8,0.9), 
		  axis.text.x=element_text(size=11), #numbers on tick marks of x axis
		  ) +
		scale_x_continuous(
			# labels = labels, 
			expand = expand_scale(add = 0)
			) +
		scale_y_continuous(
			expand = expand_scale(mult = .05)#,
			#limits=c(-2.2,6.25)
			) +
		# coord_cartesian(xlim = c(-0.07,0.04)) +
		xlab("Coefficient") + 
		ylab(expression(Density)) + 
		labs(subtitle="Growth")		
	print(p1)







# ------------- paclitaxel ---------------------------------




drug_coef_neg <- data.frame(coef=d_unique[d_unique$sig_coef < 0,"sig_coef"],type="Loci")
drug_coef_pos <- data.frame(coef=d_unique[d_unique$sig_coef >= 0,"sig_coef"],type="Loci")
drug_coef <- rbind(drug_coef_neg, drug_coef_pos)
drug_coef_null <- data.frame(coef=unlist(logP[seq(1,nrow(logP),100),c(22:27)]),type="Null")

drug_coef_neg_mean <- mean(d_unique[d_unique$sig_coef < 0,"sig_coef"])
drug_coef_pos_mean <- mean(d_unique[d_unique$sig_coef >= 0,"sig_coef"])
drug_coef_null_mean <- mean(unlist(logP[seq(1,nrow(logP),100),c(22:27)]))

drug_coef_neg_sd <- sd(d_unique[d_unique$sig_coef < 0,"sig_coef"])
drug_coef_pos_sd <- sd(d_unique[d_unique$sig_coef >= 0,"sig_coef"])
drug_coef_null_sd <- sd(unlist(logP[seq(1,nrow(logP),100),c(22:27)]))


density_scale_drug <- 0.012


bimodal <- function(x,rel_density1,mean1,sd1,mean2,sd2) {rel_density1 * dnorm(x, mean=mean1, sd=sd1)   +   dnorm(x, mean=mean2, sd=sd2) / (rel_density1+1)}
	
funcx_drug <- function(x) {
	density_scale_drug * bimodal(x,sum(d_unique$sig_coef >= 0)/length(d_unique$sig_coef),drug_coef_pos_mean, drug_coef_pos_sd,drug_coef_neg_mean, drug_coef_neg_sd)
	}
	
funcx_drug_null <- function(x) {
	density_scale_drug * bimodal(x,0,0,0,drug_coef_null_mean, drug_coef_null_sd)
	}



colores_drug <- gg_color_hue(length(unique(c("Loci","Null"))))
names(colores_drug) <- c("Loci","Null")




p2 <- ggplot() +
	  geom_histogram(
		  data = drug_coef, 
		  aes(
			  x = coef, 
			  y = density_scale_drug*..density..,
			  fill = type
			  ),
		  show.legend=FALSE,
		  binwidth=0.005,
		  alpha=0.1
		  ) +
	  geom_histogram(
		  data = drug_coef_null, 
		  aes(
			  x = coef, 
			  y = density_scale_drug*..density..,
			  fill = type
			  ),
		  show.legend=FALSE,
		  binwidth=0.005,
		  alpha=0.1
		  ) +
	stat_function(
		fun = funcx_drug, 
		aes(colour = colores_drug[2]),
		lwd =0.2
		)  +
	stat_function(
		fun = funcx_drug_null, 
		aes(colour = colores_drug[1]),
		lwd =0.2
		)  +
	scale_colour_manual(
		name = '', 
		values = as.vector(colores_drug),
		labels = names(colores_drug)
		) + 
	  # geom_line(
		  # data = growth_coef, 
		  # lwd =0.2, 
		  # aes(
			  # x = coef,
			  # y = ..density..,
			  # colour = type,
			  # group = type
			  # ),
		  # show.legend=TRUE,
		  # stat = 'density'
	  	  # ) + 
	  	# scale_linetype_manual(
			# values=as.vector(colores_1),
			# labels=names(colores_1),
			# name=NULL		
			# ) +
		# guides(
			# shape=FALSE,
		 		# fill = guide_legend(
				 		# override.aes = list(
				 		# #fill=NA,
				 		# shape=NA,
				 		# size=0.1
				 		# ),
			 		# ncol=1,
			 		# byrow=TRUE
			 		# )
		 		 # ) + 
	  theme2 + 
	  theme(
		  legend.position=c(0.25,0.9),
		  axis.text.x=element_text(size=9), #numbers on tick marks of x axis
		  ) +
		scale_x_continuous(
			# labels = labels, 
			expand = expand_scale(add = 0)
			) +
		scale_y_continuous(
			expand = expand_scale(mult = .05)#,
			#limits=c(-2.2,6.25)
			) +
		# coord_cartesian(xlim = c(-0.05,0.03)) +
		xlab("Coefficient") + 
		ylab(expression(Density)) + 
		labs(subtitle="Paclitaxel")		
	print(p2)



# ------------- Ix ---------------------------------




Ix_coef_neg <- data.frame(coef=Ix[Ix$sig_coef < 0,"sig_coef"],type="Loci")
Ix_coef_pos <- data.frame(coef=Ix[Ix$sig_coef >= 0,"sig_coef"],type="Loci")
Ix_coef <- rbind(Ix_coef_neg, Ix_coef_pos)
Ix_coef_null <- data.frame(coef=unlist(logP[seq(1,nrow(logP),100),c(28)]),type="Null")

Ix_coef_neg_mean <- mean(Ix[Ix$sig_coef < 0,"sig_coef"])
Ix_coef_pos_mean <- mean(Ix[Ix$sig_coef >= 0,"sig_coef"])
Ix_coef_null_mean <- mean(unlist(logP[seq(1,nrow(logP),100),c(28)]))

Ix_coef_neg_sd <- sd(Ix[Ix$sig_coef < 0,"sig_coef"])
Ix_coef_pos_sd <- sd(Ix[Ix$sig_coef >= 0,"sig_coef"])
Ix_coef_null_sd <- sd(unlist(logP[seq(1,nrow(logP),100),c(28)]))

density_scale_Ix <- 0.00362


bimodal <- function(x,rel_density1,mean1,sd1,mean2,sd2) {rel_density1 * dnorm(x, mean=mean1, sd=sd1)   +   dnorm(x, mean=mean2, sd=sd2) / (rel_density1+1)}
	
funcx_Ix <- function(x) {
	density_scale_Ix * bimodal(x,sum(Ix$sig_coef >= 0)/length(Ix$sig_coef),Ix_coef_pos_mean, Ix_coef_pos_sd,Ix_coef_neg_mean, Ix_coef_neg_sd)
	}
	
funcx_Ix_null <- function(x) {
	density_scale_Ix * bimodal(x,0,0,0,Ix_coef_null_mean, Ix_coef_null_sd)
	}





colores_Ix <- gg_color_hue(length(unique(c("Loci","Null"))))
names(colores_Ix) <- c("Loci","Null")








p3 <- ggplot() +
	  geom_histogram(
		  data = Ix_coef, 
		  aes(
			  x = coef, 
			  y = density_scale_Ix * ..density..,
			  fill = type
			  ),
		  show.legend=FALSE,
		  binwidth=0.001,
		  alpha=0.1
		  ) +
	  geom_histogram(
		  data = Ix_coef_null, 
		  aes(
			  x = coef, 
			  y = density_scale_Ix * ..density..,
			  fill = type
			  ),
		  show.legend=FALSE,
		  binwidth=0.001,
		  alpha=0.1
		  ) +
	stat_function(
		fun = funcx_Ix, 
		aes(colour = colores_Ix[2]),
		lwd =0.2
		)  +
	stat_function(
		fun = funcx_Ix_null, 
		aes(colour = colores_Ix[1]),
		lwd =0.2
		)  +
	scale_colour_manual(
		name = '', 
		values = as.vector(colores_Ix),
		labels = names(colores_Ix)
		) + 
	  # geom_line(
		  # data = Ix_coef, 
		  # lwd =0.2, 
		  # aes(
			  # x = coef,
			  # y = ..density..,
			  # colour = type,
			  # group = type
			  # ),
		  # show.legend=TRUE,
		  # stat = 'density'
	  	  # ) + 
	  	# scale_linetype_manual(
			# values=as.vector(colores_Ix),
			# labels=names(colores_Ix),
			# name=NULL		
			# ) +
		# guides(
			# shape=FALSE,
		 		# fill = guide_legend(
				 		# override.aes = list(
				 		# #fill=NA,
				 		# shape=NA,
				 		# size=0.1
				 		# ),
			 		# ncol=1,
			 		# byrow=TRUE
			 		# )
		 		 # ) + 
	  theme2 + 
	  theme(
		  axis.text.x=element_text(size=10), #numbers on tick marks of x axis
		  ) +
		scale_x_continuous(
			# labels = labels, 
			expand = expand_scale(add = 0)
			) +
		scale_y_continuous(
			expand = expand_scale(mult = .05)#,
			#limits=c(-2.2,6.25)
			) +
		# coord_cartesian(xlim = c(-0.07,0.04)) +
		xlab("Coefficient") + 
		ylab(expression(Density)) + 
		labs(subtitle="Ix")		
	print(p3)








# --------------- Combine panels ---------------------


pdf("coefs_1.pdf", width=7.5, height=3.33, useDingbats = FALSE)
plot_grid(p1, p2, p3, ncol = 3, nrow = 1, labels=c("A", "B", "C"), label_size = 16, align="h")
dev.off()


# png("coefs_1.png",width=7.5,height=10,units="in",res=300)
# plot_grid(p1, p2, p3,  ncol = 3, nrow = 3, labels=c("A", "B", "C"), label_size = 14, align="h")
# dev.off()


# ------------------- calx --------------------------




# ---------- g, d and Ix copy number changes -----------------



# Seven g_unique copy number go up

sum(g_unique$sig_coef < 0)
# [1] 852

sum(g_unique$sig_coef >= 0)
# [1] 7 <<<<<<<<<<<<< use in paper

length(g_unique$sig_coef < 0)
# [1] 859 <<<<<<<<<<<<< use in paper


# <<<<<<<<<< use in paper >>>>>>>>>>>>>>>>>>>>

g_unique[g_unique$sig_coef > 0,c("Chromosome","pos","conc","log10P","sig_coef","geneSymbol","gene_description")]
    # Chromosome       pos conc    log10P   sig_coef geneSymbol                                                          gene_description
# 397       chr7  84450000    0  7.529661 0.17798101     SEMA3A                         semaphorin 3A [Source:HGNC Symbol;Acc:HGNC:10723]
# 482       chr9 129820000    8 11.331008 0.19145432      TOR1A               torsin family 1 member A [Source:HGNC Symbol;Acc:HGNC:3098]
# 627      chr13  70180000    0  8.772300 0.12692569    ATXN8OS          ATXN8 opposite strand lncRNA [Source:HGNC Symbol;Acc:HGNC:10561]
# 661      chr15  20100000    0  9.705184 0.07755620  RN7SL584P RNA, 7SL, cytoplasmic 584, pseudogene [Source:HGNC Symbol;Acc:HGNC:46600]
# 742      chr17  78460000    8  7.532623 0.05351961     DNAH17         dynein axonemal heavy chain 17 [Source:HGNC Symbol;Acc:HGNC:2946]
# 743      chr17  78500000    0  7.729760 0.06255591 DNAH17-AS1                DNAH17 antisense RNA 1 [Source:HGNC Symbol;Acc:HGNC:48594]
# 826       chrX  56180000    0  9.032053 0.08650395 AL050309.1                                                          novel transcript

# ^^^^^^^^^^^^^^^^^ use in paper ^^^^^^^^^^^^^^^^^^^^^^^



# mean, sem, neg sig coef g loci
mean(g_unique[g_unique$sig_coef < 0,"sig_coef"])
# [1] -0.2376499

sem(g_unique[g_unique$sig_coef < 0,"sig_coef"])
# [1] 0.002407105

# mean, sem, pos sig coef g loci
mean(g_unique[g_unique$sig_coef >= 0,"sig_coef"])
# [1] 0.1109281

sem(g_unique[g_unique$sig_coef >= 0,"sig_coef"])
# [1] 0.02103416

# mean, sem, pos null g loci
mean(unlist(logP[seq(1,nrow(logP),100),c(17:21)]))
# [1] -0.14888

sem(unlist(logP[seq(1,nrow(logP),100),c(17:21)]))
# [1] 0.0007772435









# 31 of d_unique copy number go down and 7 go up

sum(d_unique$sig_coef < 0)
# [1] 31

sum(d_unique$sig_coef >= 0)
# [1] 7 <<<<<<<<<<<<< use in paper

length(d_unique$sig_coef)
# [1] 38 <<<<<<<<<<<<< use in paper


d_unique[d_unique$sig_coef>=0,c("Chromosome","pos","geneSymbol","gene_description")]
   # Chromosome       pos geneSymbol                                                                     gene_description
# 6        chr3  27020000      NEK10                           NIMA related kinase 10 [Source:HGNC Symbol;Acc:HGNC:18592]
# 18       chr9 133350000      RPL7A                            ribosomal protein L7a [Source:HGNC Symbol;Acc:HGNC:10364]
# 19      chr10  21790000     DNAJC1 DnaJ heat shock protein family (Hsp40) member C1 [Source:HGNC Symbol;Acc:HGNC:20090]
# 20      chr10  94390000    TBC1D12                     TBC1 domain family member 12 [Source:HGNC Symbol;Acc:HGNC:29082]
# 21      chr11  10930000  ZBED5-AS1                            ZBED5 antisense RNA 1 [Source:HGNC Symbol;Acc:HGNC:48646]
# 22      chr11  11450000    GALNT18 polypeptide N-acetylgalactosaminyltransferase 18 [Source:HGNC Symbol;Acc:HGNC:30488]
# 23      chr11  13060000 AC013762.1                                                                     novel transcript





# mean, sem, neg sig coef d loci
mean(d_unique[d_unique$sig_coef < 0,"sig_coef"])
# [1] -0.02131101

sem(d_unique[d_unique$sig_coef < 0,"sig_coef"])
# [1] 0.001668191

# mean, sem, pos sig coef d loci
mean(d_unique[d_unique$sig_coef >= 0,"sig_coef"])
# [1] 0.0168023

sem(d_unique[d_unique$sig_coef >= 0,"sig_coef"])
# [1] 0.002447047

# mean, sem, pos null d loci
mean(unlist(logP[seq(1,nrow(logP),100),c(22:27)]))
# [1] -0.0001953165

sem(unlist(logP[seq(1,nrow(logP),100),c(22:27)]))
# [1] 4.471321e-05





# Number of d loci go up significantly greater than g loci

chisq.test(x = c(7, 31),p=c(7, 852),rescale.p=TRUE)

	# Chi-squared test for given probabilities

# data:  c(7, 31)
# X-squared = 145.73, df = 1, p-value < 2.2e-16

# Warning message:
# In chisq.test(x = c(26, 115), p = c(4, 1158), rescale.p = TRUE) :
  # Chi-squared approximation may be incorrect

chisq.test(x = c(7, 31),p=c(7, 852),rescale.p=TRUE)$p.value
# [1] 1.484132e-33
# Warning message:
# In chisq.test(x = c(26, 115), p = c(4, 1158), rescale.p = TRUE) :
  # Chi-squared approximation may be incorrect
  
  





# <<<<<<<<< use in paper >>>>>>>>>>>>>>>>>>>

library(XNomial)


xmulti(obs = c(7, 31),expr=c(7, 852),detail=3)$pLLR

# P value  (LLR)  =  2.413e-08
# P value (Prob)  =  2.413e-08
# P value (Chisq) =  2.413e-08

# Observed:  7 31 
# Expected ratio:  7 852 
# Total number of tables:  39 
# [1] 2.413395e-08




xmulti(obs = c(7, 31),expr=c(7, 852),detail=3)$observedLLR

# P value  (LLR)  =  2.413e-08
# P value (Prob)  =  2.413e-08
# P value (Chisq) =  2.413e-08

# Observed:  7 31 
# Expected ratio:  7 852 
# Total number of tables:  39 
# [1] -15.76937

# ^^^^^^^^^^^ use in paper ^^^^^^^^^^^^^^^^^^






# 2 out of 62 Ix copy number go up

sum(Ix$sig_coef < 0) 
# [1] 60

sum(Ix$sig_coef >= 0)
# [1] 2 <<<<<<<<<< use in paper

length(Ix$sig_coef < 0)
# [1] 62 <<<<<<<<<< use in paper





 # <<<<<<<<<<<<<< use in paper >>>>>>>>>>>>>>>>>>>

Ix[Ix$sig_coef>=0,c("Chromosome","pos","geneSymbol","gene_description")]
   # Chromosome       pos geneSymbol                                                                gene_description
# 13       chr3 155220000  LINC01487 long intergenic non-protein coding RNA 1487 [Source:HGNC Symbol;Acc:HGNC:51139]
# 35      chr11  11030000 AC111188.1                                                                novel transcript



# ^^^^^^^^^^^^^^ use in paper ^^^^^^^^^^^^^^^^




 # <<<<<<<<<<<<<< use in paper >>>>>>>>>>>>>>>>>>>

# mean, sem, neg sig coef Ix loci
mean(Ix[Ix$sig_coef < 0,"sig_coef"])
# [1] -0.005158012

sem(Ix[Ix$sig_coef < 0,"sig_coef"])
# [1] 0.0001668766

# mean, sem, pos sig coef Ix loci
mean(Ix[Ix$sig_coef >= 0,"sig_coef"])
# [1] 0.00537763

sem(Ix[Ix$sig_coef >= 0,"sig_coef"])
# [1] 0.000335809

# mean, sem, pos null Ix loci
mean(unlist(logP[seq(1,nrow(logP),100),c(28)]))
# [1] -0.0008914604

sem(unlist(logP[seq(1,nrow(logP),100),c(28)]))
# [1] 3.495612e-05


 # <<<<<<<<<<<<<< use in paper >>>>>>>>>>>>>>>>>>>





























