library(Rmisc)
library(ggplot2)
library(cowplot) #used with plot_grid 


#----------------Aesthetics ---------------------------


theme2 <- theme(
	plot.margin = unit(c(t=1,r=1,b=1,l=1), "cm"),
	panel.grid.major = element_blank(), 
	panel.grid.minor = element_blank(), 
	panel.background = element_blank(), 
	legend.position="none", 
	axis.line.x = element_line(colour = "black", size = 0.1), 
	axis.line.y = element_line(colour = "black", size = 0.1), 
	axis.ticks = element_line(colour = "black", size = 0.1),
	axis.text.x=element_text(size=12), #numbers on tick marks of x axis
	axis.text.y=element_text(size=12), #numbers on tick marks of x axis
	axis.title=element_text(size=14), #titles of x and y axes
	axis.title.y=element_text(margin=margin(0,13,0,0)), #moves y axis title by adding margin space to bottom
	axis.title.x=element_text(margin=margin(13,0,0,0)),  #moves x axis title by adding margin space to top
	# plot.title = element_text(size=32, face="bold", hjust = -0.14), #can provide "A","B", by ggtitle, but used plot_grid wch can shift more left
	plot.subtitle = element_text(size=14, face="plain", hjust = 0.5, margin=margin(0,0,13,0)) #hjust shifts right
	)

gg_color_hue <- function(n) {
  hues = seq(15, 375, length = n + 1)
  hcl(h = hues, l = 65, c = 100)[1:n]
}


size_point <- 0.3
size_hline <- 0.2


# balloon_scale <- 5 # inflation factor for significant points	
# size_point <- 0.1*(1 + balloon_scale*(logP$log10p_g_avg/max(logP$log10p_g_avg))) # scale significant points



#---------gtex_cr_summary CI plot (1)------------------------


# GTEx_Analysis_2017-06-05_v8_RNASeQCv1.1.9_gene_median_tpm.gct downloaded from GTEx website on 09/10/19


gtex <- read.delim("GTEx_Analysis_2017-06-05_v8_RNASeQCv1.1.9_gene_median_tpm.gct",sep="\t",stringsAsFactors=FALSE,header=TRUE,check.names=TRUE,fill=FALSE,skip=2)

colnames(gtex)[1] <- "gene_id"

# remove version number in gene_id
gtex$gene_id <- gsub("\\..$","",gtex$gene_id)


# gtex$mean <- 2^rowMeans(log2(gtex[,c(3:ncol(gtex))]))


gencode_gtf_ensembl_ucsc <- read.delim("gencode_gtf_ensembl_ucsc_v31.txt",header=TRUE,sep="\t",stringsAsFactors=FALSE,check.names=FALSE)

gtex <- merge(gtex, gencode_gtf_ensembl_ucsc[,c("gene_id","gene_type")])


g_unique <- read.delim("growth_loci_unique.txt",sep="\t",stringsAsFactors=FALSE,header=TRUE,check.names=FALSE)



gtex$RH <- numeric(nrow(gtex))
gtex[gtex$gene_id %in% g_unique$ensembl_gene_id,"RH"] <- 1


# Convert TPM to log2(TPM+1) for viewing
# gtex[,c(3:55)] <- log2(gtex[,c(3:55)] + 1)
# However, led to strange behavior such as median outside 95% CI using t.test for some tissues. 
# So thresholded by discarding TPM < 5 vals, ie keeping TPM >= 5 (see below). Similar to microrarray spots not above bkgrd being discarded.



# Find most significant tissue
gtex_mean_diff_cr_CI <- data.frame(
							threshold = numeric(),
							tissue = character(),
							nRHpos = numeric(),
							nRHneg = numeric(),
							median_diff = numeric(),
							mean_diff=numeric(),
							mean_CI_1 = numeric(),
							mean_CI_2 = numeric(), 
							P = numeric(), 
							stringsAsFactors=FALSE
							)








# # Use if want to explore TPM thresholds, eg
# thresh_key <- c(1:20)



thresh_key <- 5 # equiv to ~1 transcript per cell with typical 200,000 transcripts.
i_key <- c(3:(ncol(gtex)-2))




for(thresh in c(1:length(thresh_key))) {
	
	cat(paste0("\nthresh = ", thresh, "/",length(thresh_key), "\n"))
	
	for(i in c(1:length(i_key))) {
		
		
		cat(paste0("\ti = ",i,"/",length(i_key), "\n"))
		
		gtex_mean_diff_cr_CI[c((thresh-1)*length(i_key) + i),"threshold"] <- thresh_key[thresh]
		gtex_mean_diff_cr_CI[c((thresh-1)*length(i_key) + i),"tissue"] <- colnames(gtex)[i_key[i]]
		
		gtex_mean_diff_cr_CI[c((thresh-1)*length(i_key) + i),"nRHpos"] <- length(gtex[gtex[,i_key[i]] >= thresh_key[thresh],][gtex[gtex[,i_key[i]] >= thresh_key[thresh],]$gene_type == "protein_coding" & gtex[gtex[,i_key[i]] >= thresh_key[thresh],]$RH==1, i_key[i]])
		
		gtex_mean_diff_cr_CI[c((thresh-1)*length(i_key) + i),"nRHneg"] <- length(gtex[gtex[,i_key[i]] >= thresh_key[thresh],][gtex[gtex[,i_key[i]] >= thresh_key[thresh],]$gene_type == "protein_coding" & gtex[gtex[,i_key[i]] >= thresh_key[thresh],]$RH==0, i_key[i]])
		
		gtex_mean_diff_cr_CI[c((thresh-1)*length(i_key) + i),"median_diff"] <- median(log2(gtex[gtex[,i_key[i]] >= thresh_key[thresh],][gtex[gtex[,i_key[i]] >= thresh_key[thresh],]$gene_type == "protein_coding" & gtex[gtex[,i_key[i]] >= thresh_key[thresh],]$RH==1, i_key[i]]))-median(log2(gtex[gtex[,i_key[i]] >= thresh_key[thresh],][gtex[gtex[,i_key[i]] >= thresh_key[thresh],]$gene_type == "protein_coding" & gtex[gtex[,i_key[i]] >= thresh_key[thresh],]$RH==0, i_key[i]]))

			
		tryCatch ({
		
		ans <- t.test(log2(gtex[gtex[,i_key[i]] >= thresh_key[thresh],][gtex[gtex[,i_key[i]] >= thresh_key[thresh],]$gene_type == "protein_coding" & gtex[gtex[,i_key[i]] >= thresh_key[thresh],]$RH==1, i_key[i]]), log2(gtex[gtex[,i_key[i]] >= thresh_key[thresh],][gtex[gtex[,i_key[i]] >= thresh_key[thresh],]$gene_type == "protein_coding" & gtex[gtex[,i_key[i]] >= thresh_key[thresh],]$RH==0, i_key[i]]))
		
		
		gtex_mean_diff_cr_CI[c((thresh-1)*length(i_key) + i),"mean_diff"] <- ans$estimate["mean of x"]-ans$estimate["mean of y"]
		gtex_mean_diff_cr_CI[c((thresh-1)*length(i_key) + i),"mean_CI_1"] <- ans$conf.int[1]
		gtex_mean_diff_cr_CI[c((thresh-1)*length(i_key) + i),"mean_CI_2"] <- ans$conf.int[2]
		gtex_mean_diff_cr_CI[c((thresh-1)*length(i_key) + i),"P"] <- ans$p.value
		
		}, error = function(e) {cat ("Error on line ", i, ": ", conditionMessage(e),"\n")})

	}
}



# # omit pancreas, because insufficient data points for nc genes.
# gtex_mean_diff_cr_CI <- gtex_mean_diff_cr_CI[gtex_mean_diff_cr_CI$tissue != "Pancreas",]




# Zero mean diffs > 0
sum(gtex_mean_diff_cr_CI$mean_diff >= 0)
# [1] 0

# out of 54 tissues
dim(gtex_mean_diff_cr_CI)
# [1] 54  9



# gtex_mean_diff_cr_CI$fdr <- p.adjust(gtex_mean_diff_cr_CI$P,method="BH")
# gtex_mean_diff_cr_CI$fdr_sig <- 0
# gtex_mean_diff_cr_CI[gtex_mean_diff_cr_CI$fdr < 0.05,"fdr_sig"] <- 1


# gtex_mean_diff_cr_CI$fdr <- p.adjust(gtex_mean_diff_cr_CI$P,method="BH")
gtex_mean_diff_cr_CI$P_sig <- 0
gtex_mean_diff_cr_CI[gtex_mean_diff_cr_CI$P < 0.05,"P_sig"] <- 1



# ---------- Aggregate across tissues to perform summary statistics -------------------

gtex_l <- reshape(gtex, 
   varying = colnames(gtex)[3:(ncol(gtex)-2)], 
   v.names = "tpm",
   timevar = "tissue", 
   times = colnames(gtex)[3:(ncol(gtex)-2)], 
   new.row.names = 1:1e7,
   direction = "long")

dim(gtex_l)
# [1] 2322756       7



# select cr genes and threshold tpm >= 5
gtex_l_cr <- gtex_l[gtex_l$gene_type == "protein_coding" & gtex_l$tpm >= 5,]

dim(gtex_l_cr)
# [1] 127267      7


# number of tissues
length(unique(gtex_l_cr$tissue))
# [1] 54


head(gtex_l_cr)
          # gene_id Description      gene_type RH                 tissue     tpm id
# 1 ENSG00000000005        TNMD protein_coding  0 Adipose...Subcutaneous 27.2554  1
# 2 ENSG00000001561       ENPP4 protein_coding  0 Adipose...Subcutaneous 18.7328  2
# 3 ENSG00000001629      ANKIB1 protein_coding  0 Adipose...Subcutaneous 20.4222  3
# 5 ENSG00000002933    TMEM176A protein_coding  0 Adipose...Subcutaneous 55.4147  5
# 6 ENSG00000003056        M6PR protein_coding  0 Adipose...Subcutaneous 54.1797  6
# 7 ENSG00000003137     CYP26B1 protein_coding  1 Adipose...Subcutaneous 35.6353  7




gtex_l_cr_agg <- aggregate(log2(tpm)~gene_id,data= gtex_l_cr,FUN=mean)
gtex_l_cr_agg$RH <- 0
gtex_l_cr_agg[gtex_l_cr_agg$gene_id %in% g_unique$ensembl_gene_id,"RH"] <- 1



# Add summary statistics to gtex_mean_diff_cr_CI
summary_cr <- t.test(gtex_l_cr_agg[gtex_l_cr_agg$RH==1,"log2(tpm)"],gtex_l_cr_agg[gtex_l_cr_agg$RH==0,"log2(tpm)"])


new_row <- list(5,"Summary",sum(gtex_l_cr$RH==1),sum(gtex_l_cr$RH==0),NA,summary_cr$estimate[1]-summary_cr$estimate[2],summary_cr$conf.int[1],summary_cr$conf.int[2], summary_cr$p.value,2)

gtex_mean_diff_cr_CI <- rbind(gtex_mean_diff_cr_CI,new_row)
rownames(gtex_mean_diff_cr_CI) <- NULL




gtex_mean_diff_cr_CI$tissue <- factor(gtex_mean_diff_cr_CI$tissue,levels=gtex_mean_diff_cr_CI$tissue)

gtex_mean_diff_cr_CI$P_sig <- factor(gtex_mean_diff_cr_CI$P_sig,levels=c("0","1","2"))


labels_1 <- c(expression(n.s.), expression(italic(P)<0.05),expression(italic(P)==0.02))


colores_1 <- c("black","blue","red")
names(colores_1) <- c("0","1","2")



p1 <- ggplot() + 
		theme2 + 
		theme(legend.key=element_blank()) +
		# geom_line(
			# data=summary_retent_1, 
			# lwd=0.2,
			# aes(
				# x=summary_retent_1$jitter, 
				# y= median_retent_seq, 
				# colour=as.factor(summary_retent_1$pool)),
				# show.legend=FALSE
				# ) + 
		geom_linerange(
			data= gtex_mean_diff_cr_CI,
			aes(
				x=tissue, 
				ymin= mean_CI_1, 
				ymax= mean_CI_2,
				colour=P_sig
				),
			
			lwd=0.2,
			show.legend=TRUE
			) +
		geom_point(
			data= gtex_mean_diff_cr_CI, 
			aes(
				x= tissue, 
				y= mean_diff,
				colour= P_sig
				),
			shape=16,
			stroke=0.2,
			size=1.0,
			show.legend=TRUE
			) +
		geom_hline(
			yintercept=0, 
			linetype="solid", 
            color = "darkgreen", 
            size=0.2
            ) +
		scale_colour_manual(
			values=as.vector(colores_1),
			labels=labels_1,
			name=NULL		
			) +
		guides(
			shape=FALSE,
	 		fill = guide_legend(
				 		override.aes = list(
				 		#fill=NA,
				 		shape=1,
				 		size=NA
				 		),
			 		ncol=1,
			 		byrow=TRUE
			 		)
	 		 ) +
 		theme(
 			axis.title.x=element_blank(),
	        # axis.text.x=element_blank(),
	        # axis.ticks.x=element_blank(),
	 		legend.position = c(0.9,1.1),
			legend.margin=margin(t=0,r=0,b=0,l=0,unit = "pt"),
		 	legend.box.margin=margin(t=0,r=0,b=0,l=0,unit = "pt"),
		 	legend.key.height = unit(0.3, "cm"),
			legend.key.width = unit(0.3, "cm"),
		 	legend.spacing.y = unit(0.3, 'cm'),
		 	legend.spacing.x = unit(0.3, 'cm'),
		 	legend.title = element_text(size = 11),  
		 	legend.text = element_text(size = 10),
	 		legend.title.align=0.7,
	 		legend.text.align = 0,
 			axis.text.x = element_text(
	 			size = 8,
	 			angle = 90, 
	 			hjust = 1,
	 			vjust = 0.3,
	 			color =  c(rep("black",c(nrow(gtex_mean_diff_cr_CI)-1)),"red")
	 			)
 			) +
		scale_x_discrete(
			# labels = labels, 
			expand = expand_scale(add = 1.1)
			) +
		# xlab("Weeks") + 
		ylab(expression(Delta*log[2]*'('*TPM*')')) +
		labs(subtitle="GTEx cr")
print(p1)




#---------gtex_nc_summary CI plot (2)------------------------


# GTEx_Analysis_2017-06-05_v8_RNASeQCv1.1.9_gene_median_tpm.gct downloaded from GTEx website on 09/10/19


gtex <- read.delim("GTEx_Analysis_2017-06-05_v8_RNASeQCv1.1.9_gene_median_tpm.gct",sep="\t",stringsAsFactors=FALSE,header=TRUE,check.names=TRUE,fill=FALSE,skip=2)

colnames(gtex)[1] <- "gene_id"

# remove version number in gene_id
gtex$gene_id <- gsub("\\..$","",gtex$gene_id)


# gtex$mean <- 2^rowMeans(log2(gtex[,c(3:ncol(gtex))]))


gencode_gtf_ensembl_ucsc <- read.delim("gencode_gtf_ensembl_ucsc_v31.txt",header=TRUE,sep="\t",stringsAsFactors=FALSE,check.names=FALSE)

gtex <- merge(gtex, gencode_gtf_ensembl_ucsc[,c("gene_id","gene_type")])


g_unique <- read.delim("growth_loci_unique.txt",sep="\t",stringsAsFactors=FALSE,header=TRUE,check.names=FALSE)



gtex$RH <- numeric(nrow(gtex))
gtex[gtex$gene_id %in% g_unique$ensembl_gene_id,"RH"] <- 1


# Convert TPM to log2(TPM+1) for viewing
# gtex[,c(3:55)] <- log2(gtex[,c(3:55)] + 1)
# However, led to strange behavior such as median outside 95% CI using t.test for some tissues. 
# So thresholded by discarding TPM < 5 vals, ie keeping TPM >= 5 (see below). Similar to microrarray spots not above bkgrd being discarded.



# Find most significant tissue
gtex_mean_diff_nc_CI <- data.frame(
							threshold = numeric(),
							tissue = character(),
							nRHpos = numeric(),
							nRHneg = numeric(),
							median_diff = numeric(),
							mean_diff=numeric(),
							mean_CI_1 = numeric(),
							mean_CI_2 = numeric(), 
							P = numeric(), 
							stringsAsFactors=FALSE
							)








# # Use if want to explore TPM thresholds, eg
# thresh_key <- c(1:20)



thresh_key <- 5 # equiv to ~1 transcript per cell with typical 200,000 transcripts.
i_key <- c(3:(ncol(gtex)-2))




for(thresh in c(1:length(thresh_key))) {
	
	cat(paste0("\nthresh = ", thresh, "/",length(thresh_key), "\n"))
	
	for(i in c(1:length(i_key))) {
		
		
		cat(paste0("\ti = ",i,"/",length(i_key), "\n"))
		
		gtex_mean_diff_nc_CI[c((thresh-1)*length(i_key) + i),"threshold"] <- thresh_key[thresh]
		gtex_mean_diff_nc_CI[c((thresh-1)*length(i_key) + i),"tissue"] <- colnames(gtex)[i_key[i]]
		
		gtex_mean_diff_nc_CI[c((thresh-1)*length(i_key) + i),"nRHpos"] <- length(gtex[gtex[,i_key[i]] >= thresh_key[thresh],][gtex[gtex[,i_key[i]] >= thresh_key[thresh],]$gene_type != "protein_coding" & gtex[gtex[,i_key[i]] >= thresh_key[thresh],]$RH==1, i_key[i]])
		
		gtex_mean_diff_nc_CI[c((thresh-1)*length(i_key) + i),"nRHneg"] <- length(gtex[gtex[,i_key[i]] >= thresh_key[thresh],][gtex[gtex[,i_key[i]] >= thresh_key[thresh],]$gene_type != "protein_coding" & gtex[gtex[,i_key[i]] >= thresh_key[thresh],]$RH==0, i_key[i]])
		
		gtex_mean_diff_nc_CI[c((thresh-1)*length(i_key) + i),"median_diff"] <- median(log2(gtex[gtex[,i_key[i]] >= thresh_key[thresh],][gtex[gtex[,i_key[i]] >= thresh_key[thresh],]$gene_type != "protein_coding" & gtex[gtex[,i_key[i]] >= thresh_key[thresh],]$RH==1, i_key[i]]))-median(log2(gtex[gtex[,i_key[i]] >= thresh_key[thresh],][gtex[gtex[,i_key[i]] >= thresh_key[thresh],]$gene_type != "protein_coding" & gtex[gtex[,i_key[i]] >= thresh_key[thresh],]$RH==0, i_key[i]]))

			
		tryCatch ({
		
		ans <- t.test(log2(gtex[gtex[,i_key[i]] >= thresh_key[thresh],][gtex[gtex[,i_key[i]] >= thresh_key[thresh],]$gene_type != "protein_coding" & gtex[gtex[,i_key[i]] >= thresh_key[thresh],]$RH==1, i_key[i]]), log2(gtex[gtex[,i_key[i]] >= thresh_key[thresh],][gtex[gtex[,i_key[i]] >= thresh_key[thresh],]$gene_type != "protein_coding" & gtex[gtex[,i_key[i]] >= thresh_key[thresh],]$RH==0, i_key[i]]))
		
		
		gtex_mean_diff_nc_CI[c((thresh-1)*length(i_key) + i),"mean_diff"] <- ans$estimate["mean of x"]-ans$estimate["mean of y"]
		gtex_mean_diff_nc_CI[c((thresh-1)*length(i_key) + i),"mean_CI_1"] <- ans$conf.int[1]
		gtex_mean_diff_nc_CI[c((thresh-1)*length(i_key) + i),"mean_CI_2"] <- ans$conf.int[2]
		gtex_mean_diff_nc_CI[c((thresh-1)*length(i_key) + i),"P"] <- ans$p.value
		
		}, error = function(e) {cat ("Error on line ", i, ": ", conditionMessage(e),"\n")})

	}
}



# omit pancreas, because insufficient data points for nc genes.
gtex_mean_diff_nc_CI <- gtex_mean_diff_nc_CI[gtex_mean_diff_nc_CI$tissue != "Pancreas",]


# omit liver and stomach, because n =2  for nc genes and give massive distractingly large CIs that encompass zero.
gtex_mean_diff_nc_CI <- gtex_mean_diff_nc_CI[gtex_mean_diff_nc_CI$tissue != "Liver",]
gtex_mean_diff_nc_CI <- gtex_mean_diff_nc_CI[gtex_mean_diff_nc_CI$tissue != "Stomach",]




# 31 mean diffs > 0
sum(gtex_mean_diff_nc_CI$mean_diff >= 0)
# [1] 31

# out of 51 tissues
dim(gtex_mean_diff_nc_CI)
# [1] 51  7



# gtex_mean_diff_nc_CI$fdr <- p.adjust(gtex_mean_diff_nc_CI$P,method="BH")
# gtex_mean_diff_nc_CI$fdr_sig <- 0
# gtex_mean_diff_nc_CI[gtex_mean_diff_nc_CI$fdr < 0.05,"fdr_sig"] <- 1


# gtex_mean_diff_nc_CI$fdr <- p.adjust(gtex_mean_diff_nc_CI$P,method="BH")
gtex_mean_diff_nc_CI$P_sig <- 0
gtex_mean_diff_nc_CI[gtex_mean_diff_nc_CI$P < 0.05,"P_sig"] <- 1



# select nc genes and threshold tpm >= 5
gtex_l_nc <- gtex_l[gtex_l$gene_type != "protein_coding" & gtex_l$tpm >= 5,]

dim(gtex_l_nc)
# 


gtex_l_nc_agg <- aggregate(log2(tpm)~gene_id,data= gtex_l_nc,FUN=mean)
gtex_l_nc_agg$RH <- 0
gtex_l_nc_agg[gtex_l_nc_agg$gene_id %in% g_unique$ensembl_gene_id,"RH"] <- 1



# Add summary statistics to gtex_mean_diff_nc_CI
summary_nc <- t.test(gtex_l_nc_agg[gtex_l_nc_agg$RH==1,"log2(tpm)"],gtex_l_nc_agg[gtex_l_nc_agg$RH==0,"log2(tpm)"])


new_row <- list(5,"Summary",sum(gtex_l_nc$RH==1),sum(gtex_l_nc$RH==0),NA,summary_nc$estimate[1]-summary_nc$estimate[2],summary_nc$conf.int[1],summary_nc$conf.int[2], summary_nc$p.value,2)

gtex_mean_diff_nc_CI <- rbind(gtex_mean_diff_nc_CI,new_row)
rownames(gtex_mean_diff_nc_CI) <- NULL




gtex_mean_diff_nc_CI$tissue <- factor(gtex_mean_diff_nc_CI$tissue,levels=gtex_mean_diff_nc_CI$tissue)

gtex_mean_diff_nc_CI$P_sig <- factor(gtex_mean_diff_nc_CI$P_sig,levels=c("0","1","2"))




labels_2 <- c(expression(n.s.), expression(italic(P)<0.05), expression(n.s.))


colores_2 <- c("black","blue","red")
names(colores_2) <- c("0","1","2")




p2 <- ggplot() + 
		theme2 + 
		theme(legend.key=element_blank()) +
		# geom_line(
			# data=summary_retent_1, 
			# lwd=0.2,
			# aes(
				# x=summary_retent_1$jitter, 
				# y= median_retent_seq, 
				# colour=as.factor(summary_retent_1$pool)),
				# show.legend=FALSE
				# ) + 
		geom_linerange(
			data= gtex_mean_diff_nc_CI,
			aes(
				x=tissue, 
				ymin= mean_CI_1, 
				ymax= mean_CI_2,
				colour=P_sig
				),
			
			lwd=0.2,
			show.legend=TRUE
			) +
		geom_point(
			data= gtex_mean_diff_nc_CI, 
			aes(
				x= tissue, 
				y= mean_diff,
				colour= P_sig
				),
			shape=16,
			stroke=0.2,
			size=1.0,
			show.legend=TRUE
			) +
		geom_hline(
			yintercept=0, 
			linetype="solid", 
            color = "darkgreen", 
            size=0.2
            ) +
		scale_colour_manual(
			values=as.vector(colores_2),
			labels=labels_2,
			name=NULL		
			) +
		guides(
			shape=FALSE,
	 		fill = guide_legend(
				 		override.aes = list(
				 		#fill=NA,
				 		shape=1,
				 		size=NA
				 		),
			 		ncol=1,
			 		byrow=TRUE
			 		)
	 		 ) +
 		theme(
 			axis.title.x=element_blank(),
	        # axis.text.x=element_blank(),
	        # axis.ticks.x=element_blank(),
	 		legend.position = c(0.9,1.1),
			legend.margin=margin(t=0,r=0,b=0,l=0,unit = "pt"),
		 	legend.box.margin=margin(t=0,r=0,b=0,l=0,unit = "pt"),
		 	legend.key.height = unit(0.3, "cm"),
			legend.key.width = unit(0.3, "cm"),
		 	legend.spacing.y = unit(0.3, 'cm'),
		 	legend.spacing.x = unit(0.3, 'cm'),
		 	legend.title = element_text(size = 11),  
		 	legend.text = element_text(size = 10),
	 		legend.title.align=0.7,
	 		legend.text.align = 0,
 			axis.text.x = element_text(
	 			size = 8,
	 			angle = 90, 
	 			hjust = 1,
	 			vjust = 0.3,
	 			color =  c(rep("black",c(nrow(gtex_mean_diff_nc_CI)-1)),"red")
	 			)
 			) +
		scale_x_discrete(
			# labels = labels, 
			expand = expand_scale(add = 1.1)
			) +
		# xlab("Weeks") + 
		ylab(expression(Delta*log[2]*'('*TPM*')')) +
		labs(subtitle="GTEx nc")
print(p2)




#------------------Make file --------------------------

(gtex_detail_2 <- plot_grid(p1, p2, labels=c("A", "B"), ncol = 1, nrow = 2, label_size = 16))

pdf("gtex_detail_2.pdf",width=7.5,height=10)
gtex_detail_2
dev.off()

























