# Mixed model to analyze human and hamster mito reads and copy number vs growth and paclitaxel
# Chose to normalize mitochondria based on total hamster sequence alignments (essentially mean in each 1 Mb window) rather than medians because use of mean is more common practice, a better measure of central tendency and, unlike retained human DNA with centromeres and TK1, there is hardly any worry about right skewed distribution for hamster genome.


library(lme4)
library(mgcv)
library(multcomp)
library(Rmisc)
library(ggplot2)
library(cowplot)
library(sjstats)
library(performance)
library(insight)
library(emmeans)


# --------------------- color hue --------------

gg_color_hue <- function(n) {
  hues = seq(15, 375, length = n + 1)
  hcl(h = hues, l = 65, c = 100)[1:n]
}



# --------------- get_legend function ---------------------------


get_legend<-function(myggplot){
  tmp <- ggplot_gtable(ggplot_build(myggplot))
   leg <- which(sapply(tmp$grobs, function(x) x$name) == "guide-box")
   legend <- tmp$grobs[[leg]]
   return(legend)
 }
 


# ------------- sem --------------------------
 
sem <- function(x) {sqrt(var(x,na.rm=TRUE)/sum(!is.na(x)))}


# ------------ compare ------------------------

compare <- function(a,b) {
	print(t.test(a,b))
	
	print(paste0("exact P value = ", t.test(a,b)$p.value))
	
	print(paste0("mean of a = ", mean(a, na.rm = TRUE)))
	print(paste0("sem of a = ", sem(a)))
	print(paste0("sd of a = ", sd(a, na.rm = TRUE)))
	print(paste0("number in a = ", sum(!is.na(a))))
	
	print(paste0("mean of b = ", mean(b, na.rm = TRUE)))
	print(paste0("sem of b = ", sem(b)))
	print(paste0("sd of b = ", sd(b, na.rm = TRUE)))
	print(paste0("number in b = ", sum(!is.na(b))))
	
}


# ----------------------------- ggplot ---------------------------------



# from https://stackoverflow.com/questions/8197559/emulate-ggplot2-default-color-palette
# function for default ggplot2 colors

gg_color_hue <- function(n) {
  hues = seq(15, 375, length = n + 1)
  hcl(h = hues, l = 65, c = 100)[1:n]
}

theme2 <- theme(
	plot.margin = unit(c(t=1.3,r=0.7,b=1.3,l=0.7), "cm"),
	panel.grid.major = element_blank(), 
	panel.grid.minor = element_blank(), 
	panel.background = element_blank(), 
	legend.position="none", 
	axis.line.x = element_line(colour = "black", size = 0.1), 
	axis.line.y = element_line(colour = "black", size = 0.1), 
	axis.ticks = element_line(colour = "black", size = 0.1),
	axis.text=element_text(size=12), #numbers on tick marks of x and y axes
	axis.title=element_text(size=14), #titles of x and y axes
	axis.title.y=element_text(margin=margin(t=0,r=7,b=0,l=0), hjust = 0.5), # moves y axis title by adding margin space to bottom, trbl
	axis.title.x=element_text(margin=margin(t=7,r=0,b=0,l=0), hjust = 0.5),  # moves x axis title by adding margin space to top,trbl
	plot.title = element_text(size=32, face="bold", hjust = -0.14), #can provide "A","B", by ggtitle, but used plot_grid wch can shift more left
	plot.subtitle = element_text(size=14, face="plain", hjust = 0.5), #hjust shifts right
	)
	
	
size_point <- 0.3
size_hline <- 0.1





# ---------- Human reads -------------------------

# RH_human_reads.txt saved from RH_pool against_hg38_fixed-11_24_18.xlsx
# Unlike RH_human_gseq.txt, RH_human_reads.txt has chrM

RH_human <- read.table("RH_human_reads.txt", header=TRUE, sep="\t", stringsAsFactors=FALSE)

dim(RH_human)
# [1] 323560    118


RH_human[1:10,1:10]
   # Chromosome Start    End RH1_w0_0 RH1_w1_0 RH1_w1_8 RH1_w1_25 RH1_w1_75 RH1_w2_0 RH1_w2_8
# 1        chr1     0  10000        0        0        0         0         0        0        0
# 2        chr1     0  20000        5        2        2         6         5        4        2
# 3        chr1     0  30000        6        5        4         8         6        5        3
# 4        chr1     0  40000        7        6        6         9         8        6        5
# 5        chr1     0  50000        7        8        7        10         8        7        7
# 6        chr1     0  60000        8       10        8        11         9        8        8
# 7        chr1     0  70000       10       11       11        12        10        9        9
# 8        chr1     0  80000       10       14       13        13        10        9        9
# 9        chr1     0  90000       10       16       14        13        10        9        9
# 10       chr1     0 100000       11       17       15        14        10       12       11


colnames(RH_human)[2:3] <- c("posS","posE")
RH_human$pos <- round(rowMeans(RH_human[,c("posS","posE")]))
RH_human <- RH_human[,c(1:3,ncol(RH_human),4:(ncol(RH_human)-1))]


# Rename columns
colnames(RH_human) <- gsub("_0$","_d0",colnames(RH_human))
colnames(RH_human) <- gsub("_8$","_d8",colnames(RH_human))
colnames(RH_human) <- gsub("_25$","_d25",colnames(RH_human))
colnames(RH_human) <- gsub("_75$","_d75",colnames(RH_human))


RH_human[1:10,1:10]
   # Chromosome posS   posE   pos RH1_w0_d0 RH1_w1_d0 RH1_w1_d8 RH1_w1_d25 RH1_w1_d75 RH1_w2_d0
# 1        chr1    0  10000  5000         0         0         0          0          0         0
# 2        chr1    0  20000 10000         5         2         2          6          5         4
# 3        chr1    0  30000 15000         6         5         4          8          6         5
# 4        chr1    0  40000 20000         7         6         6          9          8         6
# 5        chr1    0  50000 25000         7         8         7         10          8         7
# 6        chr1    0  60000 30000         8        10         8         11          9         8
# 7        chr1    0  70000 35000        10        11        11         12         10         9
# 8        chr1    0  80000 40000        10        14        13         13         10         9
# 9        chr1    0  90000 45000        10        16        14         13         10         9
# 10       chr1    0 100000 50000        11        17        15         14         10        12



RH_human[RH_human$Chromosome == "chrM",1:10]
       # Chromosome  posS  posE   pos RH1_w0_d0 RH1_w1_d0 RH1_w1_d8 RH1_w1_d25 RH1_w1_d75 RH1_w2_d0
# 323024       chrM     0 16569  8284        16        70        31         24         49         1
# 323025       chrM 10000 16569 13284         7         1         2          0          2         0


mito_reads <- RH_human[RH_human$Chromosome == "chrM",][1,]

# remove RH_human with mito_reads
rm(RH_human)

# replace with "mature" RH_human, lacking human mito reads and other unattached contigs
RH_human <- read.table("RH_human_gseq.txt",header=TRUE,sep="\t",stringsAsFactors=FALSE)





# ------------------ Read in and prepare ancillary tables -------------------------------

cell <- read.table("cell_label_info.txt",sep="\t",stringsAsFactors=FALSE,header=TRUE)
human_sum_reads <- colSums(RH_human[,5:ncol(RH_human)])
# could add hamster_sum_reads from hamster_gseq
human_reads <- read.table("RH_pool_human_total_align.txt",sep="\t",stringsAsFactors=FALSE,header=TRUE) # uses mapped human reads, cf human_AIC_1.R
hamster_reads <- read.table("RH_pool_hamster_total_align.txt",sep="\t",stringsAsFactors=FALSE,header=TRUE)





# -------- Prepare for lmer ---------------------------



mito_reads_l <- reshape(mito_reads, 
  varying = c(colnames(mito_reads[c(5:ncol(mito_reads))])), 
  v.names = "human_mito_reads",
  timevar = "RH_ID", 
  times = c(colnames(mito_reads[c(5:ncol(mito_reads))])), 
  new.row.names = 1:1e6,
  direction = "long")
  
mito_reads_l$week <- 0
mito_reads_l[grepl("_w0_",mito_reads_l$RH_ID),]$week <- 0
mito_reads_l[grepl("_w1_",mito_reads_l$RH_ID),]$week <- 1
mito_reads_l[grepl("_w2_",mito_reads_l$RH_ID),]$week <- 2
mito_reads_l[grepl("_w3_",mito_reads_l$RH_ID),]$week <- 3
mito_reads_l[grepl("_w4_",mito_reads_l$RH_ID),]$week <- 4
mito_reads_l[grepl("_w6_",mito_reads_l$RH_ID),]$week <- 6

mito_reads_l$conc <- 0
mito_reads_l[grepl("_d0",mito_reads_l$RH_ID),]$conc <- 0
mito_reads_l[grepl("_d8",mito_reads_l$RH_ID),]$conc <- 8
mito_reads_l[grepl("_d25",mito_reads_l$RH_ID),]$conc <- 25
mito_reads_l[grepl("_d75",mito_reads_l$RH_ID),]$conc <- 75

mito_reads_l$pool <- 0
mito_reads_l[grepl("RH1_",mito_reads_l$RH_ID),]$pool <- 1
mito_reads_l[grepl("RH2_",mito_reads_l$RH_ID),]$pool <- 2
mito_reads_l[grepl("RH3_",mito_reads_l$RH_ID),]$pool <- 3
mito_reads_l[grepl("RH4_",mito_reads_l$RH_ID),]$pool <- 4
mito_reads_l[grepl("RH5_",mito_reads_l$RH_ID),]$pool <- 5
mito_reads_l[grepl("RH6_",mito_reads_l$RH_ID),]$pool <- 6


mito_reads_l <- merge(mito_reads_l,cell)
mito_reads_l$human_sum_reads <- human_sum_reads[mito_reads_l$RH_ID]
mito_reads_l  <- merge(mito_reads_l,human_reads[,c(1:5,9)])
colnames(mito_reads_l)[13] <- "human_total_reads"
mito_reads_l  <- merge(mito_reads_l,hamster_reads[,c(1:5,9)])
colnames(mito_reads_l)[14] <- "hamster_total_reads"

mito_reads_l$pool <- as.factor(mito_reads_l$pool)
mito_reads_l$cell <- as.factor(mito_reads_l$cell)




# -------------- Add hamster mito reads -------------------


# RH_hamster_reads.txt saved from RH_PICR_RAW_ordered_fixed_11_29_18.xlsx
# Unlike RH_hamster_gseq.txt, RH_hamster_reads.txt has chrM

RH_hamster <- read.table("RH_hamster_reads.txt", header=TRUE, sep="\t", stringsAsFactors=FALSE)


dim(RH_hamster)
# [1] 237825    119

hamster_mito_reads <- RH_hamster[RH_hamster$Chromosome == "chromosomeM",][1,]


colnames(hamster_mito_reads)[3:4] <- c("posS","posE")
hamster_mito_reads$pos <- round(rowMeans(hamster_mito_reads[,c("posS","posE")]))
hamster_mito_reads <- hamster_mito_reads[,c(1:4,ncol(hamster_mito_reads),5:(ncol(hamster_mito_reads)-1))]



# Rename columns
colnames(hamster_mito_reads) <- gsub("_0$","_d0",colnames(hamster_mito_reads))
colnames(hamster_mito_reads) <- gsub("_8$","_d8",colnames(hamster_mito_reads))
colnames(hamster_mito_reads) <- gsub("_25$","_d25",colnames(hamster_mito_reads))
colnames(hamster_mito_reads) <- gsub("_75$","_d75",colnames(hamster_mito_reads))

# get rid of Contig_id
hamster_mito_reads <- hamster_mito_reads[2:length(hamster_mito_reads)]


hamster_mito_reads_l <- reshape(hamster_mito_reads, 
  varying = c(colnames(hamster_mito_reads[c(5:ncol(hamster_mito_reads))])), 
  v.names = "hamster_mito_reads",
  timevar = "RH_ID", 
  times = c(colnames(hamster_mito_reads[c(5:ncol(hamster_mito_reads))])), 
  new.row.names = 1:1e6,
  direction = "long")
  
hamster_mito_reads_l$week <- 0
hamster_mito_reads_l[grepl("_w0_",hamster_mito_reads_l$RH_ID),]$week <- 0
hamster_mito_reads_l[grepl("_w1_",hamster_mito_reads_l$RH_ID),]$week <- 1
hamster_mito_reads_l[grepl("_w2_",hamster_mito_reads_l$RH_ID),]$week <- 2
hamster_mito_reads_l[grepl("_w3_",hamster_mito_reads_l$RH_ID),]$week <- 3
hamster_mito_reads_l[grepl("_w4_",hamster_mito_reads_l$RH_ID),]$week <- 4
hamster_mito_reads_l[grepl("_w6_",hamster_mito_reads_l$RH_ID),]$week <- 6

hamster_mito_reads_l$conc <- 0
hamster_mito_reads_l[grepl("_d0",hamster_mito_reads_l$RH_ID),]$conc <- 0
hamster_mito_reads_l[grepl("_d8",hamster_mito_reads_l$RH_ID),]$conc <- 8
hamster_mito_reads_l[grepl("_d25",hamster_mito_reads_l$RH_ID),]$conc <- 25
hamster_mito_reads_l[grepl("_d75",hamster_mito_reads_l$RH_ID),]$conc <- 75

hamster_mito_reads_l$pool <- 0
hamster_mito_reads_l[grepl("RH1_",hamster_mito_reads_l$RH_ID),]$pool <- 1
hamster_mito_reads_l[grepl("RH2_",hamster_mito_reads_l$RH_ID),]$pool <- 2
hamster_mito_reads_l[grepl("RH3_",hamster_mito_reads_l$RH_ID),]$pool <- 3
hamster_mito_reads_l[grepl("RH4_",hamster_mito_reads_l$RH_ID),]$pool <- 4
hamster_mito_reads_l[grepl("RH5_",hamster_mito_reads_l$RH_ID),]$pool <- 5
hamster_mito_reads_l[grepl("RH6_",hamster_mito_reads_l$RH_ID),]$pool <- 6


mito_reads_l <- merge(mito_reads_l, hamster_mito_reads_l[,c("RH_ID","hamster_mito_reads","id","week","conc","pool")])



# ---------------------- Save hamster and human mito gseqs for archive (not done) ------------------------


# write.table(mito_reads_l[,c("RH_ID", "Chromosome", "posS", "posE", "pos", "pool", "conc", "week", "cell", "hamster_mito_reads")],"hamster_mito_gseq.txt",quote=FALSE,sep="\t",row.names=FALSE,col.names=TRUE)
# write.table(mito_reads_l[,c("RH_ID", "Chromosome", "posS", "posE", "pos", "pool", "conc", "week", "cell", "human_mito_reads")],"hamster_mito_gseq.txt",quote=FALSE,sep="\t",row.names=FALSE,col.names=TRUE)





# --------- use TK1 to quantitate human mito copy number ----------------- 

# -------- find middle TK1 -------------

gencode_gtf_ensembl_ucsc <- read.delim("gencode_gtf_ensembl_ucsc_v31.txt",header=TRUE,sep="\t",stringsAsFactors=FALSE,check.names=FALSE)




gencode_gtf_ensembl_ucsc[gencode_gtf_ensembl_ucsc$geneSymbol=="TK1",]
      # Chromosome         gene_id           tx_id geneSymbol strand    geneS    geneE geneLength txLength cdsLength    5utrS    5utrE 5utrDiff    3utrS    3utrE 3utrDiff
# 49401      chr17 ENSG00000167900 ENST00000588734        TK1      - 78174091 78187233      13143     1681       804 78186995 78187233      239 78174121 78174758      638
      # exonCount      gene_type                                       gene_description
# 49401         6 protein_coding thymidine kinase 1 [Source:HGNC Symbol;Acc:HGNC:11830]



TK1_coord <- mean(gencode_gtf_ensembl_ucsc[gencode_gtf_ensembl_ucsc$geneSymbol=="TK1","geneS"],gencode_gtf_ensembl_ucsc[gencode_gtf_ensembl_ucsc$geneSymbol=="TK1","geneE"])




# ----------- find peak TK1 reads ----------------

# Restrict search to within delta of TK1, because CEN and TEL becomes larger than TK1 in some samples
delta <- 1e6

RH_human_TK1 <- RH_human[RH_human$Chromosome=="chr17" & RH_human$pos >= TK1_coord-delta & RH_human$pos <= TK1_coord+delta,]

TK1_max <- apply(RH_human_TK1[,c(5:ncol(RH_human_TK1))],2,max)

mito_reads_l$TK1_max <- TK1_max[mito_reads_l$RH_ID]




# -------- Correct (ie decrease) TK1 retention by revertant freq -----


clone <- read.table("clone.txt",header=TRUE,sep="\t",stringsAsFactors=FALSE)

clone$revert_freq <- clone$reverts/clone$colonies

revert <- aggregate(revert_freq~Pool,data=clone,FUN=mean)

mito_reads_l <- merge(mito_reads_l,revert,by.x=c("pool"),by.y=c("Pool"),all.x=TRUE)


# ------------ Human mito copy number based on hamster reads ------------


# Follows logic in A23_HEK_mito_copy_num_1.R

# Human mito is 16569 bp from "RH_pool against_hg38_fixed-11_24_18.xlsx" (this is the actual mito sequence used for alignment, not the updated version on https://mitomap.org//bin/view.pl/MITOMAP/HumanMitoSeq, which is, nicely, also 16569 bp.)

human_mito_length <- 16569

# Hamster genome is 2368906908 bp  from https://www.ncbi.nlm.nih.gov/assembly/GCA_003668045.1/

hamster_geno_length <- 2368906908

# Assuming hamster genome in RH samples is diploid and human mito is haploid, human mito copy number based on hamster genome reads is:


mito_reads_l$human_mito_copy_hamster_reads <- 2*(mito_reads_l$human_mito_reads/(mito_reads_l$hamster_total_reads-mito_reads_l$hamster_mito_reads))*(hamster_geno_length/human_mito_length)





# ------------ Human mito copy number based on TK1 ------------


# Follows logic in A23_HEK_mito_copy_num_1.R

# Human mito is 16569 bp from "RH_pool against_hg38_fixed-11_24_18.xlsx" (this is the actual mito sequence used for alignment, not the updated version on https://mitomap.org//bin/view.pl/MITOMAP/HumanMitoSeq, which is, nicely, also 16569 bp.)

human_mito_length <- 16569

# TK1 window is 1e6 bp

TK1_length <- 1e6


# Calculate human mito copy number based on TK1 reads, correcting (downwards) for revertants following approach in graph_Human_retent_1.R. Note that human TK1 is haploid as is human mitochondrion.


mito_reads_l$human_mito_copy_TK1 <- 1*(mito_reads_l$human_mito_reads/mito_reads_l$TK1_max)*(TK1_length/human_mito_length)*(1-mito_reads_l$revert_freq)


# ------------ Human mito copy number based on mean of hamster reads and TK1 ------------



mito_reads_l$human_mean_mito_copy <- rowMeans(mito_reads_l[,c("human_mito_copy_hamster_reads","human_mito_copy_TK1")])



# ------------ Hamster mito copy number based on hamster reads ------------

# Follows logic in A23_HEK_mito_copy_num_1.R

# Hamster mito is 16283 bp from RH_PICR_RAW_ordered_fixed_11_29_18.xlsx (this is the actual mito sequence used for alignment, not the updated version on https://www.ncbi.nlm.nih.gov/nuccore/CM010855.1, which is, nicely, also 16283 bp.)

hamster_mito_length <- 16283

# Hamster genome is 2368906908 bp from https://www.ncbi.nlm.nih.gov/assembly/GCA_003668045.1/

hamster_geno_length <- 2368906908

# Assuming hamster genome in RH samples is diploid and hamster mito is haploid, hamster mito copy number based on hamster genome reads is:


mito_reads_l$hamster_mito_copy_hamster_reads <- 2*(mito_reads_l$hamster_mito_reads/(mito_reads_l$hamster_total_reads-mito_reads_l$hamster_mito_reads))*(hamster_geno_length/hamster_mito_length)




	
	


# # ---------------------------- Plot hamster mito copy num vs growth time stratified by paclitaxel conc --------------------


# summary_mito_hamster_1 <- summarySE(mito_reads_l, measurevar="hamster_mito_copy_hamster_reads", groupvars=c("week","conc"))
 


# # provide common x scale jitter for two overlaid ggplots
# summary_mito_hamster_1$jitter <- jitter(summary_mito_hamster_1$week,1)


# hamster_mito <- merge(mito_reads_l, summary_mito_hamster_1[c("week","conc","jitter")])


# # replicate week 0, drug 0 nM data point four times, to allow four graphs lines linking week 0 and week 1, instead of only one
# summary_mito_hamster_1 <- rbind(summary_mito_hamster_1[1,], summary_mito_hamster_1[1,], summary_mito_hamster_1[1,], summary_mito_hamster_1)
# summary_mito_hamster_1[1:4,"conc"] <- c(0,8,25,75)





# n = length(unique(summary_mito_hamster_1$conc))
# colores_1 = gg_color_hue(n)



# p1 <- ggplot() + 
		# theme2 + 
		# theme(legend.key=element_blank()) +
		# geom_line(
			# data= summary_mito_hamster_1, 
			# lwd=0.2,
			# aes(
				# x=jitter, 
				# y= hamster_mito_copy_hamster_reads, 
				# colour=as.factor(conc)),
				# show.legend=FALSE
				# ) + 
			# # remove 3 duplicated entries for 0w_d0, so color of error bar corresponds to 0 nM paclitaxel
		# geom_linerange(
			# data= summary_mito_hamster_1[c(1,5:nrow(summary_mito_hamster_1)),],
			# aes(
				# x=jitter, 
				# ymin= hamster_mito_copy_hamster_reads-se, 
				# ymax= hamster_mito_copy_hamster_reads +se,
				# colour=as.factor(conc)
				# ),
			# lwd=0.2,
			# show.legend=FALSE
			# ) +
		# geom_point(
			# shape=1,
			# stroke=0.4,
			# data= hamster_mito, 
			# aes(
				# x=jitter, 
				# y= hamster_mito_copy_hamster_reads, 
				# colour=as.factor(conc)),
				# size=1.0
				# ) +
		# scale_color_manual(
			# values=colores_1,
			# name ="Conc (nM)", 
			# labels=c(0,8,25,75)
			# ) +
	 		# guides(
	 			# colour = guide_legend(override.aes = list(fill=NA,shape=1,size=1),ncol=2,byrow=TRUE)
	 			# ) +
	 		# theme(
	 			# legend.position = c(0.73,1.00), 
	 			# legend.title = element_text(size = 6), 
	 			# legend.text = element_text(size = 5),
	 			# legend.title.align=0.7) +
		# scale_x_continuous(breaks = c(0,1,2,3,4,6), labels = c(0,1,2,3,4,6)) +
		# coord_cartesian(ylim = c(0,1250)) +
		# ggtitle("") + 
		# xlab("Weeks") + 
		# ylab(expression("Retention via alignments (%)")) + 
		# labs(subtitle="Growth")
# print(p1)
		
		



# ------------------------ Color scheme ------------------------------------------


n = length(unique(mito_reads_l$pool))
colores_1 = gg_color_hue(n)




# ---------------------------- (1) Hamster mito copy number vs growth time --------------------

## use if wish to display points collapsed across conc
# summary_mito_hamster_points <- summarySE(mito_reads_l, measurevar="hamster_mito_copy_hamster_reads", groupvars=c("week","pool"))
# summary_mito_hamster_lines <- summarySE(summary_mito_hamster_points, measurevar="hamster_mito_copy_hamster_reads", groupvars=c("week"))

summary_mito_hamster_lines_wk <- summarySE(mito_reads_l, measurevar="hamster_mito_copy_hamster_reads", groupvars=c("week"))


# # provide common x scale jitter for two overlaid ggplots
# summary_mito_hamster_1$jitter <- jitter(summary_mito_hamster_1$week,1)


# hamster_mito <- merge(mito_reads_l, summary_mito_hamster_1[c("week","jitter")])


# # replicate week 0, drug 0 nM data point four times, to allow four graphs lines linking week 0 and week 1, instead of only one
# summary_mito_hamster_1 <- rbind(summary_mito_hamster_1[1,], summary_mito_hamster_1[1,], summary_mito_hamster_1[1,], summary_mito_hamster_1)
# summary_mito_hamster_1[1:4,"conc"] <- c(0,8,25,75)





p1 <- ggplot() + 
		theme2 + 
		theme(legend.key=element_blank()) +
		geom_line(
			data= summary_mito_hamster_lines_wk, 
			lwd=0.2,
			colour="red",
			aes(
				x=week, 
				y= hamster_mito_copy_hamster_reads),
				show.legend=FALSE
				) + 
			# remove 3 duplicated entries for 0w_d0, so color of error bar corresponds to 0 nM paclitaxel
		geom_linerange(
			data= summary_mito_hamster_lines_wk,
			aes(
				x=week, 
				ymin= hamster_mito_copy_hamster_reads-se, 
				ymax= hamster_mito_copy_hamster_reads +se
				),
			colour="red",
			lwd=0.2,
			show.legend=FALSE
			) +
		geom_point(
			shape=1,
			stroke=0.4,
			data= mito_reads_l, 
			aes(
				x=jitter(week,factor=0.2), 
				y= hamster_mito_copy_hamster_reads, 
				colour=as.factor(pool)),
				size=1.0
				) +
		scale_color_manual(
			values=colores_1,
			name ="Pool", 
			labels=c(1,2,3,4,5,6)
			) +
	 		guides(
	 			colour = guide_legend(override.aes = list(fill=NA,shape=1,size=1),ncol=1,byrow=TRUE)
	 			) +
 		theme(
	 		plot.margin = unit(c(top=0.75,right=1.75,bottom=0.75,left=0.75), "cm"),
		 	# plot.subtitle = element_text(size=9, face="plain", hjust = 0.5, margin=margin(0,0,6,0)),
		 	# axis.title.x=element_blank(),
	        # axis.text.x=element_blank(),
	        # axis.ticks.x=element_blank(),
	 		legend.position="none",
			legend.margin=margin(t=0,r=0,b=0,l=0,unit = "pt"),
		 	legend.box.margin=margin(t=0,r=0,b=0,l=0,unit = "pt"),
		 	legend.key.height = unit(0.1, "cm"),
			legend.key.width = unit(0.1, "cm"),
		 	legend.spacing.y = unit(0.1, 'cm'),
		 	legend.spacing.x = unit(0.1, 'cm'),
		 	legend.title = element_text(size = 11),  
		 	legend.text = element_text(size = 10),
	 		legend.title.align=0.0
	 		) +
		scale_x_continuous(breaks = c(0,1,2,3,4,6), labels = c(0,1,2,3,4,6)) +
		coord_cartesian(ylim = c(0,1250)) + 
		xlab("Weeks") + 
		ylab(expression("Mito copy")) + 
		labs(subtitle="Hamster")
print(p1)




# ---------------------------- (2) Hamster mito copy number vs paclitaxel conc --------------------


summary_mito_hamster_lines_conc <- summarySE(mito_reads_l, measurevar="hamster_mito_copy_hamster_reads", groupvars=c("conc"))




p2 <- ggplot() + 
		theme2 + 
		theme(legend.key=element_blank()) +
		geom_line(
			data= summary_mito_hamster_lines_conc, 
			lwd=0.2,
			colour="red",
			aes(
				x=conc, 
				y= hamster_mito_copy_hamster_reads),
				show.legend=FALSE
				) + 
			# remove 3 duplicated entries for 0w_d0, so color of error bar corresponds to 0 nM paclitaxel
		geom_linerange(
			data= summary_mito_hamster_lines_conc,
			aes(
				x=conc, 
				ymin= hamster_mito_copy_hamster_reads-se, 
				ymax= hamster_mito_copy_hamster_reads +se
				),
			colour="red",
			lwd=0.2,
			show.legend=FALSE
			) +
		geom_point(
			shape=1,
			stroke=0.4,
			data= mito_reads_l, 
			aes(
				x=jitter(conc,factor=0.2), 
				y= hamster_mito_copy_hamster_reads, 
				colour=as.factor(pool)),
				size=1.0
				) +
		scale_color_manual(
			values=colores_1,
			name ="Pool", 
			labels=c(1,2,3,4,5,6)
			) +
	 		guides(
	 			colour = guide_legend(override.aes = list(fill=NA,shape=1,size=1),ncol=1,byrow=TRUE)
	 			) +
 		theme(
	 		plot.margin = unit(c(top=0.75,right=0.75,bottom=0.75,left=0.75), "cm"),
		 	# plot.subtitle = element_text(size=9, face="plain", hjust = 0.5, margin=margin(0,0,6,0)),
		 	# axis.title.x=element_blank(),
	        # axis.text.x=element_blank(),
	        # axis.ticks.x=element_blank(),
	 		legend.position="right",
			legend.margin=margin(t=0,r=0,b=0,l=0,unit = "pt"),
		 	legend.box.margin=margin(t=0,r=0,b=0,l=0,unit = "pt"),
		 	legend.key.height = unit(0.1, "cm"),
			legend.key.width = unit(0.1, "cm"),
		 	legend.spacing.y = unit(0.1, 'cm'),
		 	legend.spacing.x = unit(0.1, 'cm'),
		 	legend.title = element_text(size = 11),  
		 	legend.text = element_text(size = 10),
	 		legend.title.align=0.0
	 		) +
		scale_x_continuous(breaks = c(0,8,25,75), labels = c(0,8,25,75)) +
		coord_cartesian(ylim = c(0,1250)) + 
		xlab("Paclitaxel (nM)") + 
		ylab(expression("Mito copy")) + 
		labs(subtitle="Hamster")
print(p2)



# ---------------------------- (3) Human mito copy number (via aligments) vs growth time --------------------

## use if wish to display points collapsed across conc
# summary_mito_hamster_points <- summarySE(mito_reads_l, measurevar="hamster_mito_copy_hamster_reads", groupvars=c("week","pool"))
# summary_mito_hamster_lines <- summarySE(summary_mito_hamster_points, measurevar="hamster_mito_copy_hamster_reads", groupvars=c("week"))

summary_mito_human_lines_reads_wk <- summarySE(mito_reads_l, measurevar="human_mito_copy_hamster_reads", groupvars=c("week"))


# # provide common x scale jitter for two overlaid ggplots
# summary_mito_hamster_1$jitter <- jitter(summary_mito_hamster_1$week,1)


# hamster_mito <- merge(mito_reads_l, summary_mito_hamster_1[c("week","jitter")])


# # replicate week 0, drug 0 nM data point four times, to allow four graphs lines linking week 0 and week 1, instead of only one
# summary_mito_hamster_1 <- rbind(summary_mito_hamster_1[1,], summary_mito_hamster_1[1,], summary_mito_hamster_1[1,], summary_mito_hamster_1)
# summary_mito_hamster_1[1:4,"conc"] <- c(0,8,25,75)





p3 <- ggplot() + 
		theme2 + 
		theme(legend.key=element_blank()) +
		geom_line(
			data= summary_mito_human_lines_reads_wk, 
			lwd=0.2,
			colour="red",
			aes(
				x=week, 
				y= human_mito_copy_hamster_reads),
				show.legend=FALSE
				) + 
			# remove 3 duplicated entries for 0w_d0, so color of error bar corresponds to 0 nM paclitaxel
		geom_linerange(
			data= summary_mito_human_lines_reads_wk,
			aes(
				x=week, 
				ymin= human_mito_copy_hamster_reads - se, 
				ymax= human_mito_copy_hamster_reads + se
				),
			colour="red",
			lwd=0.2,
			show.legend=FALSE
			) +
		geom_point(
			shape=1,
			stroke=0.4,
			data= mito_reads_l, 
			aes(
				x=jitter(week,factor=0.2), 
				y= human_mito_copy_hamster_reads, 
				colour=as.factor(pool)),
				size=1.0
				) +
		scale_color_manual(
			values=colores_1,
			name ="Pool", 
			labels=c(1,2,3,4,5,6)
			) +
	 		guides(
	 			colour = guide_legend(override.aes = list(fill=NA,shape=1,size=1),ncol=1,byrow=TRUE)
	 			) +
 		theme(
	 		plot.margin = unit(c(top=0.75,right=1.75,bottom=0.75,left=0.75), "cm"),
		 	# plot.subtitle = element_text(size=9, face="plain", hjust = 0.5, margin=margin(0,0,6,0)),
		 	# axis.title.x=element_blank(),
	        # axis.text.x=element_blank(),
	        # axis.ticks.x=element_blank(),
	 		legend.position="none",
			legend.margin=margin(t=0,r=0,b=0,l=0,unit = "pt"),
		 	legend.box.margin=margin(t=0,r=0,b=0,l=0,unit = "pt"),
		 	legend.key.height = unit(0.1, "cm"),
			legend.key.width = unit(0.1, "cm"),
		 	legend.spacing.y = unit(0.1, 'cm'),
		 	legend.spacing.x = unit(0.1, 'cm'),
		 	legend.title = element_text(size = 11),  
		 	legend.text = element_text(size = 10),
	 		legend.title.align=0.0
	 		) +
		scale_x_continuous(breaks = c(0,1,2,3,4,6), labels = c(0,1,2,3,4,6)) +
		# coord_cartesian(ylim = c(0,1250)) + 
		xlab("Weeks") + 
		ylab(expression("Mito copy (align)")) + 
		labs(subtitle="Human")
print(p3)





# ---------------------------- (4) Human mito copy number (reads) vs paclitaxel conc --------------------


summary_mito_human_lines_reads_conc <- summarySE(mito_reads_l, measurevar="human_mito_copy_hamster_reads", groupvars=c("conc"))




p4 <- ggplot() + 
		theme2 + 
		theme(legend.key=element_blank()) +
		geom_line(
			data= summary_mito_human_lines_reads_conc, 
			lwd=0.2,
			colour="red",
			aes(
				x=conc, 
				y= human_mito_copy_hamster_reads),
				show.legend=FALSE
				) + 
		geom_linerange(
			data= summary_mito_human_lines_reads_conc,
			aes(
				x=conc, 
				ymin= human_mito_copy_hamster_reads-se, 
				ymax= human_mito_copy_hamster_reads +se
				),
			colour="red",
			lwd=0.2,
			show.legend=FALSE
			) +
		geom_point(
			shape=1,
			stroke=0.4,
			data= mito_reads_l, 
			aes(
				x=jitter(conc,factor=0.2), 
				y= human_mito_copy_hamster_reads, 
				colour=as.factor(pool)),
				size=1.0
				) +
		scale_color_manual(
			values=colores_1,
			name ="Pool", 
			labels=c(1,2,3,4,5,6)
			) +
	 		guides(
	 			colour = guide_legend(override.aes = list(fill=NA,shape=1,size=1),ncol=1,byrow=TRUE)
	 			) +
 		theme(
	 		plot.margin = unit(c(top=0.75,right=1.75,bottom=0.75,left=0.75), "cm"),
		 	# plot.subtitle = element_text(size=9, face="plain", hjust = 0.5, margin=margin(0,0,6,0)),
		 	# axis.title.x=element_blank(),
	        # axis.text.x=element_blank(),
	        # axis.ticks.x=element_blank(),
	 		legend.position="none",
			legend.margin=margin(t=0,r=0,b=0,l=0,unit = "pt"),
		 	legend.box.margin=margin(t=0,r=0,b=0,l=0,unit = "pt"),
		 	legend.key.height = unit(0.1, "cm"),
			legend.key.width = unit(0.1, "cm"),
		 	legend.spacing.y = unit(0.1, 'cm'),
		 	legend.spacing.x = unit(0.1, 'cm'),
		 	legend.title = element_text(size = 11),  
		 	legend.text = element_text(size = 10),
	 		legend.title.align=0.0
	 		) +
		scale_x_continuous(breaks = c(0,8,25,75), labels = c(0,8,25,75)) +
		# coord_cartesian(ylim = c(0,1250)) + 
		xlab("Paclitaxel (nM)") + 
		ylab(expression("Mito copy (align)")) + 
		labs(subtitle="Human")
print(p4)



# ---------------------------- (5) Human mito copy number (via TK1) vs growth time --------------------


summary_mito_human_lines_tk1_wk <- summarySE(mito_reads_l, measurevar="human_mito_copy_TK1", groupvars=c("week"))




p5 <- ggplot() + 
		theme2 + 
		theme(legend.key=element_blank()) +
		geom_line(
			data= summary_mito_human_lines_tk1_wk, 
			lwd=0.2,
			colour="red",
			aes(
				x=week, 
				y= human_mito_copy_TK1
				),
			show.legend=FALSE
			) + 
		geom_linerange(
			data= summary_mito_human_lines_tk1_wk,
			aes(
				x=week, 
				ymin= human_mito_copy_TK1 - se, 
				ymax= human_mito_copy_TK1 + se
				),
			colour="red",
			lwd=0.2,
			show.legend=FALSE
			) +
		geom_point(
			shape=1,
			stroke=0.4,
			data= mito_reads_l, 
			aes(
				x=jitter(week,factor=0.2), 
				y= human_mito_copy_TK1, 
				colour=as.factor(pool)),
				size=1.0
				) +
		scale_color_manual(
			values=colores_1,
			name ="Pool", 
			labels=c(1,2,3,4,5,6)
			) +
	 		guides(
	 			colour = guide_legend(override.aes = list(fill=NA,shape=1,size=1),ncol=1,byrow=TRUE)
	 			) +
 		theme(
	 		plot.margin = unit(c(top=0.75,right=1.75,bottom=0.75,left=0.75), "cm"),
		 	# plot.subtitle = element_text(size=9, face="plain", hjust = 0.5, margin=margin(0,0,6,0)),
		 	# axis.title.x=element_blank(),
	        # axis.text.x=element_blank(),
	        # axis.ticks.x=element_blank(),
	 		legend.position="none",
			legend.margin=margin(t=0,r=0,b=0,l=0,unit = "pt"),
		 	legend.box.margin=margin(t=0,r=0,b=0,l=0,unit = "pt"),
		 	legend.key.height = unit(0.1, "cm"),
			legend.key.width = unit(0.1, "cm"),
		 	legend.spacing.y = unit(0.1, 'cm'),
		 	legend.spacing.x = unit(0.1, 'cm'),
		 	legend.title = element_text(size = 11),  
		 	legend.text = element_text(size = 10),
	 		legend.title.align=0.0
	 		) +
		scale_x_continuous(breaks = c(0,1,2,3,4,6), labels = c(0,1,2,3,4,6)) +
		# coord_cartesian(ylim = c(0,1250)) + 
		xlab("Weeks") + 
		ylab(expression("Mito copy (TK1)")) + 
		labs(subtitle="Human")
print(p5)



# ---------------------------- (6) Human mito copy number (TK1) vs paclitaxel conc --------------------


summary_mito_human_lines_tk1_conc <- summarySE(mito_reads_l, measurevar="human_mito_copy_TK1", groupvars=c("conc"))




p6 <- ggplot() + 
		theme2 + 
		theme(legend.key=element_blank()) +
		geom_line(
			data= summary_mito_human_lines_tk1_conc, 
			lwd=0.2,
			colour="red",
			aes(
				x=conc, 
				y= human_mito_copy_TK1),
				show.legend=FALSE
				) + 
		geom_linerange(
			data= summary_mito_human_lines_tk1_conc,
			aes(
				x=conc, 
				ymin= human_mito_copy_TK1-se, 
				ymax= human_mito_copy_TK1 +se
				),
			colour="red",
			lwd=0.2,
			show.legend=FALSE
			) +
		geom_point(
			shape=1,
			stroke=0.4,
			data= mito_reads_l, 
			aes(
				x=jitter(conc,factor=0.2), 
				y= human_mito_copy_TK1, 
				colour=as.factor(pool)),
				size=1.0
				) +
		scale_color_manual(
			values=colores_1,
			name ="Pool", 
			labels=c(1,2,3,4,5,6)
			) +
	 		guides(
	 			colour = guide_legend(override.aes = list(fill=NA,shape=1,size=1),ncol=1,byrow=TRUE)
	 			) +
 		theme(
	 		plot.margin = unit(c(top=0.75,right=1.75,bottom=0.75,left=0.75), "cm"),
		 	# plot.subtitle = element_text(size=9, face="plain", hjust = 0.5, margin=margin(0,0,6,0)),
		 	# axis.title.x=element_blank(),
	        # axis.text.x=element_blank(),
	        # axis.ticks.x=element_blank(),
	 		legend.position="none",
			legend.margin=margin(t=0,r=0,b=0,l=0,unit = "pt"),
		 	legend.box.margin=margin(t=0,r=0,b=0,l=0,unit = "pt"),
		 	legend.key.height = unit(0.1, "cm"),
			legend.key.width = unit(0.1, "cm"),
		 	legend.spacing.y = unit(0.1, 'cm'),
		 	legend.spacing.x = unit(0.1, 'cm'),
		 	legend.title = element_text(size = 11),  
		 	legend.text = element_text(size = 10),
	 		legend.title.align=0.0
	 		) +
		scale_x_continuous(breaks = c(0,8,25,75), labels = c(0,8,25,75)) +
		# coord_cartesian(ylim = c(0,1250)) + 
		xlab("Paclitaxel (nM)") + 
		ylab(expression("Mito copy (TK1)")) + 
		labs(subtitle="Human")
print(p6)








# ---------------------------- (7) Human mito copy number (mean of hamster seq reads and TK1) vs growth time --------------------


summary_mito_human_lines_mean_wk <- summarySE(mito_reads_l, measurevar="human_mean_mito_copy", groupvars=c("week"))




p7 <- ggplot() + 
		theme2 + 
		theme(legend.key=element_blank()) +
		geom_line(
			data= summary_mito_human_lines_mean_wk, 
			lwd=0.2,
			colour="red",
			aes(
				x=week, 
				y= human_mean_mito_copy
				),
			show.legend=FALSE
			) + 
		geom_linerange(
			data= summary_mito_human_lines_mean_wk,
			aes(
				x=week, 
				ymin= human_mean_mito_copy - se, 
				ymax= human_mean_mito_copy + se
				),
			colour="red",
			lwd=0.2,
			show.legend=FALSE
			) +
		geom_point(
			shape=1,
			stroke=0.4,
			data= mito_reads_l, 
			aes(
				x=jitter(week,factor=0.2), 
				y= human_mean_mito_copy, 
				colour=as.factor(pool)),
				size=1.0
				) +
		scale_color_manual(
			values=colores_1,
			name ="Pool", 
			labels=c(1,2,3,4,5,6)
			) +
	 		guides(
	 			colour = guide_legend(override.aes = list(fill=NA,shape=1,size=1),ncol=1,byrow=TRUE)
	 			) +
 		theme(
	 		plot.margin = unit(c(top=0.75,right=1.75,bottom=0.75,left=0.75), "cm"),
		 	# plot.subtitle = element_text(size=9, face="plain", hjust = 0.5, margin=margin(0,0,6,0)),
		 	# axis.title.x=element_blank(),
	        # axis.text.x=element_blank(),
	        # axis.ticks.x=element_blank(),
	 		legend.position="none",
			legend.margin=margin(t=0,r=0,b=0,l=0,unit = "pt"),
		 	legend.box.margin=margin(t=0,r=0,b=0,l=0,unit = "pt"),
		 	legend.key.height = unit(0.1, "cm"),
			legend.key.width = unit(0.1, "cm"),
		 	legend.spacing.y = unit(0.1, 'cm'),
		 	legend.spacing.x = unit(0.1, 'cm'),
		 	legend.title = element_text(size = 11),  
		 	legend.text = element_text(size = 10),
	 		legend.title.align=0.0
	 		) +
		scale_x_continuous(breaks = c(0,1,2,3,4,6), labels = c(0,1,2,3,4,6)) +
		# coord_cartesian(ylim = c(0,1250)) + 
		xlab("Weeks") + 
		ylab(expression("Mito copy (mean)")) + 
		labs(subtitle="Human")
print(p7)




# ---------------------------- (8) Human mito copy number (mean of hamster seq reads and TK1) vs paclitaxel conc --------------------


summary_mito_human_lines_mean_conc <- summarySE(mito_reads_l, measurevar="human_mean_mito_copy", groupvars=c("conc"))



p8 <- ggplot() + 
		theme2 + 
		theme(legend.key=element_blank()) +
		geom_line(
			data= summary_mito_human_lines_mean_conc, 
			lwd=0.2,
			colour="red",
			aes(
				x=conc, 
				y= human_mean_mito_copy),
				show.legend=FALSE
				) + 
		geom_linerange(
			data= summary_mito_human_lines_mean_conc,
			aes(
				x=conc, 
				ymin= human_mean_mito_copy-se, 
				ymax= human_mean_mito_copy +se
				),
			colour="red",
			lwd=0.2,
			show.legend=FALSE
			) +
		geom_point(
			shape=1,
			stroke=0.4,
			data= mito_reads_l, 
			aes(
				x=jitter(conc,factor=0.2), 
				y= human_mean_mito_copy, 
				colour=as.factor(pool)),
				size=1.0
				) +
		scale_color_manual(
			values=colores_1,
			name ="Pool", 
			labels=c(1,2,3,4,5,6)
			) +
	 		guides(
	 			colour = guide_legend(override.aes = list(fill=NA,shape=1,size=1),ncol=1,byrow=TRUE)
	 			) +
 		theme(
	 		plot.margin = unit(c(top=0.75,right=1.75,bottom=0.75,left=0.75), "cm"),
		 	# plot.subtitle = element_text(size=9, face="plain", hjust = 0.5, margin=margin(0,0,6,0)),
		 	# axis.title.x=element_blank(),
	        # axis.text.x=element_blank(),
	        # axis.ticks.x=element_blank(),
	 		legend.position="none",
			legend.margin=margin(t=0,r=0,b=0,l=0,unit = "pt"),
		 	legend.box.margin=margin(t=0,r=0,b=0,l=0,unit = "pt"),
		 	legend.key.height = unit(0.1, "cm"),
			legend.key.width = unit(0.1, "cm"),
		 	legend.spacing.y = unit(0.1, 'cm'),
		 	legend.spacing.x = unit(0.1, 'cm'),
		 	legend.title = element_text(size = 11),  
		 	legend.text = element_text(size = 10),
	 		legend.title.align=0.0
	 		) +
		scale_x_continuous(breaks = c(0,8,25,75), labels = c(0,8,25,75)) +
		# coord_cartesian(ylim = c(0,1250)) + 
		xlab("Paclitaxel (nM)") + 
		ylab(expression("Mito copy (mean)")) + 
		labs(subtitle="Human")
print(p8)




#------------------ Make file --------------------------


(p_comp <- plot_grid(p1, p2, p3, p4, p5, p6, p7, p8, labels=c("A", "B", "C", "D", "E", "F", "G", "H"), ncol = 2, nrow = 4, label_size = 16,align="hv",axis="lb"))


pdf("mito_1.pdf",width=7.5,height=10,useDingbats=FALSE)
p_comp
dev.off()

# png("mito_1.png", width=7.5, height=10,units="in",res=300)
# p_comp
# dev.off()





# --------------------- Human mito reads ----------------------------


# RH pools
mean(mito_reads_l[mito_reads_l$week==0 & mito_reads_l$conc==0,"human_mito_reads"])
# [1] 171.5 <<<<<<<<< use in paper

sem(mito_reads_l[mito_reads_l$week==0 & mito_reads_l$conc==0,"human_mito_reads"])
# [1] 98.75078 <<<<<<<<< use in paper


t.test(mito_reads_l[mito_reads_l$week==0 & mito_reads_l$conc==0,"human_mito_reads"])

	# One Sample t-test

# data:  mito_reads_l[mito_reads_l$week == 0 & mito_reads_l$conc == 0,     "human_mito_reads"]
# t = 1.7367, df = 5, p-value = 0.143
# alternative hypothesis: true mean is not equal to 0
# 95 percent confidence interval:
 # -82.34696 425.34696
# sample estimates:
# mean of x 
    # 171.5 


# All samples

mean(mito_reads_l[,"human_mito_reads"])
# [1] 68.10435 <<<<<<<<< use in paper

sem(mito_reads_l[,"human_mito_reads"])
# [1] 14.60492 <<<<<<<<< use in paper

t.test(mito_reads_l[,"human_mito_reads"])

	# One Sample t-test

# data:  mito_reads_l[, "human_mito_reads"]
# t = 4.6631, df = 114, p-value = 8.527e-06
# alternative hypothesis: true mean is not equal to 0
# 95 percent confidence interval:
 # 39.17212 97.03658
# sample estimates:
# mean of x 
 # 68.10435 


# --------------------- Hamster mito reads ----------------------------


# RH pools

mean(mito_reads_l[mito_reads_l$week==0 & mito_reads_l$conc==0,"hamster_mito_reads"])
# [1] 92926 

sem(mito_reads_l[mito_reads_l$week==0 & mito_reads_l$conc==0,"hamster_mito_reads"])
# [1] 6797.38 




# All samples

mean(mito_reads_l[,"hamster_mito_reads"])
# [1] 90434.34 

sem(mito_reads_l[,"hamster_mito_reads"])
# [1] 3033.163 












# ------------------ Summary data for RH pool mito copy number table in paper -----------------

sem <- function(x) {sqrt(var(x,na.rm=TRUE)/sum(!is.na(x)))}

# write paper_table to a text file to cut and paste into latex document

paper_table <- merge(human_reads[,c("RH_ID","pool","week","conc","Total_reads")], mito_reads_l[,c("RH_ID","pool","week","conc","hamster_total_reads","TK1_max","hamster_mito_reads","human_mito_reads","revert_freq","hamster_mito_copy_hamster_reads","human_mito_copy_hamster_reads","human_mito_copy_TK1","human_mean_mito_copy")])

paper_table <- paper_table[paper_table$week == 0 & paper_table$conc == 0,]
paper_table <- paper_table[,c(2,5:ncol(paper_table))]

paper_table$revert_freq <- round(paper_table$revert_freq,digits=2)
paper_table$hamster_mito_copy_hamster_reads <- round(paper_table$hamster_mito_copy_hamster_reads,digits=0)
paper_table$human_mito_copy_hamster_reads <- round(paper_table$human_mito_copy_hamster_reads,digits=2)
paper_table$human_mito_copy_TK1 <- round(paper_table$human_mito_copy_TK1,digits=2)
paper_table$human_mean_mito_copy <- round(paper_table$human_mean_mito_copy,digits=2)


# lack of space meant that "human_mean_mito_copy" column was omitted from table in paper.

paper_table
   # pool Total_reads hamster_total_reads TK1_max hamster_mito_reads human_mito_reads revert_freq hamster_mito_copy_hamster_reads human_mito_copy_hamster_reads
# 1     1    42000172            37595178     997             103116               16        0.24                             800                          0.12
# 21    2    35358581            30754890    2636             109494              474        0.20                            1040                          4.42
# 40    3    52166656            45668817    3302              96305              492        0.27                             615                          3.09
# 60    4    32756205            28092611    3125              61749               45        0.29                             641                          0.46
# 80    5    42876176            38167486    2234              89994                0        0.28                             688                          0.00
# 98    6    41449894            37193349    2573              96898                2        0.37                             760                          0.02
   # human_mito_copy_TK1 human_mean_mito_copy
# 1                 0.74                 0.43
# 21                8.67                 6.55
# 40                6.57                 4.83
# 60                0.61                 0.54
# 80                0.00                 0.00
# 98                0.03                 0.02




colMeans(paper_table[,c(2:ncol(paper_table))])
                    # Total_reads             hamster_total_reads                         TK1_max              hamster_mito_reads                human_mito_reads 
                   # 4.110128e+07                    3.624539e+07                    2.477833e+03                    9.292600e+04                    1.715000e+02 
                    # revert_freq hamster_mito_copy_hamster_reads   human_mito_copy_hamster_reads             human_mito_copy_TK1            human_mean_mito_copy 
                   # 2.750000e-01                    7.573333e+02                    1.351667e+00                    2.770000e+00                    2.061667e+00 
                   
                   
# Avoid rounding errors:
mean(paper_table$Total_reads)
# [1] 41101281

mean(paper_table$hamster_total_reads)
# [1] 36245388

                   
                   
 apply(paper_table[,c(2:ncol(paper_table))],2,sem)
                    # Total_reads             hamster_total_reads                         TK1_max              hamster_mito_reads                human_mito_reads 
                   # 2.763700e+06                    2.528698e+06                    3.358805e+02                    6.797380e+03                    9.875078e+01 
                    # revert_freq hamster_mito_copy_hamster_reads   human_mito_copy_hamster_reads             human_mito_copy_TK1            human_mean_mito_copy 
                   # 2.320201e-02                    6.333175e+01                    7.820674e-01                    1.562231e+00                    1.171971e+00  

# Avoid rounding errors:
sem(paper_table$Total_reads)
# [1] 2763700

sem(paper_table$hamster_total_reads)
# [1] 2528698






# ------------- Mixed mod calx: hamster mito copy based on hamster reads ---------------------


hamster_mito_copy_hamster_reads <- data.frame(
									Chromosome = as.character(), 
									posS = integer(), 
									posE = integer(), 
									pos = numeric(), 
									coef_g_0nM = numeric(),
									coef_g_8nM = numeric(),
									coef_g_25nM = numeric(),
									coef_g_75nM = numeric(),
									coef_g_avg = numeric(),
									coef_d_w1 = numeric(),
									coef_d_w2 = numeric(),
									coef_d_w3 = numeric(),
									coef_d_w4 = numeric(),
									coef_d_w6 = numeric(),
									coef_d_avg = numeric(),
									coef_g_d_Ix = numeric(),
									Z_g_0nM = numeric(),
									Z_g_8nM = numeric(),
									Z_g_25nM = numeric(),
									Z_g_75nM = numeric(),
									Z_g_avg = numeric(),
									Z_d_w1 = numeric(),
									Z_d_w2 = numeric(),
									Z_d_w3 = numeric(),
									Z_d_w4 = numeric(),
									Z_d_w6 = numeric(),
									Z_d_avg = numeric(),
									Z_g_d_Ix = numeric(),
									log10p_g_0nM = numeric(),
									log10p_g_8nM = numeric(),
									log10p_g_25nM = numeric(),
									log10p_g_75nM = numeric(),
									log10p_g_avg = numeric(),
									log10p_d_w1 = numeric(),
									log10p_d_w2 = numeric(),
									log10p_d_w3 = numeric(),
									log10p_d_w4 = numeric(),
									log10p_d_w6 = numeric(),
									log10p_d_avg = numeric(),
									log10p_g_d_Ix = numeric(),
									stringsAsFactors=FALSE
									)



				
		m1 <- lmer(hamster_mito_copy_hamster_reads ~ week * conc + (1|pool/cell), data = mito_reads_l, REML=TRUE)
		
# msg:		"boundary (singular) fit: see ?isSingular", see above
		
		
		summary(glht(m1),test=adjusted("none"))

	 # Simultaneous Tests for General Linear Hypotheses

# Fit: lmer(formula = hamster_mito_copy_hamster_reads ~ week * conc + 
    # (1 | pool/cell), data = mito_reads_l, REML = TRUE)

# Linear Hypotheses:
                 # Estimate Std. Error z value Pr(>|z|)    
# (Intercept) == 0 792.3864    50.2511  15.769  < 2e-16 ***
# week == 0        -34.1251    10.8892  -3.134  0.00173 ** 
# conc == 0          0.6110     1.2976   0.471  0.63774    
# week:conc == 0     0.4880     0.3284   1.486  0.13730    
# ---
# Signif. codes:  0 ‘***’ 0.001 ‘**’ 0.01 ‘*’ 0.05 ‘.’ 0.1 ‘ ’ 1
# (Adjusted p values reported -- none method)


# Test whether estimate of hamster mito copy number is different from A23 mito copy number of 2*(95975/(33234329-95975))*(2368906908/16283) (cf A23_HEK_mito_copy_num_1.R)
# No sig diff

K <- diag(1,nrow=4)

summary(glht(m1,linfct=K,rhs=c(2*(95975/(33234329-95975))*(2368906908/16283),0,0,0)),test=adjusted("none"))

	 # Simultaneous Tests for General Linear Hypotheses

# Fit: lmer(formula = hamster_mito_copy_hamster_reads ~ week * conc + 
    # (1 | pool/cell), data = mito_reads_l, REML = TRUE)

# Linear Hypotheses:
                      # Estimate Std. Error z value Pr(>|z|)   
# 1 == 842.695655620985 792.3864    50.2511  -1.001  0.31675   
# 2 == 0                -34.1251    10.8892  -3.134  0.00173 **
# 3 == 0                  0.6110     1.2976   0.471  0.63774   
# 4 == 0                  0.4880     0.3284   1.486  0.13730   
# ---
# Signif. codes:  0 ‘***’ 0.001 ‘**’ 0.01 ‘*’ 0.05 ‘.’ 0.1 ‘ ’ 1
# (Adjusted p values reported -- none method)



# or more simply:

summary(glht(m1,rhs=c(2*(95975/(33234329-95975))*(2368906908/16283),0,0,0)),test=adjusted("none"))

	 # Simultaneous Tests for General Linear Hypotheses

# Fit: lmer(formula = hamster_mito_copy_hamster_reads ~ week * conc + 
    # (1 | pool/cell), data = mito_reads_l, REML = TRUE)

# Linear Hypotheses:
                                # Estimate Std. Error z value Pr(>|z|)   
# (Intercept) == 842.695655620985 792.3864    50.2511  -1.001  0.31675   
# week == 0                       -34.1251    10.8892  -3.134  0.00173 **
# conc == 0                         0.6110     1.2976   0.471  0.63774   
# week:conc == 0                    0.4880     0.3284   1.486  0.13730   
# ---
# Signif. codes:  0 ‘***’ 0.001 ‘**’ 0.01 ‘*’ 0.05 ‘.’ 0.1 ‘ ’ 1
# (Adjusted p values reported -- none method)



# do the same thing in emmeans:

m1_shift <- lmer(hamster_mito_copy_hamster_reads - (2*(95975/(33234329-95975))*(2368906908/16283))  ~ week * conc + (1|pool/cell), data = mito_reads_l, REML=TRUE)

test(emmeans(m1_shift,lmer.df = "asym",specs=c("week","conc"),at=list(week=c(0),conc=c(0))))
 # week conc emmean   SE  df z.ratio p.value
    # 0    0  -50.3 50.3 Inf -1.001  0.3168

# kenward-roger
test(emmeans(m1_shift,lmer.df = "kenward-roger",specs=c("week","conc"),at=list(week=c(0),conc=c(0))))
 # week conc emmean   SE   df t.ratio p.value
    # 0    0  -50.3 50.3 17.3 -1.001  0.3306 






# As an alternative, ask if hamster mito copy number in RH pools different from A23 cells.

# A23 mito copy number = 2*((95975/16283)/((33234329-95975)/2368906908)) (cf A23_HEK_mito_copy_num_1.R)

A23_mito <- 2*(95975/(33234329-95975))*(2368906908/16283)

# No sig diff, similar to results from mixed model above.
# Decided more accurate to use this result, since hamster mito copy number changes with growth time and paclitaxel conc:

compare(mito_reads_l[mito_reads_l$week == 0 & mito_reads_l$conc == 0,"hamster_mito_copy_hamster_reads"], rep(A23_mito,length(mito_reads_l[mito_reads_l$week == 0 & mito_reads_l$conc == 0,"hamster_mito_copy_hamster_reads"])))

	# Welch Two Sample t-test           <<<<<<<<<<<<<<<< use in paper (Here, "compare" is used to get means and sems only, not P values, and is equivalent to One sample t-test, i.e. using mu = A23_mito. See below. However, more correct to use pooled variance to get P val from t-test of two empirical distributions where one distribution has n = 1. Therefore used "compare" to get means and sems, and P val stats from pooled variance t test below.)

# data:  a and b
# t = -1.35, df = 5, p-value = 0.2349
# alternative hypothesis: true difference in means is not equal to 0
# 95 percent confidence interval:
 # -248.18115   77.26232
# sample estimates:
# mean of x mean of y 
 # 757.2362  842.6957 

# [1] "exact P value = 0.23489762697584"
# [1] "mean of a = 757.236240261696"
# [1] "sem of a = 63.3015188095392"
# [1] "sd of a = 155.056421026563"
# [1] "number in a = 6"
# [1] "mean of b = 842.695655620985"
# [1] "sem of b = 0"
# [1] "sd of b = 0"
# [1] "number in b = 6"


# agrees with "compare" calculation above
t.test(mito_reads_l[mito_reads_l$week == 0 & mito_reads_l$conc == 0,"hamster_mito_copy_hamster_reads"],mu=A23_mito)

	# One Sample t-test

# data:  mito_reads_l[mito_reads_l$week == 0 & mito_reads_l$conc == 0,     "hamster_mito_copy_hamster_reads"]
# t = -1.35, df = 5, p-value = 0.2349
# alternative hypothesis: true mean is not equal to 842.6957
# 95 percent confidence interval:
 # 594.5145 919.9580
# sample estimates:
# mean of x 
 # 757.2362 

# pooled variance
t.test(mito_reads_l[mito_reads_l$week == 0 & mito_reads_l$conc == 0,"hamster_mito_copy_hamster_reads"], A23_mito,var.equal=TRUE)

	# Two Sample t-test <<<<<<<<<<<<<<<< use in paper (for p val)

# data:  mito_reads_l[mito_reads_l$week == 0 & mito_reads_l$conc == 0,  and A23_mito    "hamster_mito_copy_hamster_reads"] and A23_mito
# t = -0.51027, df = 5, p-value = 0.6316
# alternative hypothesis: true difference in means is not equal to 0
# 95 percent confidence interval:
 # -515.9807  345.0618
# sample estimates:
# mean of x mean of y 
 # 757.2362  842.6957 



		 
		glht_growth <- glht(m1, linfct = c(
					"week == 0",
 					"week + 8*week:conc == 0",
					"week + 25*week:conc == 0",
 					"week + 75*week:conc == 0",
 					"week + (27)*week:conc == 0"))
 					
		glht_drug <- glht(m1, linfct = c(
					"conc + 1*week:conc == 0",
  					"conc + 2*week:conc == 0",
 					"conc + 3*week:conc == 0",
  					"conc + 4*week:conc == 0",
 					"conc + 6*week:conc == 0",
 					"conc + (3.2)*week:conc == 0"))
 					
 		glht_omni <- glht(m1)
 		 					
 		
 	
 	growth_coef <- summary(glht_growth,test = adjusted("none"))$test$coefficients
 	drug_coef <- summary(glht_drug,test = adjusted("none"))$test$coefficients
 	Ix_coef <- summary(glht_omni,test = adjusted("none"))$test$coefficients["week:conc"]				
 	growth_stat <- summary(glht_growth,test = adjusted("none"))$test$tstat
 	drug_stat <- summary(glht_drug,test = adjusted("none"))$test$tstat
 	Ix_stat <- summary(glht_omni,test = adjusted("none"))$test$tstat["week:conc"]
 	
 		
 		
 		
 	hamster_mito_copy_hamster_reads[1,] <- c(
				 							hamster_mito_reads[1,1:4],
				 							growth_coef,
				 							drug_coef,
				 							Ix_coef,
				 							growth_stat,
				 							drug_stat,
				 							Ix_stat,
				 							-log10(2*pnorm(-abs(growth_stat))),
				 							-log10(2*pnorm(-abs(drug_stat))),
				 							-log10(2*pnorm(-abs(Ix_stat)))
				 							)
				 							
				 							
hamster_mito_copy_hamster_reads
# Chromosome posS  posE  pos coef_g_0nM coef_g_8nM coef_g_25nM coef_g_75nM coef_g_avg coef_d_w1 coef_d_w2 coef_d_w3 coef_d_w4 coef_d_w6 coef_d_avg coef_g_d_Ix   Z_g_0nM   Z_g_8nM  Z_g_25nM  Z_g_75nM
# 1 chromosomeM    0 16283 8142  -34.12538  -30.22119   -21.92477    2.476434  -20.94873  1.099024  1.587048  2.075072  2.563096  3.539145   2.172677   0.4880242 -3.133846 -3.166925 -2.475458 0.1234082
    # Z_g_avg   Z_d_w1   Z_d_w2   Z_d_w3   Z_d_w4   Z_d_w6  Z_d_avg Z_g_d_Ix log10p_g_0nM log10p_g_8nM log10p_g_25nM log10p_g_75nM log10p_g_avg log10p_d_w1 log10p_d_w2 log10p_d_w3 log10p_d_w4 log10p_d_w6
# 1 -2.326801 1.074969 2.011586 3.219674 3.936043 3.392085 3.435942 1.485915     2.763132     2.812311      1.875935    0.04489755     1.699495   0.5491528    1.353953     2.89165    4.081779    3.158873
  # log10p_d_avg log10p_g_d_Ix
# 1     3.228783     0.8623245	 		


# recast as normal P vals not -log10P


	hamster_mito_copy_hamster_reads <- data.frame(
												Chromosome = as.character(), 
												posS = integer(), 
												posE = integer(), 
												pos = numeric(), 
												coef_g_0nM = numeric(),
												coef_g_8nM = numeric(),
												coef_g_25nM = numeric(),
												coef_g_75nM = numeric(),
												coef_g_avg = numeric(),
												coef_d_w1 = numeric(),
												coef_d_w2 = numeric(),
												coef_d_w3 = numeric(),
												coef_d_w4 = numeric(),
												coef_d_w6 = numeric(),
												coef_d_avg = numeric(),
												coef_g_d_Ix = numeric(),
												Z_g_0nM = numeric(),
												Z_g_8nM = numeric(),
												Z_g_25nM = numeric(),
												Z_g_75nM = numeric(),
												Z_g_avg = numeric(),
												Z_d_w1 = numeric(),
												Z_d_w2 = numeric(),
												Z_d_w3 = numeric(),
												Z_d_w4 = numeric(),
												Z_d_w6 = numeric(),
												Z_d_avg = numeric(),
												Z_g_d_Ix = numeric(),
												P_g_0nM = numeric(),
												P_g_8nM = numeric(),
												P_g_25nM = numeric(),
												P_g_75nM = numeric(),
												P_g_avg = numeric(),
												P_d_w1 = numeric(),
												P_d_w2 = numeric(),
												P_d_w3 = numeric(),
												P_d_w4 = numeric(),
												P_d_w6 = numeric(),
												P_d_avg = numeric(),
												P_g_d_Ix = numeric(),
												stringsAsFactors=FALSE
												)

	
	hamster_mito_copy_hamster_reads[1,] <- c(
				 							hamster_mito_reads[1,1:4],
				 							growth_coef,
				 							drug_coef,
				 							Ix_coef,
				 							growth_stat,
				 							drug_stat,
				 							Ix_stat,
				 							2*pnorm(-abs(growth_stat)),
				 							2*pnorm(-abs(drug_stat)),
				 							2*pnorm(-abs(Ix_stat))
				 							)
	
hamster_mito_copy_hamster_reads
# Chromosome posS  posE  pos coef_g_0nM coef_g_8nM coef_g_25nM coef_g_75nM coef_g_avg coef_d_w1 coef_d_w2 coef_d_w3 coef_d_w4 coef_d_w6 coef_d_avg coef_g_d_Ix   Z_g_0nM   Z_g_8nM  Z_g_25nM  Z_g_75nM
# 1 chromosomeM    0 16283 8142  -34.12538  -30.22119   -21.92477    2.476434  -20.94873  1.099024  1.587048  2.075072  2.563096  3.539145   2.172677   0.4880242 -3.133846 -3.166925 -2.475458 0.1234082
    # Z_g_avg   Z_d_w1   Z_d_w2   Z_d_w3   Z_d_w4   Z_d_w6  Z_d_avg Z_g_d_Ix     P_g_0nM     P_g_8nM   P_g_25nM  P_g_75nM    P_g_avg    P_d_w1     P_d_w2      P_d_w3       P_d_w4       P_d_w6      P_d_avg
# 1 -2.326801 1.074969 2.011586 3.219674 3.936043 3.392085 3.435942 1.485915 0.001725312 0.001540598 0.01330654 0.9017838 0.01997585 0.2823886 0.04426359 0.001283363 8.283628e-05 0.0006936286 0.0005904966
   # P_g_d_Ix
# 1 0.1373016 







# emmeans
# Provides kenward-roger
# lmer.df="asymptotic" agrees with glht exactly

# growth

summary(contrast(emmeans(m1, lmer.df = "kenward-roger", specs="week", at=list(week=c(0,1), conc=27)),list(c(-1,1))))
# NOTE: Results may be misleading due to involvement in interactions
 # contrast estimate   SE   df t.ratio p.value
 # c(-1, 1)    -20.9 9.02 93.5 -2.322  0.0224   <<<<<<<<<<< use in paper
 
 
summary(contrast(emmeans(m1, lmer.df = "kenward-roger", specs="week", at=list(week=c(0,1), conc=27)),list(c(-1,1))))$p.value
# NOTE: Results may be misleading due to involvement in interactions
# [1] 0.02241655 <<<<<<<<<<< use in paper
 
 

# drug
summary(contrast(emmeans(m1, lmer.df = "kenward-roger", specs="conc", at=list(conc=c(0,1), week=3.2)),list(c(-1,1))))
# NOTE: Results may be misleading due to involvement in interactions
 # contrast estimate    SE   df t.ratio p.value
 # c(-1, 1)     2.17 0.638 28.1 3.407   0.0020    <<<<<<<<<<< use in paper



# Ix
summary(contrast(emmeans(m1, lmer.df = "kenward-roger", specs=c("week","conc"), at=list(week=c(0,1),conc=c(0,1))),list(c(1,-1,-1,1))))
 # contrast        estimate    SE   df t.ratio p.value
 # c(1, -1, -1, 1)    0.488 0.329 95.5 1.482   0.1418   <<<<<<<<<<< use in paper








# pool <<<<<<<<<<<< use in paper
anova(lmer(hamster_mito_copy_hamster_reads ~ week * conc + (1|pool) + (1|cell), data = mito_reads_l, REML=TRUE),lmer(hamster_mito_copy_hamster_reads ~ week * conc + (1|cell), data = mito_reads_l, REML=TRUE))
# Data: mito_reads_l
# Models:
# lmer(hamster_mito_copy_hamster_reads ~ week * conc + (1 | cell), data = mito_reads_l, REML = TRUE): hamster_mito_copy_hamster_reads ~ week * conc + (1 | cell)
# lmer(hamster_mito_copy_hamster_reads ~ week * conc + (1 | pool) + (1 | cell), data = mito_reads_l, REML = TRUE): hamster_mito_copy_hamster_reads ~ week * conc + (1 | pool) + 
# lmer(hamster_mito_copy_hamster_reads ~ week * conc + (1 | pool) + (1 | cell), data = mito_reads_l, REML = TRUE):     (1 | cell)
                                                                                                                # Df    AIC    BIC  logLik deviance  Chisq Chi Df Pr(>Chisq)  
# lmer(hamster_mito_copy_hamster_reads ~ week * conc + (1 | cell), data = mito_reads_l, REML = TRUE)               6 1532.8 1549.3 -760.41   1520.8                           
# lmer(hamster_mito_copy_hamster_reads ~ week * conc + (1 | pool) + (1 | cell), data = mito_reads_l, REML = TRUE)  7 1528.2 1547.4 -757.11   1514.2 6.6061      1    0.01016 *
# ---
# Signif. codes:  0 ‘***’ 0.001 ‘**’ 0.01 ‘*’ 0.05 ‘.’ 0.1 ‘ ’ 1



# cell <<<<<<<<<<<< use in paper
anova(lmer(hamster_mito_copy_hamster_reads ~ week * conc + (1|pool) + (1|cell), data = mito_reads_l, REML=TRUE),lmer(hamster_mito_copy_hamster_reads ~ week * conc + (1|pool), data = mito_reads_l, REML=TRUE))
# Data: mito_reads_l
# Models:
# lmer(hamster_mito_copy_hamster_reads ~ week * conc + (1 | pool), data = mito_reads_l, REML = TRUE): hamster_mito_copy_hamster_reads ~ week * conc + (1 | pool)
# lmer(hamster_mito_copy_hamster_reads ~ week * conc + (1 | pool) + (1 | cell), data = mito_reads_l, REML = TRUE): hamster_mito_copy_hamster_reads ~ week * conc + (1 | pool) + 
# lmer(hamster_mito_copy_hamster_reads ~ week * conc + (1 | pool) + (1 | cell), data = mito_reads_l, REML = TRUE):     (1 | cell)
                                                                                                                # Df    AIC    BIC  logLik deviance Chisq Chi Df Pr(>Chisq)
# lmer(hamster_mito_copy_hamster_reads ~ week * conc + (1 | pool), data = mito_reads_l, REML = TRUE)               6 1526.2 1542.7 -757.11   1514.2                        
# lmer(hamster_mito_copy_hamster_reads ~ week * conc + (1 | pool) + (1 | cell), data = mito_reads_l, REML = TRUE)  7 1528.2 1547.4 -757.11   1514.2     0      1          1


# pool and cell together

anova(lmer(hamster_mito_copy_hamster_reads ~ week * conc + (1|pool/cell), data = mito_reads_l, REML=TRUE),lm(hamster_mito_copy_hamster_reads ~ week * conc, data = mito_reads_l, REML=TRUE))
# Data: mito_reads_l
# Models:
# lm(hamster_mito_copy_hamster_reads ~ week * conc, data = mito_reads_l, REML = TRUE): hamster_mito_copy_hamster_reads ~ week * conc
# lmer(hamster_mito_copy_hamster_reads ~ week * conc + (1 | pool/cell), data = mito_reads_l, REML = TRUE): hamster_mito_copy_hamster_reads ~ week * conc + (1 | pool/cell)
                                                                                                        # Df    AIC    BIC  logLik deviance  Chisq Chi Df Pr(>Chisq)   
# lm(hamster_mito_copy_hamster_reads ~ week * conc, data = mito_reads_l, REML = TRUE)                      5 1534.2 1547.9 -762.11   1524.2                            
# lmer(hamster_mito_copy_hamster_reads ~ week * conc + (1 | pool/cell), data = mito_reads_l, REML = TRUE)  7 1528.2 1547.4 -757.11   1514.2 9.9882      2   0.006778 **
# ---
# Signif. codes:  0 ‘***’ 0.001 ‘**’ 0.01 ‘*’ 0.05 ‘.’ 0.1 ‘ ’ 1



# icc

performance::icc(m1)
# [1] NA
# Warning message:
# Can't compute random effect variances. Some variance components equal zero.
  # Solution: Respecify random structure! 
  
# pool
(get_variance(m1)$var.intercept["pool"])/(get_variance(m1)$var.intercept["pool"] + get_variance(m1)$var.intercept["cell:pool"] + get_variance(m1)$var.residual)
    # pool 
# 0.181572 <<<<<<<<<<<<<<< use in paper


# cell (RE var == 0, causing the complaint above)
(get_variance(m1)$var.intercept["cell:pool"])/(get_variance(m1)$var.intercept["pool"] + get_variance(m1)$var.intercept["cell:pool"] + get_variance(m1)$var.residual)
# cell:pool 
        # 0  <<<<<<<<<<<<<<< use in paper


# pool and cell together

(get_variance(m1)$var.intercept["pool"] + get_variance(m1)$var.intercept["cell:pool"])/(get_variance(m1)$var.intercept["pool"] + get_variance(m1)$var.intercept["cell:pool"] + get_variance(m1)$var.residual)
    # pool 
# 0.181572 









# ------------- Mixed mod calx: human mito copy based on hamster reads ---------------------


human_mito_copy_hamster_reads <- data.frame(
									Chromosome = as.character(), 
									posS = integer(), 
									posE = integer(), 
									pos = numeric(), 
									coef_g_0nM = numeric(),
									coef_g_8nM = numeric(),
									coef_g_25nM = numeric(),
									coef_g_75nM = numeric(),
									coef_g_avg = numeric(),
									coef_d_w1 = numeric(),
									coef_d_w2 = numeric(),
									coef_d_w3 = numeric(),
									coef_d_w4 = numeric(),
									coef_d_w6 = numeric(),
									coef_d_avg = numeric(),
									coef_g_d_Ix = numeric(),
									Z_g_0nM = numeric(),
									Z_g_8nM = numeric(),
									Z_g_25nM = numeric(),
									Z_g_75nM = numeric(),
									Z_g_avg = numeric(),
									Z_d_w1 = numeric(),
									Z_d_w2 = numeric(),
									Z_d_w3 = numeric(),
									Z_d_w4 = numeric(),
									Z_d_w6 = numeric(),
									Z_d_avg = numeric(),
									Z_g_d_Ix = numeric(),
									log10p_g_0nM = numeric(),
									log10p_g_8nM = numeric(),
									log10p_g_25nM = numeric(),
									log10p_g_75nM = numeric(),
									log10p_g_avg = numeric(),
									log10p_d_w1 = numeric(),
									log10p_d_w2 = numeric(),
									log10p_d_w3 = numeric(),
									log10p_d_w4 = numeric(),
									log10p_d_w6 = numeric(),
									log10p_d_avg = numeric(),
									log10p_g_d_Ix = numeric(),
									stringsAsFactors=FALSE
									)



				
		m1 <- lmer(human_mito_copy_hamster_reads ~ week * conc + (1|pool/cell), data = mito_reads_l, REML=TRUE)
	
	
# Human mito copy number not signficantly different from zero at 0 nM paclitaxel and 0 weeks:
		
		
		summary(glht(m1),test=adjusted("none"))

	 # Simultaneous Tests for General Linear Hypotheses

# Fit: lmer(formula = human_mito_copy_hamster_reads ~ week * conc + 
    # (1 | pool/cell), data = mito_reads_l, REML = TRUE)

# Linear Hypotheses:
                  # Estimate Std. Error z value Pr(>|z|)
# (Intercept) == 0  0.274369   0.368889   0.744    0.457     
# week == 0         0.108202   0.078038   1.387    0.166
# conc == 0         0.010718   0.009300   1.152    0.249
# week:conc == 0   -0.003856   0.002354  -1.638    0.101
# (Adjusted p values reported -- none method)
		

# model complains: "boundary (singular) fit: see ?isSingular", but glmmTMB gives exact same results including P vals with no warning. And ?isSingular says boundary probs common.
# used lmer for consistency with other calx in paper




# use lmer.df = "asymptotic" to agree exactly with above

test(emmeans(m1,lmer.df = "kenward-roger",specs=c("week","conc"),at=list(week=c(0),conc=c(0))))
 # week conc emmean    SE df t.ratio p.value
    # 0    0  0.274 0.369 16 0.744   0.4679  

summary(contrast(emmeans(m1, lmer.df = "kenward-roger", specs="week", at=list(week=c(0,1), conc=0)),list(c(-1,1))))
# NOTE: Results may be misleading due to involvement in interactions
 # contrast estimate     SE   df t.ratio p.value
 # c(-1, 1)    0.108 0.0782 98.2 1.384   0.1696

 summary(contrast(emmeans(m1, lmer.df = "kenward-roger", specs="conc", at=list(conc=c(0,1), week=0)),list(c(-1,1))))
# NOTE: Results may be misleading due to involvement in interactions
 # contrast estimate      SE  df t.ratio p.value
 # c(-1, 1)   0.0107 0.00935 105 1.146   0.2544 


summary(contrast(emmeans(m1, lmer.df = "kenward-roger", specs=c("week","conc"), at=list(week=c(0,1),conc=c(0,1))),list(c(1,-1,-1,1))))
 # contrast        estimate      SE   df t.ratio p.value
 # c(1, -1, -1, 1) -0.00386 0.00236 95.5 -1.634  0.1056  


# not much diff at av conditional values. However, these results are superior, cf https://stackoverflow.com/questions/52866203/hypothesis-test-for-intercepts-in-general-mixed-linear-models-with-r
# though one purpose of this calx was to compare with human mito copy number in simple mean +/- sem in RH pools, for which "at=list(week=c(0),conc=c(0))" might be better.
# However, this approach evaluates all RH samples better
test(emmeans(m1,lmer.df = "kenward-roger",specs=c("week","conc"),at=list(week=c(3.2),conc=c(27))))
 # week conc emmean    SE   df t.ratio p.value
  # 3.2   27  0.577 0.276 5.16 2.091   0.0891 <<<<<<<<<<<< use in paper




		
# Alternatively, let's look at RH pools, human mito copy number still not sig diff from zero.
# Decided to use this human mito copy number (in addition to that from mixed model at avg conditional values), as empirical (rather than model based estimate) of human mito copy number.

compare(mito_reads_l[mito_reads_l$week == 0 & mito_reads_l$conc == 0,"human_mito_copy_hamster_reads"], rep(0,length(mito_reads_l[mito_reads_l$week == 0 & mito_reads_l$conc == 0,"human_mito_copy_hamster_reads"])))

	# Welch Two Sample t-test   <<<<<<<<<<<<<<<<<<<<<<< use in paper. (Here, "compare" used for P vals, means and sems. Actually this is same as One sample t-test, equiv to using mu = 0. See below.)

# data:  a and b
# t = 1.7268, df = 5, p-value = 0.1448
# alternative hypothesis: true difference in means is not equal to 0
# 95 percent confidence interval:
 # -0.6601764  3.3622825
# sample estimates:
# mean of x mean of y 
 # 1.351053  0.000000 

# [1] "exact P value = 0.144787262323748"
# [1] "mean of a = 1.35105305961763"
# [1] "sem of a = 0.782402417094639"
# [1] "sd of a = 1.91648669540208"
# [1] "number in a = 6"
# [1] "mean of b = 0"
# [1] "sem of b = 0"
# [1] "sd of b = 0"
# [1] "number in b = 6"


# test against hypothesized mean = 0 using one sample t-test. This test is appropriate for hypothesized value = 0 of one distribution. Also, more powerful than pooled variance t test, which is appropriate since want to test counterfactual.

t.test(mito_reads_l[mito_reads_l$week == 0 & mito_reads_l$conc == 0,"human_mito_copy_hamster_reads"], mu=0)

	# One Sample t-test <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< use in paper

# data:  mito_reads_l[mito_reads_l$week == 0 & mito_reads_l$conc == 0,     "human_mito_copy_hamster_reads"]
# t = 1.7268, df = 5, p-value = 0.1448
# alternative hypothesis: true mean is not equal to 0
# 95 percent confidence interval:
 # -0.6601764  3.3622825
# sample estimates:
# mean of x 
 # 1.351053 



# As expected, pooled variance is less powerful

t.test(mito_reads_l[mito_reads_l$week == 0 & mito_reads_l$conc == 0,"human_mito_copy_hamster_reads"], 0, var.equal=TRUE)

	# Two Sample t-test

# data:  mito_reads_l[mito_reads_l$week == 0 & mito_reads_l$conc == 0,  and 0    "human_mito_copy_hamster_reads"] and 0
# t = 0.65267, df = 5, p-value = 0.5428
# alternative hypothesis: true difference in means is not equal to 0
# 95 percent confidence interval:
 # -3.970160  6.672266
# sample estimates:
# mean of x mean of y 
 # 1.351053  0.000000





		
		
		

		 
		glht_growth <- glht(m1, linfct = c(
					"week == 0",
 					"week + 8*week:conc == 0",
					"week + 25*week:conc == 0",
 					"week + 75*week:conc == 0",
 					"week + (27)*week:conc == 0"))
 					
		glht_drug <- glht(m1, linfct = c(
					"conc + 1*week:conc == 0",
  					"conc + 2*week:conc == 0",
 					"conc + 3*week:conc == 0",
  					"conc + 4*week:conc == 0",
 					"conc + 6*week:conc == 0",
 					"conc + (3.2)*week:conc == 0"))
 					
 		glht_omni <- glht(m1)
 		 					
 		
 	
 	growth_coef <- summary(glht_growth,test = adjusted("none"))$test$coefficients
 	drug_coef <- summary(glht_drug,test = adjusted("none"))$test$coefficients
 	Ix_coef <- summary(glht_omni,test = adjusted("none"))$test$coefficients["week:conc"]				
 	growth_stat <- summary(glht_growth,test = adjusted("none"))$test$tstat
 	drug_stat <- summary(glht_drug,test = adjusted("none"))$test$tstat
 	Ix_stat <- summary(glht_omni,test = adjusted("none"))$test$tstat["week:conc"]
 	
 		
 		
 		
 	human_mito_copy_hamster_reads[1,] <- c(
				 							mito_reads[1,1:4],
				 							growth_coef,
				 							drug_coef,
				 							Ix_coef,
				 							growth_stat,
				 							drug_stat,
				 							Ix_stat,
				 							-log10(2*pnorm(-abs(growth_stat))),
				 							-log10(2*pnorm(-abs(drug_stat))),
				 							-log10(2*pnorm(-abs(Ix_stat)))
				 							)
				 							
				 							
human_mito_copy_hamster_reads
  # Chromosome posS  posE  pos coef_g_0nM coef_g_8nM coef_g_25nM coef_g_75nM  coef_g_avg   coef_d_w1   coef_d_w2     coef_d_w3   coef_d_w4   coef_d_w6   coef_d_avg  coef_g_d_Ix  Z_g_0nM  Z_g_8nM  Z_g_25nM
# 1       chrM    0 16569 8284  0.1082026 0.07735061  0.01179004  -0.1810352 0.004077028 0.006861202 0.003004698 -0.0008518057 -0.00470831 -0.01242132 -0.001623107 -0.003856504 1.386524 1.131045 0.1857452
   # Z_g_75nM    Z_g_avg    Z_d_w1    Z_d_w2     Z_d_w3    Z_d_w4    Z_d_w6   Z_d_avg  Z_g_d_Ix log10p_g_0nM log10p_g_8nM log10p_g_25nM log10p_g_75nM log10p_g_avg log10p_d_w1 log10p_d_w2 log10p_d_w3
# 1 -1.258781 0.06318673 0.9363412 0.5313601 -0.1844007 -1.008846 -1.661204 -0.358133 -1.638403    0.7809742    0.5883195    0.06923195     0.6817079   0.02245115   0.4570532   0.2253595  0.06869518
  # log10p_d_w4 log10p_d_w6 log10p_d_avg log10p_g_d_Ix
# 1   0.5043881    1.014697    0.1425205     0.9942292		 		


# recast as normal P vals not -log10P


	human_mito_copy_hamster_reads <- data.frame(
									Chromosome = as.character(), 
									posS = integer(), 
									posE = integer(), 
									pos = numeric(), 
									coef_g_0nM = numeric(),
									coef_g_8nM = numeric(),
									coef_g_25nM = numeric(),
									coef_g_75nM = numeric(),
									coef_g_avg = numeric(),
									coef_d_w1 = numeric(),
									coef_d_w2 = numeric(),
									coef_d_w3 = numeric(),
									coef_d_w4 = numeric(),
									coef_d_w6 = numeric(),
									coef_d_avg = numeric(),
									coef_g_d_Ix = numeric(),
									Z_g_0nM = numeric(),
									Z_g_8nM = numeric(),
									Z_g_25nM = numeric(),
									Z_g_75nM = numeric(),
									Z_g_avg = numeric(),
									Z_d_w1 = numeric(),
									Z_d_w2 = numeric(),
									Z_d_w3 = numeric(),
									Z_d_w4 = numeric(),
									Z_d_w6 = numeric(),
									Z_d_avg = numeric(),
									Z_g_d_Ix = numeric(),
									P_g_0nM = numeric(),
									P_g_8nM = numeric(),
									P_g_25nM = numeric(),
									P_g_75nM = numeric(),
									P_g_avg = numeric(),
									P_d_w1 = numeric(),
									P_d_w2 = numeric(),
									P_d_w3 = numeric(),
									P_d_w4 = numeric(),
									P_d_w6 = numeric(),
									P_d_avg = numeric(),
									P_g_d_Ix = numeric(),
									stringsAsFactors=FALSE
									)

	
	human_mito_copy_hamster_reads[1,] <- c(
				 							mito_reads[1,1:4],
				 							growth_coef,
				 							drug_coef,
				 							Ix_coef,
				 							growth_stat,
				 							drug_stat,
				 							Ix_stat,
				 							2*pnorm(-abs(growth_stat)),
				 							2*pnorm(-abs(drug_stat)),
				 							2*pnorm(-abs(Ix_stat))
				 							)
	
human_mito_copy_hamster_reads
  # Chromosome posS  posE  pos coef_g_0nM coef_g_8nM coef_g_25nM coef_g_75nM  coef_g_avg   coef_d_w1   coef_d_w2     coef_d_w3   coef_d_w4   coef_d_w6   coef_d_avg  coef_g_d_Ix  Z_g_0nM  Z_g_8nM  Z_g_25nM
# 1       chrM    0 16569 8284  0.1082026 0.07735061  0.01179004  -0.1810352 0.004077028 0.006861202 0.003004698 -0.0008518057 -0.00470831 -0.01242132 -0.001623107 -0.003856504 1.386524 1.131045 0.1857452
   # Z_g_75nM    Z_g_avg    Z_d_w1    Z_d_w2     Z_d_w3    Z_d_w4    Z_d_w6   Z_d_avg  Z_g_d_Ix   P_g_0nM   P_g_8nM  P_g_25nM  P_g_75nM   P_g_avg    P_d_w1    P_d_w2    P_d_w3    P_d_w4     P_d_w6   P_d_avg
# 1 -1.258781 0.06318673 0.9363412 0.5313601 -0.1844007 -1.008846 -1.661204 -0.358133 -1.638403 0.1655868 0.2580361 0.8526446 0.2081096 0.9496178 0.3490975 0.5951693 0.8536991 0.3130487 0.09667255 0.7202438
   # P_g_d_Ix
# 1 0.1013376 










# emmeans
# Provides kenward-roger
# lmer.df="asymptotic" agrees with glht exactly

# growth

summary(contrast(emmeans(m1, lmer.df = "kenward-roger", specs="week", at=list(week=c(0,1), conc=27)),list(c(-1,1))))
# NOTE: Results may be misleading due to involvement in interactions
# contrast estimate     SE   df t.ratio p.value
 # c(-1, 1)  0.00408 0.0647 93.5 0.063   0.9499    <<<<<<<<<<< use in paper
 
 
summary(contrast(emmeans(m1, lmer.df = "kenward-roger", specs="week", at=list(week=c(0,1), conc=27)),list(c(-1,1))))$p.value
# NOTE: Results may be misleading due to involvement in interactions
# [1] 0.9498602 <<<<<<<<<<< use in paper
 
 

# drug
summary(contrast(emmeans(m1, lmer.df = "kenward-roger", specs="conc", at=list(conc=c(0,1), week=3.2)),list(c(-1,1))))
# NOTE: Results may be misleading due to involvement in interactions
 # contrast estimate      SE   df t.ratio p.value
 # c(-1, 1) -0.00162 0.00457 28.1 -0.355  0.7252     <<<<<<<<<<< use in paper



# Ix
summary(contrast(emmeans(m1, lmer.df = "kenward-roger", specs=c("week","conc"), at=list(week=c(0,1),conc=c(0,1))),list(c(1,-1,-1,1))))
 # contrast        estimate      SE   df t.ratio p.value
 # c(1, -1, -1, 1) -0.00386 0.00236 95.5 -1.634  0.1056    <<<<<<<<<<< use in paper




	

# pool 
anova(lmer(human_mito_copy_hamster_reads ~ week * conc + (1|pool) + (1|cell), data = mito_reads_l, REML=TRUE),lmer(human_mito_copy_hamster_reads ~ week * conc + (1|cell), data = mito_reads_l, REML=TRUE))
# Data: mito_reads_l
# Models:
# lmer(human_mito_copy_hamster_reads ~ week * conc + (1 | cell), data = mito_reads_l, REML = TRUE): human_mito_copy_hamster_reads ~ week * conc + (1 | cell)
# lmer(human_mito_copy_hamster_reads ~ week * conc + (1 | pool) + (1 | cell), data = mito_reads_l, REML = TRUE): human_mito_copy_hamster_reads ~ week * conc + (1 | pool) + (1 | 
# lmer(human_mito_copy_hamster_reads ~ week * conc + (1 | pool) + (1 | cell), data = mito_reads_l, REML = TRUE):     cell)
                                                                                                              # Df    AIC    BIC  logLik deviance  Chisq Chi Df Pr(>Chisq)    
# lmer(human_mito_copy_hamster_reads ~ week * conc + (1 | cell), data = mito_reads_l, REML = TRUE)               6 401.79 418.26 -194.90   389.79                             
# lmer(human_mito_copy_hamster_reads ~ week * conc + (1 | pool) + (1 | cell), data = mito_reads_l, REML = TRUE)  7 392.95 412.16 -189.47   378.95 10.848      1   0.000989 *** <<<<<<<<<<< use in paper
# ---
# Signif. codes:  0 ‘***’ 0.001 ‘**’ 0.01 ‘*’ 0.05 ‘.’ 0.1 ‘ ’ 1





# cell 
anova(lmer(human_mito_copy_hamster_reads ~ week * conc + (1|pool) + (1|cell), data = mito_reads_l, REML=TRUE),lmer(human_mito_copy_hamster_reads ~ week * conc + (1|pool), data = mito_reads_l, REML=TRUE))
# Data: mito_reads_l
# Models:
# lmer(human_mito_copy_hamster_reads ~ week * conc + (1 | pool), data = mito_reads_l, REML = TRUE): human_mito_copy_hamster_reads ~ week * conc + (1 | pool)
# lmer(human_mito_copy_hamster_reads ~ week * conc + (1 | pool) + (1 | cell), data = mito_reads_l, REML = TRUE): human_mito_copy_hamster_reads ~ week * conc + (1 | pool) + (1 | 
# lmer(human_mito_copy_hamster_reads ~ week * conc + (1 | pool) + (1 | cell), data = mito_reads_l, REML = TRUE):     cell)
                                                                                                              # Df    AIC    BIC  logLik deviance Chisq Chi Df Pr(>Chisq)
# lmer(human_mito_copy_hamster_reads ~ week * conc + (1 | pool), data = mito_reads_l, REML = TRUE)               6 390.95 407.42 -189.47   378.95                        
# lmer(human_mito_copy_hamster_reads ~ week * conc + (1 | pool) + (1 | cell), data = mito_reads_l, REML = TRUE)  7 392.95 412.16 -189.47   378.95     0      1          1 <<<<<<<<<<< use in paper




# pool and cell together
anova(lmer(human_mito_copy_hamster_reads ~ week * conc + (1|pool/cell), data = mito_reads_l, REML=TRUE),lm(human_mito_copy_hamster_reads ~ week * conc, data = mito_reads_l, REML=TRUE))
# boundary (singular) fit: see ?isSingular
# refitting model(s) with ML (instead of REML)
# Data: mito_reads_l
# Models:
# lm(human_mito_copy_hamster_reads ~ week * conc, data = mito_reads_l, REML = TRUE): human_mito_copy_hamster_reads ~ week * conc
# lmer(human_mito_copy_hamster_reads ~ week * conc + (1 | pool/cell), data = mito_reads_l, REML = TRUE): human_mito_copy_hamster_reads ~ week * conc + (1 | pool/cell)
                                                                                                      # Df    AIC    BIC  logLik deviance  Chisq Chi Df Pr(>Chisq)   
# lm(human_mito_copy_hamster_reads ~ week * conc, data = mito_reads_l, REML = TRUE)                      5 400.42 414.14 -195.21   390.42                            
# lmer(human_mito_copy_hamster_reads ~ week * conc + (1 | pool/cell), data = mito_reads_l, REML = TRUE)  7 392.94 412.16 -189.47   378.94 11.475      2   0.003223 **
# ---
# Signif. codes:  0 ‘***’ 0.001 ‘**’ 0.01 ‘*’ 0.05 ‘.’ 0.1 ‘ ’ 1






icc(m1)

# performance::icc(m1)
# [1] NA
# Warning message:
# Can't compute random effect variances. Some variance components equal zero.
  # Solution: Respecify random structure! 



# pool
(get_variance(m1)$var.intercept["pool"])/(get_variance(m1)$var.intercept["pool"] + get_variance(m1)$var.intercept["cell:pool"] + get_variance(m1)$var.residual)
    # pool 
# 0.198388 <<<<<<<<<<< use in paper



# cell (zero RE variance, hence complaint above)
(get_variance(m1)$var.intercept["cell:pool"])/(get_variance(m1)$var.intercept["pool"] + get_variance(m1)$var.intercept["cell:pool"] + get_variance(m1)$var.residual)
# cell:pool 
        # 0 <<<<<<<<<<< use in paper


# cell and pool together
(get_variance(m1)$var.intercept["pool"] + get_variance(m1)$var.intercept["cell:pool"])/(get_variance(m1)$var.intercept["pool"] + get_variance(m1)$var.intercept["cell:pool"] + get_variance(m1)$var.residual)
    # pool 
# 0.198388


						

# ------------- Mixed mod calx: human mito copy based on TK1 ---------------------


human_mito_copy_TK1 <- data.frame(
									Chromosome = as.character(), 
									posS = integer(), 
									posE = integer(), 
									pos = numeric(), 
									coef_g_0nM = numeric(),
									coef_g_8nM = numeric(),
									coef_g_25nM = numeric(),
									coef_g_75nM = numeric(),
									coef_g_avg = numeric(),
									coef_d_w1 = numeric(),
									coef_d_w2 = numeric(),
									coef_d_w3 = numeric(),
									coef_d_w4 = numeric(),
									coef_d_w6 = numeric(),
									coef_d_avg = numeric(),
									coef_g_d_Ix = numeric(),
									Z_g_0nM = numeric(),
									Z_g_8nM = numeric(),
									Z_g_25nM = numeric(),
									Z_g_75nM = numeric(),
									Z_g_avg = numeric(),
									Z_d_w1 = numeric(),
									Z_d_w2 = numeric(),
									Z_d_w3 = numeric(),
									Z_d_w4 = numeric(),
									Z_d_w6 = numeric(),
									Z_d_avg = numeric(),
									Z_g_d_Ix = numeric(),
									log10p_g_0nM = numeric(),
									log10p_g_8nM = numeric(),
									log10p_g_25nM = numeric(),
									log10p_g_75nM = numeric(),
									log10p_g_avg = numeric(),
									log10p_d_w1 = numeric(),
									log10p_d_w2 = numeric(),
									log10p_d_w3 = numeric(),
									log10p_d_w4 = numeric(),
									log10p_d_w6 = numeric(),
									log10p_d_avg = numeric(),
									log10p_g_d_Ix = numeric(),
									stringsAsFactors=FALSE
									)



				
		m1 <- lmer(human_mito_copy_TK1 ~ week * conc + (1|pool/cell), data = mito_reads_l, REML=TRUE)
		
		
# Human mito copy not significantly different from zero at 0 nM paclitaxel and 0 weeks:
		
		
summary(glht(m1),test=adjusted("none"))


	 # Simultaneous Tests for General Linear Hypotheses

# Fit: lmer(formula = human_mito_copy_TK1 ~ week * conc + (1 | pool/cell), 
    # data = mito_reads_l, REML = TRUE)

# Linear Hypotheses:
                  # Estimate Std. Error z value Pr(>|z|)  
# (Intercept) == 0  0.709306   0.812102   0.873   0.3824        
# week == 0         0.008581   0.203589   0.042   0.9664  
# conc == 0         0.041096   0.024710   1.663   0.0963 .
# week:conc == 0   -0.005587   0.006120  -0.913   0.3613  
# ---
# Signif. codes:  0 ‘***’ 0.001 ‘**’ 0.01 ‘*’ 0.05 ‘.’ 0.1 ‘ ’ 1
# (Adjusted p values reported -- none method)








# use lmer.df = "asymptotic" to agree exactly with above

test(emmeans(m1,lmer.df = "kenward-roger",specs=c("week","conc"),at=list(week=c(0),conc=c(0))))
 # week conc emmean    SE   df t.ratio p.value
    # 0    0  0.709 0.812 32.7 0.873   0.3889  

summary(contrast(emmeans(m1, lmer.df = "kenward-roger", specs="week", at=list(week=c(0,1), conc=0)),list(c(-1,1))))
# NOTE: Results may be misleading due to involvement in interactions
 # contrast estimate    SE   df t.ratio p.value
 # c(-1, 1)  0.00858 0.204 98.9 0.042   0.9666

 summary(contrast(emmeans(m1, lmer.df = "kenward-roger", specs="conc", at=list(conc=c(0,1), week=0)),list(c(-1,1))))
# NOTE: Results may be misleading due to involvement in interactions
 # contrast estimate     SE  df t.ratio p.value
 # c(-1, 1)   0.0411 0.0249 104 1.653   0.1014 


summary(contrast(emmeans(m1, lmer.df = "kenward-roger", specs=c("week","conc"), at=list(week=c(0,1),conc=c(0,1))),list(c(1,-1,-1,1))))
 # contrast        estimate      SE   df t.ratio p.value
 # c(1, -1, -1, 1) -0.00559 0.00614 95.5 -0.910  0.3653 
 
 
# not much diff at av conditional values. However, these results are superior, cf https://stackoverflow.com/questions/52866203/hypothesis-test-for-intercepts-in-general-mixed-linear-models-with-r
# though one purpose of this calx was to compare with human mito copy number in simple mean +/- sem in RH pools, for which "at=list(week=c(0),conc=c(0))" might be better.
# However, this approach evaluates all RH samples better
test(emmeans(m1,lmer.df = "kenward-roger",specs=c("week","conc"),at=list(week=c(3.2),conc=c(27))))
 # week conc emmean    SE   df t.ratio p.value
  # 3.2   27   1.36 0.501 5.36 2.724   0.0387  <<<<<<<<<<<< use in paper
 
 
 
 





# Alternatively, let's look at RH pools, human mito copy number still not sig diff from zero.
# Decided to use this human mito copy number (in addition to that from mixed model at avg conditional values), as empirical (rather than model based estimate) of human mito copy number.

compare(mito_reads_l[mito_reads_l$week == 0 & mito_reads_l$conc == 0,"human_mito_copy_TK1"], rep(0,length(mito_reads_l[mito_reads_l$week == 0 & mito_reads_l$conc == 0,"human_mito_copy_TK1"])))

	# Welch Two Sample t-test     <<<<<<<<<<<<<<<<<<<<<<< use in paper. (Here, "compare" used for Pvals, means and sems. Actually this is same as One sample t-test, equiv to using mu = 0. See below.)

# data:  a and b
# t = 1.7735, df = 5, p-value = 0.1363           
# alternative hypothesis: true difference in means is not equal to 0
# 95 percent confidence interval:
 # -1.245713  6.788623
# sample estimates:
# mean of x mean of y 
 # 2.771455  0.000000 

# [1] "exact P value = 0.13634010346244"
# [1] "mean of a = 2.77145490441202"
# [1] "sem of a = 1.56274650489209"
# [1] "sd of a = 3.82793153430344"
# [1] "number in a = 6"
# [1] "mean of b = 0"
# [1] "sem of b = 0"
# [1] "sd of b = 0"
# [1] "number in b = 6"



# test against hypothesized mean = 0 using one sample t-test. This test is appropriate for hypothesized value = 0 of one distribution. Also, more powerful than pooled variance t test, which is appropriate since want to test counterfactual.


t.test(mito_reads_l[mito_reads_l$week == 0 & mito_reads_l$conc == 0,"human_mito_copy_TK1"], mu = 0)

	# One Sample t-test <<<<<<<<<<<<<<<<<<<<<<< use in paper

# data:  mito_reads_l[mito_reads_l$week == 0 & mito_reads_l$conc == 0,     "human_mito_copy_TK1"]
# t = 1.7735, df = 5, p-value = 0.1363
# alternative hypothesis: true mean is not equal to 0
# 95 percent confidence interval:
 # -1.245713  6.788623
# sample estimates:
# mean of x 
 # 2.771455 
 

# As expected, two sample test with pooled variance is less powerful

t.test(mito_reads_l[mito_reads_l$week == 0 & mito_reads_l$conc == 0,"human_mito_copy_TK1"], 0, var.equal=TRUE)

	# Two Sample t-test

# data:  mito_reads_l[mito_reads_l$week == 0 & mito_reads_l$conc == 0,  and 0    "human_mito_copy_TK1"] and 0
# t = 0.6703, df = 5, p-value = 0.5324
# alternative hypothesis: true difference in means is not equal to 0
# 95 percent confidence interval:
 # -7.856972 13.399882
# sample estimates:
# mean of x mean of y 
 # 2.771455  0.000000 




		
		


		 
		glht_growth <- glht(m1, linfct = c(
					"week == 0",
 					"week + 8*week:conc == 0",
					"week + 25*week:conc == 0",
 					"week + 75*week:conc == 0",
 					"week + (27)*week:conc == 0"))
 					
		glht_drug <- glht(m1, linfct = c(
					"conc + 1*week:conc == 0",
  					"conc + 2*week:conc == 0",
 					"conc + 3*week:conc == 0",
  					"conc + 4*week:conc == 0",
 					"conc + 6*week:conc == 0",
 					"conc + (3.2)*week:conc == 0"))
 					
 		glht_omni <- glht(m1)
 		 					
 		
 	
 	growth_coef <- summary(glht_growth,test = adjusted("none"))$test$coefficients
 	drug_coef <- summary(glht_drug,test = adjusted("none"))$test$coefficients
 	Ix_coef <- summary(glht_omni,test = adjusted("none"))$test$coefficients["week:conc"]				
 	growth_stat <- summary(glht_growth,test = adjusted("none"))$test$tstat
 	drug_stat <- summary(glht_drug,test = adjusted("none"))$test$tstat
 	Ix_stat <- summary(glht_omni,test = adjusted("none"))$test$tstat["week:conc"]
 	
 		
 		
 		
 	human_mito_copy_TK1[1,] <- c(
				 							mito_reads[1,1:4],
				 							growth_coef,
				 							drug_coef,
				 							Ix_coef,
				 							growth_stat,
				 							drug_stat,
				 							Ix_stat,
				 							-log10(2*pnorm(-abs(growth_stat))),
				 							-log10(2*pnorm(-abs(drug_stat))),
				 							-log10(2*pnorm(-abs(Ix_stat)))
				 							)
				 							
				 							
human_mito_copy_TK1
  # Chromosome posS  posE  pos  coef_g_0nM  coef_g_8nM coef_g_25nM coef_g_75nM coef_g_avg  coef_d_w1  coef_d_w2  coef_d_w3 coef_d_w4   coef_d_w6 coef_d_avg  coef_g_d_Ix    Z_g_0nM    Z_g_8nM   Z_g_25nM  Z_g_75nM    Z_g_avg
# 1       chrM    0 16569 8284 0.008580772 -0.03611547   -0.131095  -0.4104465 -0.1422691 0.03550909 0.02992206 0.02433503  0.018748 0.007573935 0.02321762 -0.005587031 0.04214761 -0.2025527 -0.7950387 -1.100397 -0.8491443
    # Z_d_w1   Z_d_w2   Z_d_w3   Z_d_w4    Z_d_w6  Z_d_avg   Z_g_d_Ix log10p_g_0nM log10p_g_8nM log10p_g_25nM log10p_g_75nM log10p_g_avg log10p_d_w1 log10p_d_w2 log10p_d_w3 log10p_d_w4 log10p_d_w6 log10p_d_avg log10p_g_d_Ix
# 1 1.798864 1.916919 1.852495 1.411648 0.3746202 1.794835 -0.9129031    0.0148516   0.07598723     0.3699883     0.5667755    0.4025231    1.142425    1.257681    1.194127   0.8011958   0.1500017     1.138585     0.4421399		 		


# recast as normal P vals not -log10P


	human_mito_copy_TK1 <- data.frame(
									Chromosome = as.character(), 
									posS = integer(), 
									posE = integer(), 
									pos = numeric(), 
									coef_g_0nM = numeric(),
									coef_g_8nM = numeric(),
									coef_g_25nM = numeric(),
									coef_g_75nM = numeric(),
									coef_g_avg = numeric(),
									coef_d_w1 = numeric(),
									coef_d_w2 = numeric(),
									coef_d_w3 = numeric(),
									coef_d_w4 = numeric(),
									coef_d_w6 = numeric(),
									coef_d_avg = numeric(),
									coef_g_d_Ix = numeric(),
									Z_g_0nM = numeric(),
									Z_g_8nM = numeric(),
									Z_g_25nM = numeric(),
									Z_g_75nM = numeric(),
									Z_g_avg = numeric(),
									Z_d_w1 = numeric(),
									Z_d_w2 = numeric(),
									Z_d_w3 = numeric(),
									Z_d_w4 = numeric(),
									Z_d_w6 = numeric(),
									Z_d_avg = numeric(),
									Z_g_d_Ix = numeric(),
									P_g_0nM = numeric(),
									P_g_8nM = numeric(),
									P_g_25nM = numeric(),
									P_g_75nM = numeric(),
									P_g_avg = numeric(),
									P_d_w1 = numeric(),
									P_d_w2 = numeric(),
									P_d_w3 = numeric(),
									P_d_w4 = numeric(),
									P_d_w6 = numeric(),
									P_d_avg = numeric(),
									P_g_d_Ix = numeric(),
									stringsAsFactors=FALSE
									)

	
	human_mito_copy_TK1[1,] <- c(
				 							mito_reads[1,1:4],
				 							growth_coef,
				 							drug_coef,
				 							Ix_coef,
				 							growth_stat,
				 							drug_stat,
				 							Ix_stat,
				 							2*pnorm(-abs(growth_stat)),
				 							2*pnorm(-abs(drug_stat)),
				 							2*pnorm(-abs(Ix_stat))
				 							)
	
human_mito_copy_TK1
  # Chromosome posS  posE  pos  coef_g_0nM  coef_g_8nM coef_g_25nM coef_g_75nM coef_g_avg  coef_d_w1  coef_d_w2  coef_d_w3 coef_d_w4   coef_d_w6 coef_d_avg  coef_g_d_Ix    Z_g_0nM    Z_g_8nM   Z_g_25nM  Z_g_75nM    Z_g_avg
# 1       chrM    0 16569 8284 0.008580772 -0.03611547   -0.131095  -0.4104465 -0.1422691 0.03550909 0.02992206 0.02433503  0.018748 0.007573935 0.02321762 -0.005587031 0.04214761 -0.2025527 -0.7950387 -1.100397 -0.8491443
    # Z_d_w1   Z_d_w2   Z_d_w3   Z_d_w4    Z_d_w6  Z_d_avg   Z_g_d_Ix  P_g_0nM   P_g_8nM P_g_25nM  P_g_75nM  P_g_avg     P_d_w1     P_d_w2    P_d_w3    P_d_w4   P_d_w6    P_d_avg  P_g_d_Ix
# 1 1.798864 1.916919 1.852495 1.411648 0.3746202 1.794835 -0.9129031 0.966381 0.8394847 0.426591 0.2711593 0.395801 0.07204019 0.05524825 0.0639548 0.1580535 0.707943 0.07267994 0.3612935 





# emmeans
# Provides kenward-roger
# lmer.df="asymptotic" agrees with glht exactly

# growth

summary(contrast(emmeans(m1, lmer.df = "kenward-roger", specs="week", at=list(week=c(0,1), conc=27)),list(c(-1,1))))
# NOTE: Results may be misleading due to involvement in interactions
 # contrast estimate    SE   df t.ratio p.value
 # c(-1, 1)   -0.142 0.168 93.4 -0.847  0.3992    <<<<<<<<<<< use in paper
 
 
summary(contrast(emmeans(m1, lmer.df = "kenward-roger", specs="week", at=list(week=c(0,1), conc=27)),list(c(-1,1))))$p.value
# NOTE: Results may be misleading due to involvement in interactions
# [1] 0.3992024 <<<<<<<<<<< use in paper
 
 

# drug
summary(contrast(emmeans(m1, lmer.df = "kenward-roger", specs="conc", at=list(conc=c(0,1), week=3.2)),list(c(-1,1))))
# NOTE: Results may be misleading due to involvement in interactions
 # contrast estimate    SE   df t.ratio p.value
 # c(-1, 1)   0.0232 0.013 27.2 1.780   0.0863      <<<<<<<<<<< use in paper



# Ix
summary(contrast(emmeans(m1, lmer.df = "kenward-roger", specs=c("week","conc"), at=list(week=c(0,1),conc=c(0,1))),list(c(1,-1,-1,1))))
 # contrast        estimate      SE   df t.ratio p.value
 # c(1, -1, -1, 1) -0.00559 0.00614 95.5 -0.910  0.3653    <<<<<<<<<<< use in paper






# pool
anova(lmer(human_mito_copy_TK1 ~ week * conc + (1|pool) + (1|cell), data = mito_reads_l, REML=TRUE), lmer(human_mito_copy_TK1 ~ week * conc + (1|cell), data = mito_reads_l, REML=TRUE))
# refitting model(s) with ML (instead of REML)
# Data: mito_reads_l
# Models:
# lmer(human_mito_copy_TK1 ~ week * conc + (1 | cell), data = mito_reads_l, REML = TRUE): human_mito_copy_TK1 ~ week * conc + (1 | cell)
# lmer(human_mito_copy_TK1 ~ week * conc + (1 | pool) + (1 | cell), data = mito_reads_l, REML = TRUE): human_mito_copy_TK1 ~ week * conc + (1 | pool) + (1 | cell)
                                                                                                    # Df    AIC    BIC  logLik deviance Chisq Chi Df Pr(>Chisq)
# lmer(human_mito_copy_TK1 ~ week * conc + (1 | cell), data = mito_reads_l, REML = TRUE)               6 610.97 627.44 -299.49   598.97                        
# lmer(human_mito_copy_TK1 ~ week * conc + (1 | pool) + (1 | cell), data = mito_reads_l, REML = TRUE)  7 611.86 631.07 -298.93   597.86 1.114      1     0.2912	<<<<<<<<<<< use in paper



# cell
anova(lmer(human_mito_copy_TK1 ~ week * conc + (1|pool) + (1|cell), data = mito_reads_l, REML=TRUE), lmer(human_mito_copy_TK1 ~ week * conc + (1|pool), data = mito_reads_l, REML=TRUE))
# refitting model(s) with ML (instead of REML)
# Data: mito_reads_l
# Models:
# lmer(human_mito_copy_TK1 ~ week * conc + (1 | pool), data = mito_reads_l, REML = TRUE): human_mito_copy_TK1 ~ week * conc + (1 | pool)
# lmer(human_mito_copy_TK1 ~ week * conc + (1 | pool) + (1 | cell), data = mito_reads_l, REML = TRUE): human_mito_copy_TK1 ~ week * conc + (1 | pool) + (1 | cell)
                                                                                                    # Df    AIC    BIC  logLik deviance  Chisq Chi Df Pr(>Chisq)
# lmer(human_mito_copy_TK1 ~ week * conc + (1 | pool), data = mito_reads_l, REML = TRUE)               6 610.10 626.57 -299.05   598.10                         
# lmer(human_mito_copy_TK1 ~ week * conc + (1 | pool) + (1 | cell), data = mito_reads_l, REML = TRUE)  7 611.86 631.07 -298.93   597.86 0.2411      1     0.6234 <<<<<<<<<<< use in paper


# pool and cell together
anova(lmer(human_mito_copy_TK1 ~ week * conc + (1|pool/cell), data = mito_reads_l, REML=TRUE), lm(human_mito_copy_TK1 ~ week * conc, data = mito_reads_l, REML=TRUE))
# Data: mito_reads_l
# Models:
# lm(human_mito_copy_TK1 ~ week * conc, data = mito_reads_l, REML = TRUE): human_mito_copy_TK1 ~ week * conc
# lmer(human_mito_copy_TK1 ~ week * conc + (1 | pool/cell), data = mito_reads_l, REML = TRUE): human_mito_copy_TK1 ~ week * conc + (1 | pool/cell)
                                                                                            # Df    AIC    BIC  logLik deviance  Chisq Chi Df Pr(>Chisq)
# lm(human_mito_copy_TK1 ~ week * conc, data = mito_reads_l, REML = TRUE)                      5 610.10 623.82 -300.05   600.10                         
# lmer(human_mito_copy_TK1 ~ week * conc + (1 | pool/cell), data = mito_reads_l, REML = TRUE)  7 611.86 631.07 -298.93   597.86 2.2397      2     0.3263



# icc

performance::icc(m1)
# # Intraclass Correlation Coefficient

     # Adjusted ICC: 0.123
  # Conditional ICC: 0.119
  
  
# pool
(get_variance(m1)$var.intercept["pool"])/(get_variance(m1)$var.intercept["pool"] + get_variance(m1)$var.intercept["cell:pool"] + get_variance(m1)$var.residual)
     # pool 
# 0.0678317 <<<<<<<<<<< use in paper



# cell 
(get_variance(m1)$var.intercept["cell:pool"])/(get_variance(m1)$var.intercept["pool"] + get_variance(m1)$var.intercept["cell:pool"] + get_variance(m1)$var.residual)
 # cell:pool 
# 0.05493621 <<<<<<<<<<< use in paper


# pool and cell together
(get_variance(m1)$var.intercept["pool"] + get_variance(m1)$var.intercept["cell:pool"])/(get_variance(m1)$var.intercept["pool"] + get_variance(m1)$var.intercept["cell:pool"] + get_variance(m1)$var.residual)
     # pool 
# 0.1227679  
  




# ------------- Mixed mod calx: human mito copy based on mean of seq reads and TK1 ---------------------


human_mean_mito_copy <- data.frame(
									Chromosome = as.character(), 
									posS = integer(), 
									posE = integer(), 
									pos = numeric(), 
									coef_g_0nM = numeric(),
									coef_g_8nM = numeric(),
									coef_g_25nM = numeric(),
									coef_g_75nM = numeric(),
									coef_g_avg = numeric(),
									coef_d_w1 = numeric(),
									coef_d_w2 = numeric(),
									coef_d_w3 = numeric(),
									coef_d_w4 = numeric(),
									coef_d_w6 = numeric(),
									coef_d_avg = numeric(),
									coef_g_d_Ix = numeric(),
									Z_g_0nM = numeric(),
									Z_g_8nM = numeric(),
									Z_g_25nM = numeric(),
									Z_g_75nM = numeric(),
									Z_g_avg = numeric(),
									Z_d_w1 = numeric(),
									Z_d_w2 = numeric(),
									Z_d_w3 = numeric(),
									Z_d_w4 = numeric(),
									Z_d_w6 = numeric(),
									Z_d_avg = numeric(),
									Z_g_d_Ix = numeric(),
									log10p_g_0nM = numeric(),
									log10p_g_8nM = numeric(),
									log10p_g_25nM = numeric(),
									log10p_g_75nM = numeric(),
									log10p_g_avg = numeric(),
									log10p_d_w1 = numeric(),
									log10p_d_w2 = numeric(),
									log10p_d_w3 = numeric(),
									log10p_d_w4 = numeric(),
									log10p_d_w6 = numeric(),
									log10p_d_avg = numeric(),
									log10p_g_d_Ix = numeric(),
									stringsAsFactors=FALSE
									)



				
m1 <- lmer(human_mean_mito_copy ~ week * conc + (1|pool/cell), data = mito_reads_l, REML=TRUE)
		
		
# Human mito copy not significantly different from zero at 0 nM paclitaxel and 0 weeks:
		
		
summary(glht(m1),test=adjusted("none"))

	 # Simultaneous Tests for General Linear Hypotheses

# Fit: lmer(formula = human_mean_mito_copy ~ week * conc + (1 | pool/cell), 
    # data = mito_reads_l, REML = TRUE)

# Linear Hypotheses:
                  # Estimate Std. Error z value Pr(>|z|)  
# (Intercept) == 0  0.488795   0.540739   0.904   0.3660
# week == 0         0.052028   0.130697   0.398   0.6906  
# conc == 0         0.026072   0.015565   1.675   0.0939 .
# week:conc == 0   -0.004617   0.003941  -1.172   0.2414  
# ---
# Signif. codes:  0 ‘***’ 0.001 ‘**’ 0.01 ‘*’ 0.05 ‘.’ 0.1 ‘ ’ 1
# (Adjusted p values reported -- none method)      








# use lmer.df = "asymptotic" to agree exactly with above

test(emmeans(m1,lmer.df = "kenward-roger",specs=c("week","conc"),at=list(week=c(0),conc=c(0))))
 # week conc emmean    SE   df t.ratio p.value
    # 0    0  0.489 0.541 26.8 0.904   0.3742   

summary(contrast(emmeans(m1, lmer.df = "kenward-roger", specs="week", at=list(week=c(0,1), conc=0)),list(c(-1,1))))
# NOTE: Results may be misleading due to involvement in interactions
 # contrast estimate    SE   df t.ratio p.value
 # c(-1, 1)    0.052 0.131 98.2 0.397   0.6921 

 summary(contrast(emmeans(m1, lmer.df = "kenward-roger", specs="conc", at=list(conc=c(0,1), week=0)),list(c(-1,1))))
# NOTE: Results may be misleading due to involvement in interactions
 # contrast estimate     SE  df t.ratio p.value
 # c(-1, 1)   0.0261 0.0157 105 1.665   0.0989 


summary(contrast(emmeans(m1, lmer.df = "kenward-roger", specs=c("week","conc"), at=list(week=c(0,1),conc=c(0,1))),list(c(1,-1,-1,1))))
 # contrast        estimate      SE   df t.ratio p.value
 # c(1, -1, -1, 1) -0.00462 0.00395 95.5 -1.168  0.2457 


# not much diff at av conditional values. However, these results are superior, cf https://stackoverflow.com/questions/52866203/hypothesis-test-for-intercepts-in-general-mixed-linear-models-with-r
# though one purpose of this calx was to compare with human mito copy number in simple mean +/- sem in RH pools, for which "at=list(week=c(0),conc=c(0))" might be better.
# However, this approach evaluates all RH samples better
test(emmeans(m1,lmer.df = "kenward-roger",specs=c("week","conc"),at=list(week=c(3.2),conc=c(27))))
 # week conc emmean    SE   df t.ratio p.value
  # 3.2   27   0.96 0.352 5.27 2.726   0.0393   <<<<<<<<<<<< use in paper







# Alternatively, let's look at RH pools, human mito copy number still not sig diff from zero.
# Decided to use this human mito copy number (in addition to that from mixed model at avg conditional values), as empirical (rather than model based estimate) of human mito copy number.

compare(mito_reads_l[mito_reads_l$week == 0 & mito_reads_l$conc == 0,"human_mean_mito_copy"], rep(0,length(mito_reads_l[mito_reads_l$week == 0 & mito_reads_l$conc == 0,"human_mean_mito_copy"])))

	# Welch Two Sample t-test #   <<<<<<<<<<<<<<<<<<<<<<< (Here, "compare" used for Pvals, means and sems. Actually this is same as One sample t-test, equiv to using mu = 0. See below.)


# data:  a and b
# t = 1.7592, df = 5, p-value = 0.1389
# alternative hypothesis: true difference in means is not equal to 0
# 95 percent confidence interval:
 # -0.9506692  5.0731771
# sample estimates:
# mean of x mean of y 
 # 2.061254  0.000000 

# [1] "exact P value = 0.138863605606064"
# [1] "mean of a = 2.06125398201483"
# [1] "sem of a = 1.17168926221912"
# [1] "sd of a = 2.87004082953492"
# [1] "number in a = 6"
# [1] "mean of b = 0"
# [1] "sem of b = 0"
# [1] "sd of b = 0"
# [1] "number in b = 6"	






# test against hypothesized mean = 0 using one sample t-test. This test is appropriate for hypothesized value = 0 of one distribution. Also, more powerful than pooled variance t test, which is appropriate since want to test counterfactual.

t.test(mito_reads_l[mito_reads_l$week == 0 & mito_reads_l$conc == 0,"human_mean_mito_copy"], mu = 0)

	# One Sample t-test <<<<<<<<<<<<<<<<<<<<<<< use in paper

# data:  mito_reads_l[mito_reads_l$week == 0 & mito_reads_l$conc == 0,     "human_mean_mito_copy"]
# t = 1.7592, df = 5, p-value = 0.1389
# alternative hypothesis: true mean is not equal to 0
# 95 percent confidence interval:
 # -0.9506692  5.0731771
# sample estimates:
# mean of x 
 # 2.061254 
 
 


# As expected, two sample test with pooled variance less powerful

t.test(mito_reads_l[mito_reads_l$week == 0 & mito_reads_l$conc == 0,"human_mean_mito_copy"], 0, var.equal=TRUE)

	# Two Sample t-test

# data:  mito_reads_l[mito_reads_l$week == 0 & mito_reads_l$conc == 0,  and 0    "human_mean_mito_copy"] and 0
# t = 0.66492, df = 5, p-value = 0.5355
# alternative hypothesis: true difference in means is not equal to 0
# 95 percent confidence interval:
 # -5.907546 10.030054
# sample estimates:
# mean of x mean of y 
 # 2.061254  0.000000





		
		


		 
		glht_growth <- glht(m1, linfct = c(
					"week == 0",
 					"week + 8*week:conc == 0",
					"week + 25*week:conc == 0",
 					"week + 75*week:conc == 0",
 					"week + (27)*week:conc == 0"))
 					
		glht_drug <- glht(m1, linfct = c(
					"conc + 1*week:conc == 0",
  					"conc + 2*week:conc == 0",
 					"conc + 3*week:conc == 0",
  					"conc + 4*week:conc == 0",
 					"conc + 6*week:conc == 0",
 					"conc + (3.2)*week:conc == 0"))
 					
 		glht_omni <- glht(m1)
 		 					
 		
 	
 	growth_coef <- summary(glht_growth,test = adjusted("none"))$test$coefficients
 	drug_coef <- summary(glht_drug,test = adjusted("none"))$test$coefficients
 	Ix_coef <- summary(glht_omni,test = adjusted("none"))$test$coefficients["week:conc"]				
 	growth_stat <- summary(glht_growth,test = adjusted("none"))$test$tstat
 	drug_stat <- summary(glht_drug,test = adjusted("none"))$test$tstat
 	Ix_stat <- summary(glht_omni,test = adjusted("none"))$test$tstat["week:conc"]
 	
 		
 		
 		
 	human_mean_mito_copy[1,] <- c(
				 							mito_reads[1,1:4],
				 							growth_coef,
				 							drug_coef,
				 							Ix_coef,
				 							growth_stat,
				 							drug_stat,
				 							Ix_stat,
				 							-log10(2*pnorm(-abs(growth_stat))),
				 							-log10(2*pnorm(-abs(drug_stat))),
				 							-log10(2*pnorm(-abs(Ix_stat)))
				 							)
				 							
				 							
human_mean_mito_copy
  # Chromosome posS  posE  pos coef_g_0nM coef_g_8nM coef_g_25nM coef_g_75nM  coef_g_avg  coef_d_w1  coef_d_w2 coef_d_w3   coef_d_w4    coef_d_w6 coef_d_avg
# 1       chrM    0 16569 8284 0.05202832  0.0150918 -0.06339831  -0.2942516 -0.07263244 0.02145503 0.01683796 0.0122209 0.007603831 -0.001630299 0.01129748
   # coef_g_d_Ix   Z_g_0nM   Z_g_8nM   Z_g_25nM  Z_g_75nM    Z_g_avg   Z_d_w1   Z_d_w2   Z_d_w3    Z_d_w4    Z_d_w6  Z_d_avg  Z_g_d_Ix log10p_g_0nM log10p_g_8nM
# 1 -0.004617065 0.3980828 0.1317644 -0.5964625 -1.222095 -0.6722444 1.749675 1.779576 1.581006 0.9732649 -0.130193 1.489574 -1.171539    0.1607928   0.04809417
  # log10p_g_25nM log10p_g_75nM log10p_g_avg log10p_d_w1 log10p_d_w2 log10p_d_w3 log10p_d_w4 log10p_d_w6 log10p_d_avg log10p_g_d_Ix
# 1     0.2589538     0.6542894    0.2997913    1.095964    1.124098   0.9435653   0.4809315  0.04749149    0.8653882     0.6172947

	 		


# recast as normal P vals not -log10P


	human_mean_mito_copy <- data.frame(
									Chromosome = as.character(), 
									posS = integer(), 
									posE = integer(), 
									pos = numeric(), 
									coef_g_0nM = numeric(),
									coef_g_8nM = numeric(),
									coef_g_25nM = numeric(),
									coef_g_75nM = numeric(),
									coef_g_avg = numeric(),
									coef_d_w1 = numeric(),
									coef_d_w2 = numeric(),
									coef_d_w3 = numeric(),
									coef_d_w4 = numeric(),
									coef_d_w6 = numeric(),
									coef_d_avg = numeric(),
									coef_g_d_Ix = numeric(),
									Z_g_0nM = numeric(),
									Z_g_8nM = numeric(),
									Z_g_25nM = numeric(),
									Z_g_75nM = numeric(),
									Z_g_avg = numeric(),
									Z_d_w1 = numeric(),
									Z_d_w2 = numeric(),
									Z_d_w3 = numeric(),
									Z_d_w4 = numeric(),
									Z_d_w6 = numeric(),
									Z_d_avg = numeric(),
									Z_g_d_Ix = numeric(),
									P_g_0nM = numeric(),
									P_g_8nM = numeric(),
									P_g_25nM = numeric(),
									P_g_75nM = numeric(),
									P_g_avg = numeric(),
									P_d_w1 = numeric(),
									P_d_w2 = numeric(),
									P_d_w3 = numeric(),
									P_d_w4 = numeric(),
									P_d_w6 = numeric(),
									P_d_avg = numeric(),
									P_g_d_Ix = numeric(),
									stringsAsFactors=FALSE
									)

	
	human_mean_mito_copy[1,] <- c(
				 							mito_reads[1,1:4],
				 							growth_coef,
				 							drug_coef,
				 							Ix_coef,
				 							growth_stat,
				 							drug_stat,
				 							Ix_stat,
				 							2*pnorm(-abs(growth_stat)),
				 							2*pnorm(-abs(drug_stat)),
				 							2*pnorm(-abs(Ix_stat))
				 							)
	
human_mean_mito_copy
  # Chromosome posS  posE  pos coef_g_0nM coef_g_8nM coef_g_25nM coef_g_75nM  coef_g_avg  coef_d_w1  coef_d_w2 coef_d_w3   coef_d_w4    coef_d_w6 coef_d_avg
# 1       chrM    0 16569 8284 0.05202832  0.0150918 -0.06339831  -0.2942516 -0.07263244 0.02145503 0.01683796 0.0122209 0.007603831 -0.001630299 0.01129748
   # coef_g_d_Ix   Z_g_0nM   Z_g_8nM   Z_g_25nM  Z_g_75nM    Z_g_avg   Z_d_w1   Z_d_w2   Z_d_w3    Z_d_w4    Z_d_w6  Z_d_avg  Z_g_d_Ix   P_g_0nM   P_g_8nM  P_g_25nM
# 1 -0.004617065 0.3980828 0.1317644 -0.5964625 -1.222095 -0.6722444 1.749675 1.779576 1.581006 0.9732649 -0.130193 1.489574 -1.171539 0.6905692 0.8951706 0.5508663
   # P_g_75nM   P_g_avg     P_d_w1     P_d_w2    P_d_w3    P_d_w4    P_d_w6   P_d_avg  P_g_d_Ix
# 1 0.2216719 0.5014281 0.08017448 0.07514538 0.1138767 0.3304216 0.8964137 0.1363364 0.2413822 # 









# emmeans
# Provides kenward-roger
# lmer.df="asymptotic" agrees with glht exactly

# growth

summary(contrast(emmeans(m1, lmer.df = "kenward-roger", specs="week", at=list(week=c(0,1), conc=27)),list(c(-1,1))))
# NOTE: Results may be misleading due to involvement in interactions
 # contrast estimate    SE   df t.ratio p.value
 # c(-1, 1)  -0.0726 0.108 93.5 -0.671  0.5040     <<<<<<<<<<< use in paper
 
 
summary(contrast(emmeans(m1, lmer.df = "kenward-roger", specs="week", at=list(week=c(0,1), conc=27)),list(c(-1,1))))$p.value
# NOTE: Results may be misleading due to involvement in interactions
# [1] 0.5040421 <<<<<<<<<<< use in paper
 
 

# drug
summary(contrast(emmeans(m1, lmer.df = "kenward-roger", specs="conc", at=list(conc=c(0,1), week=3.2)),list(c(-1,1))))
# NOTE: Results may be misleading due to involvement in interactions
 # contrast estimate      SE   df t.ratio p.value
 # c(-1, 1)   0.0113 0.00765 28.2 1.476   0.1510       <<<<<<<<<<< use in paper



# Ix
summary(contrast(emmeans(m1, lmer.df = "kenward-roger", specs=c("week","conc"), at=list(week=c(0,1),conc=c(0,1))),list(c(1,-1,-1,1))))
 # contrast        estimate      SE   df t.ratio p.value
 # c(1, -1, -1, 1) -0.00462 0.00395 95.5 -1.168  0.2457   <<<<<<<<<<< use in paper








# pool
anova(lmer(human_mean_mito_copy ~ week * conc + (1|pool) + (1|cell), data = mito_reads_l, REML=TRUE), lmer(human_mean_mito_copy ~ week * conc + (1|cell), data = mito_reads_l, REML=TRUE))
# boundary (singular) fit: see ?isSingular
# refitting model(s) with ML (instead of REML)
# Data: mito_reads_l
# Models:
# lmer(human_mean_mito_copy ~ week * conc + (1 | cell), data = mito_reads_l, REML = TRUE): human_mean_mito_copy ~ week * conc + (1 | cell)
# lmer(human_mean_mito_copy ~ week * conc + (1 | pool) + (1 | cell), data = mito_reads_l, REML = TRUE): human_mean_mito_copy ~ week * conc + (1 | pool) + (1 | cell)
                                                                                                     # Df    AIC    BIC  logLik deviance Chisq Chi Df Pr(>Chisq)  
# lmer(human_mean_mito_copy ~ week * conc + (1 | cell), data = mito_reads_l, REML = TRUE)               6 509.48 525.95 -248.74   497.48                          
# lmer(human_mean_mito_copy ~ week * conc + (1 | pool) + (1 | cell), data = mito_reads_l, REML = TRUE)  7 508.22 527.43 -247.11   494.22 3.262      1     0.0709 <<<<<<<<<<< use in paper
# ---
# Signif. codes:  0 ‘***’ 0.001 ‘**’ 0.01 ‘*’ 0.05 ‘.’ 0.1 ‘ ’ 1
	



# cell
anova(lmer(human_mean_mito_copy ~ week * conc + (1|pool) + (1|cell), data = mito_reads_l, REML=TRUE), lmer(human_mean_mito_copy ~ week * conc + (1|pool), data = mito_reads_l, REML=TRUE))

# boundary (singular) fit: see ?isSingular
# refitting model(s) with ML (instead of REML)
# Data: mito_reads_l
# Models:
# lmer(human_mean_mito_copy ~ week * conc + (1 | pool), data = mito_reads_l, REML = TRUE): human_mean_mito_copy ~ week * conc + (1 | pool)
# lmer(human_mean_mito_copy ~ week * conc + (1 | pool) + (1 | cell), data = mito_reads_l, REML = TRUE): human_mean_mito_copy ~ week * conc + (1 | pool) + (1 | cell)
                                                                                                     # Df    AIC    BIC  logLik deviance Chisq Chi Df Pr(>Chisq)
# lmer(human_mean_mito_copy ~ week * conc + (1 | pool), data = mito_reads_l, REML = TRUE)               6 506.22 522.69 -247.11   494.22                        
# lmer(human_mean_mito_copy ~ week * conc + (1 | pool) + (1 | cell), data = mito_reads_l, REML = TRUE)  7 508.22 527.43 -247.11   494.22     0      1          1 <<<<<<<<<<< use in paper



# pool and cell together
anova(lmer(human_mean_mito_copy ~ week * conc + (1|pool/cell), data = mito_reads_l, REML=TRUE), lm(human_mean_mito_copy ~ week * conc, data = mito_reads_l, REML=TRUE))
# Data: mito_reads_l
# Models:
# lm(human_mean_mito_copy ~ week * conc, data = mito_reads_l, REML = TRUE): human_mean_mito_copy ~ week * conc
# lmer(human_mean_mito_copy ~ week * conc + (1 | pool/cell), data = mito_reads_l, REML = TRUE): human_mean_mito_copy ~ week * conc + (1 | pool/cell)
                                                                                             # Df    AIC    BIC  logLik deviance  Chisq Chi Df Pr(>Chisq)
# lm(human_mean_mito_copy ~ week * conc, data = mito_reads_l, REML = TRUE)                      5 508.23 521.95 -249.11   498.23                         
# lmer(human_mean_mito_copy ~ week * conc + (1 | pool/cell), data = mito_reads_l, REML = TRUE)  7 508.22 527.43 -247.11   494.22 4.0084      2     0.1348





# icc


performance::icc(m1)
# # Intraclass Correlation Coefficient

     # Adjusted ICC: 0.107
  # Conditional ICC: 0.104
  
  
  
  
# pool
(get_variance(m1)$var.intercept["pool"])/(get_variance(m1)$var.intercept["pool"] + get_variance(m1)$var.intercept["cell:pool"] + get_variance(m1)$var.residual)
    # pool 
# 0.106935 <<<<<<<<<<< use in paper




# cell
(get_variance(m1)$var.intercept["cell:pool"])/(get_variance(m1)$var.intercept["pool"] + get_variance(m1)$var.intercept["cell:pool"] + get_variance(m1)$var.residual)
   # cell:pool 
# 4.929416e-09 <<<<<<<<<<< use in paper


# pool and cell together
(get_variance(m1)$var.intercept["pool"] + get_variance(m1)$var.intercept["cell:pool"])/(get_variance(m1)$var.intercept["pool"] + get_variance(m1)$var.intercept["cell:pool"] + get_variance(m1)$var.residual)
     # pool 
# 0.1069351




































