# Because very complicated using mgcv to get ICC (cf clone_count_3.R), decided to use lmer for ICC.
# Tried glmmTMB, which is more flexible than lmer and allows neg binom models which are not needed here. But glmmTMB gave warning that it could not converge (though it did give results that were close to lmer and mgcv, but not exactly the same).


library(ggplot2)
library(cowplot)
library(mgcv)
library(multcomp)
library(glmmTMB)
library(sjstats)
library(lme4)
library(Rmisc)
library(performance)
library(insight)
library(emmeans)


# ----------- Enable glht with glmmTMB cf https://stackoverflow.com/questions/51886334/planned-contrasts-on-glmmtmb -------------


glht_glmmTMB <- function (model, ..., component="cond") {
    glht(model, ...,
         coef. = function(x) fixef(x)[[component]],
         vcov. = function(x) vcov(x)[[component]],
         df = NULL)
}
modelparm.glmmTMB <- function (model, coef. = function(x) fixef(x)[[component]],
                               vcov. = function(x) vcov(x)[[component]],
                               df = NULL, component="cond", ...) {
    multcomp:::modelparm.default(model, coef. = coef., vcov. = vcov.,
                        df = df, ...)
}



# --------------- get_legend function ---------------------------


get_legend<-function(myggplot){
  tmp <- ggplot_gtable(ggplot_build(myggplot))
   leg <- which(sapply(tmp$grobs, function(x) x$name) == "guide-box")
   legend <- tmp$grobs[[leg]]
   return(legend)
 }




# ----------------------------- ggplot ---------------------------------



# from https://stackoverflow.com/questions/8197559/emulate-ggplot2-default-color-palette
# function for default ggplot2 colors

gg_color_hue <- function(n) {
  hues = seq(15, 375, length = n + 1)
  hcl(h = hues, l = 65, c = 100)[1:n]
}

theme2 <- theme(
	plot.margin = unit(c(0.8,0.8,0.8,0.8), "cm"),
	panel.grid.major = element_blank(), 
	panel.grid.minor = element_blank(), 
	panel.background = element_blank(), 
	legend.position="none", 
	axis.line.x = element_line(colour = "black", size = 0.1), 
	axis.line.y = element_line(colour = "black", size = 0.1), 
	axis.ticks = element_line(colour = "black", size = 0.1),
	axis.text=element_text(size=12), #numbers on tick marks of x and y axes
	axis.title=element_text(size=14), #titles of x and y axes
	axis.title.y=element_text(margin=margin(t=0,r=7,b=0,l=0), hjust = 0.5), # moves y axis title by adding margin space to bottom, trbl
	axis.title.x=element_text(margin=margin(t=7,r=0,b=0,l=0), hjust = 0.5),  # moves x axis title by adding margin space to top,trbl
	plot.title = element_text(size=32, face="bold", hjust = -0.14), #can provide "A","B", by ggtitle, but used plot_grid wch can shift more left
	plot.subtitle = element_text(size=14, face="plain", hjust = 0.5), #hjust shifts right
	)
	
	
size_point <- 0.3
size_hline <- 0.1


#----------------- TK accuracy ---------------------


# read in RH human sequence reads:
RH_human <- read.table("RH_human_gseq.txt",header=TRUE,sep="\t",stringsAsFactors=FALSE)

# Get rows at beginning of each chromosome:
RH_human_start <- RH_human[RH_human$posS == 0 & RH_human$posE == 1e6,]

# Get rid of ramp ups and ramp downs:
RH_human <- RH_human[c(0,diff(RH_human$pos)) == 1e4,]

# combine RH_human without ramps and RH_human_start:
RH_human <- rbind(RH_human_start,RH_human)

# Sort:
chrOrder<-paste("chr",c(1:22,"X","Y"),sep="")
RH_human$Chromosome <-factor(RH_human$Chromosome, levels=chrOrder)
RH_human <- RH_human[order(RH_human$Chromosome, RH_human$pos), ]
RH_human$Chromosome <- as.character(RH_human$Chromosome)



# get rid of chrY
RH_human <- RH_human[RH_human$Chromosome != "chrY",]


# -------- find middle TK1 -------------

gencode_gtf_ensembl_ucsc <- read.delim("gencode_gtf_ensembl_ucsc_v31.txt",header=TRUE,sep="\t",stringsAsFactors=FALSE,check.names=FALSE)


gencode_gtf_ensembl_ucsc[gencode_gtf_ensembl_ucsc$geneSymbol=="TK1",]
      # geneSymbol           tx_id Chromosome    geneS    geneE strand         gene_id      gene_type geneLength txLength cdsLength                                       gene_description    5utrS    5utrE
# 54647        TK1 ENST00000588734      chr17 78174075 78187233      - ENSG00000167900 protein_coding      13159     1681       804 thymidine kinase 1 [Source:HGNC Symbol;Acc:HGNC:11830] 78186995 78187233
      # 5utrDiff    3utrS    3utrE 3utrDiff exonCount
# 54647      239 78174121 78174758      638         6



TK1_coord <- mean(gencode_gtf_ensembl_ucsc[gencode_gtf_ensembl_ucsc$geneSymbol=="TK1","geneS"],gencode_gtf_ensembl_ucsc[gencode_gtf_ensembl_ucsc$geneSymbol=="TK1","geneE"])


# ----------- find peak TK1 coords and calcuate delta from middle TK1 ----------------

# Restrict search to within delta of TK1, because CEN and TEL becomes larger than TK1 in some samples
delta <- 1e6

RH_human_TK1 <- RH_human[RH_human$Chromosome=="chr17" & RH_human$pos >= TK1_coord-delta & RH_human$pos <= TK1_coord+delta,]



TK1_map <- RH_human_TK1[apply(RH_human_TK1[,c(5:ncol(RH_human_TK1))],2,which.max),"pos"]
names(TK1_map) <- colnames(RH_human_TK1[,c(5:ncol(RH_human_TK1))])


TK1_map <- TK1_map - TK1_coord


# ------------------ Read in and prepare ancillary tables -------------------------------

cell <- read.table("cell_label_info.txt",sep="\t",stringsAsFactors=FALSE,header=TRUE)


# -------- Prepare TK1_map for gam loop ---------------------------

# automatically puts into long form:
TK1_map_l <- as.data.frame(TK1_map)

colnames(TK1_map_l) <- "delta"
TK1_map_l$RH_ID <- row.names(TK1_map_l)
row.names(TK1_map_l) <- NULL



# # No need for reshape
# TK1_map_l <- reshape(TK1_map, 
  # varying = c(colnames(TK1_map)), 
  # v.names = "delta",
  # timevar = "RH_ID", 
  # times = c(colnames(TK1_map)), 
  # new.row.names = 1:200,
  # direction = "long")
  
TK1_map_l$week <- 0
TK1_map_l[grepl("_w0_",TK1_map_l$RH_ID),]$week <- 0
TK1_map_l[grepl("_w1_",TK1_map_l$RH_ID),]$week <- 1
TK1_map_l[grepl("_w2_",TK1_map_l$RH_ID),]$week <- 2
TK1_map_l[grepl("_w3_",TK1_map_l$RH_ID),]$week <- 3
TK1_map_l[grepl("_w4_",TK1_map_l$RH_ID),]$week <- 4
TK1_map_l[grepl("_w6_",TK1_map_l$RH_ID),]$week <- 6

TK1_map_l$conc <- 0
TK1_map_l[grepl("_d0",TK1_map_l$RH_ID),]$conc <- 0
TK1_map_l[grepl("_d8",TK1_map_l$RH_ID),]$conc <- 8
TK1_map_l[grepl("_d25",TK1_map_l$RH_ID),]$conc <- 25
TK1_map_l[grepl("_d75",TK1_map_l$RH_ID),]$conc <- 75

TK1_map_l$pool <- 0
TK1_map_l[grepl("RH1_",TK1_map_l$RH_ID),]$pool <- 1
TK1_map_l[grepl("RH2_",TK1_map_l$RH_ID),]$pool <- 2
TK1_map_l[grepl("RH3_",TK1_map_l$RH_ID),]$pool <- 3
TK1_map_l[grepl("RH4_",TK1_map_l$RH_ID),]$pool <- 4
TK1_map_l[grepl("RH5_",TK1_map_l$RH_ID),]$pool <- 5
TK1_map_l[grepl("RH6_",TK1_map_l$RH_ID),]$pool <- 6


TK1_map_l <- merge(TK1_map_l,cell)


TK1_map_l$pool <- as.factor(TK1_map_l$pool)
TK1_map_l$cell <- as.factor(TK1_map_l$cell)



# ------------------ (1) Plot TK1 individual samples ------------------------



n = length(unique(TK1_map_l$pool))
colores_1 = gg_color_hue(n)


p1 <- ggplot() + 
		theme2 + 
		theme(legend.key=element_blank()) +
		geom_boxplot(
			data= TK1_map_l,
			alpha=0.1,
			outlier.shape=NA, 
			notch=FALSE,
			lwd=0.1,
			fatten = 2,
			show.legend=FALSE,
			aes(x=0,y=delta) # change to fill = pool if want colored bars by pool
			) + 
		geom_point(
			size=1.0,
			stroke=0.4,
			shape=1,
			data= TK1_map_l, 
			aes(
				x=jitter(as.numeric(rep(0,nrow(TK1_map_l))),factor=2.0), 
				y=delta, 
				colour=as.factor(pool)
				)
			) +
		scale_color_manual(
			values=colores_1,
			name ="Pool", 
			labels=c(1,2,3,4,5,6)
			) +
	 	# scale_shape_manual(
	 		# values=c(1,2,3,4),
			# name =expression("Conc (nM)"), 
			# labels=c(0,8,25,75)
			# ) +				
		# scale_fill_manual( # # for color filled bar legends
			# values=colores_1,
			# name ="Pool", 
			# labels=c(1,2,3,4,5,6)
			# ) +
	 	guides(
	 		colour = guide_legend(override.aes = list(fill=NA,shape=1,size=1),nrow=6,ncol=1,byrow=TRUE,order=1,title.position = "top", title.vjust = 0.5)# ,
	 		# shape = guide_legend(override.aes = list(fill=NA,size=1,colour="darkgrey"),nrow=4,ncol=1,byrow=TRUE,order=2,title.position = "top", title.vjust = 0.5)
	 		) +
		theme(
	 		plot.margin = unit(c(top=0.5,right=0.5,bottom=0.5,left=0.5), "cm"),
		 	# plot.subtitle = element_text(size=9, face="plain", hjust = 0.5, margin=margin(0,0,6,0)),
		 	axis.title.x=element_blank(),
	        axis.text.x=element_blank(),
	        axis.ticks.x=element_blank(),
	        axis.line.x=element_blank(),
	 		# legend.position=c(0,0),
			legend.position="none",
			legend.margin=margin(t=0,r=0,b=0,l=0,unit = "pt"),
		 	legend.box.margin=margin(t=0,r=0,b=0,l=0,unit = "pt"),
		 	legend.key.height = unit(0.1, "cm"),
			legend.key.width = unit(0.1, "cm"),
		 	legend.spacing.y = unit(0.1, 'cm'),
		 	legend.spacing.x = unit(0.1, 'cm'),
		 	legend.title = element_text(size = 11),  
		 	legend.text = element_text(size = 10),
	 		legend.title.align=0.5
	 		) +
 		scale_x_discrete(
	 		# breaks = NULL,
			# labels = NULL, 
			expand = expand_scale(add = .6)
			) +
		scale_y_continuous(
			expand = expand_scale(mult = .05),
			breaks = seq(-2e5, 5e5, 2e5),
			labels = seq(-0.2,0.4,0.2)
			) +
		ylab(expression(Distance~(Mb))) +
		coord_cartesian(ylim = c(-3e5,5e5)) +
		labs(subtitle="TK1")
		# ggtitle("") + 
		# xlab("Cell") +
print(p1)


# legend_pool <- get_legend(p1)
# p1 <- p1 + theme(legend.position="none")


# ---------------- (3) TK1 collapsed to RH pools -----------------------------


TK1_map_sum <- summarySE(data= TK1_map_l,measurevar="delta",groupvars="pool")



p3 <- ggplot() + 
		theme2 + 
		theme(legend.key=element_blank()) +
		geom_boxplot(
			data= TK1_map_sum,
			alpha=0.1,
			outlier.shape=NA, 
			notch=FALSE,
			lwd=0.1,
			fatten = 2,
			show.legend=FALSE,
			aes(x=0,y=delta) # change to fill = pool if want colored bars by pool
			) + 
		geom_point(
			size=1.0,
			stroke=0.4,
			shape=1,
			data= TK1_map_sum, 
			aes(
				x=jitter(as.numeric(rep(0,nrow(TK1_map_sum))),factor=3.0), 
				y=delta, 
				colour=as.factor(pool)
				)
			) +
		scale_color_manual(
			values=colores_1,
			name ="Pool", 
			labels=c(1,2,3,4,5,6)
			) +
	 	# scale_shape_manual(
	 		# values=c(1,2,3,4),
			# name =expression("Conc (nM)"), 
			# labels=c(0,8,25,75)
			# ) +				
		# scale_fill_manual( # # for color filled bar legends
			# values=colores_1,
			# name ="Pool", 
			# labels=c(1,2,3,4,5,6)
			# ) +
	 	guides(
	 		colour = guide_legend(override.aes = list(fill=NA,shape=1,size=1),nrow=6,ncol=1,byrow=TRUE,order=1,title.position = "top", title.vjust = 0.5)# ,
	 		# shape = guide_legend(override.aes = list(fill=NA,size=1,colour="darkgrey"),nrow=4,ncol=1,byrow=TRUE,order=2,title.position = "top", title.vjust = 0.5)
	 		) +
		theme(
	 		plot.margin = unit(c(top=0.5,right=0.5,bottom=0.5,left=0.5), "cm"),
		 	# plot.subtitle = element_text(size=9, face="plain", hjust = 0.5, margin=margin(0,0,6,0)),
		 	axis.title.x=element_blank(),
	        axis.text.x=element_blank(),
	        axis.ticks.x=element_blank(),
	        axis.line.x=element_blank(),
	 		legend.position="none",
			legend.margin=margin(t=0,r=0,b=0,l=0,unit = "pt"),
		 	legend.box.margin=margin(t=0,r=0,b=0,l=0,unit = "pt"),
		 	legend.key.height = unit(0.1, "cm"),
			legend.key.width = unit(0.1, "cm"),
		 	legend.spacing.y = unit(0.1, 'cm'),
		 	legend.spacing.x = unit(0.1, 'cm'),
		 	legend.title = element_text(size = 11),  
		 	legend.text = element_text(size = 10),
	 		legend.title.align=0.5
	 		) +
 		scale_x_discrete(
			# labels = labels, 
			expand = expand_scale(add = .6)
			) +
		scale_y_continuous(
			expand = expand_scale(mult = .05),
			breaks = seq(-2e5, 2e5, 2e5),
			labels = seq(-0.2, 0.2, 0.2)
			) +
		ylab(expression(Distance~(Mb))) +
		labs(subtitle="TK1")
		# coord_cartesian(ylim = c(-3e5,5e5)) +
		# ggtitle("") + 
		# xlab("Cell") +
print(p3)





#----------------- CEN accuracy ---------------------


cen <- read.table("hg38_centromere.txt",header=TRUE,sep="\t",stringsAsFactors=FALSE)

cen_min <- aggregate(chromStart~chrom,data=cen,min)
cen_max <- aggregate(chromEnd~chrom,data=cen,max)
cen_lim <- merge(cen_min,cen_max)
cen_lim$cen_coord <- round(rowMeans(cen_lim[,c("chromStart","chromEnd")]))



# Sort:
chrOrder<-paste("chr",c(1:22,"X","Y"),sep="")
cen_lim$chrom <-factor(cen_lim$chrom, levels=chrOrder)
cen_lim <- cen_lim[order(cen_lim$chrom), ]
cen_lim$chrom <- as.character(cen_lim$chrom)
row.names(cen_lim) <- NULL


# get rid of chrY
cen_lim <- cen_lim[cen_lim$chrom != "chrY",]




# ---------- Compact RH_human to regions within delta of CEN ------------------

delta <- 0e6 # 0 Mb to left and right of beginning and end of CEN, respectively. Chose 0 Mb because genes close to outside of CEN messed up peak identification.

RH_human_cen <- data.frame()

# ignore chrY

for(i in c(1:22,"X")) {

RH_temp <- RH_human[RH_human$Chromosome == paste0("chr",i) & RH_human$pos >= cen_lim[cen_lim$chrom==paste0("chr",i),"chromStart"]-delta & RH_human$pos <= cen_lim[cen_lim$chrom==paste0("chr",i),"chromEnd"]+delta,]

RH_human_cen <- rbind(RH_human_cen,RH_temp)

	}


# ---------------- Pick max regions in CEN --------------------


RH_human_cen_map <- RH_human_cen[0,c(1,5:ncol(RH_human_cen))]

chr_names <- c(1:22,"X")



for(i in c(1:23)) {

RH_human_cen_map[i,1] <- paste0("chr",chr_names[i])
RH_human_cen_map[i,2:ncol(RH_human_cen_map)] <- RH_human_cen[RH_human_cen$Chromosome == paste0("chr",chr_names[i]),][apply(RH_human_cen[RH_human_cen$Chromosome == paste0("chr",chr_names[i]),c(5:ncol(RH_human_cen))],2,which.max),"pos"]

}


RH_human_cen_map[,c(2:ncol(RH_human_cen_map))] <- RH_human_cen_map[,c(2:ncol(RH_human_cen_map))] - cen_lim$cen_coord



# -------- Prepare RH_human_cen_map for gam loop and ggplot ---------------------------




RH_human_cen_map_l <- reshape(RH_human_cen_map, 
  varying = c(colnames(RH_human_cen_map[,c(2:ncol(RH_human_cen_map))])), 
  v.names = "delta",
  timevar = "RH_ID", 
  times = c(colnames(RH_human_cen_map[,c(2:ncol(RH_human_cen_map))])), 
  new.row.names = 1:3e3,
  direction = "long")
  

  
RH_human_cen_map_l$week <- 0
RH_human_cen_map_l[grepl("_w0_",RH_human_cen_map_l$RH_ID),]$week <- 0
RH_human_cen_map_l[grepl("_w1_",RH_human_cen_map_l$RH_ID),]$week <- 1
RH_human_cen_map_l[grepl("_w2_",RH_human_cen_map_l$RH_ID),]$week <- 2
RH_human_cen_map_l[grepl("_w3_",RH_human_cen_map_l$RH_ID),]$week <- 3
RH_human_cen_map_l[grepl("_w4_",RH_human_cen_map_l$RH_ID),]$week <- 4
RH_human_cen_map_l[grepl("_w6_",RH_human_cen_map_l$RH_ID),]$week <- 6

RH_human_cen_map_l$conc <- 0
RH_human_cen_map_l[grepl("_d0",RH_human_cen_map_l$RH_ID),]$conc <- 0
RH_human_cen_map_l[grepl("_d8",RH_human_cen_map_l$RH_ID),]$conc <- 8
RH_human_cen_map_l[grepl("_d25",RH_human_cen_map_l$RH_ID),]$conc <- 25
RH_human_cen_map_l[grepl("_d75",RH_human_cen_map_l$RH_ID),]$conc <- 75

RH_human_cen_map_l$pool <- 0
RH_human_cen_map_l[grepl("RH1_",RH_human_cen_map_l$RH_ID),]$pool <- 1
RH_human_cen_map_l[grepl("RH2_",RH_human_cen_map_l$RH_ID),]$pool <- 2
RH_human_cen_map_l[grepl("RH3_",RH_human_cen_map_l$RH_ID),]$pool <- 3
RH_human_cen_map_l[grepl("RH4_",RH_human_cen_map_l$RH_ID),]$pool <- 4
RH_human_cen_map_l[grepl("RH5_",RH_human_cen_map_l$RH_ID),]$pool <- 5
RH_human_cen_map_l[grepl("RH6_",RH_human_cen_map_l$RH_ID),]$pool <- 6


RH_human_cen_map_l <- merge(RH_human_cen_map_l,cell)


RH_human_cen_map_l$pool <- as.factor(RH_human_cen_map_l$pool)
RH_human_cen_map_l$cell <- as.factor(RH_human_cen_map_l$cell)

chrOrder<-paste("chr",c(1:22,"X","Y"),sep="")
RH_human_cen_map_l$Chromosome <-factor(RH_human_cen_map_l$Chromosome, levels=chrOrder)


# ---------------- (2) Plot CEN accuracy vs chromosome number ----------------------------


p2 <- ggplot() + 
		theme2 + 
		theme(legend.key=element_blank()) +
		geom_boxplot(
			data= RH_human_cen_map_l,
			alpha=0.1,
			outlier.shape=NA, 
			notch=FALSE,
			lwd=0.1,
			fatten = 2,
			show.legend=FALSE,
			aes(x=Chromosome,y=delta) # change to fill = pool if want colored bars by pool
			) + 
		geom_point(
			size=0.5,
			stroke=0.4,
			shape=1,
			data= RH_human_cen_map_l, 
			aes(
				x=jitter(as.numeric(Chromosome),factor=0.3), 
				y=delta, 
				colour=as.factor(pool)
				)
			) +
		scale_color_manual(
			values=colores_1,
			name ="Pool", 
			labels=c(1,2,3,4,5,6)
			) +
	 	# scale_shape_manual(
	 		# values=c(1,2,3,4),
			# name =expression("Conc (nM)"), 
			# labels=c(0,8,25,75)
			# ) +				
		# scale_fill_manual( # # for color filled bar legends
			# values=colores_1,
			# name ="Pool", 
			# labels=c(1,2,3,4,5,6)
			# ) +
	 	guides(
	 		colour = guide_legend(override.aes = list(fill=NA,shape=1,size=1),nrow=6,ncol=1,byrow=TRUE,order=1,title.position = "top", title.vjust = 0.5)# ,
	 		# shape = guide_legend(override.aes = list(fill=NA,size=1,colour="darkgrey"),nrow=4,ncol=1,byrow=TRUE,order=2,title.position = "top", title.vjust = 0.5)
	 		) +
		theme(
	 		plot.margin = unit(c(top=0.5,right=0.5,bottom=0.5,left=0.5), "cm"),
		 	# plot.subtitle = element_text(size=9, face="plain", hjust = 0.5, margin=margin(0,0,6,0)),
		 	# axis.title.x=element_blank(),
	        # axis.text.x=element_blank(),
	        # axis.ticks.x=element_blank(),
	        # axis.line.x=element_blank(),
	 		legend.position="right",
			legend.margin=margin(t=0,r=0,b=0,l=10,unit = "pt"),
		 	legend.box.margin=margin(t=0,r=0,b=0,l=10,unit = "pt"),
		 	legend.key.height = unit(0.1, "cm"),
			legend.key.width = unit(0.1, "cm"),
		 	legend.spacing.y = unit(0.1, 'cm'),
		 	legend.spacing.x = unit(0.1, 'cm'),
		 	legend.title = element_text(size = 11),  
		 	legend.text = element_text(size = 10),
	 		legend.title.align=0.5
	 		) +
 		scale_x_discrete(
	 		breaks = levels(RH_human_cen_map_l$Chromosome)[seq(1,23,2)],
			labels = c(seq(1,21,2),"X"), 
			expand = expand_scale(add = 1.2)
			) +
		scale_y_continuous(
			expand = expand_scale(mult = .05),
			breaks = seq(-2e6, 2e6, 2e6),
			labels = seq(-2,2,2)
			) +
		ylab(expression(Distance~(Mb))) +
		labs(subtitle="CEN")
		# coord_cartesian(ylim = c(-3e5,5e5)) +
		# ggtitle("") + 
		# xlab("Cell") +
print(p2)





# ---------------- (4) CEN collapsed to RH pools -----------------------------


RH_human_cen_map_sum <- summarySE(data= RH_human_cen_map_l,measurevar="delta",groupvars="pool")



p4 <- ggplot() + 
		theme2 + 
		theme(legend.key=element_blank()) +
		geom_boxplot(
			data= RH_human_cen_map_sum,
			alpha=0.1,
			outlier.shape=NA, 
			notch=FALSE,
			lwd=0.1,
			fatten = 2,
			show.legend=FALSE,
			aes(x=0,y=delta) # change to fill = pool if want colored bars by pool
			) + 
		geom_point(
			size=1.0,
			stroke=0.4,
			shape=1,
			data= RH_human_cen_map_sum, 
			aes(
				x=jitter(as.numeric(rep(0,nrow(RH_human_cen_map_sum))),factor=3.0), 
				y=delta, 
				colour=as.factor(pool)
				)
			) +
		scale_color_manual(
			values=colores_1,
			name ="Pool", 
			labels=c(1,2,3,4,5,6)
			) +
	 	# scale_shape_manual(
	 		# values=c(1,2,3,4),
			# name =expression("Conc (nM)"), 
			# labels=c(0,8,25,75)
			# ) +				
		# scale_fill_manual( # # for color filled bar legends
			# values=colores_1,
			# name ="Pool", 
			# labels=c(1,2,3,4,5,6)
			# ) +
	 	guides(
	 		colour = guide_legend(override.aes = list(fill=NA,shape=1,size=1),nrow=6,ncol=1,byrow=TRUE,order=1,title.position = "top", title.vjust = 0.5)# ,
	 		# shape = guide_legend(override.aes = list(fill=NA,size=1,colour="darkgrey"),nrow=4,ncol=1,byrow=TRUE,order=2,title.position = "top", title.vjust = 0.5)
	 		) +
		theme(
	 		plot.margin = unit(c(top=0.5,right=0.5,bottom=0.5,left=0.5), "cm"),
		 	# plot.subtitle = element_text(size=9, face="plain", hjust = 0.5, margin=margin(0,0,6,0)),
		 	axis.title.x=element_blank(),
	        axis.text.x=element_blank(),
	        axis.ticks.x=element_blank(),
	        axis.line.x=element_blank(),
	 		legend.position="right",
			legend.margin=margin(t=0,r=0,b=0,l=0,unit = "pt"),
		 	legend.box.margin=margin(t=0,r=0,b=0,l=5,unit = "pt"),
		 	legend.key.height = unit(0.1, "cm"),
			legend.key.width = unit(0.1, "cm"),
		 	legend.spacing.y = unit(0.1, 'cm'),
		 	legend.spacing.x = unit(0.1, 'cm'),
		 	legend.title = element_text(size = 11),  
		 	legend.text = element_text(size = 10),
	 		legend.title.align=0.5
	 		) +
 		scale_x_discrete(
			# labels = labels, 
			expand = expand_scale(add = .6)
			) +
		scale_y_continuous(
			expand = expand_scale(mult = .05),
			breaks = seq(-1e5, 0.5e5, 0.5e5),
			labels = seq(-100,50,50)
			) +
		ylab(expression(Distance~(kb))) +
		labs(subtitle="CEN")
		# coord_cartesian(ylim = c(-3e5,5e5)) +
		# ggtitle("") + 
		# xlab("Cell") +
print(p4)





# ---------------- (5) CEN collapsed to chromosome -----------------------------


RH_human_cen_map_sum_chr <- summarySE(data= RH_human_cen_map_l,measurevar="delta",groupvars="Chromosome")


n_sum_chr = length(unique(RH_human_cen_map_sum_chr$Chromosome))
colores_2 = gg_color_hue(n_sum_chr)



p5 <- ggplot() + 
		theme2 + 
		theme(legend.key=element_blank()) +
		geom_boxplot(
			data= RH_human_cen_map_sum_chr,
			alpha=0.1,
			outlier.shape=NA, 
			notch=FALSE,
			lwd=0.1,
			fatten = 2,
			show.legend=FALSE,
			aes(x=0,y=delta) # change to fill = pool if want colored bars by pool
			) + 
		geom_point(
			size=1.0,
			stroke=0.4,
			shape=1,
			data= RH_human_cen_map_sum_chr, 
			aes(
				x=jitter(as.numeric(rep(0,nrow(RH_human_cen_map_sum_chr))),factor=3.0), 
				y=delta, 
				colour=as.factor(Chromosome)
				)
			) +
		scale_color_manual(
			values=colores_2,
			name ="Chromosome", 
			labels=c(1:22,"X")
			) +
	 	# scale_shape_manual(
	 		# values=c(1,2,3,4),
			# name =expression("Conc (nM)"), 
			# labels=c(0,8,25,75)
			# ) +				
		# scale_fill_manual( # # for color filled bar legends
			# values=colores_1,
			# name ="Pool", 
			# labels=c(1,2,3,4,5,6)
			# ) +
	 	guides(
	 		colour = guide_legend(override.aes = list(fill=NA,shape=1,size=1),ncol=2,byrow=TRUE,order=1,title.position = "top", title.vjust = 0.5)# ,
	 		# shape = guide_legend(override.aes = list(fill=NA,size=1,colour="darkgrey"),nrow=4,ncol=1,byrow=TRUE,order=2,title.position = "top", title.vjust = 0.5)
	 		) +
		theme(
	 		plot.margin = unit(c(top=0.5,right=0.5,bottom=0.5,left=0.5), "cm"),
		 	# plot.subtitle = element_text(size=9, face="plain", hjust = 0.5, margin=margin(0,0,6,0)),
		 	axis.title.x=element_blank(),
	        axis.text.x=element_blank(),
	        axis.ticks.x=element_blank(),
	        axis.line.x=element_blank(),
	 		legend.position="right",
			legend.margin=margin(t=0,r=0,b=0,l=0,unit = "pt"),
		 	legend.box.margin=margin(t=0,r=0,b=0,l=0,unit = "pt"),
		 	legend.key.height = unit(0.1, "cm"),
			legend.key.width = unit(0.1, "cm"),
		 	legend.spacing.y = unit(0.1, 'cm'),
		 	legend.spacing.x = unit(0.1, 'cm'),
		 	legend.title = element_text(size = 10),  
		 	legend.text = element_text(size = 9),
	 		legend.title.align=0.5
	 		) +
 		scale_x_discrete(
			# labels = labels, 
			expand = expand_scale(add = .6)
			) +
		scale_y_continuous(
			expand = expand_scale(mult = .05),
			breaks = seq(-5e5, 5e5, 5e5),
			labels = seq(-0.5,0.5,0.5)
			) +
		ylab(expression(Distance~(Mb))) +
		labs(subtitle="CEN")
		# coord_cartesian(ylim = c(-3e5,5e5)) +
		# ggtitle("") + 
		# xlab("Cell") +
print(p5)





#------------------Make file --------------------------


dev.new(width=7.5,height=5.5)	
	
(p_comp <- ggdraw() +
	draw_plot(p1, x = 0.0, y = 0.57, width = 0.23, height = 0.43) +
 	draw_plot(p2, x = 0.3, y = 0.5, width = 0.65, height = 0.5) +
 	
 	draw_plot(p3, x = 0.0, y = 0.0, width = 0.23, height = 0.47) + 
 	draw_plot(p4, x = 0.28, y = 0.0, width = 0.31, height = 0.47) +
 	# draw_plot(legend_pool, x = 0.4, y = 0.0, width = 0.5, height = 0.5) +
 	draw_plot(p5, x = 0.63, y = 0.0, width = 0.36, height = 0.47) +
 	
 	
 		
 	draw_plot_label(
	 	c("A", "B", "C", "D", "", "E"), 
	 	x = c(0.0, 0.3, 0.0, 0.28, 0.6,0.63), 
	 	y = c(1.0, 1.0, 0.48, 0.48, 0.48, 0.48), 
	 	size = c(16, 16, 16, 16, 0, 16)
	 	)
)


pdf("map_acc_1.pdf",width=7.5,height=5.5,useDingbats=FALSE)
p_comp
dev.off()

# png("map_acc_1.png", width=7.5, height=5.5,units="in",res=300)
# p_comp
# dev.off()





# ------------------------------------------------------
# ----------- TK1 mix mod calx -------------------------
# ------------------------------------------------------


# m1_gam <- gam(delta ~ week * conc + s(pool, bs = "re") + s(cell, bs = "re"), data = TK1_map_l, family = gaussian, method = "REML")

# m1 <- glmmTMB(delta ~ week * conc + (1|pool/cell), data = TK1_map_l, family = gaussian, REML=TRUE)

m1 <- lmer(delta ~ week * conc + (1|pool/cell), data = TK1_map_l,  REML=TRUE)


summary(m1)


# Linear mixed model fit by REML ['lmerMod']
# Formula: delta ~ week * conc + (1 | pool/cell)
   # Data: TK1_map_l

# REML criterion at convergence: 3084.7

# Scaled residuals: 
    # Min      1Q  Median      3Q     Max 
# -2.1801 -0.5203 -0.1120  0.3691  5.0708 

# Random effects:
 # Groups    Name        Variance Std.Dev.
 # cell:pool (Intercept) 8.44e+09  91869  
 # pool      (Intercept) 2.21e+10 148677  
 # Residual              4.01e+10 200243  
# Number of obs: 115, groups:  cell:pool, 30; pool, 6

# Fixed effects:
            # Estimate Std. Error t value
# (Intercept) -64184.0    78298.8  -0.820
# week         26341.4    13122.4   2.007
# conc          1576.9     1659.9   0.950
# week:conc    -1043.4      392.3  -2.660

# Correlation of Fixed Effects:
          # (Intr) week   conc  
# week      -0.466              
# conc      -0.363  0.420       
# week:conc  0.299 -0.611 -0.811



# get p vals at zero vals

K <- diag(1,4,4)

summary(glht(m1,linfct=K),test=adjusted("none"))

	 # Simultaneous Tests for General Linear Hypotheses

# Fit: lmer(formula = delta ~ week * conc + (1 | pool/cell), data = TK1_map_l, 
    # REML = TRUE)

# Linear Hypotheses:
       # Estimate Std. Error z value Pr(>|z|)   
# 1 == 0 -64184.0    78298.8  -0.820  0.41237   
# 2 == 0  26341.4    13122.4   2.007  0.04471 * 
# 3 == 0   1576.9     1659.9   0.950  0.34211   
# 4 == 0  -1043.4      392.3  -2.660  0.00782 **
# ---
# Signif. codes:  0 ‘***’ 0.001 ‘**’ 0.01 ‘*’ 0.05 ‘.’ 0.1 ‘ ’ 1
# (Adjusted p values reported -- none method)


# Alternatively, and more simply:
summary(glht(m1),test=adjusted("none"))

	 # Simultaneous Tests for General Linear Hypotheses

# Fit: lmer(formula = delta ~ week * conc + (1 | pool/cell), data = TK1_map_l, 
    # REML = TRUE)

# Linear Hypotheses:
                 # Estimate Std. Error z value Pr(>|z|)   
# (Intercept) == 0 -64184.0    78298.8  -0.820  0.41237   
# week == 0         26341.4    13122.4   2.007  0.04471 * 
# conc == 0          1576.9     1659.9   0.950  0.34211   
# week:conc == 0    -1043.4      392.3  -2.660  0.00782 **
# ---
# Signif. codes:  0 ‘***’ 0.001 ‘**’ 0.01 ‘*’ 0.05 ‘.’ 0.1 ‘ ’ 1
# (Adjusted p values reported -- none method)



# use lmer.df = "asymptotic" to agree exactly with above

test(emmeans(m1,lmer.df = "kenward-roger",specs=c("week","conc"),at=list(week=c(0),conc=c(0))))
 # week conc emmean    SE   df t.ratio p.value
    # 0    0 -64184 78307 9.83 -0.820  0.4318 

summary(contrast(emmeans(m1, lmer.df = "kenward-roger", specs="week", at=list(week=c(0,1), conc=0)),list(c(-1,1))))
# NOTE: Results may be misleading due to involvement in interactions
 # contrast estimate    SE   df t.ratio p.value
 # c(-1, 1)    26341 13180 99.5 1.999   0.0484 

 summary(contrast(emmeans(m1, lmer.df = "kenward-roger", specs="conc", at=list(conc=c(0,1), week=0)),list(c(-1,1))))
# NOTE: Results may be misleading due to involvement in interactions
 # contrast estimate   SE   df t.ratio p.value
 # c(-1, 1)     1577 1667 96.1 0.946   0.3466 


summary(contrast(emmeans(m1, lmer.df = "kenward-roger", specs=c("week","conc"), at=list(week=c(0,1),conc=c(0,1))),list(c(1,-1,-1,1))))
 # contrast        estimate  SE   df t.ratio p.value
 # c(1, -1, -1, 1)    -1043 394 94.7 -2.650  0.0094 
 
 
# not much diff at av conditional values. However, this approach advisable, cf https://stackoverflow.com/questions/52866203/hypothesis-test-for-intercepts-in-general-mixed-linear-models-with-r
test(emmeans(m1,lmer.df = "kenward-roger",specs=c("week","conc"),at=list(week=c(3.2),conc=c(27))))
 # week conc emmean    SE  df t.ratio p.value
  # 3.2   27 -27465 66391 5.1 -0.414  0.6959 <<<<<<<<<<<< use in paper





		glht_growth <- glht(m1, linfct = c(
					"week == 0",
 					"week + 8*week:conc == 0",
					"week + 25*week:conc == 0",
 					"week + 75*week:conc == 0",
 					"week + (27)*week:conc == 0"))
 					
		glht_drug <- glht(m1, linfct = c(
					"conc + 1*week:conc == 0",
  					"conc + 2*week:conc == 0",
 					"conc + 3*week:conc == 0",
  					"conc + 4*week:conc == 0",
 					"conc + 6*week:conc == 0",
 					"conc + (3.2)*week:conc == 0"))
 					
 		glht_omni <- glht(m1)
 		
 		
 		 					
 		
 					
 	growth_stat <- summary(glht_growth,test = adjusted("none"))$test$tstat
 	drug_stat <- summary(glht_drug,test = adjusted("none"))$test$tstat
 	Ix_stat <- summary(glht_omni,test = adjusted("none"))$test$tstat["week:conc"]
 	growth_coef <- summary(glht_growth,test = adjusted("none"))$test$coefficients
 	drug_coef <- summary(glht_drug,test = adjusted("none"))$test$coefficients
 	Ix_coef <- summary(glht_omni,test = adjusted("none"))$test$coefficients["week:conc"]
 	
 	

 	
 	
 	
 	
growth_stat
                   # week    week + 8 * week:conc   week + 25 * week:conc   week + 75 * week:conc week + (27) * week:conc 
             # 2.00736041              1.56764210              0.02428486             -2.18131351             -0.17096945  

drug_stat
   # conc + 1 * week:conc     conc + 2 * week:conc     conc + 3 * week:conc     conc + 4 * week:conc     conc + 6 * week:conc conc + (3.2) * week:conc 
               # 0.3919402               -0.4545061               -1.5757192               -2.6061176               -3.3468023               -1.8064913   

Ix_stat
 # week:conc 
# -2.659676 



growth_coef
                   # week    week + 8 * week:conc   week + 25 * week:conc   week + 75 * week:conc week + (27) * week:conc 
              # 26341.394               17994.104                 256.115              -51914.442               -1830.707 

drug_coef
    # conc + 1 * week:conc     conc + 2 * week:conc     conc + 3 * week:conc     conc + 4 * week:conc     conc + 6 * week:conc conc + (3.2) * week:conc 
                # 533.5360                -509.8752               -1553.2863               -2596.6975               -4683.5197               -1761.9685 


Ix_coef
 # week:conc 
# -1043.41

 	
 	
 	
 	

2*pnorm(-abs(growth_stat))
                   # week    week + 8 * week:conc   week + 25 * week:conc   week + 75 * week:conc week + (27) * week:conc 
             # 0.04471130              0.11696468              0.98062539              0.02916024              0.86424779   


2*pnorm(-abs(drug_stat))
    # conc + 1 * week:conc     conc + 2 * week:conc     conc + 3 * week:conc     conc + 4 * week:conc     conc + 6 * week:conc conc + (3.2) * week:conc 
             # 0.695102386              0.649464619              0.115090531              0.009157500              0.000817495              0.070841637  


2*pnorm(-abs(Ix_stat))
  # week:conc 
# 0.007821595 






# emmeans
# Provides kenward-roger
# lmer.df="asymptotic" agrees with glht exactly

# growth

summary(contrast(emmeans(m1, lmer.df = "kenward-roger", specs="week", at=list(week=c(0,1), conc=27)),list(c(-1,1))))
# NOTE: Results may be misleading due to involvement in interactions
 # contrast estimate    SE   df t.ratio p.value
 # c(-1, 1)    -1831 10738 92.5 -0.170  0.8650  <<<<<<<<<<< use in paper
 
 
summary(contrast(emmeans(m1, lmer.df = "kenward-roger", specs="week", at=list(week=c(0,1), conc=27)),list(c(-1,1))))$p.value
# NOTE: Results may be misleading due to involvement in interactions
# [1] 0.8649922 <<<<<<<<<<< use in paper
 
 

# drug
summary(contrast(emmeans(m1, lmer.df = "kenward-roger", specs="conc", at=list(conc=c(0,1), week=3.2)),list(c(-1,1))))
# NOTE: Results may be misleading due to involvement in interactions
 # contrast estimate  SE   df t.ratio p.value
 # c(-1, 1)    -1762 981 25.6 -1.796  0.0843   <<<<<<<<<<< use in paper



# Ix
summary(contrast(emmeans(m1, lmer.df = "kenward-roger", specs=c("week","conc"), at=list(week=c(0,1),conc=c(0,1))),list(c(1,-1,-1,1))))
 # contrast        estimate  SE   df t.ratio p.value
 # c(1, -1, -1, 1)    -1043 394 94.7 -2.650  0.0094  <<<<<<<<<<< use in paper




 

# pool and cell together
anova(lmer(delta ~ week * conc + (1|pool/cell), data = TK1_map_l,  REML=TRUE),lm(delta ~ week * conc, data = TK1_map_l,  REML=TRUE))

D# ata: TK1_map_l
# Models:
# lm(delta ~ week * conc, data = TK1_map_l, REML = TRUE): delta ~ week * conc
# lmer(delta ~ week * conc + (1 | pool/cell), data = TK1_map_l, REML = TRUE): delta ~ week * conc + (1 | pool/cell)
                                                                           # Df    AIC    BIC  logLik deviance Chisq Chi Df Pr(>Chisq)    
# lm(delta ~ week * conc, data = TK1_map_l, REML = TRUE)                      5 3202.0 3215.8 -1596.0   3192.0                            
# lmer(delta ~ week * conc + (1 | pool/cell), data = TK1_map_l, REML = TRUE)  7 3172.2 3191.5 -1579.1   3158.2 33.79      2  4.598e-08 ***
# ---
# Signif. codes:  0 ‘***’ 0.001 ‘**’ 0.01 ‘*’ 0.05 ‘.’ 0.1 ‘ ’ 1


# pool alone
# "delta ~ week * conc + (1|pool) + (1|cell)" == "delta ~ week * conc + (1|pool/cell)" because of nested labeling of "cell" term
anova(lmer(delta ~ week * conc + (1|pool) + (1|cell), data = TK1_map_l,  REML=TRUE),lmer(delta ~ week * conc + (1|cell), data = TK1_map_l,  REML=TRUE))
# Models:
# lmer(delta ~ week * conc + (1 | cell), data = TK1_map_l, REML = TRUE): delta ~ week * conc + (1 | cell)
# lmer(delta ~ week * conc + (1 | pool) + (1 | cell), data = TK1_map_l, REML = TRUE): delta ~ week * conc + (1 | pool) + (1 | cell)
                                                                                   # Df    AIC    BIC  logLik deviance  Chisq Chi Df Pr(>Chisq)   
# lmer(delta ~ week * conc + (1 | cell), data = TK1_map_l, REML = TRUE)               6 3178.0 3194.4 -1583.0   3166.0                            
# lmer(delta ~ week * conc + (1 | pool) + (1 | cell), data = TK1_map_l, REML = TRUE)  7 3172.2 3191.5 -1579.1   3158.2 7.7338      1    0.00542 ** <<<<<<<<<<< use in paper




#cell alone 
anova(lmer(delta ~ week * conc + (1|pool) + (1|cell), data = TK1_map_l,  REML=TRUE),lmer(delta ~ week * conc + (1|pool), data = TK1_map_l,  REML=TRUE))
# Data: TK1_map_l
# Models:
# lmer(delta ~ week * conc + (1 | pool), data = TK1_map_l, REML = TRUE): delta ~ week * conc + (1 | pool)
# lmer(delta ~ week * conc + (1 | pool) + (1 | cell), data = TK1_map_l, REML = TRUE): delta ~ week * conc + (1 | pool) + (1 | cell)
                                                                                   # Df    AIC    BIC  logLik deviance Chisq Chi Df Pr(>Chisq)  
# lmer(delta ~ week * conc + (1 | pool), data = TK1_map_l, REML = TRUE)               6 3173.2 3189.7 -1580.6   3161.2                          
# lmer(delta ~ week * conc + (1 | pool) + (1 | cell), data = TK1_map_l, REML = TRUE)  7 3172.2 3191.5 -1579.1   3158.2 2.951      1    0.08582 <<<<<<<<<<< use in paper
# ---
# Signif. codes:  0 ‘***’ 0.001 ‘**’ 0.01 ‘*’ 0.05 ‘.’ 0.1 ‘ ’ 1


 
# total random effects (pool + cell) icc
performance::icc(m1)
# # Intraclass Correlation Coefficient

     # Adjusted ICC: 0.432
  # Conditional ICC: 0.402

get_variance(m1)
# $var.fixed
# [1] 5275028308

# $var.random
# [1] 30544788852

# $var.residual
# [1] 40097199350

# $var.distribution
# [1] 40097199350

# $var.dispersion
# [1] 0

# $var.intercept
  # cell:pool        pool 
 # 8439871933 22104916919 


# icc for pool only

(get_variance(m1)$var.intercept["pool"])/(get_variance(m1)$var.intercept["pool"] + get_variance(m1)$var.intercept["cell:pool"] + get_variance(m1)$var.residual)
     # pool 
# 0.3129147 <<<<<<<<<< use in paper



# icc for cell only
(get_variance(m1)$var.intercept["cell:pool"])/(get_variance(m1)$var.intercept["pool"] + get_variance(m1)$var.intercept["cell:pool"] + get_variance(m1)$var.residual)
# cell:pool 
# 0.1194739 <<<<<<<<<< use in paper


# icc for pool and cell together

(get_variance(m1)$var.intercept["pool"] + get_variance(m1)$var.intercept["cell:pool"])/(get_variance(m1)$var.intercept["pool"] + get_variance(m1)$var.intercept["cell:pool"] + get_variance(m1)$var.residual)
     # pool 
# 0.4323886 



# -------------------------------------------------------------------------------
# ------------------------ CEN mix mod calx -------------------------------------
# -------------------------------------------------------------------------------



# m1_cen <- gam(delta ~ week * conc + s(Chromosome, bs = "re") + s(pool, bs = "re") + s(cell, bs = "re"), data = RH_human_cen_map_l, family = gaussian, method = "REML")
m1_cen <- lmer(delta ~ week * conc + (1|Chromosome) + (1|pool/cell), data = RH_human_cen_map_l,  REML=TRUE)


# mgcv converged and gave nearly exactly same vals as lmer. glmmTMB failed to converge and gave different p vals. So, lmer OK and gives ICC simply

summary(m1_cen)

# Linear mixed model fit by REML ['lmerMod']
# Formula: delta ~ week * conc + (1 | Chromosome) + (1 | pool/cell)
   # Data: RH_human_cen_map_l

# REML criterion at convergence: 78233

# Scaled residuals: 
    # Min      1Q  Median      3Q     Max 
# -3.8604 -0.2570  0.0085  0.3322  5.6818 

# Random effects:
 # Groups     Name        Variance  Std.Dev.
 # cell:pool  (Intercept) 6.724e+08  25931  
 # Chromosome (Intercept) 2.360e+11 485781  
 # pool       (Intercept) 3.222e+09  56766  
 # Residual               4.052e+11 636552  
# Number of obs: 2645, groups:  cell:pool, 30; Chromosome, 23; pool, 6

# Fixed effects:
             # Estimate Std. Error t value
# (Intercept) -13066.41  108213.55  -0.121
# week         -4909.36    8520.09  -0.576
# conc           -72.85    1027.55  -0.071
# week:conc       12.43     256.51   0.048

# Correlation of Fixed Effects:
          # (Intr) week   conc  
# week      -0.229              
# conc      -0.160  0.454       
# week:conc  0.145 -0.603 -0.862
# convergence code: 0
# Model failed to converge with max|grad| = 0.0100628 (tol = 0.002, component 1)



# get p vals at zero vals

K <- diag(1,4,4)

summary(glht(m1_cen,linfct=K),test=adjusted("none"))

	 # Simultaneous Tests for General Linear Hypotheses

# Fit: lmer(formula = delta ~ week * conc + (1 | Chromosome) + (1 | 
    # pool/cell), data = RH_human_cen_map_l, REML = TRUE)

# Linear Hypotheses:
        # Estimate Std. Error z value Pr(>|z|)
# 1 == 0 -13066.41  108213.55  -0.121    0.904
# 2 == 0  -4909.36    8520.09  -0.576    0.564
# 3 == 0    -72.85    1027.55  -0.071    0.943
# 4 == 0     12.43     256.51   0.048    0.961
# (Adjusted p values reported -- none method)


# Alternatively, and more simply:

summary(glht(m1_cen),test=adjusted("none"))

	 # Simultaneous Tests for General Linear Hypotheses

# Fit: lmer(formula = delta ~ week * conc + (1 | Chromosome) + (1 | 
    # pool/cell), data = RH_human_cen_map_l, REML = TRUE)

# Linear Hypotheses:
                  # Estimate Std. Error z value Pr(>|z|)
# (Intercept) == 0 -13066.41  108213.55  -0.121    0.904   
# week == 0         -4909.36    8520.09  -0.576    0.564
# conc == 0           -72.85    1027.55  -0.071    0.943
# week:conc == 0       12.43     256.51   0.048    0.961
# (Adjusted p values reported -- none method)





test(emmeans(m1_cen,lmer.df = "kenward-roger",specs=c("week","conc"),at=list(week=c(0),conc=c(0))))
# week conc emmean     SE   df t.ratio p.value
    # 0    0 -13066 108216 27.2 -0.121  0.9048  

summary(contrast(emmeans(m1_cen, lmer.df = "kenward-roger", specs="week", at=list(week=c(0,1), conc=0)),list(c(-1,1))))
# NOTE: Results may be misleading due to involvement in interactions
 # contrast estimate   SE   df t.ratio p.value
 # c(-1, 1)    -4909 8538 1572 -0.575  0.5654 

 summary(contrast(emmeans(m1_cen, lmer.df = "kenward-roger", specs="conc", at=list(conc=c(0,1), week=0)),list(c(-1,1))))
# NOTE: Results may be misleading due to involvement in interactions
# contrast estimate   SE  df t.ratio p.value
 # c(-1, 1)    -72.9 1033 309 -0.071  0.9438 


summary(contrast(emmeans(m1_cen, lmer.df = "kenward-roger", specs=c("week","conc"), at=list(week=c(0,1),conc=c(0,1))),list(c(1,-1,-1,1))))
 # contrast        estimate  SE   df t.ratio p.value
 # c(1, -1, -1, 1)     12.4 257 2053 0.048   0.9615 
 
 
# not much diff at av conditional values. However, these results are superior, cf https://stackoverflow.com/questions/52866203/hypothesis-test-for-intercepts-in-general-mixed-linear-models-with-r
test(emmeans(m1_cen,lmer.df = "kenward-roger",specs=c("week","conc"),at=list(week=c(3.2),conc=c(27))))
 # week conc emmean     SE df t.ratio p.value
  # 3.2   27 -29669 104854 24 -0.283  0.7796 <<<<<<<<<<<< use in paper







		glht_growth <- glht(m1_cen, linfct = c(
					"week == 0",
 					"week + 8*week:conc == 0",
					"week + 25*week:conc == 0",
 					"week + 75*week:conc == 0",
 					"week + (27)*week:conc == 0"))
 					
		glht_drug <- glht(m1_cen, linfct = c(
					"conc + 1*week:conc == 0",
  					"conc + 2*week:conc == 0",
 					"conc + 3*week:conc == 0",
  					"conc + 4*week:conc == 0",
 					"conc + 6*week:conc == 0",
 					"conc + (3.2)*week:conc == 0"))
 					
 		glht_omni <- glht(m1_cen)
 		
 		
 		 					
 		
 					
 	growth_stat <- summary(glht_growth,test = adjusted("none"))$test$tstat
 	drug_stat <- summary(glht_drug,test = adjusted("none"))$test$tstat
 	Ix_stat <- summary(glht_omni,test = adjusted("none"))$test$tstat["week:conc"]
 	growth_coef <- summary(glht_growth,test = adjusted("none"))$test$coefficients
 	drug_coef <- summary(glht_drug,test = adjusted("none"))$test$coefficients
 	Ix_coef <- summary(glht_omni,test = adjusted("none"))$test$coefficients["week:conc"]
 	
 	

 	
 	
 	
 	
growth_stat
                   # week    week + 8 * week:conc   week + 25 * week:conc   week + 75 * week:conc week + (27) * week:conc 
             # -0.5762094              -0.6444526              -0.6652495              -0.2541225              -0.6510537  

drug_stat
    # conc + 1 * week:conc     conc + 2 * week:conc     conc + 3 * week:conc     conc + 4 * week:conc     conc + 6 * week:conc conc + (3.2) * week:conc 
            # -0.073978466             -0.074960766             -0.066695386             -0.042924937              0.002088144             -0.063075469   

Ix_stat
 # week:conc 
# 0.04846773 





growth_coef
                   # week    week + 8 * week:conc   week + 25 * week:conc   week + 75 * week:conc week + (27) * week:conc 
              # -4909.355               -4809.895               -4598.541               -3976.912               -4573.676 
              
              
              
drug_coef
    # conc + 1 * week:conc     conc + 2 * week:conc     conc + 3 * week:conc     conc + 4 * week:conc     conc + 6 * week:conc conc + (3.2) * week:conc 
              # -60.419720               -47.987139               -35.554559               -23.121978                 1.743183               -33.068043



Ix_coef
# week:conc 
 # 12.43258 
 	
 	
 	
 	

2*pnorm(-abs(growth_stat))
                   # week    week + 8 * week:conc   week + 25 * week:conc   week + 75 * week:conc week + (27) * week:conc 
              # 0.5644736               0.5192820               0.5058909               0.7994009               0.5150118  


2*pnorm(-abs(drug_stat))
    # conc + 1 * week:conc     conc + 2 * week:conc     conc + 3 * week:conc     conc + 4 * week:conc     conc + 6 * week:conc conc + (3.2) * week:conc 
               # 0.9410275                0.9402459                0.9468242                0.9657614                0.9983339                0.9497064  


2*pnorm(-abs(Ix_stat))
# week:conc 
# 0.9613435  







# emmeans
# Provides kenward-roger
# lmer.df="asymptotic" agrees with glht exactly

# growth

summary(contrast(emmeans(m1_cen, lmer.df = "kenward-roger", specs="week", at=list(week=c(0,1), conc=27)),list(c(-1,1))))
# NOTE: Results may be misleading due to involvement in interactions
 # contrast estimate   SE   df t.ratio p.value
 # c(-1, 1)    -4574 7039 2293 -0.650  0.5159  <<<<<<<<<<< use in paper
 
 
summary(contrast(emmeans(m1_cen, lmer.df = "kenward-roger", specs="week", at=list(week=c(0,1), conc=27)),list(c(-1,1))))$p.value
# NOTE: Results may be misleading due to involvement in interactions
# [1] 0.5159145 <<<<<<<<<<< use in paper
 
 

# drug
summary(contrast(emmeans(m1_cen, lmer.df = "kenward-roger", specs="conc", at=list(conc=c(0,1), week=3.2)),list(c(-1,1))))
# NOTE: Results may be misleading due to involvement in interactions
 # contrast estimate  SE   df t.ratio p.value
 # c(-1, 1)    -33.1 528 27.6 -0.063  0.9505    <<<<<<<<<<< use in paper



# Ix
summary(contrast(emmeans(m1_cen, lmer.df = "kenward-roger", specs=c("week","conc"), at=list(week=c(0,1),conc=c(0,1))),list(c(1,-1,-1,1))))
 # contrast        estimate  SE   df t.ratio p.value
 # c(1, -1, -1, 1)     12.4 257 2053 0.048   0.9615   <<<<<<<<<<< use in paper








# Chromosome alone
anova(lmer(delta ~ week * conc + (1|Chromosome) + (1|pool) + (1|cell), data = RH_human_cen_map_l,  REML=TRUE),lmer(delta ~ week * conc + (1|pool)  + (1|cell), data = RH_human_cen_map_l,  REML=TRUE))
# boundary (singular) fit: see ?isSingular
# refitting model(s) with ML (instead of REML)
# Data: RH_human_cen_map_l
# Models:
# lmer(delta ~ week * conc + (1 | pool) + (1 | cell), data = RH_human_cen_map_l, REML = TRUE): delta ~ week * conc + (1 | pool) + (1 | cell)
# lmer(delta ~ week * conc + (1 | Chromosome) + (1 | pool) + (1 | cell), data = RH_human_cen_map_l, REML = TRUE): delta ~ week * conc + (1 | Chromosome) + (1 | pool) + (1 | cell)
                                                                                                               # Df   AIC   BIC logLik deviance  Chisq Chi Df Pr(>Chisq)    
# lmer(delta ~ week * conc + (1 | pool) + (1 | cell), data = RH_human_cen_map_l, REML = TRUE)                     7 79396 79437 -39691    79382                             
# lmer(delta ~ week * conc + (1 | Chromosome) + (1 | pool) + (1 | cell), data = RH_human_cen_map_l, REML = TRUE)  8 78321 78368 -39152    78305 1077.2      1  < 2.2e-16 ***  <<<<<<<<<<<<<< use in paper
# ---
# Signif. codes:  0 ‘***’ 0.001 ‘**’ 0.01 ‘*’ 0.05 ‘.’ 0.1 ‘ ’ 1


anova(lmer(delta ~ week * conc + (1|Chromosome) + (1|pool) + (1|cell), data = RH_human_cen_map_l,  REML=TRUE),lmer(delta ~ week * conc + (1|pool)  + (1|cell), data = RH_human_cen_map_l,  REML=TRUE))$Pr
# [1]            NA 3.032582e-236 <<<<<<<<<<<<<< use in paper






# pool alone
anova(lmer(delta ~ week * conc + (1|Chromosome) + (1|pool) + (1|cell), data = RH_human_cen_map_l,  REML=TRUE),lmer(delta ~ week * conc + (1|Chromosome) + (1|cell), data = RH_human_cen_map_l,  REML=TRUE))

# Data: RH_human_cen_map_l <<<<<<<<<<< use in paper
# Models:
# lmer(delta ~ week * conc + (1 | Chromosome) + (1 | cell), data = RH_human_cen_map_l, REML = TRUE): delta ~ week * conc + (1 | Chromosome) + (1 | cell)
# lmer(delta ~ week * conc + (1 | Chromosome) + (1 | pool) + (1 | cell), data = RH_human_cen_map_l, REML = TRUE): delta ~ week * conc + (1 | Chromosome) + (1 | pool) + (1 | cell)
                                                                                                               # Df   AIC   BIC logLik deviance  Chisq Chi Df Pr(>Chisq)  
# lmer(delta ~ week * conc + (1 | Chromosome) + (1 | cell), data = RH_human_cen_map_l, REML = TRUE)               7 78323 78364 -39155    78309                           
# lmer(delta ~ week * conc + (1 | Chromosome) + (1 | pool) + (1 | cell), data = RH_human_cen_map_l, REML = TRUE)  8 78321 78368 -39152    78305 4.4661      1    0.03457 *
# ---
# Signif. codes:  0 ‘***’ 0.001 ‘**’ 0.01 ‘*’ 0.05 ‘.’ 0.1 ‘ ’ 1




#cell alone 
anova(lmer(delta ~ week * conc + (1|Chromosome) + (1|pool) + (1|cell), data = RH_human_cen_map_l,  REML=TRUE),lmer(delta ~ week * conc + (1|Chromosome)  + (1|pool), data = RH_human_cen_map_l,  REML=TRUE))

# Data: RH_human_cen_map_l
# Models:
# lmer(delta ~ week * conc + (1 | Chromosome) + (1 | pool), data = RH_human_cen_map_l, REML = TRUE): delta ~ week * conc + (1 | Chromosome) + (1 | pool)
# lmer(delta ~ week * conc + (1 | Chromosome) + (1 | pool) + (1 | cell), data = RH_human_cen_map_l, REML = TRUE): delta ~ week * conc + (1 | Chromosome) + (1 | pool) + (1 | cell)
                                                                                                               # Df   AIC   BIC logLik deviance  Chisq Chi Df Pr(>Chisq)
# lmer(delta ~ week * conc + (1 | Chromosome) + (1 | pool), data = RH_human_cen_map_l, REML = TRUE)               7 78319 78360 -39152    78305                         
# lmer(delta ~ week * conc + (1 | Chromosome) + (1 | pool) + (1 | cell), data = RH_human_cen_map_l, REML = TRUE)  8 78321 78368 -39152    78305 0.0935      1     0.7598 <<<<<<<<<<<<< use in paper



# pool and cell together

anova(lmer(delta ~ week * conc + (1|Chromosome) + (1|pool/cell), data = RH_human_cen_map_l,  REML=TRUE),lmer(delta ~ week * conc + (1|Chromosome), data = RH_human_cen_map_l,  REML=TRUE))

# Data: RH_human_cen_map_l
# Models:
# lmer(delta ~ week * conc + (1 | Chromosome), data = RH_human_cen_map_l, REML = TRUE): delta ~ week * conc + (1 | Chromosome)
# lmer(delta ~ week * conc + (1 | Chromosome) + (1 | pool/cell), data = RH_human_cen_map_l, REML = TRUE): delta ~ week * conc + (1 | Chromosome) + (1 | pool/cell)
                                                                                                       # Df   AIC   BIC logLik deviance  Chisq Chi Df Pr(>Chisq)   
# lmer(delta ~ week * conc + (1 | Chromosome), data = RH_human_cen_map_l, REML = TRUE)                    6 78327 78362 -39157    78315                            
# lmer(delta ~ week * conc + (1 | Chromosome) + (1 | pool/cell), data = RH_human_cen_map_l, REML = TRUE)  8 78321 78368 -39152    78305 10.201      2   0.006092 **
# ---
# Signif. codes:  0 ‘***’ 0.001 ‘**’ 0.01 ‘*’ 0.05 ‘.’ 0.1 ‘ ’ 1




 
 # icc for all random effects summed
performance::icc(m1_cen)
# # Intraclass Correlation Coefficient

     # Adjusted ICC: 0.372
  # Conditional ICC: 0.372



# icc for each random effect
get_variance(m1_cen)

# $var.fixed
# [1] 77892630

# $var.random
# [1] 239878496425

# $var.residual
# [1] 405198110461

# $var.distribution
# [1] 405198110461

# $var.dispersion
# [1] 0

# $var.intercept
   # cell:pool   Chromosome         pool 
   # 672437961 235983646912   3222411553 




# icc for Chromosome

(get_variance(m1_cen)$var.intercept["Chromosome"])/(get_variance(m1_cen)$var.intercept["pool"] + get_variance(m1_cen)$var.intercept["cell:pool"] + get_variance(m1_cen)$var.intercept["Chromosome"] + get_variance(m1_cen)$var.residual)
# Chromosome 
 # 0.3658227 <<<<<<< use in paper

(get_variance(m1_cen)$var.intercept["Chromosome"])/(get_variance(m1_cen)$var.random + get_variance(m1_cen)$var.residual)
# Chromosome 
 # 0.3658227 




# icc for pool

(get_variance(m1_cen)$var.intercept["pool"])/(get_variance(m1_cen)$var.intercept["pool"] + get_variance(m1_cen)$var.intercept["cell:pool"] + get_variance(m1_cen)$var.intercept["Chromosome"] + get_variance(m1_cen)$var.residual)
       # pool 
# 0.004995394 <<<<<<< use in paper

(get_variance(m1_cen)$var.intercept["pool"])/(get_variance(m1_cen)$var.random + get_variance(m1_cen)$var.residual)
       # pool 
# 0.004995394 






# icc for cell

(get_variance(m1_cen)$var.intercept["cell:pool"])/(get_variance(m1_cen)$var.intercept["pool"] + get_variance(m1_cen)$var.intercept["cell:pool"] + get_variance(m1_cen)$var.intercept["Chromosome"] + get_variance(m1_cen)$var.residual)
  # cell:pool 
# 0.001042416 <<<<<<< use in paper

(get_variance(m1_cen)$var.intercept["cell:pool"])/(get_variance(m1_cen)$var.random + get_variance(m1_cen)$var.residual)
  # cell:pool 
# 0.001042416 




# icc for pool and cell together

(get_variance(m1_cen)$var.intercept["pool"] + get_variance(m1_cen)$var.intercept["cell:pool"])/(get_variance(m1_cen)$var.intercept["pool"] + get_variance(m1_cen)$var.intercept["cell:pool"] + get_variance(m1_cen)$var.intercept["Chromosome"] + get_variance(m1_cen)$var.residual)
       # pool 
# 0.006037809 

(get_variance(m1_cen)$var.intercept["pool"] + get_variance(m1_cen)$var.intercept["cell:pool"])/(get_variance(m1_cen)$var.random + get_variance(m1_cen)$var.residual)
       # pool 
# 0.006037809 















