## Manhattan plot for hum_to_hum and ham_to_ham alignments in both HEK, A23 and pool averaged RH cells.
## Also, find correlations between HEK293 and human DNA in RH pools as well as correlations between A23 and hamster DNA in RH pools.

#install.packages("ggplot2")
library(ggplot2)
library(cowplot) #used with plot_grid 
library(pdftools)
library(magick)

#----------------Aesthetics ---------------------------


theme2 <- theme(
	plot.margin = unit(c(t=1.2,r=0.4,b=1.2,l=0.4), "cm"),
	panel.grid.major = element_blank(), 
	panel.grid.minor = element_blank(), 
	panel.background = element_blank(), 
	legend.position="none", 
	axis.line.x = element_line(colour = "black", size = 0.1), 
	axis.line.y = element_line(colour = "black", size = 0.1), 
	axis.ticks = element_line(colour = "black", size = 0.1),
	axis.text=element_text(size=12), #numbers on tick marks of x and y axes
	axis.title=element_text(size=14), #titles of x and y axes
	axis.title.y=element_text(margin=margin(0,13,0,0)), #moves y axis title by adding margin space to bottom
	axis.title.x=element_text(margin=margin(10,0,0,0)),  #moves x axis title by adding margin space to top
	plot.title = element_text(size=32, face="bold", hjust = -0.14), #can provide "A","B", by ggtitle, but used plot_grid wch can shift more left
	plot.subtitle = element_text(size=14, face="plain", hjust = 0.5) #hjust shifts right
	)




# darkest two hues from 3-class PuBuGn in color brewer
# cb1<-rep(c("#1c9099", "#a6bddb"), 12)

# # darkest two hues from 3-class PuBu in color brewer
# cb1<-rep(c("#2b8cbe", "#a6bddb"), 12)


# #attractive pinks, greys
# cb1<-c("#999999", "#E69F00", "#56B4E9", "#009E73", "#F0E442", "#0072B2", "#D55E00", "#CC79A7","#999999", "#E69F00", "#56B4E9", "#E69F00", "#009E73", "#F0E442", "#0072B2", "#56B4E9", "#009E73", "#F0E442", "#0072B2", "#D55E00", "#CC79A7","#999999", "#D55E00", "#CC79A7")

# cb1_rev <- c("#CC79A7", "#D55E00", "#0072B2", "#F0E442", "#009E73", "#56B4E9", "#E69F00","#999999", "#CC79A7", "#D55E00", "#0072B2", "#D55E00", "#F0E442", "#009E73", "#56B4E9", "#0072B2", "#F0E442", "#009E73", "#56B4E9", "#E69F00","#999999", "#CC79A7", "#E69F00","#999999")

# #'4-class RdBu'
# cb2 <- c('#ca0020','#f4a582','#92c5de','#0571b0','#ca0020','#f4a582','#92c5de','#0571b0','#ca0020','#f4a582','#92c5de','#f4a582','#0571b0','#ca0020','#f4a582','#92c5de','#0571b0','#ca0020','#f4a582','#92c5de','#0571b0','#ca0020','#92c5de','#0571b0')

# #'4-class RdYlBu'
# cb3 <- c('#d7191c','#fdae61','#abd9e9','#2c7bb6','#d7191c','#fdae61','#abd9e9','#2c7bb6','#d7191c','#fdae61','#abd9e9','#fdae61','#2c7bb6','#d7191c','#fdae61','#abd9e9','#2c7bb6','#d7191c','#fdae61','#abd9e9','#2c7bb6','#d7191c','#abd9e9','#2c7bb6')
	
	
size_point <- 0.3
size_hline <- 0.1

# If desired, modify balloon code. Probably not a good idea in this context, though.
# balloon_scale <- 0.8 # inflation factor for significant points	
# # scale significant points beginning wiht 0.8 pt
# size_point <- 0.8*(1 + balloon_scale*(bleed$A23_T_HUM_ratio_norm/max(bleed$A23_T_HUM_ratio_norm, na.rm=TRUE)))



#----------------- A23 align to hamster ---------------------


# read in A23 sequence reads:
A23 <- read.table("A23_gseq.txt",header=TRUE,sep="\t",stringsAsFactors=FALSE)

# Get rows at beginning of each chromosome:
A23_start <- A23[A23$posS == 0 & A23$posE == 1e6,]

# Get rid of ramp ups and ramp downs (though note that hamster has ramp downs, not ramp ups):
A23 <- A23[c(0,diff(A23$pos)) == 1e4,]

# combine A23 without ramps and A23_start:
A23 <- rbind(A23_start,A23)

# # get rid of contigs with only one entry. Has no action, since A23_gseq.txt already has smallest 5% of contigs removed.
# A23 <- A23[!(A23$Contig_ID %in% aggregate(pos ~ Contig_ID, 
          # data = A23, 
          # FUN = function(x){NROW(x)})[aggregate(pos ~ Contig_ID, 
          # data = A23, 
          # FUN = function(x){NROW(x)})$pos==1,"Contig_ID"]),]


# Sort:
chrOrder<-paste("chr",c(1:10,"X"),sep="")
A23$Chromosome <-factor(A23$Chromosome, levels=chrOrder)
A23 <- A23[order(A23$Chromosome, A23$pos), ]
A23$Chromosome <- as.character(A23$Chromosome)


# Transform reads into mean ratios
A23$read_ratio <- A23$reads/mean(A23$reads)


# Transform chr1 etc. to numbers
A23$Chromosome <- gsub('chr', '', A23$Chromosome)
A23[A23$Chromosome == "X","Chromosome"] <- 11
chrOrder<-c(1:11)
A23$Chromosome <-factor(A23$Chromosome, levels=chrOrder)
A23 <- A23[order(A23$Chromosome, A23$pos), ]
A23$Chromosome <- as.numeric(A23$Chromosome)

# Compute chromosome size
gen_coord <- aggregate(pos~Chromosome,FUN=max,data=A23)
colnames(gen_coord)[2] <- "chr_size"
gen_coord$Chromosome <-factor(gen_coord$Chromosome, levels=chrOrder)
gen_coord <- gen_coord[order(gen_coord$Chromosome), ]
gen_coord$Chromosome <- as.numeric(gen_coord$Chromosome)

# Use cumsum to make genome coordinates
gen_coord$coord <- c(0,cumsum(gen_coord$chr_size)[-11])

# merge genome coordinates with A23
A23 <- merge(A23,gen_coord[,c("Chromosome","coord")])
A23$Chromosome <-factor(A23$Chromosome, levels=chrOrder)
A23 <- A23[order(A23$Chromosome, A23$pos), ]
A23$Chromosome <- as.numeric(A23$Chromosome)

A23$coord <- A23$pos + A23$coord

# find midpoints of chromosomes for breaks in ggplot
mid <- function(x) {(max(x)+min(x))/2}
chr_mid <- aggregate(coord~Chromosome,FUN = mid,data=A23)
colnames(chr_mid)[2] <- "mid"
chr_mid$Chromosome <-factor(chr_mid$Chromosome, levels=chrOrder)
chr_mid <- chr_mid[order(chr_mid$Chromosome), ]
chr_mid$Chromosome <- as.numeric(chr_mid$Chromosome)

# Define breaks as mid-points chromosomes
breaks <- chr_mid$mid


# attractive grey and skyblue color scheme
cb1<-c(rep(c("grey", "skyblue"), 5),"grey")

labels <- as.character(c(1:8,"","","X"))


p1 <- ggplot(data = A23, aes(x = coord, y = read_ratio, color=as.factor(Chromosome))) + 
	geom_point(size= size_point,stroke=0) +
	scale_color_manual(values=cb1) +
	theme2 +
	scale_x_continuous(breaks = breaks, labels = labels) +
	# ggtitle("") + 
	xlab("Chromosome") + 
	ylab("Copy") + 
	labs(subtitle="A23")+
	scale_y_continuous(breaks=c(0,1,2),limit = c(0, 2))
print(p1)





#----------------- HAMSTER retain mean RH pools ---------------------


RH_hamster <- read.table("RH_hamster_gseq.txt",header=TRUE,sep="\t",stringsAsFactors=FALSE)

# Get rows at beginning of each chromosome:
RH_hamster_start <- RH_hamster[RH_hamster$posS == 0 & RH_hamster$posE == 1e6,]

# Get rid of ramp ups and ramp downs (though note that hamster has ramp downs, not ramp ups):
RH_hamster <- RH_hamster[c(0,diff(RH_hamster$pos)) == 1e4,]

# combine RH_hamster without ramps and RH_hamster_start:
RH_hamster <- rbind(RH_hamster_start,RH_hamster)


# # get rid of contigs with only one entry. Has no action, since RH_hamster_gseq.txt already has smallest 5% of contigs removed.
# RH_hamster <- RH_hamster[!(RH_hamster$Contig_ID %in% aggregate(pos ~ Contig_ID, 
          # data = RH_hamster, 
          # FUN = function(x){NROW(x)})[aggregate(pos ~ Contig_ID, 
          # data = RH_hamster, 
          # FUN = function(x){NROW(x)})$pos==1,"Contig_ID"]),]

# Sort:
chrOrder<-paste("chr",c(1:10,"X"),sep="")
RH_hamster$Chromosome <-factor(RH_hamster$Chromosome, levels=chrOrder)
RH_hamster <- RH_hamster[order(RH_hamster$Chromosome, RH_hamster$pos), ]
RH_hamster$Chromosome <- as.character(RH_hamster$Chromosome)


# Transform reads into mean ratios for RH pools only
RH_hamster$read_ratio <- rowMeans(apply(RH_hamster[,c(grep("_w0_d0$",colnames(RH_hamster)))],2,FUN=function(x) {x/mean(x)}))


# Transform chr1 etc. to numbers
RH_hamster$Chromosome <- gsub('chr', '', RH_hamster$Chromosome)
RH_hamster[RH_hamster$Chromosome == "X","Chromosome"] <- 11
chrOrder<-c(1:11)
RH_hamster$Chromosome <-factor(RH_hamster$Chromosome, levels=chrOrder)
RH_hamster <- RH_hamster[order(RH_hamster$Chromosome, RH_hamster$pos), ]
RH_hamster$Chromosome <- as.numeric(RH_hamster$Chromosome)

# Compute chromosome size
gen_coord <- aggregate(pos~Chromosome,FUN=max,data=RH_hamster)
colnames(gen_coord)[2] <- "chr_size"
gen_coord$Chromosome <-factor(gen_coord$Chromosome, levels=chrOrder)
gen_coord <- gen_coord[order(gen_coord$Chromosome), ]
gen_coord$Chromosome <- as.numeric(gen_coord$Chromosome)

# Use cumsum to make genome coordinates
gen_coord$coord <- c(0,cumsum(gen_coord$chr_size)[-11])

# merge genome coordinates with RH_hamster
RH_hamster <- merge(RH_hamster,gen_coord[,c("Chromosome","coord")])
RH_hamster$Chromosome <-factor(RH_hamster$Chromosome, levels=chrOrder)
RH_hamster <- RH_hamster[order(RH_hamster$Chromosome, RH_hamster$pos), ]
RH_hamster$Chromosome <- as.numeric(RH_hamster$Chromosome)

RH_hamster$coord <- RH_hamster$pos + RH_hamster$coord

# find midpoints of chromosomes for breaks in ggplot
mid <- function(x) {(max(x)+min(x))/2}
chr_mid <- aggregate(coord~Chromosome,FUN = mid,data=RH_hamster)
colnames(chr_mid)[2] <- "mid"
chr_mid$Chromosome <-factor(chr_mid$Chromosome, levels=chrOrder)
chr_mid <- chr_mid[order(chr_mid$Chromosome), ]
chr_mid$Chromosome <- as.numeric(chr_mid$Chromosome)

# Define breaks as mid-points chromosomes
breaks <- chr_mid$mid


# attractive grey and skyblue color scheme
cb1<-c(rep(c("grey", "skyblue"), 5),"grey")

labels <- as.character(c(1:8,"","","X"))


p2 <- ggplot(data = RH_hamster, aes(x = coord, y = read_ratio, color=as.factor(Chromosome))) + 
	geom_point(size= size_point,stroke=0) +
	scale_color_manual(values=cb1) +
	theme2 +
	scale_x_continuous(breaks = breaks, labels = labels) +
	# ggtitle("") + 
	xlab("Chromosome") + 
	ylab("Copy") + 
	labs(subtitle="Hamster genome in RH pools")+
	scale_y_continuous(breaks=c(0,1,2),limit = c(0, 2))
print(p2)


#----------------- HEK align to human ---------------------

# read in HEK293 sequence reads:
HEK <- read.table("HEK293_gseq.txt",header=TRUE,sep="\t",stringsAsFactors=FALSE)

# Get rows at beginning of each chromosome:
HEK_start <- HEK[HEK$posS == 0 & HEK$posE == 1e6,]

# Get rid of ramp ups and ramp downs:
HEK <- HEK[c(0,diff(HEK$pos)) == 1e4,]

# combine HEK without ramps and HEK_start:
HEK <- rbind(HEK_start,HEK)

# Sort:
chrOrder<-paste("chr",c(1:22,"X","Y"),sep="")
HEK$Chromosome <-factor(HEK$Chromosome, levels=chrOrder)
HEK <- HEK[order(HEK$Chromosome, HEK$pos), ]
HEK$Chromosome <- as.character(HEK$Chromosome)


# Transform reads into mean ratios
HEK$read_ratio <- HEK$reads/mean(HEK$reads)


# Transform chr1 etc. to numbers
HEK$Chromosome <- gsub('chr', '', HEK$Chromosome)
HEK[HEK$Chromosome == "X","Chromosome"] <- 23
HEK[HEK$Chromosome == "Y","Chromosome"] <- 24
chrOrder<-c(1:24)
HEK$Chromosome <-factor(HEK$Chromosome, levels=chrOrder)
HEK <- HEK[order(HEK$Chromosome, HEK$pos), ]
HEK$Chromosome <- as.numeric(HEK$Chromosome)

# Compute chromosome size
gen_coord <- aggregate(pos~Chromosome,FUN=max,data=HEK)
colnames(gen_coord)[2] <- "chr_size"
gen_coord$Chromosome <-factor(gen_coord$Chromosome, levels=chrOrder)
gen_coord <- gen_coord[order(gen_coord$Chromosome), ]
gen_coord$Chromosome <- as.numeric(gen_coord$Chromosome)

# Use cumsum to make genome coordinates
gen_coord$coord <- c(0,cumsum(gen_coord$chr_size)[-24])

# merge genome coordinates with HEK
HEK <- merge(HEK,gen_coord[,c("Chromosome","coord")])
HEK$Chromosome <-factor(HEK$Chromosome, levels=chrOrder)
HEK <- HEK[order(HEK$Chromosome, HEK$pos), ]
HEK$Chromosome <- as.numeric(HEK$Chromosome)

HEK$coord <- HEK$pos + HEK$coord

# get rid of chrY
HEK <- HEK[HEK$Chromosome != 24,]

# find midpoints of chromosomes for breaks in ggplot
mid <- function(x) {(max(x)+min(x))/2}
chr_mid <- aggregate(coord~Chromosome,FUN = mid,data=HEK)
colnames(chr_mid)[2] <- "mid"
chr_mid$Chromosome <-factor(chr_mid$Chromosome, levels=chrOrder)
chr_mid <- chr_mid[order(chr_mid$Chromosome), ]
chr_mid$Chromosome <- as.numeric(chr_mid$Chromosome)

# Define breaks as mid-points chromosomes
breaks <- chr_mid$mid


# attractive grey and skyblue color scheme
cb1<-rep(c("grey", "skyblue"), 12)

labels <- as.character(c(1:9,"",11,"",13,"","",16,"","","",20,"","","X"))


p3 <- ggplot(data = HEK, aes(x = coord, y = read_ratio, color=as.factor(Chromosome))) + 
	geom_point(size= size_point,stroke=0) +
	scale_color_manual(values=cb1) +
	theme2 +
	scale_x_continuous(breaks = breaks, labels = labels) +
	# ggtitle("") + 
	xlab("Chromosome") + 
	ylab("Copy") + 
	labs(subtitle="HEK293")+
	scale_y_continuous(breaks=c(0,2,4),limit = c(0, 5))
print(p3)


#-----------------HUMAN retain mean RH pools---------------------

# Used seq reads for retention graphs to compare on equal footing with hamster and HEK293 "retentions"

# read in RH human sequence reads:
RH_human <- read.table("RH_human_gseq.txt",header=TRUE,sep="\t",stringsAsFactors=FALSE)

# Get rows at beginning of each chromosome:
RH_human_start <- RH_human[RH_human$posS == 0 & RH_human$posE == 1e6,]

# Get rid of ramp ups and ramp downs:
RH_human <- RH_human[c(0,diff(RH_human$pos)) == 1e4,]

# combine RH_human without ramps and RH_human_start:
RH_human <- rbind(RH_human_start,RH_human)

# Sort:
chrOrder<-paste("chr",c(1:22,"X","Y"),sep="")
RH_human$Chromosome <-factor(RH_human$Chromosome, levels=chrOrder)
RH_human <- RH_human[order(RH_human$Chromosome, RH_human$pos), ]
RH_human$Chromosome <- as.character(RH_human$Chromosome)



# Mean human reads for RH pools:
RH_human$read_ratio <- rowMeans(apply(RH_human[,c(grep("_w0_d0$",colnames(RH_human)))],2,FUN=function(x) {x/mean(x)}))



#  find middle TK1

gencode_gtf_ensembl_ucsc <- read.delim("gencode_gtf_ensembl_ucsc_v31.txt",header=TRUE,sep="\t",stringsAsFactors=FALSE,check.names=FALSE)


gencode_gtf_ensembl_ucsc[gencode_gtf_ensembl_ucsc$geneSymbol=="TK1",]
      # geneSymbol           tx_id Chromosome    geneS    geneE strand         gene_id      gene_type geneLength txLength cdsLength                                       gene_description    5utrS    5utrE
# 54647        TK1 ENST00000588734      chr17 78174075 78187233      - ENSG00000167900 protein_coding      13159     1681       804 thymidine kinase 1 [Source:HGNC Symbol;Acc:HGNC:11830] 78186995 78187233
      # 5utrDiff    3utrS    3utrE 3utrDiff exonCount
# 54647      239 78174121 78174758      638         6


TK1_coord <- mean(gencode_gtf_ensembl_ucsc[gencode_gtf_ensembl_ucsc$geneSymbol=="TK1","geneS"],gencode_gtf_ensembl_ucsc[gencode_gtf_ensembl_ucsc$geneSymbol=="TK1","geneE"])



#  find peak TK1 reads 

# For more accurate search of TK1 peak, restrict search to within delta of TK1, because CEN and TEL becomes larger than TK1 in some samples. (Though not RH pools; other higher peaks usually occur with later time and higher conc samples.) In fact, in RH pools TK1 is highest peak on whole of chr17. But for generality use this approach with delta_1.

delta_1 <- 1e6

TK1_max <- max(RH_human[RH_human$Chromosome=="chr17" & RH_human$pos >= TK1_coord-delta_1 & RH_human$pos <= TK1_coord+delta_1,"read_ratio"])


TK1_max
# [1] 10.58925


# Normalize using peak copy number at TK1:
RH_human$read_ratio <- RH_human$read_ratio/TK1_max


# If desired can normalize using simpler approach of finding max seq reads on chr17:
# RH_human$read_ratio <- RH_human$read_ratio/max(RH_human[RH_human$Chromosome == "chr17","read_ratio"])




# Transform chr1 etc. to numbers
RH_human$Chromosome <- gsub('chr', '', RH_human$Chromosome)
RH_human[RH_human$Chromosome == "X","Chromosome"] <- 23
RH_human[RH_human$Chromosome == "Y","Chromosome"] <- 24
chrOrder<-c(1:24)
RH_human$Chromosome <-factor(RH_human$Chromosome, levels=chrOrder)
RH_human <- RH_human[order(RH_human$Chromosome, RH_human$pos), ]
RH_human$Chromosome <- as.numeric(RH_human$Chromosome)

# Compute chromosome size
gen_coord <- aggregate(pos~Chromosome,FUN=max,data=RH_human)
colnames(gen_coord)[2] <- "chr_size"
gen_coord$Chromosome <-factor(gen_coord$Chromosome, levels=chrOrder)
gen_coord <- gen_coord[order(gen_coord$Chromosome), ]
gen_coord$Chromosome <- as.numeric(gen_coord$Chromosome)

# Use cumsum to make genome coordinates
gen_coord$coord <- c(0,cumsum(gen_coord$chr_size)[-24])

# merge genome coordinates with RH_human
RH_human <- merge(RH_human,gen_coord[,c("Chromosome","coord")])
RH_human$Chromosome <-factor(RH_human$Chromosome, levels=chrOrder)
RH_human <- RH_human[order(RH_human$Chromosome, RH_human$pos), ]
RH_human$Chromosome <- as.numeric(RH_human$Chromosome)

RH_human$coord <- RH_human$pos + RH_human$coord

# get rid of chrY
RH_human <- RH_human[RH_human$Chromosome != 24,]

# find midpoints of chromosomes for breaks in ggplot
mid <- function(x) {(max(x)+min(x))/2}
chr_mid <- aggregate(coord~Chromosome,FUN = mid,data=RH_human)
colnames(chr_mid)[2] <- "mid"
chr_mid$Chromosome <-factor(chr_mid$Chromosome, levels=chrOrder)
chr_mid <- chr_mid[order(chr_mid$Chromosome), ]
chr_mid$Chromosome <- as.numeric(chr_mid$Chromosome)

# Define breaks as mid-points chromosomes
breaks <- chr_mid$mid


# attractive grey and skyblue color scheme
cb1<-rep(c("grey", "skyblue"), 12)

labels <- as.character(c(1:9,"",11,"",13,"","",16,"","","",20,"","","X"))


p4 <- ggplot(data = RH_human, aes(x = coord, y = read_ratio, color=as.factor(Chromosome))) + 
	geom_point(size= size_point,stroke=0) +
	geom_text(data = RH_human[RH_human$Chromosome==17,][which.max(RH_human[RH_human$Chromosome==17,"read_ratio"]),], aes(x = coord, y = read_ratio), label="TK1", colour = "black", fontface = "italic", size = 4, nudge_y=0.15) +
	scale_color_manual(values=cb1) +
	theme2 +
	scale_x_continuous(breaks = breaks, labels = labels) +
	# ggtitle("") + 
	xlab("Chromosome") + 
	ylab("Copy") + 
	labs(subtitle="Human genome in RH pools")+
	scale_y_continuous(breaks=c(0,1),limit = c(0, 1.5))
print(p4)


# --------------- Recast RH_human to include "chr1" etc for use in UCSC genome browser -----------------------

RH_human[RH_human$Chromosome == 23, "Chromosome"] <- "X"
RH_human$Chromosome <- paste0("chr",RH_human$Chromosome)





# --------------- TK1, chr17 ---------------------


# # If desired, and if RH_pool_retent_ratio.txt is available, begin from START HERE to save time.

# # show whole chr17 for TK1

# RH_pool_retent <- RH_human[,c("Chromosome","pos","read_ratio")]


# dim(RH_pool_retent)
# # [1] 300814    3

# RH_pool_retent$posS <- RH_pool_retent$pos + 1
# RH_pool_retent <- RH_pool_retent[,c("Chromosome","pos","posS","read_ratio")]

# RH_pool_retent$pos <- format(RH_pool_retent$pos,scientific=FALSE)
# RH_pool_retent$posS <- format(RH_pool_retent$posS,scientific=FALSE)
# # RH_pool_retent$read_ratio <- round(RH_pool_retent$read_ratio,4)

# head(RH_pool_retent)
  # # Chromosome       pos      posS read_ratio
# # 1       chr1    500000    500001 0.07535384
# # 2       chr1    510000    510001 0.07557400
# # 3       chr1    520000    520001 0.07256259
# # 4       chr1    530000    530001 0.07255724
# # 5       chr1    540000    540001 0.07253440
# # 6       chr1    550000    550001 0.07205277




# # write.table(RH_pool_retent, "RH_pool_retent_ratio.txt",quote=FALSE,sep="\t",row.names=FALSE,col.names=FALSE)


# START HERE
# # Place following header at top of RH_pool_retent_ratio.txt and use in bedGraph format on ucsc genome browser
# browser position chr17:1-100000000
# track type=bedGraph name="Copy" description="chr17" visibility=full color=0,0,255 altColor=255,0,0 priority=20





# --------- Custom UCSC track settings for all -----------------

# Custom track settings
# Display mode: full
# Type of graph: points
# Track height: 128 pixels
# Data view scaling: auto-scale to data view
# Always include zero: ON
# Vertical viewing range:  min: 0; max: 1000  (range: 0 to 1000) (greyed out)
# Transform function: Transform data points by: NONE
# Windowing function: mean
# Smoothing window: OFF
# Negate values: not selected
# Draw y indicator lines: 
# at y = 0.0: ON at y =0 OFF (no threshold for retention)


# Configure Image page on ucsc genome browser:
# image width:	400	pixels
# label area width:	10	characters	(7 for TK1)
# text size: 12

# # Remember to get rid of splicoforms and nc genes by clicking GENCODE v31 bar on left of diagram, if desired. Then click on wrench and select "Transcript class": coding, "Transcript Annotation Method": All, "Transcript Biotype": protein_coding, "Tag":appris_principal_1, "Support Level": All. Leave "Highlight items" unselected.

# # Add squished centromeres.




# --------------- chr2, CEN2 ---------------------

# Use RH_pool_retent_ratio.txt (above) and upload to UCSC genome browser. Focus on chr2, add squished centromeres.

# # Place following header at top of RH_pool_retent_ratio.txt and use in bedGraph format on ucsc genome browser
# browser position chr2:1-1000000000
# track type=bedGraph name="Copy" description="chr2" visibility=full color=0,0,255 altColor=255,0,0 priority=20





#------------------Make file --------------------------


p5 <- ggdraw() + draw_image(magick::image_read_pdf("CEN2.pdf", density = 300),scale=0.95) + coord_cartesian(clip = "off") # + draw_label("Paclitaxel", fontface='plain', size=12, x=0.55,y=0.92) + draw_label("D", fontface='bold',x=0.05,y=0.98)
p6 <- ggdraw() + draw_image(magick::image_read_pdf("TK1.pdf", density = 300),scale=0.95) + coord_cartesian(clip = "off")


pdf("HEK_A23_align_retain.pdf",width=7.5,height=10, useDingbats=FALSE)
plot_grid(p1, p2, p3, p4, p5, p6, labels=c("A", "B", "C", "D", "E", "F"), ncol = 2, nrow = 3, label_size = 16)
dev.off()


tiff("HEK_A23_align_retain.tif",width=7.5,height=10,units="in",res=300)
plot_grid(p1, p2, p3, p4, p5, p6, labels=c("A", "B", "C", "D", "E", "F"), ncol = 2, nrow = 3, label_size = 16)
dev.off()



png("HEK_A23_align_retain.png",width=7.5,height=10,units="in",res=300)
plot_grid(p1, p2, p3, p4, p5, p6, labels=c("A", "B", "C", "D", "E", "F"), ncol = 2, nrow = 3, label_size = 16)
dev.off()


p5 <- ggdraw() + draw_image(magick::image_read_pdf("CEN2.pdf", density = 1200),scale=0.95) + coord_cartesian(clip = "off") # + draw_label("Paclitaxel", fontface='plain', size=12, x=0.55,y=0.92) + draw_label("D", fontface='bold',x=0.05,y=0.98)
p6 <- ggdraw() + draw_image(magick::image_read_pdf("TK1.pdf", density = 1200),scale=0.95) + coord_cartesian(clip = "off")



pdf("HEK_A23_align_retain_hi_res.pdf",width=7.5,height=10, useDingbats=FALSE)
plot_grid(p1, p2, p3, p4, p5, p6, labels=c("A", "B", "C", "D", "E", "F"), ncol = 2, nrow = 3, label_size = 16)
dev.off()





png("HEK_A23_align_retain_hi_res.png",width=7.5,height=10,units="in",res=1200)
plot_grid(p1, p2, p3, p4, p5, p6, labels=c("A", "B", "C", "D", "E", "F"), ncol = 2, nrow = 3, label_size = 16)
dev.off()

















