/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */
package org.rhwlab.LMS.dataframe.investigator;

import java.io.File;
import java.io.FileOutputStream;
import java.io.OutputStream;
import java.io.PrintStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Observable;
import java.util.TreeMap;
import org.jdom2.Document;
import org.jdom2.Element;
import org.jdom2.input.SAXBuilder;
import org.jdom2.output.Format;
import org.jdom2.output.XMLOutputter;
import org.rhwlab.LMS.dataframe.DataFrame;
import org.rhwlab.LMS.dataframe.DataSet;
import org.rhwlab.LMS.dataframe.FileDataFrame;
import org.rhwlab.LMS.dataframe.investigator.transform.Normalization;
import org.rhwlab.LMS.dataframe.investigator.transform.Transform;

/**
 *
 * @author gevirl
 */
public class Investigation extends Observable {

   List<DataFrame> dfs = new ArrayList<>();
    HashMap<DataFrame, List<DataSet>> dataSets = new HashMap<>();
    HashMap<DataFrame, List<Clustering>> clusterings = new HashMap<>();
    File file;

    public Investigation(){
        
    }
    
    public Investigation(File xml) throws Exception {
        this();
        init(xml);
    }
    
    public Investigation(DataSet ds, int rownameCol) {
        this();
        init(ds,rownameCol);
    }

    public void init(DataSet ds, int rownameCol){
        DataFrame df = ds.getDataFrame();
        df.setRowNameColumn(rownameCol);
        
        ArrayList<DataSet> dsList = new ArrayList<>();
        dsList.add(ds);
        dataSets.put(df, dsList); 
        addDataFrame(df);
    }
    
    public void init(File xml) throws Exception {
        this.file = xml;
        SAXBuilder saxBuilder = new SAXBuilder();
        Document doc = saxBuilder.build(xml);
        Element root = doc.getRootElement();

        if (root.getAttributeValue("dataframe") == null) {
            for (Element dfEle : root.getChildren("DataFrame")) {
                dataFrameFromXML(dfEle);
            }
        } else {
            dataFrameFromXML(root);
        } 

    }
    public void combineWith(Investigation otherInv) {
        for (DataFrame df : otherInv.dfs) {
            addDataFrame(df);
            dataSets.put(df, otherInv.dataSets.get(df));
            clusterings.put(df, otherInv.clusterings.get(df));
        }
    }

    private void dataFrameFromXML(Element root) throws Exception {
        
        String dfFile = root.getAttributeValue("dataframe");
        DataFrame df = new FileDataFrame(new File(dfFile));
        String rownameCol = root.getAttributeValue("rownameColumn");
        if (rownameCol != null) {
            df.setRowNameColumn(Integer.valueOf(rownameCol));
        } else {
            df.setRowNameColumn(1);
        }        
        addDataFrame(df);
        
        String dfname = root.getAttributeValue("name");
        if (dfname != null){
            df.setName(dfname);
        }
        
        List<DataSet> dataSetList = this.dataSets.get(df);
        if (dataSetList == null) {
            dataSetList = new ArrayList<>();
            this.dataSets.put(df, dataSetList);
        }
        for (Element dsEle : root.getChildren("DataSet")) {
            DataSet ds = new DataSet(dsEle, df);
            this.addDataSet(ds);
        }

        List<Clustering> clusterList = this.clusterings.get(df);
        if (clusterList == null) {
            clusterList = new ArrayList<>();
            this.clusterings.put(df, clusterList);
        }
        for (Element clEle : root.getChildren("Clustering")) {
            Clustering cl = new Clustering(clEle);
            this.addClustering(cl, df);
        }


        

    }

    public void addDataFrame(DataFrame df){
        dfs.add(df);
        this.setChanged();
        this.notifyObservers(df);
    }
    

    // add the columns from a dataframe to this ivnvestigation's dataframe

    public void mergeDataFrame(String title, DataFrame source, DataFrame dest) {
        DataFrame df = dest;
        int[] cols = new int[source.getColumnCount()];
        for (int c = 0; c <cols.length; ++c) {
            String hd = source.getColumnName(c);
            String[] colData = source.getColumn(c);
            cols[c] = df.addColumn(colData, hd);
        }

        // make a data set for the new columns
        DataSet ds = null;
        String[] attrs = null;
        if (cols.length >= 2) {
            //make the dataset a coordinate system
            attrs = new String[1];
            attrs[0] = "Coord";
            ds = new DataSet(title, df, cols, attrs);
            ds.setAttribute(attrs[0], 0, "X");
            ds.setAttribute(attrs[0], 1, "Y");
            if (cols.length == 3){
                ds.setAttribute(attrs[0], 2, "Z");
            }

        } else {
            ds = new DataSet(title, df, cols, attrs);  // no attributes known
        }
        this.addDataSet(ds);
        this.setChanged();
        this.notifyObservers(df);        
    }

    public int getRownameColumn() {
        return getRownameColumn(0);
    }

    
    public int getRownameColumn(int dfIndex) {
        return this.dfs.get(dfIndex).getRowNameColumn();
    }

    public String[] getDataRowNames(DataFrame df){
        return df.getColumn(df.getRowNameColumn());    
    }
    public String[] getDataRowNames() {
        return getDataRowNames(0);
    }

    public String[] getDataRowNames(int dfIndex) {
        DataFrame df = this.dfs.get(dfIndex);
        return getDataRowNames(df);
    }

    public void save() throws Exception {
        if (file != null) {
            saveAs(file);
        }
    }

    public void saveAs(File xml) throws Exception {

        Element root = new Element("Investigation");
        for (int i = 0; i < this.dfs.size(); ++i) {
            DataFrame df = dfs.get(i);
            Element dfEle = new Element("DataFrame");
            root.addContent(dfEle);
            String dfFile = xml.getPath().replace(".xml", String.format("%s.df", df.getName()));
            dfEle.setAttribute("dataframe", dfFile);
            dfEle.setAttribute("rownameColumn", Integer.toString(df.getRowNameColumn()));
            dfEle.setAttribute("name", df.getName());
            PrintStream dfStream = new PrintStream(dfFile);
            df.save(dfStream);
            dfStream.close();

            for (DataSet ds : dataSets.get(df)) {
                dfEle.addContent(ds.toXML());
            }
            List<Clustering> clusteringsList = clusterings.get(df);
            if (clusteringsList != null) {
                for (Clustering clustering : clusterings.get(df)) {
                    dfEle.addContent(clustering.toXML());
                }
            }
        }
        OutputStream stream = new FileOutputStream(xml);
        XMLOutputter out = new XMLOutputter(Format.getPrettyFormat());
        out.output(root, stream);
        stream.close();

        this.file = xml;
    }

    public DataSet getDataSet(String title) {
        for (DataSet ds : getDataSets()) {
            if (ds.toString().equals(title)) {
                return ds;
            }
        }
        return null;
    }

    public List<DataFrame> getDataFrames() {
        return this.dfs;
    }

    public List<DataSet> getDataSets() {
        return getDataSets(dfs.get(0));
    }

    public List<DataSet> getDataSets(DataFrame df) {
        return dataSets.get(df);
    }

    public void addDataSet(DataSet ds) {
        DataFrame df = ds.getDataFrame();
        dataSets.get(df).add(ds);
        this.setChanged();
        this.notifyObservers(df);
    }

    public void addClustering(Clustering cl) {
        addClustering(cl, 0);
    }

    public void addClustering(Clustering cl, int dfIndex) {
        DataFrame df = dfs.get(dfIndex);
        addClustering(cl,df);
    }

    public void addClustering(Clustering cl, DataFrame df) {
        clusterings.get(df).add(cl);
        this.setChanged();
        this.notifyObservers(df);        
        
    }    
    public File getFile() {
        return file;
    }

    public List<Clustering> getClusterings() {
        return getClusterings(0);
    }

    public List<Clustering> getClusterings(int dfIndex) {
        return getClusterings(dfs.get(dfIndex));
    }

    public List<Clustering> getClusterings(DataFrame df){
        return this.clusterings.get(df);
    }
    
    public TreeMap<String, String> characterizeRows(DataSet ds, Clustering clustering, String attr, double thresh) {
        return characterizeRows(ds, clustering, attr, thresh, 0);
    }

    public TreeMap<String, String> characterizeRows(DataSet ds, Clustering clustering, String attr, double thresh, int dfIndex) {
        DataFrame df = dfs.get(dfIndex);
        TreeMap<String, String> ret = new TreeMap<>();

        int[][] clusters = clustering.getClusters();
        for (int cl = 0; cl < clusters.length; ++cl) {
            String characterization = ds.characterizeByMeans(clusters[cl], thresh, attr);
            for (int row : clusters[cl]) {
                String rowname = df.getRow(row).get(df.getRowNameColumn());
                ret.put(rowname, characterization);
            }

        }
        return ret;
    }
  
    
    
    public DataSet getDataSet(String dfName,String dsName){
        for (DataFrame df : this.dfs){
            if (df.getName().equals(dfName)){
                for (DataSet ds : this.getDataSets(df)){
                    if (ds.toString().equals(dsName)){
                        return ds;
                    }
                }
            }
        }
        return null;
    }
}
