/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */
package org.rhwlab.LMS.RNASeq.merged;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.PrintStream;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.TreeMap;
import org.rhwlab.LMS.dataframe.FACSSamples;
import org.rhwlab.alignment.PairedEndBAM;
import org.rhwlab.gene.model.Annotation;
import org.rhwlab.gene.model.Exon;
import org.rhwlab.gene.model.exons.NonOverlapping;

/**
 *
 * @author gevirl
 */
public class ExonSummary {

    TreeMap<String, List<String>> dataMap;  // store the data by columns

    public ExonSummary() {
        dataMap = new TreeMap<>();

    }

    // ExonSummary from a list of single sample ExonSummary files
    public ExonSummary(File[] files, String[] sampleIDs, String[] commonColumns, String dataColumn) throws Exception {
        this();

        // read all the exon summaries to combine
        ExonSummary[] summaries = new ExonSummary[files.length];
        for (int i = 0; i < files.length; ++i) {
            summaries[i] = new ExonSummary(files[i]);
        }

        // add the common columns
        for (String head : commonColumns) {
            this.addColumn(head, summaries[0].getColumn(head));
        }

        // add all the data columns
        for (int i = 0; i < sampleIDs.length; ++i) {
            this.addColumn(sampleIDs[i], summaries[i].getColumn(dataColumn));
        }
    }

    // ExonSummary from a single file - can have multipe sample columns
    public ExonSummary(File file) throws Exception {
        this();
        BufferedReader reader = new BufferedReader(new FileReader(file));
        String[] heads = reader.readLine().split(",");
        for (String head : heads) {
            dataMap.put(head, new ArrayList<>());
        }
        String line = reader.readLine();
        while (line != null) {
            String[] tokens = line.split(",");
            for (int i = 0; i < heads.length; ++i) {
                List<String> list = this.dataMap.get(heads[i]);
                list.add(tokens[i]);
            }
            line = reader.readLine();
        }
        reader.close();
    }

    public void addColumn(String head, List<String> list) {
        this.dataMap.put(head, list);
    }

    public List<String> getColumn(String head) {
        return this.dataMap.get(head);
    }

    public void saveTo(File file) throws Exception {
        PrintStream stream = new PrintStream(file);

        // print the headers
        boolean first = true;
        for (String head : dataMap.keySet()) {
            if (first) {
                stream.print(head);
                first = false;
            } else {
                stream.printf(",%s", head);
            }
        }
        stream.println();

        // print all the data
        int n = dataMap.firstEntry().getValue().size();
        for (int i = 0; i < n; ++i) {
            first = true;
            for (String head : dataMap.keySet()) {
                if (first) {
                    stream.print(dataMap.get(head).get(i));
                    first = false;
                } else {
                    stream.printf(",%s", dataMap.get(head).get(i));
                }
            }
            stream.println();
        }

        stream.close();
    }

    public void averageFACSReps(int dataStart) {
        int n = dataMap.firstEntry().getValue().size();
        
        TreeMap<String, List<String>> toAvg = new TreeMap<>();
        for (String head : dataMap.keySet()) {
            if (head.contains("Merged")) {
                String key = FACSSamples.tissueName(head) + FACSSamples.timeString(head);
                List<String> list = toAvg.get(key);
                if (list == null) {
                    list = new ArrayList<>();
                    toAvg.put(key, list);
                }
                list.add(head);
            }

        }
        // average the data
        for (String head : toAvg.keySet()){
            List<String> avg = new ArrayList<>();
            
            double[] sum = new double[n];
            for (String sample : toAvg.get(head)){
                for (int i=0 ; i<n ; ++i){
                    double v = Double.valueOf(dataMap.get(sample).get(i));
                    sum[i] = sum[i] + v;
                }
            }
            
            int d = toAvg.get(head).size();
            for (int i=0 ; i<n ; ++i){
                avg.add(String.format("%.3f",sum[i]/d));
            }
            
            this.addColumn(head, avg);
            for (String sample : toAvg.get(head)){
                this.dataMap.remove(sample);
            }
        }

    }

    // calculate the exon summary for the given bam file in args[0]
    // saves the result csv in the same directory as the bam file
    static public void main(String[] args) throws Exception {

        File bamFile = new File(args[0]);
        NonOverlapping non = new NonOverlapping(new File(args[1]));
        int n = PairedEndBAM.nonOverlappingExonReadCounts(bamFile, non);  // accumulates the read counts in the exon annotations of the non overlapping model
        File csv = new File(bamFile.getParent(), bamFile.getName().replace(".bam", ".exon.csv"));
        PrintStream stream = new PrintStream(csv);
        double sum = 0.0;
        for (Annotation annot : non.getAll()) {
            Exon exon = (Exon) annot;
            int len = exon.getEnd() - exon.getStart() + 1;
            double v = (double) exon.count / (double) len;
            sum = sum + v;
        }
        double f = 1000000.0 / sum;

        stream.println("ExonID,WBGene,SequenceName,GeneName,Chromosome,Start,End,Length,Readcount,TPM");
        for (Annotation annot : non.getAll()) {
            Exon exon = (Exon) annot;
            String id = (String) exon.getAttributeValue("Name");
            String wbGene = (String) exon.getAttributeValue("Parent");
            String seqName = (String) exon.getAttributeValue("sequence_name");
            String geneName = (String) exon.getAttributeValue("gene_name");
            if (geneName == null) {
                Object locus = exon.getAttributeValue("locus");
                if (locus != null) {
                    geneName = (String) locus;
                } else {
                    geneName = seqName;
                }
            }
            String chr = exon.getChromosome();
            int len = exon.getEnd() - exon.getStart() + 1;
            double v = f * (double) exon.count / (double) len;
            stream.printf("%s,%s,%s,%s,%s,%d,%d,%d,%d,%.3f\n", id, wbGene, seqName, geneName, chr, exon.getStart(), exon.getEnd(), len, exon.count, v);
        }
        System.out.printf("Alignments aggregated %d\n", n);
        stream.close();
    }
}
