/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.rhwlab.beans;
import java.util.*;
import java.io.*;
import javax.swing.tree.*;
import javax.swing.event.ChangeListener;
import org.rhwlab.expression.*;
import org.rhwlab.db.MySql;
import org.rhwlab.imaging.binarytree.BinaryTreeNode;
import java.sql.*;
import org.rhwlab.db.beans.Imaged;
import org.rhwlab.universalimaging.Cell;


/**
 *
 * @author gevirl
 */
// represents a single cell in an embryo
// 
public class EmbryoCell extends DefaultMutableTreeNode implements Comparable<EmbryoCell> , BinaryTreeNode , org.rhwlab.universalimaging.Cell{
    public EmbryoCell(String name,int end){
        this(name);
        this.endTime = end;
    }
    public EmbryoCell(String name){
        super(name);
        nuclei = null;
        expType = null;
        endTime = -1;
        startTime = 0;
        expr = null;
    }
    // construct an embryo cell from the given TimePointNucleus
    // this constructs all progency cells recursively
    public EmbryoCell(SeriesEmbryo embryo,TimePointNucleus nuc,Integer limit,String type){
        super(nuc.getName());
        this.embryo = embryo;
        nuclei = new ArrayList<TimePointNucleus>();
        endTime = this.addNucleus(nuc,limit,type);
        startTime = endTime - nuclei.size() + 1;
        expr = new CellDataTimeSeries(this,type);
        expType = type;
    }
    static public List<Map<String,List<EmbryoCell>>> terminalGroups(Map<String,EmbryoCell> cellMap){
        ArrayList<Map<String,List<EmbryoCell>>> ret = new ArrayList<>();
        
        // put any leaves into the return map
        boolean noLeaves = true;
        for (String label : cellMap.keySet()){
            EmbryoCell cell = cellMap.get(label);
            if (cell.isLeaf()){
                noLeaves = false;
                break;
            }
        }
        
        if (noLeaves){
            // no leaves were found
            
            // build a map of less daughters and find terminal groups
            HashMap<String,EmbryoCell> lessMap = new HashMap<>();
            for (String label : cellMap.keySet()){
                EmbryoCell cell = cellMap.get(label);
                EmbryoCell less = (EmbryoCell)cell.getLessDaughter();
                lessMap.put(label, less);
            }
            List<Map<String,List<EmbryoCell>>> lessRet = EmbryoCell.terminalGroups(lessMap);
            
            // build a map of greater daughters and find terminal groups
            HashMap<String,EmbryoCell> greatMap = new HashMap<>();
            for (String label : cellMap.keySet()){
                EmbryoCell cell = cellMap.get(label);
                EmbryoCell great = (EmbryoCell)cell.getGreaterDaughter();
                greatMap.put(label,great);
            }
            List<Map<String,List<EmbryoCell>>> greatRet = EmbryoCell.terminalGroups(greatMap);      
            
            // combine results from both daughters
            ret.addAll(lessRet);
            ret.addAll(greatRet);
            
        } else {
            // add all the leaves of each cell
            TreeMap<String,List<EmbryoCell>> map = new TreeMap<>();
            for (String label : cellMap.keySet()){
                
                EmbryoCell cell = cellMap.get(label);
                EmbryoCell[] leaves = cell.getLeaves();
                ArrayList<EmbryoCell> list = new ArrayList<>();
                for (EmbryoCell leaf : leaves){
                    list.add(leaf);
                }
                map.put(label,list);
            }
            ret.add(map);
        }
        
        return ret;
    }
    //builds a consensus tree from a set of cells, cells need to be sulstonized
    // this is a tree made from the deepest leaves in the input trees
    // the tree is made from newly built cells, not from cells from the input trees
    static public EmbryoCell ConsensusTree(Collection<EmbryoCell> rootCollection){
        EmbryoCell[] roots = rootCollection.toArray(new EmbryoCell[0]);
        
        EmbryoCell ret = new EmbryoCell(roots[0].getName());
        ret.sulston = roots[0].sulston;
        ret.endTime = roots[0].endTime;
        ret.startTime = roots[0].startTime;
        
        ArrayList<EmbryoCell> lessCells = new ArrayList<>();
        ArrayList<EmbryoCell> greatCells = new ArrayList<>();
        for (EmbryoCell root : roots){
            EmbryoCell less = (EmbryoCell)root.getLessDaughter();
            EmbryoCell great = (EmbryoCell)root.getGreaterDaughter();
            if (less != null){
                lessCells.add(less);
                greatCells.add(great);
            } else {
                // root is a leaf

                if (root.sulston == null || ret.sulston == null){
                    System.out.printf("Cell: %s not sulston in series: %s\n", root.getName(),root.fromSeries());
                }                
                else if (ret.sulston.end < root.sulston.end){
                    ret.sulston.end = root.sulston.end;
                }
                
            }
        }
        if (!lessCells.isEmpty()){
            ret.insert(ConsensusTree(lessCells), 0);
            ret.insert(ConsensusTree(greatCells), 1);
        }
        
        return ret;
    }
    public BinaryTreeNode getGreaterDaughter(){
        if (this.getChildCount()>0){
            return (BinaryTreeNode)this.getChildAt(1);
        }
        return null;
    }
    public BinaryTreeNode getLessDaughter(){
        if (this.getChildCount()>0){
            return (BinaryTreeNode)this.getChildAt(0);
        }
        return null;
    }
    public int[] getValues(){
        return expr.getValues();
    }
    public double[] getLogExpressionValues(){
        if (logValues == null){
            double minExp = embryo.minExpression();
            int[] expValues = this.getExpressionValues();
            logValues = new double[expValues.length];
            for (int i=0 ; i<logValues.length;++i){
                logValues[i] = Math.log(expValues[i]-minExp+1.0);
            }
        }
        return this.logValues;
    }
    public int[] getExpressionValues(){
        return getValues();
    }
    public String getLabel(){
        return this.getName();
    }
    
    public int compareTo(EmbryoCell other){
        return this.getName().compareTo(other.getName());
    }
    // add an onset listener to receive events any time an onset changes in this cell or any of it's children
    public void addOnsetListener(ChangeListener listen){
        expr.addOnsetListener(listen);
        int n = this.getChildCount();
        for (int i=0 ; i<n ; ++i){
            EmbryoCell child = (EmbryoCell)this.getChildAt(i);
            child.addOnsetListener(listen);
        }
    }
    // return the cell which contains the onset for this cell
    // the onset can come from this cell or any of its ancestors
    // will return null if this cell and ancestors do not have an onset
    public EmbryoCell getOnsetCell(){
        ExpressionOnset onset = expr.getExprOnset();
        if (onset != null) return this;  // this cell contains the onset
        
        EmbryoCell parentCell = this.getParentCell();
        if (parentCell == null) return null; // this is a root cell and has no onset
        
        return parentCell.getOnsetCell();  // look in the parent for the onset
        
    }
    // add the given nucleus and all it's progeny to this cell
    // this creates any child cells necessary for this cell
    // returns the end time after the nucleus and it's progeny are added
    public int addNucleus(TimePointNucleus nuc,Integer limit,String type){
        nuclei.add(nuc);
        if (limit != null){
            if (nuc.getTime()==limit) return nuc.getTime();
        }
        TimePointNucleus[] succs = nuc.getSuccessors();
        // does this nucleus give rise to zero, one or two nuclei in the next time frame?
        if (succs.length==0) return nuc.getTime();  // no successors 
        
        if (succs.length == 1) {
            // one successor 
            return addNucleus(succs[0],limit,type);
        } else {
            // two successors make two new cells
            if (nuc.getName().equals("EMS")){
                if (succs[0].getName().compareTo(succs[1].getName())>0){
                    
                    this.insert(new EmbryoCell(embryo,succs[1],limit,type),0);
                    this.insert(new EmbryoCell(embryo,succs[0],limit,type),1);
                } else {
                    this.insert(new EmbryoCell(embryo,succs[0],limit,type),0);
                    this.insert(new EmbryoCell(embryo,succs[1],limit,type),1);                     
                }
            }
            else if (succs[0].getName().compareTo(succs[1].getName())>01){
                this.insert(new EmbryoCell(embryo,succs[0],limit,type),0);
                this.insert(new EmbryoCell(embryo,succs[1],limit,type),1);                
            } else {
                
                this.insert(new EmbryoCell(embryo,succs[1],limit,type),0);  
                this.insert(new EmbryoCell(embryo,succs[0],limit,type),1);
            }

            return nuc.getTime();
        }
    }

    // returns a list of cells from the subtree rooted at this cell that exist at the given time
    public EmbryoCell[] getCellsAtTime(int time){
        ArrayList<EmbryoCell> cells = getCellsAtTime(this,time);
        EmbryoCell[] ret = new EmbryoCell[cells.size()];
        return cells.toArray(ret);    
    }
    public static ArrayList<EmbryoCell> getCellsAtTime(EmbryoCell cell,int time){
        ArrayList<EmbryoCell> cells = new ArrayList<EmbryoCell>();
        // does the cell exist at the given time?
        if (cell.getStartTime()<=time && time <= cell.endTime){
            cells.add(cell);
            return cells;
        }
        // no - get the cells from the children
        int n = cell.getChildCount();
        for (int i=0 ; i<n ; ++i){
            EmbryoCell child = (EmbryoCell)cell.getChildAt(i);
            cells.addAll(getCellsAtTime(child,time));
        }
        return cells;
    } 
    // return all the cells in the subtree rooted at this cell
    public TreeSet<EmbryoCell> getAllCells(){
        TreeSet<EmbryoCell> ret = new TreeSet<EmbryoCell>();
        ret.add(this);
        if (this.isLeaf()){
            return ret;
        }
        ret.addAll(((EmbryoCell)this.getLessDaughter()).getAllCells());
        ret.addAll(((EmbryoCell)this.getGreaterDaughter()).getAllCells());
        return ret;
    }
    // return all the cells in the subtree rooted at this cell
    public ArrayList<EmbryoCell> getAllCellsBreath(){
        ArrayList<EmbryoCell> ret = new ArrayList<EmbryoCell>();
        ret.add(this);
        if (this.isLeaf()){
            return ret;
        }
        ret.addAll(((EmbryoCell)this.getLessDaughter()).getAllCells());
        ret.addAll(((EmbryoCell)this.getGreaterDaughter()).getAllCells());
        return ret;
    }
    @Override
    public List<Cell> getLeaves(int time){
        ArrayList<Cell> ret = new ArrayList<>();
        if (this.isLeaf()){
            ret.add(this);
        } else if (time <= this.getEndTime()){
            ret.add(this);
        } else {
            Cell less = (Cell)this.getLessDaughter();
            ret.addAll(less.getLeaves(time));
            Cell great = (Cell)this.getGreaterDaughter();
            ret.addAll(great.getLeaves(time));
        }
        return ret;
    }
    // returns all the leaves of this cell
    // returns this cell if it is a leaf
    public EmbryoCell[] getLeaves(){
        ArrayList<EmbryoCell> leaves = getLeaves(this);
        
        EmbryoCell[] ret = new EmbryoCell[leaves.size()];
        return leaves.toArray(ret);
    }
    // gets all the leaves of the given cell
    public static ArrayList<EmbryoCell> getLeaves(EmbryoCell cell){
        ArrayList<EmbryoCell> leaves = new ArrayList<EmbryoCell>();
        if (cell.isLeaf()){
            leaves.add(cell);
            return leaves;            
        }
        leaves.addAll(EmbryoCell.getLeaves((EmbryoCell)cell.getLessDaughter()));
        leaves.addAll(EmbryoCell.getLeaves((EmbryoCell)cell.getGreaterDaughter()));
        return leaves;
    } 
    // find the nearest leaf in the subtree rooted at this cell that is nearest to the cell provided
    public EmbryoCell nearestLeaf(EmbryoCell other){
        EmbryoCell cell = this.getDescendent(other.getName());
        if (cell != null){
            if (cell.isLeaf()) return cell;
            return cell.leastLeaf();
        }
        EmbryoCell otherParent = other.getParentCell();
        if (otherParent != null){
            return this.nearestLeaf(otherParent);
        }
        return null;
        
    }
    public EmbryoCell leastLeaf(){
        if (this.isLeaf()) return this;
        return ((EmbryoCell)this.getLessDaughter()).leastLeaf();
    }
/*
    public static TreeSet<EmbryoCell> getLeaves(EmbryoCell cell){
        TreeSet<EmbryoCell> leaves = new TreeSet<EmbryoCell>(new Comparator<EmbryoCell>(){
            public int compare(EmbryoCell c1,EmbryoCell c2){
                return -c1.compareTo(c2);
            }
        });
        int n = cell.getChildCount();
        if (n == 0){
            leaves.add(cell);
            return leaves;
        }
        
        for (int i=0 ; i<n ; ++i){
            EmbryoCell child = (EmbryoCell)cell.getChildAt(i);
            leaves.addAll(getLeaves(child));
        }
        return leaves;
    }
*/
    // get all leaves of this cell except the leaves in the "except" subtree
    public ArrayList<EmbryoCell> getLeavesExcept(EmbryoCell except){
        ArrayList<EmbryoCell> leaves = new ArrayList<EmbryoCell>();
        if (this.isLeaf()) {
            if (!except.getName().equals(this.getName())) {
                leaves.add(this);
            }
        } else {
            int n = this.getChildCount();
            for (int i=0 ; i<n ;++i){
                EmbryoCell child = (EmbryoCell)this.getChildAt(i);
                if (!child.getName().equals(except.getName()))
                    leaves.addAll(child.getLeavesExcept(except));
            }
        }
        return leaves;
    }
    // count the number of nuclei in this cell and all the descendent cells
    public long countNuclei() {
        long count = nuclei.size();
        int n = this.getChildCount();
        for (int i=0 ; i<n ; ++i){
            EmbryoCell child = (EmbryoCell)this.getChildAt(i);
            count = count + child.countNuclei();
        }
        return count;
    }
    
    // report the maximum number of children in this cell or any descendent
    public int maxChildren() {
        int n = this.getChildCount();
        int ret = n;
        for (int i=0 ; i<n ; ++i){
            EmbryoCell child = (EmbryoCell)this.getChildAt(i);
            int childCount = child.maxChildren();
            if ( childCount > ret) ret = childCount;
        }
        return ret;
    }
    // count the number of leaf cells under this cell
    // returns one if this cell is a leaf
    private static int leafCount(EmbryoCell cell){
        int n = cell.getChildCount();
        if (n == 0) return 1;
        
        int count = 0;
        for (int i=0 ; i<n ; ++i){
            EmbryoCell child = (EmbryoCell)cell.getChildAt(i);
            count = count + leafCount(child);
        }
        return count;
    }
    // get the longest endtime of all the children
    // return this cell's endtime if there are no children
    public int maxEndTime() {
        if (this.maxTime == null){
            this.maxTime = new Integer(maxEndTime(this));
        }
        return maxTime;
    }
    private static int maxEndTime(EmbryoCell cell) {
        int n = cell.getChildCount();
        if (n == 0) return cell.endTime;
        
        int max = 0;
        for (int i=0 ; i<n ; ++i){
            EmbryoCell child = (EmbryoCell)cell.getChildAt(i);
            int endT = maxEndTime(child);
            if (endT > max) max = endT;
        }
        return max;
    }
    public int getEndTime(){
        return endTime;
    }
    public int getStartTime(){
        return this.startTime;
/*        
        Object par = this.getParent();
        if (par instanceof EmbryoCell){
            // have a parent - so start time is 1 plus parents endtime
            EmbryoCell cell = (EmbryoCell)par;
            return cell.endTime+1;
        } else {
            // do not have a parent - calculate start time based on number of data points
            try {
                return endTime - nuclei.size() + 1;
            } catch (Exception exc){
                int sjnfi=0;
            }
        }
        return -1;
        * */
    }
    public String getName(){
        return (String)this.getUserObject();
    }
    // get the expression(type specific) values for this cell
    public int[] getExpression(String type){
        int[] ret = new int[nuclei.size()];
        for ( int i=0 ; i<ret.length ; ++i){
            TimePointNucleus nuc = nuclei.get(i);
            ret[i] = nuc.getExpression(type);
        }
        return ret;
    }
    // get a descendent cell of the given name
    // returns null if no cell matches the name
    public EmbryoCell getDescendent(String desc){
        try {
        String name = (String)this.getUserObject();
        if (name.equals(desc)){
            return this;
        }
        }catch (Exception exc) {
           int uiosdhgf=0; 
        }
        int n = this.getChildCount();
        for (int i=0 ; i<n ; ++i){
            EmbryoCell child = (EmbryoCell)this.getChildAt(i);
            EmbryoCell ret = child.getDescendent(desc);
            if (ret != null) return ret;
        }
        return null;
    }
    public CellDataTimeSeries getExpTimeSeries(){
        return expr;
    }
    public String toString(){
        return getName();
    }
    
    // dump the entire contents of this cell to a file
    // dumps all the children too
    public void dump(PrintWriter write){
        dump(this,write);
    }
    
    static private void dump(EmbryoCell cell,PrintWriter write){
        int time = cell.getStartTime();
        write.printf("%s,%d,%d\n",cell.getName(),time,cell.getEndTime());
        write.flush();
/*        
        // dump all the Nuclei data
        for (TimePointNucleus nuc : cell.nuclei){
            nuc.dump(time,write);
            ++time;
        }
*/
        int n = cell.getChildCount();
        for (int i=0 ; i<n ; ++i){
            EmbryoCell child = (EmbryoCell)cell.getChildAt(i);
            child.dump(write);
        }
 
    }
    // determine if the given cellname is contained in the subtree rooted at this cell
    public boolean hasCell(String name) {
        if (name.equals(this.getName())) {
            return true;
        }
        int n = this.getChildCount();
        for (int i=0 ; i< n ; ++i){
            EmbryoCell child = (EmbryoCell)this.getChildAt(i);
            if (child.hasCell(name)) {
                return true;
            }
        }
        return false;
    }
    public String getType(){
        return expType;
    }

    public void setExpressionType(String type){
        ExpressionOnset onset = expr.getExprOnset();
        boolean change = expr.getChanged();
        
        expr = new CellDataTimeSeries(this,type);
        expType = type;
        expr.setExprOnset(onset,change);
        int n = this.getChildCount();
        for (int i=0 ; i< n ; ++i){
            EmbryoCell child = (EmbryoCell)this.getChildAt(i);
            child.setExpressionType(type);
        }
    }
    // count the number of onsets set in this subtree
    public int countOnsets(){
        int count = 0;
        if (expr.getExprOnset() != null) count = 1;
        int n = this.getChildCount();
        for (int i=0 ; i<n ; ++i){
            EmbryoCell child = (EmbryoCell)this.getChildAt(i);
            count = count + child.countOnsets();
        }
        return count;
    }
    // test if this cell is a root cell of the embryo
    public boolean isRoot(){
        TreeNode parent = this.getParent();
        if (parent instanceof EmbryoCell) {
            return false;
        }
        return true;
    }
    
    // test if this cell is a leaf in the embryo
    public boolean isLeaf(){
        int n = this.getChildCount();
        if (n == 0){
            return true;
        }
        return false;
    }

    public void saveToDB(MySql db,String series)throws Exception {
        String sql = String.format("Insert into Embryos (SeriesID,Cell,StartTime,EndTime) values (\'%s\',\'%s\',%d,%d)",
                series,this.getName(),this.getStartTime(),endTime);
        db.execute(sql);
        
        //update the parent value if not a root cell
        TreeNode parent = this.getParent();
        if (parent instanceof EmbryoCell){
            EmbryoCell parentCell = (EmbryoCell)parent;
            sql = String.format("Update Embryos set Parent=\'%s\' where SeriesID=\'%s\' and Cell=\'%s\'",
                    parentCell.getName(),series,this.getName());
            db.execute(sql);
        }
        
        // insert all the children cells into the database
        int n = this.getChildCount();
        for (int i=0 ; i<n ; ++i){
            EmbryoCell child = (EmbryoCell)this.getChildAt(i);
            if (child != null){
                child.saveToDB(db, series);
            }
        }
    }
    
    public void saveLinkage(MySql db) throws Exception {
        ArrayList<EmbryoCell> ancestors = this.getAncestors();
        for (EmbryoCell ancestor : ancestors){
            // insert a linkage record for each ancestor
            String sql = String.format("Insert into Linkages (Cell,Ancestor) values (\'%s\',\'%s\') ", 
                    this.getName(),ancestor.getName());
            db.execute(sql);
            
        }
        int n = this.getChildCount();
        for (int i=0 ; i< n ; ++i){
            EmbryoCell child = (EmbryoCell)this.getChildAt(i);
            child.saveLinkage(db);
        }
    }
    public ArrayList<EmbryoCell> getAncestorsInclusiveUpto(String stopWith) {
        ArrayList<EmbryoCell> ret = getAncestorsUpto(stopWith);
        ret.add(this);
        return ret;
    }    
    public ArrayList<EmbryoCell> getAncestorsUpto(String stopWith){
        ArrayList<EmbryoCell> ret = new ArrayList<>();
        Object obj = this.getParent();
        if (obj instanceof EmbryoCell){
            EmbryoCell par = (EmbryoCell)obj;
            if (!stopWith.equals(par.getName())){
                ret.addAll(par.getAncestorsUpto(stopWith));
            }
            ret.add(par);
        }
        return ret;        
    }
    // get a list of all the ancestors of this cell
    //the list does not include the cell itself
    public ArrayList<EmbryoCell> getAncestors(){
        ArrayList<EmbryoCell> ret = new ArrayList<EmbryoCell>();
        Object obj = this.getParent();
        if (obj instanceof EmbryoCell){
            EmbryoCell par = (EmbryoCell)obj;
           
            ret.addAll(par.getAncestors());
             ret.add(par);
        }
        return ret;
    }
    public ArrayList<EmbryoCell> getAncestorsInclusive() {
        ArrayList<EmbryoCell> ret = getAncestors();
        ret.add(this);
        return ret;
    }
    // find the earliest cell in this embroy that is complete, ie start and endtimes truely represent 
    // when the cell began and ended.  The returned cell is the first generation cell after the 
    // root for which this cell is a descendent
    public EmbryoCell earliestCompleteAncestor(){
        EmbryoCell parentCell = (EmbryoCell)this.getParent();
        if (parentCell.isRoot()){
            return this;
        }
        return parentCell.earliestCompleteAncestor();
    }
    // sulstonize this cell
    // provide the sulston embryo and the average time ratio between this embryo and the sulton embryo
    // ratio = sulton/embryo time
    // returns true if the cell could be sulstonized
    public boolean sulstonize(org.rhwlab.universalimaging.Sulston sulstonEmbryo,double averageRatio){

        if (this.getName().equals("P0")) {
            return false;
        } // cannot sulstonize P0
       
        Cell sulstonCell = sulstonEmbryo.getCell(this.getName());
        if (sulstonCell == null){
            return false; 
        }
        this.sulston = new org.rhwlab.beans.Sulston(sulstonCell.getStartTime(),sulstonCell.getEndTime(),averageRatio);
        return true;
    }
    // determine the embryo time coresponding to a sulston time for this cell
    public double embryoTime(double sulstonTime)throws Exception {
        if (this.sulston == null){
            throw new Exception("Cell not sulstonized");
        }
        if (this.sulston.start <= sulstonTime && sulstonTime <= this.sulston.end){
            if (this.isRoot()){
                return this.endTime - (this.sulston.end - sulstonTime)/this.sulston.ratio;  // could be negative 
            }else {
                double ratio = sulston.ratio;
                if (!this.isLeaf()){
                    ratio = (double)(sulston.end-sulston.start)/(double)(this.getEndTime()-this.getStartTime());
                }
                double delsulT = sulstonTime-this.sulston.start;
                double delembT = delsulT/ratio;
                return this.getStartTime() + delembT;  // could be past cell end
            }
        }
        String msg = String.format("Sulston time out of range of cell %s: %d,%d  %d,%d   %f",this.getName(),this.getStartTime(),this.getEndTime(),this.sulston.start,this.sulston.end,sulstonTime);
        throw new Exception(msg);
    }
    // the cell must be sulstonized before this call or it will fail
    // input time is experiment time
    public float toSulston(int time) throws Exception {
        if (time < this.startTime || time > this.endTime){
            throw new Exception("time is out of range for the cell");
        }
        if (sulston==null){
            return -1;
        }
        float ratio = (float)sulston.getRatio(); // overall ratio (used in roots and leafs)
        if (!this.isRoot()){
            // interpolate from the  start time
            if (!this.isLeaf()){
                // compute a ratio for this cell
                ratio = (float)(sulston.end-sulston.start)/(float)(this.getEndTime()-this.getStartTime());
            }   
            float ret = (float)sulston.getStart() + ratio*(time-this.getStartTime());  
            if (ret > sulston.getEnd()) ret = sulston.getEnd();
            return ret;
        }
        
        float ret = sulston.getEnd() - ratio*(this.getEndTime()-time);  // interpolate from the end time
        if (ret < sulston.start) ret = sulston.start;
        return ret;
    }

 /*   
    // convert an index time to standard Sulston time
    public float toSulston(MySql db,int time) throws Exception {
        if (db == null){
            //connect to the database
            db = MySql.getMySql();
        }
//        System.out.printf("\ntoSulston: cell:%s, time:%d\n",this.getName(),time);
        Sulston suls = getSulston(db,this.isLeaf());
//        suls.dump();
        float ret = 0;
        if (this.isLeaf()){
            ret = (float)suls.getStart() + (float)suls.getRatio()*(time-this.getStartTime());
            
        }else if (this.isRoot()){
            // the root is not necessarliy complete - it will likely have started later than the sulston cell
            // compute the average sulston ratio of all the leaves of the root
            EmbryoCell[] leaves = this.getLeaves();
            float ratio = 0;
            for (EmbryoCell leaf : leaves){
                float r = (float)leaf.getSulston(db,true).getRatio();
                if (r == Float.POSITIVE_INFINITY){
                    System.err.println("ratio of infinity in toSulston");
                }
                ratio = ratio + r;
            }
            ratio = ratio/leaves.length;
            
            // interpolate based on endtime
            ret = suls.getEnd() - ratio*(this.getEndTime()-time);
            
        }else {
            ret =  (float)suls.getStart() + (float)suls.getRatio()*(time-this.getStartTime());
        }
//        System.out.printf("toSulston: sulstonTime = %f\n",ret);
        return ret;
    }
   
    // determine the ratio of sulston time to measured time in this embryo above this cell
    public double sulstonRatio(MySql db) throws Exception {
        EmbryoCell early = this.earliestCompleteAncestor();
        EmbryoCell par = (EmbryoCell)this.getParent();
        
        String sql = String.format("Select * from Embryos where SeriesID=\'20081128_sulston\' and Cell=\'%s\'",early.getName());
        ResultSet rs = db.execute(sql);
        rs.next();
        float sulstonStart = rs.getInt("StartTime");
        
        sql = String.format("Select * from Embryos where SeriesID=\'20081128_sulston\' and Cell=\'%s\'",par.getName());
        rs = db.execute(sql);
        if (rs.next()){
            float sulstonEnd=0;
            try {
                sulstonEnd = rs.getInt("EndTime");
            } catch (Exception exc){
                exc.printStackTrace();
                int jsadfhis=0;
            }
        
            return (sulstonEnd-sulstonStart)/(par.getEndTime()-early.getStartTime());            
        }
        return par.sulstonRatio(db);  // this cells parent is not in sulston database - move up and compute ratio

    }    
    // calculate the sulston data for this cell
    public Sulston getSulston(MySql db,boolean asLeaf) throws Exception {
        if (this.getName().equals("MSaaappp")){
            int kjsafdo=0;
        }
        if (sulston != null) return sulston;  // do not recalculate if already done      
        sulston = new Sulston();
        
        // is this cell in the sulston database
        String sql = String.format("Select * from Embryos where SeriesID=\'20081128_sulston\' and Cell=\'%s\'",this.getName());
        ResultSet rs = db.execute(sql);
        if (!rs.next()){  // cell is not in the database
            EmbryoCell par=null;
            // use the sulston of this cell's parent to calculate the sulston of this cell
            try {
                par = (EmbryoCell)this.getParent();
            } catch (Exception exc){
                exc.printStackTrace();
            }
            Sulston parentSulston = par.getSulston(db,true);  // always calculate parent's sulston as a leaf when this cell is not in sulston db
            sulston.setRatio(parentSulston.getRatio());
            sulston.setStart(parentSulston.getEnd()+1);
            sulston.setEnd((int)parentSulston.getRatio()*(this.getEndTime()-this.getStartTime())+parentSulston.getEnd());
        } else {  // cell is in sulston db
            sulston.setStart(rs.getInt("StartTime"));
            sulston.setEnd(rs.getInt("EndTime")); 
            if (asLeaf){  // this cell is a leaf - this means the endtime is not at a cell division
                sulston.setRatio(this.sulstonRatio(db));  // use all the cells above to calculate the sulston ratio
            }else {
                int d = this.getEndTime()-this.getStartTime()+1;
                sulston.setRatio((double)(sulston.getEnd()-sulston.getStart()+1)/(double)d);
            }
        }      
        return sulston;
    }
     * 
     */
    // get the root cell of this cell
    public EmbryoCell getRoot(){
        Object par = this.getParent();
        if (par instanceof EmbryoCell) {
            EmbryoCell parentCell = (EmbryoCell)par;
            return parentCell.getRoot();
        }
        return this;
    }
    
    // get the number of expressing leaves below this cell
    public int expressingLeaves(){
        if (expr.getExprOnset()!=null){
            EmbryoCell[] leaves = this.getLeaves();
            return leaves.length;
        }
        int count = 0;
        int n = this.getChildCount();
        for (int i=0 ; i<n ; ++i){
            EmbryoCell child = (EmbryoCell)this.getChildAt(i);
            count = count + child.expressingLeaves();
        }
        return count;
    }
    public EmbryoCell getSibling(){
        Object par = this.getParent();
        if (par instanceof EmbryoCell) {
            EmbryoCell parentCell = (EmbryoCell)par;
            int n = parentCell.getChildCount();
            if (n == 2){
                EmbryoCell child0 = (EmbryoCell)parentCell.getChildAt(0);
                EmbryoCell child1 = (EmbryoCell)parentCell.getChildAt(1);
                if (child0.getName().equals(this.getName())){
                    return child1;
                }else {
                    return child0;
                }
            } else {
                return null;
            }
        }
        return null;
    }

    // find an ancestor cell (may include this cell) that contains the given time
    public EmbryoCell getAncestorAtTime(int time){
       // is the time in this cell
        if (time <= this.endTime && time >= this.getStartTime()){
            return this;
        }
        if (time < this.getStartTime()){  // can the time be in a cell above?
            EmbryoCell p = this.getParentCell();
            if (p == null) return null;  
            return p.getAncestorAtTime(time);
        }
        return null;
    }
    public EmbryoCell getParentCell(){
        Object par = this.getParent();
        if (par instanceof EmbryoCell) {
            EmbryoCell parentCell = (EmbryoCell)par;
            return parentCell;
        }
        return null;
    }
    // finds a parent of this cell with the given name
    public EmbryoCell getParentByName(String name){
        Object par = this.getParent();
        if (par instanceof EmbryoCell) {
            EmbryoCell parentCell = (EmbryoCell)par;
            if (parentCell.getName().equals(name)){
                return parentCell;
            }
            return parentCell.getParentByName(name);
        } 
        return null;
    }

    // determine is a cell is a child of this cell or any of it's descendents
    public boolean hasChild(String name){
        int n = this.getChildCount();
        for (int i=0 ; i<n ;++i){
            EmbryoCell child = (EmbryoCell)this.getChildAt(i);
            if (child.hasCell(name)) return true;
        }
        return false;
    }
    public boolean onPlane(int time,int plane,double zPixelRes){
        // 
        TimePointNucleus nuc = nuclei.get(time-this.getStartTime());
       
        double radius = (double)nuc.size/2.0;  // this is in xy pixels
        // covert the radius from pixels to z planes
        radius = (radius/zPixelRes);  //zPixelRes is the number of xy pixels there would be between each z plane
 //       if (plane >= nuc.z-radius-1 && plane <= nuc.z+radius+1) return true;
        if (plane >= nuc.z-radius && plane <= nuc.z+radius) return true;
        return false;

    }
    /*
    public double nucDiameter(TimePointNucleus n, double imgPlane){
        return nucDiameter(n,imgPlane,1.0/0.9);
    }
        public boolean hasCircle(TimePointNucleus n, double imgPlane) {
        double d = nucDiameter(n, imgPlane);
        return (d > 0);
    }* 
    */
    public double nucDiameter(TimePointNucleus n, double imgPlane,double zPixelsPerPlane) {
            /*
        if (n == null) return -1; //covers some issues re currentCell and not tracking
        double r = -0.5;
        double cellPlane = n.z;
        double R = n.size/2.; //pixels
//        double y = (cellPlane - imgPlane)*iZPixRes/R;
        double y = (cellPlane - imgPlane)/R;
        double r2 = 1 - y*y;
        if (r2 >= 0.) r = Math.sqrt(r2)*R;
        return 2*r;
        * */
        double cellPlane = n.z;
 //       double R = (double)n.size; //pixels 
        double R = (double)n.size/2.0; //pixels         
        double y = Math.abs(cellPlane - imgPlane)*zPixelsPerPlane;
 //       return Math.sqrt(R*R-y*y);
        return 2.0*Math.sqrt(R*R-y*y);
    }
/*    
        public double nucDiameter(TimePointNucleus n, double imgPlane) {
        if (n == null) return -1; //covers some issues re currentCell and not tracking
        double r = -0.5;
        double cellPlane = n.z;
        double R = n.size/2.; //pixels
//        double y = (cellPlane - imgPlane)*iZPixRes/R;
        double y = (cellPlane - imgPlane)/R;
        double r2 = 1 - y*y;
        if (r2 >= 0.) r = Math.sqrt(r2)*R;
        return 2*r;
    }
*/

    
    public int xAtTime(int time){
        if (time <= endTime && time >= this.getStartTime()){
            TimePointNucleus nuc = nuclei.get(time-this.getStartTime());
            return nuc.x;
        }
        return -1;
    }
    public int yAtTime(int time){
        if (time <= endTime && time >= this.getStartTime()){
            TimePointNucleus nuc = nuclei.get(time-this.getStartTime());
            return nuc.y;
        }
        return -1;
    } 
    public int radiusAtTime(int time){
        if (time <= endTime && time >= this.getStartTime()){
            TimePointNucleus nuc = nuclei.get(time-this.getStartTime());
            return nuc.size;
        }
        return -1;
    }     
    public int timeAtLargestRadius(){
        int mx = 0;
        int ret=0;
        for (int i=0;i<nuclei.size();++i){
            TimePointNucleus nuc = nuclei.get(i);
            if (nuc.size>mx){
                mx = nuc.size;
                ret = i;
            }
        }
        return ret+this.getStartTime();
    }
    public int planeAtTime(int time){
        int t = time-this.getStartTime();
        return (int)nuclei.get(t).z;
    }
    
    public int backgroundCorrectedExpression(int time,int back){
        // get the nucleus for the given time
        TimePointNucleus nuc = this.getNucleusAtTime(time);
        return nuc.getTotalExpression() - (int)(back*nuc.getVolume()/1000.0);
    }
    
    public TimePointNucleus getNucleusAtTime(int t){
        int index = t - this.getStartTime();
        TimePointNucleus nuc = nuclei.get(index);
        return nuc;
    }
    // set the payload for this cell
    public void setPayLoad(Object obj){
        this.payLoad = obj;
    }
    public Object getPayLoad(){
        return this.payLoad;
    }
    // set the payload of this cell and all it's descendents to the given object
    public void setAllPayLoads(Object obj){
        this.setPayLoad(obj);
        if (!this.isLeaf()){
            ((EmbryoCell)this.getLessDaughter()).setAllPayLoads(obj);
            ((EmbryoCell)this.getGreaterDaughter()).setAllPayLoads(obj);
        }
        
    }
    // determine if the payload of this cell and all descendents is empty
    public boolean isPayLoadEmpty(){
        if (this.payLoad != null){
            return false;
        }
        if (this.isLeaf()) {
            return true;
        }
        if (  ((EmbryoCell)this.getLessDaughter()).isPayLoadEmpty() && ((EmbryoCell)this.getGreaterDaughter()).isPayLoadEmpty() ){
            return true;
        }
        return false;
    }
    public String fromSeries(){
        return nuclei.get(0).getSeriesID();
    }
    // returns the embryo that this cell is in
    public SeriesEmbryo getEmbryo(){
        return this.embryo;
    }

    // the average (over all its life time)  x position of this cell 
    public double averageX(){
        double total = 0.0;
        for (TimePointNucleus nuc : nuclei){
            total = total + nuc.x;
        }
        return total/nuclei.size();
    }
    public double averageY(){
        double total = 0.0;
        for (TimePointNucleus nuc : nuclei){
            total = total + nuc.y;
        }
        return total/nuclei.size();
    }  
    public double getSulstonStart()throws Exception {
        return this.toSulston(this.getStartTime());
    }
    public double getSulstonEnd()throws Exception {
        return this.toSulston(this.getEndTime());
    }
    @Override
    public double[] getPosition(int embryoTime) {
        double[] ret = new double[3];
        ret[0] = this.xAtTime(embryoTime);
        ret[1] = this.yAtTime(embryoTime);
        ret[2] = this.planeAtTime(embryoTime);
        return ret;
    }

    @Override
    public double sphereRadius(int embryoTime) {
        return this.radiusAtTime(embryoTime);
    }

    @Override
    public Cell[] getChildrenCells() {
        Cell[] ret = new Cell[this.getChildCount()];

        for (int i=0 ; i<ret.length ;++i){
            ret[i] = (Cell)this.getChildAt(i);
        }
        return ret;
    }

    @Override
    public Integer getExpressionBySulstonTime(int sulstonTime) throws Exception {
        int t = (int)this.embryoTime(sulstonTime);
        if (this.getStartTime()<=t && t<=this.endTime){
            int[] exp = this.getExpressionValues();
            return exp[t-this.getStartTime()];
        }
        return null;
    }
    static public void main(String[] args)throws Exception {
        String[] lineages = {"ABar","ABal","ABpr","ABpl","E","MS","P3","C"};
        ArrayList<String> seriesList = Imaged.publishedByGeneConstruct(args[0], args[1]);
        
        // sulstonize all the embryos
        SulstonEmbryo sulston = new SulstonEmbryo(MySql.getMySql());
        SeriesEmbryo[] embryos = new SeriesEmbryo[seriesList.size()];
        for (int i=0 ; i<seriesList.size() ;++i){
            embryos[i] = new SeriesEmbryo(MySql.getMySql(), seriesList.get(i), "zblot");
            sulston.sulstonize(embryos[i]);
        }
        
        for (String lineage : lineages){
            ArrayList<EmbryoCell> rootCells = new ArrayList<>();
            for (SeriesEmbryo embryo : embryos){
                rootCells.add(embryo.getCell(lineage));
            }
            EmbryoCell consensus = EmbryoCell.ConsensusTree(rootCells);
            
        }
    }
    Integer maxTime = null;
    int endTime;
    int startTime;
    ArrayList<TimePointNucleus> nuclei;
    CellDataTimeSeries expr;
    String expType;
    private Sulston sulston = null;
    Object payLoad;
    double[] logValues = null;
    SeriesEmbryo embryo;

    @Override
    public boolean isAfterOnset(int threshold, int ntimes) throws Exception {
        EmbryoCell parent = this.getParentCell();
        if (parent != null){
            if (parent.isAfterOnset(threshold,ntimes)) {
                return true;
            } else {
                Integer onset = this.sulstonTimeOnset(threshold,ntimes);
                return onset!=null;                
            }
        }
        Integer onset = this.sulstonTimeOnset(threshold,ntimes);
        return onset!=null;
    }
/*
    @Override
    public Integer sulstonTimeOnset(int threshold, int ntimes) throws Exception {
        int[] expValues = this.getExpressionValues();
        int first = -1;
        int count = 0;
        for (int i=0 ; i<expValues.length ; ++i){
            if (expValues[i] >=threshold){
                ++count;
                if (first == -1){
                    first = i;
                }
            }
        }
        if (count >= ntimes){       
            return (int)this.toSulston(this.getStartTime()+first);
        }
        return null;
    }
    */

    @Override
    public Integer sulstonTimeOnset(int threshold, int ntimes) throws Exception {
        if (ntimes < 0){
            ntimes = 10;
            int[] expValues = this.getExpressionValues();
            for (int i=0 ; i+ntimes<expValues.length ; ++i){
                double sum = 0.0;
                for (int j=0 ; j<ntimes ; ++j){
                    sum = sum + expValues[i+j];
                }
                sum = sum/ntimes;
                if (sum >=threshold){
                    return (int)this.toSulston(this.getStartTime()+i+ntimes/2);
                }
            }
            return null;
        } else {
            int[] expValues = this.getExpressionValues();
            int first = -1;
            int count = 0;
            for (int i=0 ; i<expValues.length ; ++i){
                if (expValues[i] >=threshold){
                    ++count;
                    if (first == -1){
                        first = i;
                    }
                }
            }
            if (count >= ntimes){       
                return (int)this.toSulston(this.getStartTime()+first);
            }
            return null;            
        }
    }
    
    @Override
    public Cell getParentAsCell() {
        return this.getParentCell();
    }



}