/*
 * Decompiled with CFR 0.152.
 */
package jsat.linear;

import java.util.Arrays;
import java.util.List;
import jsat.linear.IndexValue;
import jsat.linear.Matrix;
import jsat.linear.Vec;

public class DenseVector
extends Vec {
    private static final long serialVersionUID = -889493251793828934L;
    protected double[] array;
    private Double sumCache = null;
    private Double varianceCache = null;
    private Double minCache = null;
    private Double maxCache = null;
    private int startIndex;
    private int endIndex;

    public DenseVector(int length) {
        if (length < 0) {
            throw new ArithmeticException("You can not have a negative dimension vector");
        }
        this.array = new double[length];
        this.startIndex = 0;
        this.endIndex = this.array.length;
    }

    public DenseVector(List<Double> list) {
        this.array = new double[list.size()];
        for (int i = 0; i < list.size(); ++i) {
            this.array[i] = list.get(i);
        }
        this.startIndex = 0;
        this.endIndex = this.array.length;
    }

    public DenseVector(double[] array) {
        this(array, 0, array.length);
    }

    public DenseVector(double[] array, int start, int end) {
        this.array = array;
        this.startIndex = start;
        this.endIndex = end;
    }

    public DenseVector(Vec toCopy) {
        this(toCopy.length());
        for (IndexValue iv : toCopy) {
            this.set(iv.getIndex(), iv.getValue());
        }
    }

    private void clearCaches() {
        this.sumCache = null;
        this.varianceCache = null;
        this.minCache = null;
        this.maxCache = null;
    }

    @Override
    public int length() {
        return this.endIndex - this.startIndex;
    }

    @Override
    public double get(int index) {
        return this.array[index + this.startIndex];
    }

    @Override
    public void set(int index, double val) {
        this.clearCaches();
        this.array[index + this.startIndex] = val;
    }

    @Override
    public double min() {
        if (this.minCache != null) {
            return this.minCache;
        }
        double result = this.array[this.startIndex];
        for (int i = this.startIndex + 1; i < this.endIndex; ++i) {
            result = Math.min(result, this.array[i]);
        }
        this.minCache = result;
        return this.minCache;
    }

    @Override
    public double max() {
        if (this.maxCache != null) {
            return this.maxCache;
        }
        double result = this.array[this.startIndex];
        for (int i = this.startIndex + 1; i < this.endIndex; ++i) {
            result = Math.max(result, this.array[i]);
        }
        this.maxCache = result;
        return this.maxCache;
    }

    @Override
    public double sum() {
        if (this.sumCache != null) {
            return this.sumCache;
        }
        double sum = 0.0;
        double c = 0.0;
        for (int i = this.startIndex; i < this.endIndex; ++i) {
            double y = this.array[i] - c;
            double t = sum + y;
            c = t - sum - y;
            sum = t;
        }
        this.sumCache = sum;
        return this.sumCache;
    }

    @Override
    public double median() {
        double[] copy = Arrays.copyOfRange(this.array, this.startIndex, this.endIndex);
        Arrays.sort(copy);
        if (copy.length % 2 == 1) {
            return copy[copy.length / 2];
        }
        return copy[copy.length / 2] / 2.0 + copy[copy.length / 2 + 1] / 2.0;
    }

    @Override
    public double skewness() {
        double mean = this.mean();
        double tmp = 0.0;
        for (int i = this.startIndex; i < this.endIndex; ++i) {
            tmp += Math.pow(this.array[i] - mean, 3.0);
        }
        double s1 = tmp / (Math.pow(this.standardDeviation(), 3.0) * (double)(this.array.length - 1));
        if (this.array.length >= 3) {
            return Math.sqrt(this.array.length * (this.array.length - 1)) / (double)(this.array.length - 2) * s1;
        }
        return s1;
    }

    @Override
    public double kurtosis() {
        double mean = this.mean();
        double tmp = 0.0;
        for (int i = this.startIndex; i < this.endIndex; ++i) {
            tmp += Math.pow(this.array[i] - mean, 4.0);
        }
        return (tmp /= (double)this.length()) / Math.pow(this.variance(), 2.0) - 3.0;
    }

    @Override
    public DenseVector sortedCopy() {
        double[] copy = Arrays.copyOfRange(this.array, this.startIndex, this.endIndex);
        Arrays.sort(copy);
        return new DenseVector(copy);
    }

    @Override
    public double variance() {
        if (this.varianceCache != null) {
            return this.varianceCache;
        }
        double mu = this.mean();
        double tmp = 0.0;
        double N = this.length();
        for (int i = this.startIndex; i < this.endIndex; ++i) {
            tmp += Math.pow(this.array[i] - mu, 2.0) / N;
        }
        this.varianceCache = tmp;
        return this.varianceCache;
    }

    @Override
    public double dot(Vec v) {
        if (this.length() != v.length()) {
            throw new ArithmeticException("Vectors must have the same length");
        }
        if (v.isSparse()) {
            return v.dot(this);
        }
        double dot = 0.0;
        for (int i = this.startIndex; i < this.endIndex; ++i) {
            dot += this.array[i] * v.get(i - this.startIndex);
        }
        return dot;
    }

    public DenseVector deepCopy() {
        return new DenseVector(Arrays.copyOf(this.array, this.array.length));
    }

    @Override
    public void multiply(double c, Matrix A, Vec b) {
        if (this.length() != A.rows()) {
            throw new ArithmeticException("Vector x Matrix dimensions do not agree [1," + this.length() + "] x [" + A.rows() + ", " + A.cols() + "]");
        }
        if (b.length() != A.cols()) {
            throw new ArithmeticException("Destination vector is not the right size");
        }
        for (int i = 0; i < this.length(); ++i) {
            double this_i = c * this.array[i + this.startIndex];
            for (int j = 0; j < A.cols(); ++j) {
                b.increment(j, this_i * A.get(i, j));
            }
        }
    }

    @Override
    public void mutableAdd(double c) {
        this.clearCaches();
        int i = this.startIndex;
        while (i < this.endIndex) {
            int n = i++;
            this.array[n] = this.array[n] + c;
        }
    }

    @Override
    public void mutableAdd(double c, Vec b) {
        if (this.length() != b.length()) {
            throw new ArithmeticException("Can not add vectors of unequal length");
        }
        this.clearCaches();
        if (b.isSparse()) {
            for (IndexValue iv : b) {
                int n = iv.getIndex();
                this.array[n] = this.array[n] + c * iv.getValue();
            }
        } else {
            for (int i = this.startIndex; i < this.endIndex; ++i) {
                int n = i;
                this.array[n] = this.array[n] + c * b.get(i);
            }
        }
    }

    @Override
    public void mutableSubtract(double c) {
        this.clearCaches();
        int i = this.startIndex;
        while (i < this.endIndex) {
            int n = i++;
            this.array[n] = this.array[n] - c;
        }
    }

    @Override
    public void mutableMultiply(double c) {
        this.clearCaches();
        int i = this.startIndex;
        while (i < this.endIndex) {
            int n = i++;
            this.array[n] = this.array[n] * c;
        }
    }

    @Override
    public void mutableDivide(double c) {
        this.clearCaches();
        int i = this.startIndex;
        while (i < this.endIndex) {
            int n = i++;
            this.array[n] = this.array[n] / c;
        }
    }

    @Override
    public double pNormDist(double p, Vec y) {
        if (this.length() != y.length()) {
            throw new ArithmeticException("Vectors must be of the same length");
        }
        double norm = 0.0;
        if (y.isSparse()) {
            int lastIndx = -1;
            for (IndexValue iv : y) {
                for (int i = lastIndx + 1; i < iv.getIndex(); ++i) {
                    norm += Math.pow(Math.abs(this.array[i]), p);
                }
                lastIndx = iv.getIndex();
                norm += Math.pow(Math.abs(this.array[iv.getIndex()] - iv.getValue()), p);
            }
            for (int i = lastIndx + 1; i < y.length(); ++i) {
                norm += Math.pow(Math.abs(this.array[i]), p);
            }
        } else {
            for (int i = this.startIndex; i < this.endIndex; ++i) {
                norm += Math.pow(Math.abs(this.array[i] - y.get(i)), p);
            }
        }
        return Math.pow(norm, 1.0 / p);
    }

    @Override
    public double pNorm(double p) {
        if (p <= 0.0) {
            throw new IllegalArgumentException("norm must be a positive value, not " + p);
        }
        double result = 0.0;
        if (p == 1.0) {
            for (int i = this.startIndex; i < this.endIndex; ++i) {
                result += Math.abs(this.array[i]);
            }
        } else if (p == 2.0) {
            for (int i = this.startIndex; i < this.endIndex; ++i) {
                result += this.array[i] * this.array[i];
            }
            result = Math.sqrt(result);
        } else if (Double.isInfinite(p)) {
            for (int i = this.startIndex; i < this.endIndex; ++i) {
                result = Math.max(result, Math.abs(this.array[i]));
            }
        } else {
            for (int i = this.startIndex; i < this.endIndex; ++i) {
                result += Math.pow(Math.abs(this.array[i]), p);
            }
            result = Math.pow(result, 1.0 / p);
        }
        return result;
    }

    @Override
    public DenseVector clone() {
        DenseVector copy = new DenseVector(this.length());
        System.arraycopy(this.array, this.startIndex, copy.array, 0, this.length());
        return copy;
    }

    @Override
    public void normalize() {
        double sum = 0.0;
        for (int i = this.startIndex; i < this.endIndex; ++i) {
            sum += this.array[i] * this.array[i];
        }
        sum = Math.sqrt(sum);
        this.mutableDivide(Math.max(sum, 1.0E-10));
    }

    @Override
    public void mutablePairwiseMultiply(Vec b) {
        if (this.length() != b.length()) {
            throw new ArithmeticException("Vectors must have the same length");
        }
        for (int i = this.startIndex; i < this.endIndex; ++i) {
            int n = i;
            this.array[n] = this.array[n] * b.get(i);
        }
    }

    @Override
    public void mutablePairwiseDivide(Vec b) {
        if (this.length() != b.length()) {
            throw new ArithmeticException("Vectors must have the same length");
        }
        for (int i = this.startIndex; i < this.endIndex; ++i) {
            int n = i;
            this.array[n] = this.array[n] / b.get(i);
        }
    }

    @Override
    public boolean equals(Object obj) {
        if (!(obj instanceof Vec)) {
            return false;
        }
        Vec otherVec = (Vec)obj;
        if (this.length() != otherVec.length()) {
            return false;
        }
        for (int i = this.startIndex; i < this.endIndex; ++i) {
            if (this.get(i) == otherVec.get(i)) continue;
            return Double.isNaN(this.get(i)) && Double.isNaN(otherVec.get(i));
        }
        return true;
    }

    @Override
    public boolean equals(Object obj, double range) {
        if (!(obj instanceof Vec)) {
            return false;
        }
        Vec otherVec = (Vec)obj;
        range = Math.abs(range);
        if (this.length() != otherVec.length()) {
            return false;
        }
        for (int i = this.startIndex; i < this.endIndex; ++i) {
            if (!(Math.abs(this.get(i) - otherVec.get(i)) > range)) continue;
            return Double.isNaN(this.get(i)) && Double.isNaN(otherVec.get(i));
        }
        return true;
    }

    public static DenseVector toDenseVec(double ... array) {
        return new DenseVector(array);
    }

    @Override
    public double[] arrayCopy() {
        return Arrays.copyOfRange(this.array, this.startIndex, this.endIndex);
    }

    @Override
    public boolean isSparse() {
        return false;
    }
}

