// Copyright (c) 2017 10X Genomics, Inc. All rights reserved.
use serde_json;
use std::fs::File;
use std::io::BufReader;
use std::io::BufRead;
use std::io::Read;
use std::collections::HashMap;
use ::correction::Correction;
use bincode::{serialize, deserialize_from, Infinite};

#[derive(Serialize, Deserialize)]
pub struct BarcodeCount {
    pub bad_bc_count: i32,
    pub bc_error_rate: f64,
    pub bc_counts: Vec<u32>,
}

pub struct BarcodeValidator {
    pub whitelist_exists: bool,
    pub whitelist: Vec<String>,
    pub barcode_2_idx: HashMap<String, usize>,
    pub ec_table_left: HashMap<String, Correction>,
    pub ec_table_right: HashMap<String, Correction>,
    pub max_expected_errors: f64,
    pub barcode_rates: HashMap<u8, Vec<f64>>,
    pub bc_confidence_threshold: f64,
    pub max_num_err: i32,
}


impl BarcodeValidator{
    pub fn new(whitelist_filename: String, ec_table_left_filename: String,
        ec_table_right_filename: String, barcode_counts_filename: String,
        max_expected_barcode_errors: f64, bc_confidence_threshold: f64, max_num_err:i32, pseudo_count: f64)
        -> BarcodeValidator {

        // load in the left ec_table
        let mut file = File::open(ec_table_left_filename).expect("Fail to open the left ec_table json file.");
        let mut data = String::new();
        file.read_to_string(&mut data).expect("Fail to read the whole file from the left ec_table file.");
        let ec_table_left: HashMap<String, Correction> =
            serde_json::from_str(&data).expect("Fail to convert the json file string to the left ec_table.");


        let mut file = File::open(ec_table_right_filename).expect("Fail to open the right ec_table json file.");
        let mut data = String::new();
        file.read_to_string(&mut data).expect("Fail to read the whole file from the right ec_table file.");
        let ec_table_right: HashMap<String, Correction> =
            serde_json::from_str(&data).expect("Fail to convert the json file string to the right ec_table.");

        // load in the whitelist barcodes
        let file = File::open(whitelist_filename).expect("no such file");
        let buf = BufReader::new(file);
        let mut whitelist: Vec<String> = buf.lines().map(|l| l.expect("Could not parse line")).collect();

        whitelist.sort();
        let mut barcode_2_idx: HashMap<String, usize> = HashMap::new();
        for (index, bc) in whitelist.iter().enumerate() {
            barcode_2_idx.insert(bc.clone(), index);
        }

        // read bc_count from the rsd format generated by bincode
        println!("Starting reading barcode_counts_filename");
        println!("Number of bcs {}", whitelist.len());
        let mut file = BufReader::new(File::open(barcode_counts_filename).expect("Fail to open the bc count rsd file."));
        let bc_count_hashmap: HashMap<(u8, u32), u32> = deserialize_from(&mut file, Infinite).unwrap();
        let mut barcode_count: HashMap<u8, BarcodeCount> = HashMap::new();
        for (b, c) in bc_count_hashmap {
            let gem_group = b.0;
            let entry = barcode_count.entry(gem_group).or_insert_with(|| {
                BarcodeCount{
                    bad_bc_count:   0,
                    bc_error_rate:  0.0,
                    bc_counts:      vec![0_u32; whitelist.len()],
                }
            });
            if b.1 > 0 {
                entry.bc_counts[b.1 as usize - 1] = c;
            }
        }

        println!("Ending reading barcode_counts_filename");

        let mut barcode_rates: HashMap<u8, Vec<f64>> = HashMap::new();
        for (gem_group, counts) in barcode_count {
            let sum: u32 = counts.bc_counts.iter().sum();
            let gem_group_rates = counts.bc_counts.iter().map(|x| (*x as f64 + pseudo_count) / sum as f64).collect();
            barcode_rates.insert(gem_group, gem_group_rates);
        }

        BarcodeValidator{whitelist_exists:true, whitelist:whitelist,
            ec_table_left:ec_table_left, ec_table_right:ec_table_right,
            max_expected_errors: max_expected_barcode_errors, barcode_rates: barcode_rates,
            bc_confidence_threshold: bc_confidence_threshold, barcode_2_idx:barcode_2_idx,
            max_num_err: max_num_err}
    }
}

#[derive(Debug)]
pub enum CorrectionRst {
    UNCORRECTABLE,
    LOWPROB,
    OLDBC,
    CORRECT,
    DIFFBC(String),
    NEWBC(String),
    NA,
}
