function Regression_Analysis_GUI

% Load Datasets
AllProfiles = [];
load('Average_Dyad_Density_Profiles.mat', 'Label', 'AllProfiles');

% Create and then hide the GUI as it is being constructed.
MainFigHandle = figure(...       % The main GUI figure
    'Visible', 'off', ...
    'MenuBar', 'none', ...
    'Toolbar', 'none', ...
    'HandleVisibility', 'on', ...
    'Color', get(0,'defaultuicontrolbackgroundcolor'), ...
    'Position', [10000,10000,1800,790], ...
    'Resize', 'on');

% Construct the components

% Save button
mSave = uimenu(MainFigHandle, 'Label', 'Save figure...', ...
    'Accelerator', 'S', ...
    'HandleVisibility', 'off', ...
    'Callback', {@PrintFigure_Callback});

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Popups for data sets
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

hPanel1 = uipanel('Title','Data set 1','FontSize',18,...
    'units','pixels',...
    'Position',[1 121 600 60]);
hPopupData1 = uicontrol('Style', 'popup',...
    'Parent', hPanel1, ...
    'String', Label,...
    'Position',[5 -41 585 70], ...
    'FontSize', 16, ...
    'Value', 1, ...
    'Callback', @RedrawPlot);

hPanel2 = uipanel('Title','Data set 2','FontSize',18,...
    'units','pixels',...
    'Position',[601 121 600 60]);
hPopupData2 = uicontrol('Style', 'popup',...
    'Parent', hPanel2, ...
    'String', ['None', Label],...
    'Position',[5 -41 585 70], ...
    'FontSize', 16, ...
    'Value', 4, ...
    'Callback', @RedrawPlot);

hPanel3 = uipanel('Title','Data set 3','FontSize',18,...
    'units','pixels',...
    'Position',[1201 121 600 60]);
hPopupData3 = uicontrol('Style', 'popup',...
    'Parent', hPanel3, ...
    'String', ['None', Label],...
    'Position',[5 -41 585 70], ...
    'FontSize', 16, ...
    'Value', 6, ...
    'Callback', @RedrawPlot);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Button Box for gene selection
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

hButtonsGeneSelection = uibuttongroup(...
    'Visible', 'on', ...
    'Units', 'Pixels', ...
    'Position',[1 81 1798 40],...
    'SelectionChangeFcn',@RedrawPlot);

% Buttons for gene selection
uicontrol('Style','text',...
    'Position',[10 5 200 30],...
    'parent',hButtonsGeneSelection,...
    'String','Gene selection:', ...
    'HorizontalAlignment', 'Left', ...
    'FontSize', 20);
GeneRB1 = uicontrol('Style','Radio','String','All genes',...
    'pos',[250 5 250 30],...
    'parent',hButtonsGeneSelection,...
    'HandleVisibility','off', ...
    'FontSize', 16);
GeneRB2 = uicontrol('Style','Radio','String','Divergent genes only',...
    'pos',[450 5 250 30],...
    'parent',hButtonsGeneSelection,...
    'HandleVisibility','off', ...
    'FontSize', 16);
GeneRB3 = uicontrol('Style','Radio','String','Tandem genes only',...
    'pos',[750 5 250 30],...
    'parent',hButtonsGeneSelection,...
    'HandleVisibility','off', ...
    'FontSize', 16);

set(hButtonsGeneSelection, 'SelectedObject', GeneRB1);

%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Button Box: Transcript filter
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

hButtonsTranscriptFilter = uibuttongroup(...
    'Visible', 'on', ...
    'Units', 'Pixels', ...
    'Position',[1 41 500 40],...
    'SelectionChangeFcn',@RedrawPlot);

% Buttons for Transcript filter
uicontrol('Style','text',...
    'Position',[10 5 390 30],...
    'parent',hButtonsTranscriptFilter,...
    'String','Transcript filter:', ...
    'HorizontalAlignment', 'Left', ...
    'FontSize', 20);
TFRB1 = uicontrol('Style','Radio','String','on',...
    'pos',[450 5 50 30],...
    'parent',hButtonsTranscriptFilter,...
    'HandleVisibility','off', ...
    'FontSize', 16);
TFRB2 = uicontrol('Style','Radio','String','off',...
    'pos',[520 5 50 30],...
    'parent',hButtonsTranscriptFilter,...
    'HandleVisibility','off', ...
    'FontSize', 16);
set(hButtonsTranscriptFilter, 'SelectedObject', TFRB1);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Button Box: No. of upstream nucs.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

hButtonsNoOfUpstreamNucs = uibuttongroup(...
    'Visible', 'on', ...
    'Units', 'Pixels', ...
    'Position',[501 41 1298 40],...
    'SelectionChangeFcn',@RedrawPlot);

% Buttons for No. of upstream nucs.
uicontrol('Style','text',...
    'Position',[10 5 450 30],...
    'parent',hButtonsNoOfUpstreamNucs,...
    'String','Number of upstream peaks:', ...
    'HorizontalAlignment', 'Left', ...
    'FontSize', 20);
% UpRB1 = uicontrol('Style','Radio','String','1',...
%                 'pos',[400 5 50 30],...
%                 'parent',hButtonsNoOfUpstreamNucs,...
%                 'HandleVisibility','off', ...
%                 'FontSize', 16);
UpRB2 = uicontrol('Style','Radio','String','2',...
    'pos',[470 5 50 30],...
    'parent',hButtonsNoOfUpstreamNucs,...
    'HandleVisibility','off', ...
    'FontSize', 16);
UpRB3 = uicontrol('Style','Radio','String','3',...
    'pos',[520 5 50 30],...
    'parent',hButtonsNoOfUpstreamNucs,...
    'HandleVisibility','off', ...
    'FontSize', 16);
UpRB4 = uicontrol('Style','Radio','String','4',...
    'pos',[570 5 50 30],...
    'parent',hButtonsNoOfUpstreamNucs,...
    'HandleVisibility','off', ...
    'FontSize', 16);
UpRB5 = uicontrol('Style','Radio','String','5',...
    'pos',[620 5 50 30],...
    'parent',hButtonsNoOfUpstreamNucs,...
    'HandleVisibility','off', ...
    'FontSize', 16);
set(hButtonsNoOfUpstreamNucs, 'SelectedObject', UpRB4);

%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Button Box: Dephasing elements filter
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

hButtonsDephasingElemFilter = uibuttongroup(...
    'Visible', 'on', ...
    'Units', 'Pixels', ...
    'Position',[1 1 500 40],...
    'SelectionChangeFcn',@RedrawPlot);

% Buttons for Dephasing elements filter
uicontrol('Style','text',...
    'Position',[10 5 390 30],...
    'parent',hButtonsDephasingElemFilter,...
    'String','Dephasing elements filter:', ...
    'HorizontalAlignment', 'Left', ...
    'FontSize', 20);
DFRB1 = uicontrol('Style','Radio','String','on',...
    'pos',[450 5 50 30],...
    'parent',hButtonsDephasingElemFilter,...
    'HandleVisibility','off', ...
    'FontSize', 16);
DFRB2 = uicontrol('Style','Radio','String','off',...
    'pos',[520 5 50 30],...
    'parent',hButtonsDephasingElemFilter,...
    'HandleVisibility','off', ...
    'FontSize', 16);

set(hButtonsDephasingElemFilter, 'SelectedObject', DFRB1);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Button Box: No. of downstream nucs.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

hButtonsNoOfDownstreamNucs = uibuttongroup(...
    'Visible', 'on', ...
    'Units', 'Pixels', ...
    'Position',[501 1 1298 40],...
    'SelectionChangeFcn',@RedrawPlot);

% Buttons for No. of downstream nucs.
uicontrol('Style','text',...
    'Position',[10 5 450 30],...
    'parent',hButtonsNoOfDownstreamNucs,...
    'String','Number of downstream peaks:', ...
    'HorizontalAlignment', 'Left', ...
    'FontSize', 20);
% DownRB1 = uicontrol('Style','Radio','String','1',...
%                 'pos',[400 5 50 30],...
%                 'parent',hButtonsNoOfDownstreamNucs,...
%                 'HandleVisibility','off', ...
%                 'FontSize', 16);
DownRB2 = uicontrol('Style','Radio','String','2',...
    'pos',[470 5 50 30],...
    'parent',hButtonsNoOfDownstreamNucs,...
    'HandleVisibility','off', ...
    'FontSize', 16);
DownRB3 = uicontrol('Style','Radio','String','3',...
    'pos',[520 5 50 30],...
    'parent',hButtonsNoOfDownstreamNucs,...
    'HandleVisibility','off', ...
    'FontSize', 16);
DownRB4 = uicontrol('Style','Radio','String','4',...
    'pos',[570 5 50 30],...
    'parent',hButtonsNoOfDownstreamNucs,...
    'HandleVisibility','off', ...
    'FontSize', 16);
DownRB5 = uicontrol('Style','Radio','String','5',...
    'pos',[620 5 50 30],...
    'parent',hButtonsNoOfDownstreamNucs,...
    'HandleVisibility','off', ...
    'FontSize', 16);
DownRB6 = uicontrol('Style','Radio','String','6',...
    'pos',[670 5 50 30],...
    'parent',hButtonsNoOfDownstreamNucs,...
    'HandleVisibility','off', ...
    'FontSize', 16);
DownRB7 = uicontrol('Style','Radio','String','7',...
    'pos',[720 5 50 30],...
    'parent',hButtonsNoOfDownstreamNucs,...
    'HandleVisibility','off', ...
    'FontSize', 16);
DownRB8 = uicontrol('Style','Radio','String','8',...
    'pos',[770 5 50 30],...
    'parent',hButtonsNoOfDownstreamNucs,...
    'HandleVisibility','off', ...
    'FontSize', 16);
DownRB9 = uicontrol('Style','Radio','String','9',...
    'pos',[820 5 50 30],...
    'parent',hButtonsNoOfDownstreamNucs,...
    'HandleVisibility','off', ...
    'FontSize', 16);
DownRB10 = uicontrol('Style','Radio','String','10',...
    'pos',[870 5 50 30],...
    'parent',hButtonsNoOfDownstreamNucs,...
    'HandleVisibility','off', ...
    'FontSize', 16);
set(hButtonsNoOfDownstreamNucs, 'SelectedObject', DownRB5);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Axes
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

hPlotAxes1 = axes(...    % Axes for plotting the selected plot
    'Parent', MainFigHandle, ...
    'Units', 'Pixels', ...
    'HandleVisibility', 'on', ...
    'OuterPosition', [1,202,898,587]);
hPlotAxes2 = axes(...    % Axes for plotting the selected plot
    'Parent', MainFigHandle, ...
    'Units', 'Pixels', ...
    'HandleVisibility', 'on', ...
    'OuterPosition', [901,202,898,587]);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Start main program
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%
% Initialize the GUI.
% Change units to normalized so components resize automatically.
set([MainFigHandle, hPlotAxes1, hPlotAxes2, hPanel1, hPanel2, hPanel3, ...
    hPopupData1, hPopupData2, hPopupData3, hButtonsGeneSelection, ...
    hButtonsTranscriptFilter, hButtonsDephasingElemFilter, ...
    hButtonsNoOfUpstreamNucs, hButtonsNoOfDownstreamNucs], 'Units', 'normalized');

% Assign the GUI a name to appear in the window title.
set(MainFigHandle, 'Name', 'Regression analysis GUI', 'NumberTitle', 'off')

% Move the GUI to the center of the screen.
% movegui(MainFigHandle, 'center')

% Make the GUI visible and track the keys that are pressed
set(MainFigHandle, 'Visible', 'on');
maxfig(MainFigHandle, 1);

beforeRef = 1000;
set(MainFigHandle, 'WindowKeyPressFcn', @buttonPressed_Callback);
titleString = [];

RedrawPlot;

    function RedrawPlot(~,~)
        dataset1Idx = get(hPopupData1, 'Value');
        titleString = Label{dataset1Idx};
        
        dataset2Idx = get(hPopupData2, 'Value') - 1;
        if dataset2Idx == 0
            set(hPopupData3, 'Value', 1);
        end
        dataset3Idx = get(hPopupData3, 'Value') - 1;
        if dataset2Idx >= 1
            titleString = [titleString, ' - ', Label{dataset2Idx}];
            if dataset3Idx >= 1
                titleString = [titleString, ' - ', Label{dataset3Idx}];
            end
        end
        
        
        SelectedGenes = get(hButtonsGeneSelection,'SelectedObject');
        switch SelectedGenes
            case GeneRB1
                geneIdx = 1;
                titleString = [titleString, ' All genes'];
            case GeneRB2
                geneIdx = 2;
                titleString = [titleString, ' Divergent genes'];
            case GeneRB3
                geneIdx = 3;
                titleString = [titleString, ' Tandem genes'];
        end
        
        TFSelection = get(hButtonsTranscriptFilter,'SelectedObject');
        DFSelection = get(hButtonsDephasingElemFilter,'SelectedObject');
        switch TFSelection
            case TFRB2
                switch DFSelection
                    case DFRB2 % off, off
                        profileIdx = 1;
                        titleString = [titleString, ' TF off DF off'];
                    case DFRB1 % off, on
                        profileIdx = 3;
                        titleString = [titleString, ' TF off DF on'];
                end
            case TFRB1
                switch DFSelection
                    case DFRB2 % on, off
                        profileIdx = 2;
                        titleString = [titleString, ' TF on DF off'];
                    case DFRB1 % on, on
                        profileIdx = 4;
                        titleString = [titleString, ' TF on DF on'];
                end
        end
        
        UpstreamSelection = get(hButtonsNoOfUpstreamNucs,'SelectedObject');
        switch UpstreamSelection
            case UpRB2
                noUpstreamPeaks = 2;
            case UpRB3
                noUpstreamPeaks = 3;
            case UpRB4
                noUpstreamPeaks = 4;
            case UpRB5
                noUpstreamPeaks = 5;
        end
        
        DownstreamSelection = get(hButtonsNoOfDownstreamNucs,'SelectedObject');
        switch DownstreamSelection
            case DownRB2
                noDownstreamPeaks = 2;
            case DownRB3
                noDownstreamPeaks = 3;
            case DownRB4
                noDownstreamPeaks = 4;
            case DownRB5
                noDownstreamPeaks = 5;
            case DownRB6
                noDownstreamPeaks = 6;
            case DownRB7
                noDownstreamPeaks = 7;
            case DownRB8
                noDownstreamPeaks = 8;
            case DownRB9
                noDownstreamPeaks = 9;
            case DownRB10
                noDownstreamPeaks = 10;
        end
        
        PlotRegressionFigure(dataset1Idx, dataset2Idx, dataset3Idx, geneIdx, profileIdx, noUpstreamPeaks, noDownstreamPeaks)
    end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Plot Functions
    function PlotRegressionFigure(dataset1Idx, dataset2Idx, dataset3Idx, geneIdx, profileIdx, noUpstreamPeaks, noDownstreamPeaks)
        
        Profile1 = AllProfiles{dataset1Idx}{geneIdx, profileIdx};
        
        %% Detect peaks
        [maxpks, maxlocs] = findpeaks(Profile1, 'minpeakdistance', 130);
        
        UpstreamPeaks = maxlocs < beforeRef - 150;
        % Keep only the first noUpstreamPeaks upstream peaks
        UpstreamPeaks = find(UpstreamPeaks == 1, noUpstreamPeaks, 'last');
        UpstreamPeakLocations = maxlocs(UpstreamPeaks) - (1 + beforeRef);
        UpstreamPeakHeights =  maxpks(UpstreamPeaks);
        
        DownstreamPeaks = maxlocs > beforeRef - 50;
        % Keep only the first noDownstreamPeaks downstream peaks
        DownstreamPeaks = find(DownstreamPeaks == 1, noDownstreamPeaks, 'first');
        DowstreamPeakLocations = maxlocs(DownstreamPeaks) - (1 + beforeRef);
        DownsreamPeakHeights =  maxpks(DownstreamPeaks);
        
        %% Phasing
        axes(hPlotAxes1)
        cla
        hold all
        line1 = plot(Profile1, 'Color', [0 0.5 1], 'LineWidth', 3);
        plot((1 + beforeRef)+UpstreamPeakLocations, UpstreamPeakHeights, 'o', ...
            'markersize', 14, ...
            'linewidth', 3, ...
            'markeredgecolor', [0 0 1], 'markerfacecolor', [1 1 1]);
        plot((1 + beforeRef)+DowstreamPeakLocations, DownsreamPeakHeights, 'o', ...
            'markersize', 14, ...
            'markeredgecolor', [0 0 1], 'markerfacecolor', [0 0 1]);
        
        xlim(1+beforeRef+[-1000, 1500])
        ylim([0 3.5])
        
        set(gca, 'FontSize', 20, 'XTick', 1+beforeRef+[-2500:500:2500], ...
            'YTick', [0:0.5:3.5], ...
            'XTickLabel', {'-2500', '-2000', '-1500', '-1000', '-500', '0', '500', '1000', '1500', '2000', '2500'}, ...
            'XGrid', 'on', 'GridLineStyle', '--')
        xlabel('Position relative to +1 nuc. (bp)', 'FontSize', 24)
        ylabel('Nucleosome dyad density', 'FontSize', 24)
        
        legend(line1, {Label{dataset1Idx}}, 'location', 'NE', 'FontSize', 20, 'interpreter', 'none')
        
        hold off
        %% Regression
        x_up = - noUpstreamPeaks : -1; x_up = x_up';
        x_down = 1 : noDownstreamPeaks; x_down = x_down';
        
        % FigHandle = figure('Position', [50, 50, 750, 400]);
        axes(hPlotAxes2)
        cla
        hold all
        p_up = polyfit(x_up, UpstreamPeakLocations,1);
        f = polyval(p_up, [x_up; 0]);
        plot([x_up; 0], f, 'Color', [0 0.5 1], 'linewidth', 3);
        plot(x_up, UpstreamPeakLocations, 'o', ...
            'markersize', 14, ...
            'linewidth', 3, ...
            'markeredgecolor', [0 0 1], 'markerfacecolor', [1 1 1]);
        
        p_down = polyfit(x_down, DowstreamPeakLocations, 1);
        f = polyval(p_down, [0;x_down]);
        plot([0;x_down], f, 'Color', [0 0.5 1], 'linewidth', 3);
        plot(x_down, DowstreamPeakLocations, 'ko', ...
            'markersize', 14, ...
            'markeredgecolor', [0 0 1], 'markerfacecolor', [0 0 1]);
        
        yL = [UpstreamPeakLocations(1)-100, DowstreamPeakLocations(end)+100];
        ylim(yL)
        text(((-noUpstreamPeaks-1.5) + (noDownstreamPeaks+0.5))/2, yL(1)+0.25*diff(yL), sprintf('y(x) = %0.1f x - %0.1f', p_up(1), abs(p_up(2))), ...
            'FontSize', 20, 'Color', [0 0.5 1], ...
            'HorizontalAlignment', 'left', ...
            'VerticalAlignment', 'middle')
        text(((-noUpstreamPeaks+0.5) + (noDownstreamPeaks+0.5))/2, yL(1)+0.95*diff(yL), sprintf('y(x) = %0.1f x - %0.1f', p_down(1), abs(p_down(2))), ...
            'FontSize', 20, 'Color', [0 0.5 1], ...
            'HorizontalAlignment', 'right', ...
            'VerticalAlignment', 'middle')
        
        set(gca, 'XTick', -5:10)
        set(gca, 'XTickLabel', {'-5', '-4', '-3', '-2', '-1', '', '+1', '+2', '+3', '+4', '+5', '+6', '+7', '+8', '+9', '+10'});
        set(gca, 'YTick', -1000:500:2000);
        set(gca, 'YTickLabel', {'-1000', '-500', '0', '500', '1000', '1500', '2000'});
        set(gca, 'FontSize', 20);
        xlim([-noUpstreamPeaks-0.5 noDownstreamPeaks+0.5])
        
        xlabel('Nucleosome', 'FontSize', 24);
        ylabel('Peak position relative to +1 nuc. (bp)', 'FontSize', 24);
        
        hold off
        
        %%%%%%%%%%%%%
        % Profile 2 %
        %%%%%%%%%%%%%
        if dataset2Idx >= 1
            Profile2 = AllProfiles{dataset2Idx}{geneIdx, profileIdx};
            
            %% Detect peaks
            [maxpks, maxlocs] = findpeaks(Profile2, 'minpeakdistance', 130);
            
            UpstreamPeaks = maxlocs < beforeRef - 150;
            % Keep only the first noUpstreamPeaks upstream peaks
            UpstreamPeaks = find(UpstreamPeaks == 1, noUpstreamPeaks, 'last');
            UpstreamPeakLocations = maxlocs(UpstreamPeaks) - (1 + beforeRef);
            UpstreamPeakHeights =  maxpks(UpstreamPeaks);
            
            DownstreamPeaks = maxlocs > beforeRef - 50;
            % Keep only the first noDownstreamPeaks downstream peaks
            DownstreamPeaks = find(DownstreamPeaks == 1, noDownstreamPeaks, 'first');
            DowstreamPeakLocations = maxlocs(DownstreamPeaks) - (1 + beforeRef);
            DownsreamPeakHeights =  maxpks(DownstreamPeaks);
            
            %% Phasing
            axes(hPlotAxes1)
            hold all
            line2 = plot(Profile2, 'Color', [1 0.4 0], 'LineWidth', 3);
            plot((1 + beforeRef)+UpstreamPeakLocations, UpstreamPeakHeights, 'o', ...
                'markersize', 14, ...
                'linewidth', 3, ...
                'markeredgecolor', [1 0 0], 'markerfacecolor', [1 1 1]);
            plot((1 + beforeRef)+DowstreamPeakLocations, DownsreamPeakHeights, 'o', ...
                'markersize', 14, ...
                'markeredgecolor', [1 0 0], 'markerfacecolor', [1 0 0]);
            
            xlim(1+beforeRef+[-1000, 1500])
            ylim([0 3.5])
            
            set(gca, 'FontSize', 20, 'XTick', 1+beforeRef+[-2500:500:2500], ...
                'YTick', [0:0.5:3.5], ...
                'XTickLabel', {'-2500', '-2000', '-1500', '-1000', '-500', '0', '500', '1000', '1500', '2000', '2500'}, ...
                'XGrid', 'on', 'GridLineStyle', '--')
            xlabel('Position relative to +1 nuc. (bp)', 'FontSize', 24)
            ylabel('Nucleosome dyad density', 'FontSize', 24)
            
            legend([line1, line2], {Label{dataset1Idx}, Label{dataset2Idx}}, 'location', 'NE', 'FontSize', 20, 'interpreter', 'none')
            
            hold off
            %% Regression
            x_up = - noUpstreamPeaks : -1; x_up = x_up';
            x_down = 1 : noDownstreamPeaks; x_down = x_down';
            
            axes(hPlotAxes2)
            hold all
            p_up = polyfit(x_up, UpstreamPeakLocations,1);
            f = polyval(p_up, [x_up; 0]);
            plot([x_up; 0], f, 'Color', [1 0.4 0], 'linewidth', 3);
            plot(x_up, UpstreamPeakLocations, 'o', ...
                'markersize', 14, ...
                'linewidth', 3, ...
                'markeredgecolor', [1 0 0], 'markerfacecolor', [1 1 1]);
            
            p_down = polyfit(x_down, DowstreamPeakLocations, 1);
            f = polyval(p_down, [0;x_down]);
            plot([0;x_down], f, 'Color', [1 0.4 0], 'linewidth', 3);
            plot(x_down, DowstreamPeakLocations, 'ko', ...
                'markersize', 14, ...
                'markeredgecolor', [1 0 0], 'markerfacecolor', [1 0 0]);
            
%             yL = [UpstreamPeakLocations(1)-100, DowstreamPeakLocations(end)+100];
%             ylim(yL)
            text(((-noUpstreamPeaks-1.5) + (noDownstreamPeaks+0.5))/2, yL(1)+0.15*diff(yL), sprintf('y(x) = %0.1f x - %0.1f', p_up(1), abs(p_up(2))), ...
                'FontSize', 20, 'Color', [1 0.4 0], ...
                'HorizontalAlignment', 'left', ...
                'VerticalAlignment', 'middle')
            text(((-noUpstreamPeaks+0.5) + (noDownstreamPeaks+0.5))/2, yL(1)+0.85*diff(yL), sprintf('y(x) = %0.1f x - %0.1f', p_down(1), abs(p_down(2))), ...
                'FontSize', 20, 'Color', [1 0.4 0], ...
                'HorizontalAlignment', 'right', ...
                'VerticalAlignment', 'middle')
            
            set(gca, 'XTick', -5:10)
            set(gca, 'XTickLabel', {'-5', '-4', '-3', '-2', '-1', '', '+1', '+2', '+3', '+4', '+5', '+6', '+7', '+8', '+9', '+10'});
            set(gca, 'YTick', -1000:500:2000);
            set(gca, 'YTickLabel', {'-1000', '-500', '0', '500', '1000', '1500', '2000'});
            set(gca, 'FontSize', 20);
            xlim([-noUpstreamPeaks-0.5 noDownstreamPeaks+0.5])
            
            xlabel('Nucleosome', 'FontSize', 24);
            ylabel('Peak position relative to +1 nuc. (bp)', 'FontSize', 24);
            
            hold off
            
            %%%%%%%%%%%%%
            % Profile 3 %
            %%%%%%%%%%%%%
            if dataset3Idx >= 1
                Profile3 = AllProfiles{dataset3Idx}{geneIdx, profileIdx};
                
                %% Detect peaks
                [maxpks, maxlocs] = findpeaks(Profile3, 'minpeakdistance', 130);
                
                UpstreamPeaks = maxlocs < beforeRef - 150;
                % Keep only the first noUpstreamPeaks upstream peaks
                UpstreamPeaks = find(UpstreamPeaks == 1, noUpstreamPeaks, 'last');
                UpstreamPeakLocations = maxlocs(UpstreamPeaks) - (1 + beforeRef);
                UpstreamPeakHeights =  maxpks(UpstreamPeaks);
                
                DownstreamPeaks = maxlocs > beforeRef - 50;
                % Keep only the first noDownstreamPeaks downstream peaks
                DownstreamPeaks = find(DownstreamPeaks == 1, noDownstreamPeaks, 'first');
                DowstreamPeakLocations = maxlocs(DownstreamPeaks) - (1 + beforeRef);
                DownsreamPeakHeights =  maxpks(DownstreamPeaks);
                
                %% Phasing
                axes(hPlotAxes1)
                hold all
                line3 = plot(Profile3, 'Color', [0 0.8 0], 'LineWidth', 3);
                plot((1 + beforeRef)+UpstreamPeakLocations, UpstreamPeakHeights, 'o', ...
                    'markersize', 14, ...
                    'linewidth', 3, ...
                    'markeredgecolor', [0 0.6 0], 'markerfacecolor', [1 1 1]);
                plot((1 + beforeRef)+DowstreamPeakLocations, DownsreamPeakHeights, 'o', ...
                    'markersize', 14, ...
                    'markeredgecolor', [0 0.6 0], 'markerfacecolor', [0 0.6 0]);
                
                xlim(1+beforeRef+[-1000, 1500])
                ylim([0 3.5])
                
                set(gca, 'FontSize', 20, 'XTick', 1+beforeRef+[-2500:500:2500], ...
                    'YTick', [0:0.5:3.5], ...
                    'XTickLabel', {'-2500', '-2000', '-1500', '-1000', '-500', '0', '500', '1000', '1500', '2000', '2500'}, ...
                    'XGrid', 'on', 'GridLineStyle', '--')
                xlabel('Position relative to +1 nuc. (bp)', 'FontSize', 24)
                ylabel('Nucleosome dyad density', 'FontSize', 24)
                
                legend([line1, line2, line3], {Label{dataset1Idx}, Label{dataset2Idx}, Label{dataset3Idx}}, 'location', 'NE', 'FontSize', 20, 'interpreter', 'none')
                
                hold off
                %% Regression
                x_up = - noUpstreamPeaks : -1; x_up = x_up';
                x_down = 1 : noDownstreamPeaks; x_down = x_down';
                
                axes(hPlotAxes2)
                hold all
                p_up = polyfit(x_up, UpstreamPeakLocations,1);
                f = polyval(p_up, [x_up; 0]);
                plot([x_up; 0], f, 'Color', [0 0.8 0], 'linewidth', 3);
                plot(x_up, UpstreamPeakLocations, 'o', ...
                    'markersize', 14, ...
                    'linewidth', 3, ...
                    'markeredgecolor', [0 0.6 0], 'markerfacecolor', [1 1 1]);
                
                p_down = polyfit(x_down, DowstreamPeakLocations, 1);
                f = polyval(p_down, [0;x_down]);
                plot([0;x_down], f, 'Color', [0 0.8 0], 'linewidth', 3);
                plot(x_down, DowstreamPeakLocations, 'ko', ...
                    'markersize', 14, ...
                    'markeredgecolor', [0 0.6 0], 'markerfacecolor', [0 0.6 0]);
                
%                 yL = [UpstreamPeakLocations(1)-100, DowstreamPeakLocations(end)+100];
%                 ylim(yL)
                text(((-noUpstreamPeaks-1.5) + (noDownstreamPeaks+0.5))/2, yL(1)+0.05*diff(yL), sprintf('y(x) = %0.1f x - %0.1f', p_up(1), abs(p_up(2))), ...
                    'FontSize', 20, 'Color', [0 0.8 0], ...
                    'HorizontalAlignment', 'left', ...
                    'VerticalAlignment', 'middle')
                text(((-noUpstreamPeaks+0.5) + (noDownstreamPeaks+0.5))/2, yL(1)+0.75*diff(yL), sprintf('y(x) = %0.1f x - %0.1f', p_down(1), abs(p_down(2))), ...
                    'FontSize', 20, 'Color', [0 0.8 0], ...
                    'HorizontalAlignment', 'right', ...
                    'VerticalAlignment', 'middle')
                
                set(gca, 'XTick', -5:10)
                set(gca, 'XTickLabel', {'-5', '-4', '-3', '-2', '-1', '', '+1', '+2', '+3', '+4', '+5', '+6', '+7', '+8', '+9', '+10'});
                set(gca, 'YTick', -1000:500:2000);
                set(gca, 'YTickLabel', {'-1000', '-500', '0', '500', '1000', '1500', '2000'});
                set(gca, 'FontSize', 20);
                xlim([-noUpstreamPeaks-0.5 noDownstreamPeaks+0.5])
                
                xlabel('Nucleosome', 'FontSize', 24);
                ylabel('Peak position relative to +1 nuc. (bp)', 'FontSize', 24);
                
                hold off
            end
            
        end
        
        %     [ax, titleHandle] = suplabel(titleString  ,'t');
        %     set(titleHandle, 'FontSize', 24)
    end

    function PrintFigure_Callback(varargin)
        [filename, pathname] = uiputfile2(...
            {'*.eps', 'Encapsulated PostScript (.eps)'}, ...
            'Save Image', strrep(titleString, ' ', '_'));
        if ~isequal(filename, 0)
            set(MainFigHandle, 'PaperPositionMode', 'auto')
            filename = [pathname, filename];
            print(MainFigHandle, '-depsc', '-painters', filename);
        end
    end
end    % The main program ends here!
