package main.java.plot;

import com.compomics.util.experiment.biology.Ion;
import com.compomics.util.experiment.biology.NeutralLoss;
import com.compomics.util.experiment.biology.Peptide;
import com.compomics.util.experiment.biology.ions.PeptideFragmentIon;
import com.compomics.util.experiment.identification.matches.IonMatch;
import com.compomics.util.experiment.identification.matches.ModificationMatch;
import com.compomics.util.experiment.identification.spectrum_annotation.AnnotationSettings;
import com.compomics.util.experiment.identification.spectrum_annotation.SpecificAnnotationSettings;
import com.compomics.util.experiment.identification.spectrum_annotation.SpectrumAnnotator;
import com.compomics.util.experiment.identification.spectrum_annotation.spectrum_annotators.PeptideSpectrumAnnotator;
import com.compomics.util.experiment.identification.spectrum_assumptions.PeptideAssumption;
import com.compomics.util.experiment.massspectrometry.Charge;
import com.compomics.util.experiment.massspectrometry.MSnSpectrum;
import com.compomics.util.experiment.massspectrometry.Peak;
import com.compomics.util.experiment.massspectrometry.SpectrumFactory;
import main.java.OpenModificationSearch.ModificationDB;
import main.java.PSMMatch.HyperscoreMatch;
import main.java.pg.CParameter;
import main.java.pg.PeptideSearchMT;
import main.java.util.Cloger;
import org.apache.commons.cli.*;
import org.apache.commons.lang3.StringUtils;
import uk.ac.ebi.jmzml.xml.io.MzMLUnmarshallerException;

import java.io.*;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;

/**
 * Peptide spectrum annotation
 */
public class PeakAnnotation {


    public static double intensityLimit = 0.03;


    public static void main(String[] args) throws IOException, MzMLUnmarshallerException, ParseException {

        Cloger.init();
        Cloger.logger.info("Start analysis");
        Cloger.logger.debug(StringUtils.join(args," "));

        HyperscoreMatch.generateFactorialValues(60);

        Options options = new Options();


        options.addOption("ms", true, "Spectrum file used for identification, mgf format");

        options.addOption("i", true, "Peptide identification txt format file");
        options.addOption("o", true, "Output annotation file");
         options.addOption("itol", true, "Fragment ion m/z tolerance, default is 0.6da");

        options.addOption("h", false, "Help");


        CommandLineParser parser = new PosixParser();
        CommandLine cmd = parser.parse(options, args);
        if (cmd.hasOption("h") || cmd.hasOption("help") || args.length == 0) {
            HelpFormatter f = new HelpFormatter();

            f.printHelp("Options", options);
            System.exit(0);
        }

        String psm_file = cmd.getOptionValue("i");
        String mgf = cmd.getOptionValue("ms");
        String outfile = cmd.getOptionValue("o");
        double itol = 0.6;
        if(cmd.hasOption("itol")){
            itol = Double.valueOf(cmd.getOptionValue("itol"));
        }
        CParameter.itol = itol;
        plotPSM(psm_file,mgf,outfile);
    }

    public static String getPeakAnnotation(Peptide objPeptide, MSnSpectrum spectrum, boolean lossWaterNH3){

        PeptideSpectrumAnnotator peptideSpectrumAnnotator = new PeptideSpectrumAnnotator();

        String spectrumKey = spectrum.getSpectrumKey();
        Charge objCharge = new Charge(Charge.PLUS, spectrum.getPrecursor().getPossibleCharges().get(0).value);
        PeptideAssumption peptideAssumption = new PeptideAssumption(objPeptide, 1, 1, objCharge, 1.0);
        SpecificAnnotationSettings specificAnnotationPreferences = new SpecificAnnotationSettings(spectrumKey, peptideAssumption);

        ArrayList<Integer> charges = new ArrayList<>(4);
        int precursorCharge = peptideAssumption.getIdentificationCharge().value;
        if (precursorCharge == 1) {
            charges.add(precursorCharge);
        } else {
            //for (int c = 1; c < precursorCharge; c++) {
            for (int c = 1; c <= precursorCharge; c++) {
                charges.add(c);
            }
        }
        specificAnnotationPreferences.setSelectedCharges(charges);

        specificAnnotationPreferences.addIonType(Ion.IonType.PEPTIDE_FRAGMENT_ION,PeptideFragmentIon.B_ION);
        specificAnnotationPreferences.addIonType(Ion.IonType.PEPTIDE_FRAGMENT_ION,PeptideFragmentIon.Y_ION);
        specificAnnotationPreferences.setFragmentIonAccuracy(CParameter.itol);
        specificAnnotationPreferences.setFragmentIonPpm(false);
        //ע⣬specificAnnotationPreferencesprivate boolean neutralLossesAuto = true;
        specificAnnotationPreferences.setNeutralLossesAuto(false);
        specificAnnotationPreferences.clearNeutralLosses();

        if(lossWaterNH3) {
            specificAnnotationPreferences.addNeutralLoss(NeutralLoss.H2O);
            specificAnnotationPreferences.addNeutralLoss(NeutralLoss.NH3);
        }

        AnnotationSettings annotationSettings = new AnnotationSettings();
        annotationSettings.setTiesResolution(SpectrumAnnotator.TiesResolution.mostIntense);
        annotationSettings.setFragmentIonAccuracy(CParameter.itol);
        annotationSettings.setFragmentIonPpm(false);
        annotationSettings.setIntensityLimit(intensityLimit);
        annotationSettings.setNeutralLossesSequenceAuto(false);

        if(lossWaterNH3) {
            annotationSettings.addNeutralLoss(NeutralLoss.H2O);
            annotationSettings.addNeutralLoss(NeutralLoss.NH3);
        }

        ArrayList<IonMatch> matches = peptideSpectrumAnnotator.getSpectrumAnnotation(annotationSettings, specificAnnotationPreferences, spectrum, objPeptide);


        if (matches.isEmpty() && PeptideSearchMT.debug) {
            System.err.println("No ions matched!");
            return (null);
        }

        StringBuilder mz_all = new StringBuilder();
        StringBuilder int_all = new StringBuilder();
        StringBuilder mz_match = new StringBuilder();
        StringBuilder int_match = new StringBuilder();
        StringBuilder m_label = new StringBuilder();

        HashMap<Double,Double> validPeakMz = new HashMap<>();

        int i =0;
        for(Peak jPeak : spectrum.getPeakList()){
            validPeakMz.put(jPeak.getMz(),jPeak.getIntensity());
            if(i==0){
                mz_all.append(jPeak.getMz());
                int_all.append(jPeak.getIntensity());
            }else{
                mz_all.append(";"+jPeak.getMz());
                int_all.append(";"+jPeak.getIntensity());
            }
            i++;
        }
        Iterator<IonMatch> pListIterator = matches.iterator();
        HashSet<Double> ionHitSet = new HashSet<>();
        while (pListIterator.hasNext()) {
            IonMatch ionMatch = pListIterator.next();
            if (!validPeakMz.containsKey(ionMatch.peak.mz) || ionHitSet.contains(ionMatch.peak.mz) || ionMatch.ion.getNeutralLosses().length>=2) {
                pListIterator.remove();

            }else{
                ionHitSet.add(ionMatch.peak.mz);
            }
        }



        for(i=0;i<matches.size();i++){
            IonMatch ionMatch = matches.get(i);
            String label = ionMatch.getPeakAnnotation();

            if(i==0){
                mz_match.append(ionMatch.peak.mz);
                int_match.append(ionMatch.peak.intensity);
                m_label.append(label);
            }else{
                mz_match.append(";"+ionMatch.peak.mz);
                int_match.append(";"+ionMatch.peak.intensity);
                m_label.append(";"+label);
            }

        }

        //IonFactory.getInstance().getDefaultNeutralLosses().clear();

        StringBuilder outline = new StringBuilder();
        outline.append(objPeptide.getSequence()).append("\t");
        outline.append(spectrum.getSpectrumTitle()).append("\t");
        outline.append(objPeptide.getMass()).append("\t");
        outline.append(spectrum.getPrecursor().getMz()).append("\t");
        outline.append(spectrum.getPrecursor().getPossibleCharges().get(0).value).append("\t");
        outline.append(objPeptide.getSequenceWithLowerCasePtms()).append("\t");
        outline.append(m_label).append("\t");
        outline.append(mz_match).append("\t");
        outline.append(int_match).append("\t");
        outline.append(mz_all).append("\t");
        outline.append(int_all);

        return(outline.toString());
    }


    public static String getPeakAnnotationHeader(){
        StringBuilder outline = new StringBuilder();
        outline.append("peptide").append("\t");
        outline.append("Query").append("\t");
        outline.append("calc_mr").append("\t");
        outline.append("observed_mz").append("\t");
        outline.append("charge").append("\t");
        outline.append("pepSeq").append("\t");
        outline.append("m_label").append("\t");
        outline.append("m_mz").append("\t");
        outline.append("m_intensity").append("\t");
        outline.append("mz").append("\t");
        outline.append("intensity");

        return(outline.toString());
    }


    public static void plot(String input, String figfile, double limit) throws InterruptedException, IOException {

        InputStream is = PeakAnnotation.class.getResourceAsStream("/main/java/plot/plot_spectrum.r");
        BufferedReader br = new BufferedReader(new InputStreamReader(is));

        String prog_dir = PeakAnnotation.class.getProtectionDomain().getCodeSource().getLocation().getFile();
        File prog_dir_f = new File(prog_dir);
        String sfile = prog_dir_f.getParent();
        sfile = sfile.replaceAll("\\\\", "/");

        String rcode = "arg=c('" + input + "','" + figfile + "','" + limit +"')\n";
        String line = "";
        while ((line = br.readLine()) != null) {
            rcode = rcode + "\n" + line;
        }
        br.close();


        try {

            String rbin= "R";

            Process p = null;
            if(System.getProperty("os.name").equals("Linux")){
                rbin= ""+rbin+"";
                rbin = rbin + " --vanilla --slave";
                p = Runtime.getRuntime().exec(rbin);
            }else if (System.getProperty("os.name").startsWith("Windows")) {
                rbin= "\""+rbin+"\"";
                rbin = rbin + " --vanilla --slave";
                rbin = "cmd /c "+rbin;
                System.out.println(rbin);
                p = Runtime.getRuntime().exec(rbin);
            }else{
                System.out.println("Don't support current System!");
                System.exit(0);
            }

            BufferedReader reader = new BufferedReader(new InputStreamReader(
                    p.getInputStream()));
            BufferedWriter writer = new BufferedWriter(new OutputStreamWriter(
                    p.getOutputStream()));

            BufferedInputStream errin = new BufferedInputStream(p.getErrorStream());
            BufferedReader errbr = new BufferedReader(new InputStreamReader(errin));

            writer.write(rcode + "\n");
            writer.flush();
            writer.close();
            String logline;
            p.waitFor();

            while ((logline = reader.readLine()) != null) {
                System.out.println(logline);
            }


            while ((logline = errbr.readLine()) != null) {
                System.out.println(logline);
            }


        } catch (IOException e) {
            e.printStackTrace();
        }

    }


    public static void plotPSM(String psm_file,String mgf,String outfile) throws IOException, MzMLUnmarshallerException {

        ModificationDB.importPTMsFromUnimod();

        File mgfFile = new File(mgf);
        SpectrumFactory spectrumFactory = SpectrumFactory.getInstance();
        spectrumFactory.addSpectra(mgfFile, null);


        BufferedWriter bWriter = new BufferedWriter(new FileWriter(new File(outfile)));
        bWriter.write(getPeakAnnotationHeader()+"\n");

        BufferedReader bReader = new BufferedReader(new FileReader(new File(psm_file)));

        String headline = bReader.readLine().trim();
        String hd[] = headline.split("\t");
        HashMap<String,Integer> headMap = new HashMap<>();
        for(int i=0;i<hd.length;i++){
            headMap.put(hd[i],i);
        }
        String line;
        while((line = bReader.readLine())!=null){
            String d[] = line.trim().split("\t");
            String peptide = d[headMap.get("peptide")];
            String mod = d[headMap.get("modification")];
            String spectrum_title = d[headMap.get("spectrum_title")];
            MSnSpectrum spectrum = (MSnSpectrum) spectrumFactory.getSpectrum(mgfFile.getName(),spectrum_title);

            ArrayList<ModificationMatch> modificationMatches = new ArrayList<>();
            System.out.print(peptide+"\t"+spectrum_title+"\t");
            if(!mod.equalsIgnoreCase("-")){
                String mods[] = mod.split(";");
                for(String modification : mods){
                    // CLIP_TRAQ_4 of K@13[244.1015]
                    String name = modification.replaceAll("@.*$","");
                    String pos = modification.replaceAll(".*@(\\d+).*$","$1");
                    int modpos = Integer.valueOf(pos);
                    System.out.print(name+"@"+modpos+",");
                    ModificationMatch modificationMatch = new ModificationMatch(name,true,modpos);
                    modificationMatches.add(modificationMatch);
                }
            }
            System.out.println();
            Peptide ePeptide = new Peptide(peptide,modificationMatches);
            String psmMatch_tmp = PeakAnnotation.getPeakAnnotation(ePeptide, spectrum, true);
            bWriter.write(psmMatch_tmp+"\n");

        }

        bReader.close();
        bWriter.close();

    }
}
