package main.java.pg;

import com.compomics.util.experiment.biology.*;
import com.compomics.util.experiment.identification.matches.ModificationMatch;
import com.compomics.util.experiment.massspectrometry.MSnSpectrum;
import com.compomics.util.experiment.massspectrometry.Peak;
import main.java.OpenModificationSearch.ModificationDB;
import main.java.plot.PeakAnnotation;
import main.java.PSMMatch.*;
import main.java.util.Cloger;
import org.apache.commons.cli.*;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import java.io.*;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * This class is the main class for peptide-centric database searching engine
 */
public class PeptideSearchMT {

    public static boolean debug = false;


    /**
     * Whether or not to use GPD method to calculate the permutation test p-value
     */
    public static boolean useGPD = false;

    public static void main(String[] args) throws ParseException, IOException, InterruptedException, SQLException, ClassNotFoundException {

        Cloger.init();
        Cloger.logger.info("Start analysis");
        Cloger.logger.debug(StringUtils.join(args," "));

        HyperscoreMatch.generateFactorialValues(60);

        Options options = new Options();

        options.addOption("ms", true, "Spectrum file used for identification, mgf format");
        options.addOption("pep", true, "Peptide sequence which you want to search");

        // take protein, DNA or VCF as input
        options.addOption("i", true, "Take protein, DNA or VCF as input");
        options.addOption("t",true,"Input type: 1=>protein,2=>DNA,3=>VCF,4=>BED,5=>GTF");
        options.addOption("f",true,"The frame to translate DNA sequence to protein. The right format is like this: \"1,2,3,4,5,6\",\"1,2,3\",\"1\". \"0\" means to keep the longest frame. In default, for each frame only the longest protein is used.");
        options.addOption("anno",true,"Annotation files folder for VCF/BED/GTF");
        options.addOption("tp",false,"Whether or not to perform target protein identification. If you set this parameter, then the input value for -i is a protein ID from the input reference protein database (-db)");
        options.addOption("decoy",false,"In target protein identification mode, try to identity the decoy version of the selected target protein. Default is false.");

        options.addOption("prefix", true, "Output file prefix");
        options.addOption("o", true, "Output dir");
        options.addOption("tol", true, "Precursor ion m/z tolerance, default is 10");
        options.addOption("tolu", true, "The unit of precursor ion m/z tolerance, default is ppm");
        options.addOption("itol", true, "Fragment ion m/z tolerance, default is 0.6da");
        options.addOption("db", true, "Fasta format database file");

        options.addOption("fragmentMethod", true, "1: CID/HCD (default), 2: ETD");
        options.addOption("fixMod",true,"Fixed modification, the format is like : 1,2,3. Default is 6 (Carbamidomethylation(C)[57.02])");
        options.addOption("varMod",true,"Variable modification, the format is like : 1,2,3. Default is 107 (Oxidation(M)[15.99])");
        options.addOption("maxVar",true,"Max number of variable modifications, default is 3");
        options.addOption("aa",false,"Whether or not to consider aa substitution modifications when perform modification filtering. In default, don't consider.");

        options.addOption("printPTM",false,"Print PTMs");

        options.addOption("m",true,"Scoring method: 1=HyperScore (default), 2=MVH");

        options.addOption("e",true,"1:Trypsin (default), 2:Trypsin (no P rule), 3:Arg-C, 4:Arg-C (no P rule), 5:Arg-N, 6:Glu-C, 7:Lys-C");
        options.addOption("c",true,"The max missed cleavages, default is 2");

        // Unrestricted modification peptide identification
        options.addOption("um",false,"Validation with unrestricted modification searching");
        // When perform unrestricted modification searching, how to filter the result.
        // When the value is true, then filtering is score(ptm) >= score(target peptide).
        // When the value is false, then filtering is score(ptm) > score(target peptide). This is the default value.
        // We recommend to set it as true when require very stringent filtering,
        // for example performing missing proteinidentification.
        options.addOption("hc",false,"When perform validation with unrestricted modification searching (UMS), whether or not to use more stringent criterion. TRUE: score(UMS)>=score(targetPSM); FALSE: score(UMS)>score(targetPSM), default");

        //
        options.addOption("n",true,"The number of random peptides, default is 1000");
        options.addOption("cpu",true,"The number of cpus used, default is 1");

        // spectra pre-processing
        options.addOption("minPeaks",true,"Min peaks in spectrum, default is 10");

        // In general, the larger the better for the score
        options.addOption("minScore",true,"Minimum score to consider for peptide searching, default is 5");


        options.addOption("minCharge", true, "The minimum charge to consider if the charge state is not available, default is 2");
        options.addOption("maxCharge", true, "The maximum charge to consider if the charge state is not available, default is 3");

        options.addOption("h", false, "Help");

        CommandLineParser parser = new PosixParser();
        CommandLine cmd = parser.parse(options, args);
        if (cmd.hasOption("h") || cmd.hasOption("help") || args.length == 0) {
            HelpFormatter f = new HelpFormatter();

            System.out.println("java -Xmx2G -jar pepquery.jar");
            System.out.println("Version 1.0.0");
            // System.out.println(version);
            f.printHelp("Options", options);
            System.exit(0);
        }

        // Print modification list
        if(cmd.hasOption("printPTM")){
            PTMFactory ptmFactory = PTMFactory.getInstance();
            ArrayList<String> ptmNames = ptmFactory.getPTMs();
            //System.out.println(ptmNames.size());
            int i=0;
            for(String name : ptmNames){
                i++;
                PTM ptm = ptmFactory.getPTM(name);

                System.out.println(i+"\t"+ptm.getName()+"\t"+ptm.getMass()+"\t"+ptm.getType());
            }
            System.exit(0);
        }

        String msmsfile = null;
        if(cmd.hasOption("ms")){
            msmsfile = cmd.getOptionValue("ms");
        }else{
            System.out.println("Please provide MS/MS file!");
            System.exit(0);
        }

        String db = null;
        if(cmd.hasOption("db")){
            db = cmd.getOptionValue("db");
        }

        // set output directory
        String outdir = "./";
        if(cmd.hasOption("o")){
            outdir = cmd.getOptionValue("o");
            File DIR = new File(outdir);
            if(!DIR.isDirectory()){
                DIR.mkdirs();
            }
        }
        File DIR = new File(outdir);
        outdir = DIR.getAbsolutePath()+"/";

        if(cmd.hasOption("e")){
            // DatabaseInput.enzymeIndex = Integer.valueOf(cmd.getOptionValue("e"));
            CParameter.enzyme = Integer.valueOf(cmd.getOptionValue("e"));
        }
        if(cmd.hasOption("c")){
            int missCleavages = Integer.valueOf(cmd.getOptionValue("c"));
            if(missCleavages<0){
                missCleavages = 0;
            }
            CParameter.maxMissedCleavages = missCleavages;
        }

        if(cmd.hasOption("tol")){
            CParameter.tol = Double.valueOf(cmd.getOptionValue("tol"));
        }

        if(cmd.hasOption("tolu")){
            if(!cmd.getOptionValue("tolu").equalsIgnoreCase("ppm")){
                CParameter.tolu = false;
            }
        }


        if(cmd.hasOption("itol")){
            CParameter.itol = Double.valueOf(cmd.getOptionValue("itol"));
        }

        //JMatch.setMs2tol(itol);

        //int method = 1; // hyperscore
        if(cmd.hasOption("m")){
            CParameter.scoreMethod = Integer.valueOf(cmd.getOptionValue("m"));
        }

        //int nRandomPeptides = 1000;
        if(cmd.hasOption("n")){
            CParameter.nRandomPeptides = Integer.valueOf(cmd.getOptionValue("n"));
        }


        //int cpu = 1;
        if(cmd.hasOption("cpu")){
            CParameter.cpu = Integer.valueOf(cmd.getOptionValue("cpu"));
            if(CParameter.cpu==0){
                CParameter.cpu = Runtime.getRuntime().availableProcessors();
            }
        }else{
            CParameter.cpu = Runtime.getRuntime().availableProcessors();
        }


        ////////////////////////////////////////////////////////////////////////////////////////////////////////////////

        //double minScore = 5;
        if(cmd.hasOption("minScore")){
            CParameter.minScore = Double.valueOf(cmd.getOptionValue("minScore"));
        }

        //int minPeaks = 10;
        if(cmd.hasOption("minPeaks")){
            CParameter.minScore = Integer.valueOf(cmd.getOptionValue("minPeaks"));
        }

        //double limit = 0.03;
        if(cmd.hasOption("limit")){
            CParameter.intensityThreshold = Double.valueOf(cmd.getOptionValue("limit"));
        }

        // int maxVar = 3;
        if(cmd.hasOption("maxVar")){
            CParameter.maxVarMods = Integer.valueOf(cmd.getOptionValue("maxVar"));
        }

        //String fixModOptionValue = "6";
        //String varModOptionValue = "107";

        if(cmd.hasOption("fixMod")){
            CParameter.fixMods = cmd.getOptionValue("fixMod");
        }
        if(cmd.hasOption("varMod")){
            CParameter.varMods = cmd.getOptionValue("varMod");
        }

        // For most of the applications, we don't need to consider these modifications in
        // unrestricted modification-based filtering.
        if(cmd.hasOption("aa")){
            CParameter.addAAsubstitutionMods = true;
        }


        if(cmd.hasOption("hc")){
            CParameter.unrestrictedSearchWithEqualAndBetterScore = true;
        }


        CParameter.print();


        String peptideSequence = null;
        ArrayList<String> pepSeqs = new ArrayList<>();
        if(cmd.hasOption("pep")) {
            // take peptide as input
            peptideSequence = cmd.getOptionValue("pep");
            File pFile = new File(peptideSequence);
            if (pFile.isFile()) {

                HashSet<String> removeDupPep = new HashSet<>();
                BufferedReader pepReader = new BufferedReader(new FileReader(pFile));
                String line;

                while ((line = pepReader.readLine()) != null) {

                    String pep[] = line.split("\t");
                    if (removeDupPep.contains(pep[0].toUpperCase())) {


                    } else {
                        if(!pep[0].isEmpty()) {
                            pepSeqs.add(pep[0].toUpperCase());
                            removeDupPep.add(pep[0].toUpperCase());
                        }
                    }
                }

                pepReader.close();
            } else {
                pepSeqs.add(peptideSequence.toUpperCase());
            }

        }else if(cmd.hasOption("i")){
            // take protein, DNA or VCF as input
            String inputSeq = cmd.getOptionValue("i");
            InputProcessor inputProcessor = new InputProcessor();

            if(cmd.hasOption("tp")){
                // Perform target protein identification. In this case, the value for "-i" is a protein ID.
                TargetProteinID targetProteinID = new TargetProteinID();
                if(cmd.hasOption("decoy")){
                    targetProteinID.only_identity_decoy_version = true;
                }
                targetProteinID.prepareDB(db,inputSeq,outdir);
                db = targetProteinID.getRefdb();
                inputSeq = targetProteinID.getTargetProteinSequence();
            }

            if(cmd.hasOption("t")){
                inputProcessor.setInputType(Integer.valueOf(cmd.getOptionValue("t")));
            }else{
                inputProcessor.setInputType(InputProcessor.InputTypeProtein);
            }

            if(cmd.hasOption("f")){
                inputProcessor.frames = cmd.getOptionValue("f");
            }
            if(cmd.hasOption("anno")) {
                inputProcessor.annotationFolder = cmd.getOptionValue("anno");
            }
            inputProcessor.outdir = outdir;
            inputProcessor.referenceProDB = db;
            ArrayList<CPeptide> cPeptides = inputProcessor.run(inputSeq);
            for(CPeptide cPeptide: cPeptides){
                if(cPeptide.nhit==0) {
                    pepSeqs.add(cPeptide.peptideSequence.toUpperCase());
                }
            }

        }else{
            System.out.println("Please provide peptide sequence which you want to search!");
            System.exit(0);
        }
        Cloger.logger.info("Valid target peptides: "+pepSeqs.size());


        ArrayList<PeptideInput> peptideInputs = generatePeptideInputs(pepSeqs);


        if(msmsfile.toLowerCase().endsWith(".mgf")){

            SpectraInput.readMSMSfast(msmsfile,peptideInputs);

        }else {

            SpectraInput.readMSMSdb2(msmsfile, peptideInputs);
        }

        Cloger.logger.info("Time elapsed: "+Cloger.getRunTime());


        ////////////////////////////////////////////////////////////////////////////////////////////////////////////////

        String outfile = outdir + "/psm.txt";
        BufferedWriter bufferedWriter = new BufferedWriter(new FileWriter(new File(outfile)));

        if(useGPD) {
            bufferedWriter.write("peptide\tmodification\tn\tspectrum_title\tcharge\texp_mass\tppm\tpep_mass\tmz\tscore\tn_db\ttotal_db\tn_random\ttotal_random\tpvalue\trandom_score1\trandom_score2\tscore1\tscore2\n");
        }else{
            bufferedWriter.write("peptide\tmodification\tn\tspectrum_title\tcharge\texp_mass\tppm\tpep_mass\tmz\tscore\tn_db\ttotal_db\tn_random\ttotal_random\tpvalue\n");
        }
        String outDetailfile = outdir + "/detail.txt";
        BufferedWriter dWriter = new BufferedWriter(new FileWriter(new File(outDetailfile)));
        dWriter.write("spectrum_title\tpeptide\tmodification\tpep_mass\tscore\n");
        //bufferedWriter.write("peptide\tpeptide_mod\tspectrum_title\tcharge\texp_mass\tppm\tpep_mass\tmz\tscore\tevalue\tn\ttotal\n");

        String psmannofile = outdir + "/psm_annotation.txt";
        BufferedWriter annoWriter = new BufferedWriter(new FileWriter(new File(psmannofile)));



        ////////////////////////////////////////////////////////////////////////////////////////////////////////////////

        if(!SpectraInput.isTargetPeptide2spectrumScored) {
            Cloger.logger.info("Start to score PSMs ...");

            ExecutorService fixedThreadPoolScore = Executors.newFixedThreadPool(CParameter.cpu);

            ////////////////////////////////////////////////////////////////////////////////////////////////////////////////

            for (PeptideInput peptideInput : peptideInputs) {
                fixedThreadPoolScore.execute(new ScoreWorker(peptideInput));
            }
            fixedThreadPoolScore.shutdown();
            fixedThreadPoolScore.awaitTermination(Long.MAX_VALUE, TimeUnit.HOURS);
            Cloger.logger.info("Start to score PSMs done.");
        }



        ////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        // search all the spectra against the reference protein database.

        DatabaseInput databaseInput = new DatabaseInput(db);
        //databaseInput.mSnSpectrums = SpectraInput.spectraMap;

        //if (cmd.hasOption("fixMod")) {
        String ifixmod[] = CParameter.fixMods.split(",");
        for (String i : ifixmod) {
            int id = Integer.valueOf(i);
            databaseInput.addFixedModifications(id);
        }
        //}

        //if (cmd.hasOption("varMod")) {
        String ivarmod[] = CParameter.varMods.split(",");
        for (String i : ivarmod) {
            int id = Integer.valueOf(i);
            databaseInput.addVarModifications(id);
        }

        if(db.toLowerCase().endsWith(".fa") || db.toLowerCase().endsWith(".fasta")) {

            String sqldb_file = db+".sqldb";
            File SQLDB = new File(sqldb_file);
            if(SQLDB.isFile()){
                System.out.println("Use indexed database:"+sqldb_file);
                databaseInput.db = sqldb_file;
                databaseInput.readDBSQL();
            }else {
                System.out.println("Don't find indexed database:"+sqldb_file);
                System.out.println("Use database:"+db);
                databaseInput.readDB();
            }
        }else{
            System.err.println("Please provide valid protein database: .fa or .fasta format.");
            System.exit(0);
        }


        Cloger.logger.info("Time elapsed: "+Cloger.getRunTime());


        boolean psmAnnoDataHasGenerated = false;

        if(pepSeqs.size()>=2) {


            Iterator<PeptideInput> inputIterator = peptideInputs.iterator();
            while(inputIterator.hasNext()){
                PeptideInput peptideInput = inputIterator.next();
                boolean rm = true;
                if(peptideInput.jPeptide.spectraIndexs.size()>0){
                    rm = false;
                }

                Iterator<JPeptide> jPeptideIterator = peptideInput.getPtmIsoforms().iterator();
                while(jPeptideIterator.hasNext()){
                    JPeptide jPeptide = jPeptideIterator.next();
                    if(jPeptide.spectraIndexs.size()==0){
                        jPeptideIterator.remove();
                    }else{
                        rm = false;
                    }
                }
                if(rm){
                    inputIterator.remove();
                }
            }

            System.out.println("Peptides with matched spectra: "+peptideInputs.size());

            if(peptideInputs.size()==0){
                System.out.println("Done!");
                System.exit(0);
            }

            ExecutorService fixedThreadPool = Executors.newFixedThreadPool(CParameter.cpu);

            ////////////////////////////////////////////////////////////////////////////////////////////////////////////////
            int ind = 1;
            for (PeptideInput peptideInput : peptideInputs) {
                fixedThreadPool.execute(new SearchWorker(peptideInput, databaseInput, ind));
                ind++;
            }
            fixedThreadPool.shutdown();
            fixedThreadPool.awaitTermination(Long.MAX_VALUE, TimeUnit.HOURS);

        }else{

            ExecutorService fixedThreadPool = Executors.newFixedThreadPool(CParameter.cpu);

            ////////////////////////////////////////////////////////////////////////////////////////////////////////////////

            PeptideInput peptideInput = peptideInputs.get(0);
            peptideInput.setRandomPeptides(PeptideInput.generateRandomPeptidesFast(peptideInput.jPeptide.peptide,CParameter.nRandomPeptides,true));
            //for (int i=0;i<peptideInput.jPeptide.mSnSpectrums.size();i++) {
            for (int i=0;i<peptideInput.jPeptide.spectraIndexs.size();i++) {
                // public PsmScoringWorker (PeptideInput pi, DatabaseInput di, int index, double itol, int method,String fixModOptionValue, JPeptide jPeptide){
                JPeptide jPeptide = peptideInput.jPeptide;
                fixedThreadPool.execute(new PsmScoringWorker(peptideInput, databaseInput, i, jPeptide));

            }

            for(JPeptide jPeptide : peptideInput.getPtmIsoforms()){
                //for (int i=0;i<jPeptide.mSnSpectrums.size();i++) {
                for (int i=0;i<jPeptide.spectraIndexs.size();i++) {
                    // public PsmScoringWorker (PeptideInput pi, DatabaseInput di, int index, double itol, int method,String fixModOptionValue, JPeptide jPeptide){
                    fixedThreadPool.execute(new PsmScoringWorker(peptideInput, databaseInput, i, jPeptide));
                }
            }

            fixedThreadPool.shutdown();
            fixedThreadPool.awaitTermination(Long.MAX_VALUE, TimeUnit.HOURS);

            psmAnnoDataHasGenerated = true;


        }
        annoWriter.write(PeakAnnotation.getPeakAnnotationHeader()+"\n");

        for(PeptideInput peptideInput : peptideInputs) {
            if(peptideInput.getOutLines().size() >= 1){
                bufferedWriter.write(StringUtils.join(peptideInput.getOutLines(),"\n")+"\n");
            }else{
                System.err.println(peptideInput.jPeptide.peptide.getSequence()+" don't have outputs!");
            }

            if(peptideInput.getOutAdditionalLines().size() >= 1){
                dWriter.write(StringUtils.join(peptideInput.getOutAdditionalLines(),"\n")+"\n");
            }


            if(psmAnnoDataHasGenerated) {
                if (peptideInput.getOutPeakAnnotations().size() >= 1) {
                    annoWriter.write(StringUtils.join(peptideInput.getOutPeakAnnotations(), "\n") + "\n");
                }
            }else{
                for(String title: peptideInput.jPeptide.spectraIndexs) {
                    MSnSpectrum spectrum = SpectraInput.spectraMap.get(title);
                    Peptide peptide = peptideInput.jPeptide.peptide;
                    String psmMatch_tmp = PeakAnnotation.getPeakAnnotation(peptide, spectrum, true);
                    if (psmMatch_tmp != null) {
                        annoWriter.write(psmMatch_tmp + "\n");
                    }
                }
                for(JPeptide jPeptide: peptideInput.getPtmIsoforms()){
                    for(String title: jPeptide.spectraIndexs) {
                        MSnSpectrum spectrum = SpectraInput.spectraMap.get(title);
                        Peptide peptide = jPeptide.peptide;
                        String psmMatch_tmp = PeakAnnotation.getPeakAnnotation(peptide, spectrum, true);
                        if (psmMatch_tmp != null) {
                            annoWriter.write(psmMatch_tmp + "\n");
                        }
                    }
                }
            }


        }



        bufferedWriter.close();
        dWriter.close();

        //ranScoreWriter.close();

        String psm_rank_file = RankPSM.rankPSMs(outfile);

        exportMGF(psm_rank_file,peptideInputs);

        if(cmd.hasOption("um")){

            String ofile = ModificationDB.doPTMValidation(psm_rank_file,peptideInputs,db,databaseInput.fixedModifications,outdir);
            //System.out.println("ofile:"+ofile);
            summariseResult(psm_rank_file,ofile);

            // annotate best PSM from unrestricted modification searching
            annotateBestPsmFromModSearch(ofile, annoWriter);

        }

        annoWriter.close();



        if(cmd.hasOption("plot")){
            String outfig = outdir + "/psm.pdf";
            PeakAnnotation.plot(psmannofile,outfig,CParameter.intensityThreshold);
        }

        Cloger.logger.info("Time elapsed: "+Cloger.getRunTime());
        Cloger.logger.info("End.");

    }

    public static void annotateBestPsmFromModSearch(String ptm_file, BufferedWriter annoWriter) throws IOException {

        System.out.println("Annotate best PSMs from unrestricted modification searching ...");

        File ptm_F = new File(ptm_file);
        if(!ptm_F.exists()){
            System.out.println("File:"+ptm_file+" doesn't exist!");
            return;
        }

        // ptm.txt
        // spectrum_title  peptide charge  exp_mass        pep_mass        modification    score
        // read PSM data
        BufferedReader psmReader = new BufferedReader(new FileReader(new File(ptm_file)));
        String headline = psmReader.readLine().trim();
        String h[] = headline.split("\t");
        HashMap<String,Integer> hMap = new HashMap<>();
        for(int i=0;i<h.length;i++){
            hMap.put(h[i],i);
        }
        String line;
        HashMap<String, RankPSM> bestPsmMap = new HashMap<>();

        while((line = psmReader.readLine())!=null){
            line = line.trim();
            String d[] = line.split("\t");
            String spectrum_title = d[hMap.get("spectrum_title")];
            double score = Double.valueOf(d[hMap.get("score")]);

            if(bestPsmMap.containsKey(spectrum_title)){
                if(bestPsmMap.get(spectrum_title).score < score){
                    bestPsmMap.get(spectrum_title).score = score;
                    bestPsmMap.get(spectrum_title).line = line;
                }
            }else{
                RankPSM rankPSM = new RankPSM();
                rankPSM.score = score;
                rankPSM.line = line;
                bestPsmMap.put(spectrum_title,rankPSM);
            }


        }

        psmReader.close();

        //
        //PTMFactory ptmFactory = PTMFactory.getInstance();
        Pattern pattern = Pattern.compile("^(.*)@(\\d+)");
        for(String spectrum_title: bestPsmMap.keySet()){
            String d[] = bestPsmMap.get(spectrum_title).line.split("\t");
            String peptide_seq = d[hMap.get("peptide")];
            String modification = d[hMap.get("modification")];
            ArrayList<ModificationMatch> modificationMatches = new ArrayList<>();
            if(!modification.startsWith("-")){
                String mods[] = modification.split(";");
                // TMT2plex of S@4[225.1558]
                for(String mod : mods){
                    Matcher matcher = pattern.matcher(mod);
                    while (matcher.find()) {
                        String ptm_name = matcher.group(1);
                        int pos = Integer.valueOf(matcher.group(2));
                        // PTM ptm = ptmFactory.getPTM(ptm_name);
                        ModificationMatch mm = new ModificationMatch(ptm_name, true, pos);
                        modificationMatches.add(mm);
                    }
                }
            }

            Peptide peptide = new Peptide(peptide_seq,modificationMatches);
            MSnSpectrum spectrum = SpectraInput.spectraMap.get(spectrum_title);
            String psmMatch_tmp = PeakAnnotation.getPeakAnnotation(peptide, spectrum, true);
            if (psmMatch_tmp != null) {
                annoWriter.write(psmMatch_tmp + "\n");

            }

        }

        System.out.println("Generate annotation data for best matches from unrestricted modification searching:"+bestPsmMap.size());


    }


    /**
     *
     * @param psm_rank_file psm_rank.txt
     * @param ptm_file  ptm.txt. This file is from the unrestricted modification searching.
     */
    static void summariseResult(String psm_rank_file, String ptm_file){
        // P-value <= 0.01, rank == 1, n_db == 0
        int identifiedPSMs = 0;

        HashMap<String,ArrayList<ScoreResult>> peptideRes = new HashMap<>();
        HashMap<String,ScoreResult> spectraRes = new HashMap<>();

        String psm_rank_file_header = null;

        try {
            BufferedReader bReader = new BufferedReader(new FileReader(new File(psm_rank_file)));

            String line = bReader.readLine().trim();
            psm_rank_file_header = line;
            String headLine[] = line.split("\t");
            HashMap<String,Integer> headMap = new HashMap<>();
            for(int i=0;i<headLine.length;i++){
                headMap.put(headLine[i],i);
            }



            while((line = bReader.readLine())!=null){
                line = line.trim();
                String d[] = line.split("\t");
                String spectrum_title  = d[headMap.get("spectrum_title")];
                String peptideSequence = d[headMap.get("peptide")];

                double pvalue = Double.valueOf(d[headMap.get("pvalue")]);
                int rank = Integer.valueOf(d[headMap.get("rank")]);
                int n_db = Integer.valueOf(d[headMap.get("n_db")]);
                double score = Double.valueOf(d[headMap.get("score")]);


                if(pvalue <= 0.01 && n_db == 0 && rank == 1){
                    identifiedPSMs++;
                    ScoreResult scoreResult = new ScoreResult();
                    scoreResult.score = score;
                    scoreResult.peptideSequence = peptideSequence;
                    scoreResult.spectrum_title = spectrum_title;
                    scoreResult.eInfo = line;

                    if(CParameter.scoreMethod == 0){
                        double score2 = Double.valueOf(d[headMap.get("score2")]);
                        scoreResult.score2 = score2;
                    }

                    if(peptideRes.containsKey(peptideSequence)){
                        peptideRes.get(peptideSequence).add(scoreResult);
                    }else{
                        ArrayList<ScoreResult> scoreResults = new ArrayList<>();
                        scoreResults.add(scoreResult);
                        peptideRes.put(peptideSequence,scoreResults);
                    }

                    spectraRes.put(spectrum_title,scoreResult);
                }

            }

            bReader.close();

        } catch (FileNotFoundException e) {
            e.printStackTrace();
        } catch (IOException e) {
            e.printStackTrace();
        }

        System.out.println("Identified PSMs: "+spectraRes.size());
        System.out.println("Identified peptides: "+peptideRes.size());



        if(identifiedPSMs>=1){
            File PT = new File(ptm_file);
            if(PT.isFile()){
                try {

                    HashMap<String,ArrayList<String>> spectraPtmRes = new HashMap<>();

                    // ptm.txt. This file is from the unrestricted modification searching.
                    BufferedReader bReader = new BufferedReader(new FileReader(PT));

                    String line = bReader.readLine().trim();
                    String headLine[] = line.split("\t");
                    HashMap<String, Integer> headMap = new HashMap<>();
                    for (int i = 0; i < headLine.length; i++) {
                        headMap.put(headLine[i], i);
                    }


                    while ((line = bReader.readLine()) != null) {
                        line = line.trim();
                        String d[] = line.split("\t");
                        String spectrum_title = d[headMap.get("spectrum_title")];
                        //String peptideSequence = d[headMap.get("peptide")];


                        // compare the score from PTM searching with the score from matching to target peptide.

                        if(CParameter.scoreMethod == 0){
                            double score = Double.valueOf(d[headMap.get("score1")]);
                            double score2 = Double.valueOf(d[headMap.get("score2")]);
                            // Ҫ

                            // very stringent
                            if(CParameter.unrestrictedSearchWithEqualAndBetterScore) {
                                if (score >= spectraRes.get(spectrum_title).score || score2 >= spectraRes.get(spectrum_title).score2) {

                                    if (spectraPtmRes.containsKey(spectrum_title)) {
                                        spectraPtmRes.get(spectrum_title).add(line);
                                    } else {
                                        ArrayList<String> tmp = new ArrayList<>();
                                        tmp.add(line);
                                        spectraPtmRes.put(spectrum_title, tmp);
                                    }
                                }
                            // it's normal cutoff
                            }else {
                                if (score > spectraRes.get(spectrum_title).score || score2 > spectraRes.get(spectrum_title).score2) {

                                    if (spectraPtmRes.containsKey(spectrum_title)) {
                                        spectraPtmRes.get(spectrum_title).add(line);
                                    } else {
                                        ArrayList<String> tmp = new ArrayList<>();
                                        tmp.add(line);
                                        spectraPtmRes.put(spectrum_title, tmp);
                                    }
                                }
                            }
                        }else{
                            double score = Double.valueOf(d[headMap.get("score")]);

                            // very stringent
                            if(CParameter.unrestrictedSearchWithEqualAndBetterScore) {
                                if (score >= spectraRes.get(spectrum_title).score) {
                                    if (spectraPtmRes.containsKey(spectrum_title)) {
                                        spectraPtmRes.get(spectrum_title).add(line);
                                    } else {
                                        ArrayList<String> tmp = new ArrayList<>();
                                        tmp.add(line);
                                        spectraPtmRes.put(spectrum_title, tmp);
                                    }
                                }
                            }else {
                                if (score > spectraRes.get(spectrum_title).score) {
                                    if (spectraPtmRes.containsKey(spectrum_title)) {
                                        spectraPtmRes.get(spectrum_title).add(line);
                                    } else {
                                        ArrayList<String> tmp = new ArrayList<>();
                                        tmp.add(line);
                                        spectraPtmRes.put(spectrum_title, tmp);
                                    }
                                }
                            }
                        }

                    }

                    bReader.close();

                    System.out.println("Spectra with higher score from PTM searching: "+spectraPtmRes.size());

                    if(spectraPtmRes.size()>=1) {

                        String ofile = ptm_file;
                        ofile = ofile.replaceAll(".txt$","_detail.txt");
                        BufferedWriter bWriter = new BufferedWriter(new FileWriter(new File(ofile)));
                        for (int i = 0; i < headLine.length; i++) {
                            headLine[i] = "ptm_" + headLine[i];
                        }
                        bWriter.write(psm_rank_file_header + "\t" + StringUtils.join(headLine, "\t") + "\n");
                        for (String title : spectraPtmRes.keySet()) {
                            for (String ll : spectraPtmRes.get(title)) {
                                bWriter.write(spectraRes.get(title).eInfo + "\t" + ll+"\n");
                            }
                        }


                        bWriter.close();

                        int nFinalPeps = 0;
                        int nFinalPsms = 0;
                        for(String pep: peptideRes.keySet()){
                            Iterator<ScoreResult> iterator = peptideRes.get(pep).iterator();
                            while(iterator.hasNext()){
                                ScoreResult scoreResult = iterator.next();
                                if(spectraPtmRes.containsKey(scoreResult.spectrum_title)){
                                    iterator.remove();
                                }
                            }

                            if(peptideRes.get(pep).size()>=1){
                                nFinalPeps++;
                                nFinalPsms = nFinalPsms + peptideRes.get(pep).size();
                            }

                        }

                        System.out.println("Identified peptides after PTM searching: "+nFinalPeps);
                        System.out.println("Identified PSMs after PTM searching: "+nFinalPsms);


                    }

                    // Based on the PTM searching result, filtering psm_rank.txt file
                    String old_psm_rank_file = psm_rank_file.replaceAll("txt$","") + "tmp";
                    FileUtils.copyFile(new File(psm_rank_file), new File(old_psm_rank_file));

                    BufferedReader psm = new BufferedReader(new FileReader(new File(old_psm_rank_file)));
                    PrintWriter writer = new PrintWriter(psm_rank_file, "UTF-8");
                    String hl= psm.readLine().trim();
                    String hls[] = hl.split("\t");
                    HashMap<String, Integer> hMap = new HashMap<>();
                    for (int i = 0; i < hls.length; i++) {
                        hMap.put(hls[i], i);
                    }
                    writer.println(hl+"\tn_ptm");

                    while((line = psm.readLine())!=null){
                        line = line.trim();
                        String d[] = line.split("\t");
                        String title = d[hMap.get("spectrum_title")];
                        double pvalue = Double.valueOf(d[hMap.get("pvalue")]);
                        int rank = Integer.valueOf(d[hMap.get("rank")]);
                        int n_db = Integer.valueOf(d[hMap.get("n_db")]);
                        int n_ptm = -1;
                        if(pvalue <= 0.01 && n_db == 0 && rank == 1){
                            if(spectraPtmRes.containsKey(title)){
                                n_ptm = spectraPtmRes.get(title).size();
                            }else{
                                n_ptm = 0;
                            }

                        }else{
                            n_ptm = -1;
                        }

                        writer.println(line+"\t"+n_ptm);
                    }
                    psm.close();
                    writer.close();


                } catch (FileNotFoundException e) {
                    e.printStackTrace();
                } catch (IOException e) {
                    e.printStackTrace();
                }

            }
        }



    }

    public static ScoreResult scorePeptide2Spectrum(Peptide peptide, MSnSpectrum msnSpectrum){
        JSpectrum jSpectrum = new JSpectrum();
        for (Peak p : msnSpectrum.getPeakList()) {
            JPeak jPeak = new JPeak(p.getMz(), p.getIntensity());
            jSpectrum.addRawPeak(jPeak);
        }
        jSpectrum.resetPeaks();
        jSpectrum.sortPeaksByMZ();

        JPeptideSpectrumMatch psm = new JPeptideSpectrumMatch();
        psm.setCalculatedMassToCharge(msnSpectrum.getPrecursor().getMz());
        psm.setCharge(msnSpectrum.getPrecursor().getPossibleCharges().get(0).value);
        psm.setExperimentalMassToCharge(msnSpectrum.getPrecursor().getMz());
        psm.setSpectrumID(msnSpectrum.getSpectrumTitle());
        psm.setPepSeq(peptide.getSequence());

        //JMatch.ms2tol = itol;

        double score = -1.0;
        ScoreResult scoreResult = new ScoreResult();

        if(CParameter.scoreMethod == 0){
            ////////////////////////////////////////////////////////////////////////////////////////////////////////////////
            // The first scoring algorithm: hyperscore
            HyperscoreMatch hyperscoreMatch = new HyperscoreMatch(msnSpectrum, jSpectrum, psm);
            hyperscoreMatch.objPeptide = peptide;

            try {
                hyperscoreMatch.initialize(false, 2);
            } catch (ClassNotFoundException e) {
                e.printStackTrace();
            } catch (IOException e) {
                e.printStackTrace();
            } catch (InterruptedException e) {
                e.printStackTrace();
            } catch (SQLException e) {
                e.printStackTrace();
            }

            hyperscoreMatch.calcHyperScore();


            score = hyperscoreMatch.getHyperScore();

            ////////////////////////////////////////////////////////////////////////////////////////////////////////////////
            // The second scoring algorithm: mvh

            // We need to reset the peaks
            jSpectrum.resetPeaks();
            jSpectrum.sortPeaksByMZ();

            MVHMatch MVHMatch = new MVHMatch(msnSpectrum,jSpectrum, psm);
            MVHMatch.objPeptide = peptide;
            try {
                MVHMatch.initialize(false, 3);
            } catch (ClassNotFoundException e) {
                e.printStackTrace();
            } catch (IOException e) {
                e.printStackTrace();
            } catch (InterruptedException e) {
                e.printStackTrace();
            } catch (SQLException e) {
                e.printStackTrace();
            }
            try {
                MVHMatch.calcMVH();
            } catch (ClassNotFoundException e) {
                e.printStackTrace();
            } catch (IOException e) {
                e.printStackTrace();
            } catch (InterruptedException e) {
                e.printStackTrace();
            } catch (SQLException e) {
                e.printStackTrace();
            }
            double score2 = MVHMatch.getMvh();


            scoreResult.score = score;
            scoreResult.score2 = score2;

        }else if(CParameter.scoreMethod == 1){
            HyperscoreMatch hyperscoreMatch = new HyperscoreMatch(msnSpectrum, jSpectrum, psm);
            hyperscoreMatch.objPeptide = peptide;

            try {
                hyperscoreMatch.initialize(false, 2);
            } catch (ClassNotFoundException e) {
                e.printStackTrace();
            } catch (IOException e) {
                e.printStackTrace();
            } catch (InterruptedException e) {
                e.printStackTrace();
            } catch (SQLException e) {
                e.printStackTrace();
            }

            hyperscoreMatch.calcHyperScore();


            score = hyperscoreMatch.getHyperScore();
            scoreResult.score = score;

        }else if(CParameter.scoreMethod == 2){
            MVHMatch MVHMatch = new MVHMatch(msnSpectrum,jSpectrum, psm);
            MVHMatch.objPeptide = peptide;
            try {
                MVHMatch.initialize(false, 3);
            } catch (ClassNotFoundException e) {
                e.printStackTrace();
            } catch (IOException e) {
                e.printStackTrace();
            } catch (InterruptedException e) {
                e.printStackTrace();
            } catch (SQLException e) {
                e.printStackTrace();
            }
            try {
                MVHMatch.calcMVH();
            } catch (ClassNotFoundException e) {
                e.printStackTrace();
            } catch (IOException e) {
                e.printStackTrace();
            } catch (InterruptedException e) {
                e.printStackTrace();
            } catch (SQLException e) {
                e.printStackTrace();
            }
            score = MVHMatch.getMvh();
            scoreResult.score = score;
        }else{
            System.err.println("Please provide valid score method: 1 or 2. Your input is "+CParameter.scoreMethod);
            System.exit(1);
        }


        return(scoreResult);
    }


    public static void exportMGF(String psm_rank_file, ArrayList<PeptideInput> peptideInputs) throws IOException {

        HashSet<String> spectraTitleSet = new HashSet<>();


        BufferedReader bReader = new BufferedReader(new FileReader(new File(psm_rank_file)));

        String line = bReader.readLine().trim();
        String headLine[] = line.split("\t");
        HashMap<String,Integer> headMap = new HashMap<>();
        for(int i=0;i<headLine.length;i++){
            headMap.put(headLine[i],i);
        }

        while((line = bReader.readLine())!=null){
            line = line.trim();
            String d[] = line.split("\t");
            String spectrum_title  = d[headMap.get("spectrum_title")];
            spectraTitleSet.add(spectrum_title);

        }

        bReader.close();

        if(spectraTitleSet.size()>=1){

            HashSet<String> findSpectraSet = new HashSet<>();

            String omgf = psm_rank_file.replaceAll(".txt$",".mgf");

            BufferedWriter mWriter = new BufferedWriter(new FileWriter(new File(omgf)));

            for (PeptideInput peptideInput : peptideInputs) {
                if (peptideInput.getOutLines().size() >= 1) {
                    //String peptideSequence = peptideInput.jPeptide.peptide.getSequence();
                    //for (MSnSpectrum spectrum : peptideInput.jPeptide.mSnSpectrums) {
                    for (String title: peptideInput.jPeptide.spectraIndexs) {
                        MSnSpectrum spectrum = SpectraInput.spectraMap.get(title);
                        if (spectraTitleSet.contains(spectrum.getSpectrumTitle())) {
                            if(!findSpectraSet.contains(spectrum.getSpectrumTitle())) {
                                findSpectraSet.add(spectrum.getSpectrumTitle());
                                mWriter.write(spectrum.asMgf());
                            }

                        }
                    }

                    for(JPeptide jPeptide:peptideInput.getPtmIsoforms()){
                        //for (MSnSpectrum spectrum : jPeptide.mSnSpectrums) {
                        for (String title : jPeptide.spectraIndexs) {
                            MSnSpectrum spectrum = SpectraInput.spectraMap.get(title);
                            if (spectraTitleSet.contains(spectrum.getSpectrumTitle())) {
                                if(!findSpectraSet.contains(spectrum.getSpectrumTitle())) {
                                    findSpectraSet.add(spectrum.getSpectrumTitle());
                                    mWriter.write(spectrum.asMgf());
                                }

                            }
                        }
                    }
                }
            }

            mWriter.close();

            if(spectraTitleSet.size()==findSpectraSet.size()){
                System.out.println("All spectra have been exported to file: "+omgf);
            }else{
                System.out.println("Warnings: "+(spectraTitleSet.size()-findSpectraSet.size())+" left!");
            }


        }
    }



    public static void generateSpectraAnnotation(PeptideInput peptideInput, Peptide peptide, MSnSpectrum spectrum){
        String psmMatch_tmp = PeakAnnotation.getPeakAnnotation(peptide, spectrum, true);
        if (psmMatch_tmp != null) {
            peptideInput.addOutPeakAnnotations(psmMatch_tmp);
        }
    }


    public static ArrayList<PeptideInput> generatePeptideInputs(ArrayList<String> pepSeqs) throws InterruptedException {

        Cloger.logger.info("Generate peptide objects ...");
        ArrayList<PeptideInput> peptideInputs = new ArrayList<>();

        ModificationGear.getInstance();

        int cpu = CParameter.cpu;

        if(cpu > pepSeqs.size()){
            cpu = pepSeqs.size();
        }

        System.out.println("CPU: "+cpu);
        ExecutorService fixedThreadPoolScore = Executors.newFixedThreadPool(cpu);

        ConcurrentHashMap<String,PeptideInput> peptideInputConcurrentHashMap = new ConcurrentHashMap<>();

        for(String pseq : pepSeqs){
            fixedThreadPoolScore.execute(new CreatePeptideInputWorker(pseq,peptideInputConcurrentHashMap));

        }

        fixedThreadPoolScore.shutdown();

        fixedThreadPoolScore.awaitTermination(Long.MAX_VALUE, TimeUnit.HOURS);

        for(String pep: peptideInputConcurrentHashMap.keySet()){
            peptideInputs.add(peptideInputConcurrentHashMap.get(pep));
        }

        Cloger.logger.info("Generate peptide objects done.");

        return(peptideInputs);
    }


}
