package main.java.pg;

import com.compomics.util.experiment.biology.PTM;
import com.compomics.util.experiment.biology.PTMFactory;
import com.compomics.util.experiment.biology.Peptide;
import com.compomics.util.experiment.identification.matches.ModificationMatch;
import com.compomics.util.preferences.SequenceMatchingPreferences;
import main.java.PSMMatch.JPeptide;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.math3.util.CombinatoricsUtils;
import org.paukov.combinatorics3.Generator;

import java.io.IOException;
import java.sql.SQLException;
import java.util.*;
import java.util.concurrent.ThreadLocalRandom;
import java.util.stream.Collectors;


public final class PeptideInput {

    public JPeptide jPeptide = null;

    /**
     * different ptm isoform of the peptide
     */
    private ArrayList<JPeptide> isoforms = new ArrayList<JPeptide>();

    /**
     * Random peptides
     */
    public ArrayList<String> randomPeptides = new ArrayList<>();


    private static SequenceMatchingPreferences sequenceMatchingPreferences = new SequenceMatchingPreferences();
    private static SequenceMatchingPreferences ptmSequenceMatchingPreferences = new SequenceMatchingPreferences();


    private ArrayList<String> outLines = new ArrayList<>();
    private ArrayList<String> outAdditionalLines = new ArrayList<>();
    private ArrayList<String> outPeakAnnotations = new ArrayList<>();


    public PeptideInput(JPeptide pep){
        this.jPeptide = pep;
    }

    public PeptideInput(Peptide pep){
        this.jPeptide = new JPeptide(pep);
    }

    public ArrayList<JPeptide> getPtmIsoforms(){
        return(this.isoforms);
    }

    public void setPtmIsoforms(ArrayList<Peptide> ps){
        for(Peptide p:ps){
            JPeptide jPeptide = new JPeptide(p);
            this.isoforms.add(jPeptide);
        }
    }



    /**
     * According to variable modifications to generate peptide isoforms for a peptide
     * @param pepObj Peptide object
     * @param varMods Variable modifications
     * @param maxVarMods Max number of variable modifications in a peptide
     * @return An array list of Peptide
     */
    public static ArrayList<Peptide> calcPeptideIsoforms(Peptide pepObj, ArrayList<PTM> varMods, int maxVarMods) throws InterruptedException, ClassNotFoundException, SQLException, IOException {
        String peptideSequence = pepObj.getSequence();
        // Firstly, we need to get the possible modification sites for the peptide according to varMods.
        // If an AA has had a fixed modification, this AA cannot have variable modification.

        ArrayList<Peptide> peptides = new ArrayList<>();
        if(varMods.size()>=1) {
            HashMap<Integer, PTM> position2ptm = new HashMap<>();

            //int peptideLength = peptideSequence.length();
            for (PTM ptm : varMods) {
                ArrayList<Integer> possibleSites = pepObj.getPotentialModificationSites(ptm, sequenceMatchingPreferences, ptmSequenceMatchingPreferences);
                for (Integer i : possibleSites) {
                    position2ptm.put(i, ptm);
                }
            }
            if(position2ptm.size()>=1) {
                // >=1 possible modification site
                Set<Integer> ptmPosition = position2ptm.keySet();
                // the max number of modifications to consider
                int maxNumPtms = position2ptm.size()>=maxVarMods?maxVarMods:position2ptm.size();
                for(int k=1;k<=maxNumPtms;k++){
                    Iterator<List<Integer>> iter = Generator.combination(ptmPosition).simple(k).iterator();
                    while(iter.hasNext()){
                        List<Integer> iCombination = iter.next();
                        Peptide modPeptide = new Peptide(pepObj.getSequence(), pepObj.getModificationMatches());
                        for(Integer pos:iCombination){
                            modPeptide.addModificationMatch(new ModificationMatch(position2ptm.get(pos).getName(),true,pos));
                        }
                        peptides.add(modPeptide);
                        //System.out.println("====="+modPeptide.getSequence()+"\t"+modPeptide.getSequenceWithLowerCasePtms()+"\t"+modPeptide.getNModifications());

                    }
                }
            }
        }
        return(peptides);
    }

    /**
     * Add fixed modification into the peptide
     * @param modification Fixed modification index, the format is like : 1,2,3
     * @throws InterruptedException
     * @throws ClassNotFoundException
     * @throws SQLException
     * @throws IOException
     */
    public void addFixedModification(String modification) throws InterruptedException, ClassNotFoundException, SQLException, IOException {
        String [] mods = modification.split(",");
        if(mods.length>=1) {
            for (String mod : mods) {
                PTM ptm = ModificationGear.getInstance().getPTM(Integer.valueOf(mod));
                ArrayList<Integer> possibleSites = this.jPeptide.peptide.getPotentialModificationSites(ptm, sequenceMatchingPreferences, ptmSequenceMatchingPreferences);
                for (Integer k : possibleSites) {
                    this.jPeptide.peptide.addModificationMatch(new ModificationMatch(ptm.getName(),false,k));
                }
            }
        }
    }


    public static void main(String[] args) {

        PTMFactory ptmFactory = PTMFactory.getInstance();
        ArrayList<String> ptmNames = ptmFactory.getPTMs();
        System.out.println(ptmNames.size());
        for(String name : ptmNames){
            PTM ptm = ptmFactory.getPTM(name);
            System.out.println(ptm.getName());
        }

    }


    /**
     * Generate random peptides for a target peptide
     * @param pepObj The object of target peptide
     * @param num The number of random peptides
     * @param fixCtermAA Fix the c-terminal amino acid
     * @return Random peptide sequences
     */
    public static ArrayList<String> generateRandomPeptides(Peptide pepObj, int num, boolean fixCtermAA){
        ArrayList<String> randomPeptides = new ArrayList<>();
        String peptideSequence = pepObj.getSequence();
        int peptideLength = peptideSequence.length();

        // No chage for c-term amino acid
        if(fixCtermAA){
            peptideLength = peptideLength - 1;
            // remove c-term amino acid
            peptideSequence = peptideSequence.substring(0,peptideLength);
        }

        // Calculate the max number of combinations
        HashMap<String,Integer> aaCount = new HashMap<>();
        for(int i=0;i<peptideLength;i++){
            String aa = String.valueOf(peptideSequence.charAt(i));
            if(aaCount.containsKey(aa)){
                aaCount.put(aa,1+aaCount.get(aa));
            }else{
                aaCount.put(aa,1);
            }
        }


        // sort aaCount, key=aavalue=int
        // Sort by key: low to high
        List<HashMap.Entry<String,Integer>> aaList=new ArrayList<>(aaCount.size());
        aaList.addAll(aaCount.entrySet());
        HashMapValueCompare vc = new HashMapValueCompare();
        Collections.sort(aaList,vc);

        // The total number of random peptides
        long nComb = 1;
        int rn = peptideLength;

        for(HashMap.Entry<String,Integer> it : aaList) {
            String aa = it.getKey();
            //
            int n = aaCount.get(aa);
            long c = CombinatoricsUtils.binomialCoefficient(rn, n);

            // if the number > Long.MAX_VALUE, the result may be a negative value (will cause problem)
            // so set a max value
            if(nComb >= 10000000){
                nComb = 10000000;
            }else{
                nComb = nComb * c;
            }
            rn = rn - n;
            //System.out.println(aa+"\t"+n+"\t"+c+"\t"+rn);
        }


        // generate random peptides
        HashSet<String> pSet = new HashSet<>(num);
        pSet.add(pepObj.getSequence());

        num = num>nComb? (int) nComb :num;
        //System.out.println("Combination: "+nComb+","+num);

        // Save the result for combinations from the first amino acid
        // lowest frequency.
        Set<Integer> pos1aa = new HashSet<>(peptideLength);
        for(int j=1;j<=peptideLength;j++){
            pos1aa.add(j);
        }
        List<List<Integer>> pp1aa = Generator.combination(pos1aa).simple(aaList.get(0).getValue()).stream().collect(Collectors.toList());

        for(int i=1;i<=num;i++){

            Set<Integer> pos = new HashSet<>(peptideLength);
            for(int j=1;j<=peptideLength;j++){
                pos.add(j);
            }
            String pep[] = new String[pepObj.getSequence().length()];
            for(int k=0;k<aaList.size();k++) {

                String aa = aaList.get(k).getKey();
                int  n = aaCount.get(aa);
                long c = CombinatoricsUtils.binomialCoefficient(pos.size(),n);
                int  r = new Random().nextInt((int)c);

                List<Integer> pp;
                if(k==0){
                    // speed up
                    pp = pp1aa.get(r);
                }else{
                    pp = Generator.combination(pos).simple(aaCount.get(aa)).stream().collect(Collectors.toList()).get(r);
                }

                for(int indpp : pp){
                    pep[indpp] = aa;
                }
                //pp.forEach(k -> pep[k]=aa);

                pos.removeAll(pp);
                //System.out.println(aa+"\t"+n+"\t"+c+"\t"+r+ "\t"+pp.stream().map(k -> String.valueOf(k)).collect(Collectors.joining("_")));

            }
            String rpep = StringUtils.join(pep,"")+pepObj.getSequence().charAt(peptideLength);

            if(pSet.contains(rpep)){
                continue;
            }else{
                pSet.add(rpep);
                randomPeptides.add(rpep);
            }

            //System.out.println(rpep);

        }

        return(randomPeptides);
    }



    /**
     * Generate random peptides for a target peptide. This is the version used currently.
     * @param pepObj The object of target peptide
     * @param num The number of random peptides
     * @param fixCtermAA Fix the c-terminal amino acid
     * @return peptide sequences
     */
    public static ArrayList<String> generateRandomPeptidesFast(Peptide pepObj, int num, boolean fixCtermAA){
        ArrayList<String> randomPeptides = new ArrayList<>();
        String peptideSequence = pepObj.getSequence();
        int peptideLength = peptideSequence.length();

        // No chage for c-term amino acid
        if(fixCtermAA){
            peptideLength = peptideLength - 1;
            peptideSequence = peptideSequence.substring(0,peptideLength);
        }

        HashMap<String,Integer> aaCount = new HashMap<>();
        for(int i=0;i<peptideLength;i++){
            String aa = String.valueOf(peptideSequence.charAt(i));
            if(aaCount.containsKey(aa)){
                aaCount.put(aa,1+aaCount.get(aa));
            }else{
                aaCount.put(aa,1);
            }
        }

        List<HashMap.Entry<String,Integer>> aaList=new ArrayList<>(aaCount.size());
        aaList.addAll(aaCount.entrySet());
        HashMapValueCompare vc = new HashMapValueCompare();
        Collections.sort(aaList,vc);

        long nComb = 1;
        int rn = peptideLength;

        for(HashMap.Entry<String,Integer> it : aaList) {
            String aa = it.getKey();
            int n = aaCount.get(aa);
            long c = CombinatoricsUtils.binomialCoefficient(rn, n);

            if(nComb >= 10000000){
                nComb = 10000000;
            }else{
                nComb = nComb * c;
            }
            rn = rn - n;
            //System.out.println(aa+"\t"+n+"\t"+c+"\t"+rn);
        }

        HashSet<String> pSet = new HashSet<>(num);
        pSet.add(pepObj.getSequence());

        num = num>nComb? (int) nComb :num;
        //System.out.println("Combination: "+nComb+","+num);

        Set<Integer> pos1aa = new HashSet<>(peptideLength);
        for(int j=1;j<=peptideLength;j++){
            pos1aa.add(j);
        }
        List<List<Integer>> pp1aa = Generator.combination(pos1aa).simple(aaList.get(0).getValue()).stream().collect(Collectors.toList());

        // control the random with random seed
        Random random = new Random();
        random.setSeed(12457);

        for(int i=1;i<=num;i++){

            Set<Integer> pos = new HashSet<>(peptideLength);
            for(int j=1;j<=peptideLength;j++){
                pos.add(j);
            }
            String pep[] = new String[pepObj.getSequence().length()];
            for(int k=0;k<aaList.size();k++) {

                String aa = aaList.get(k).getKey();
                int  n = aaCount.get(aa);

                for(int rindex=0;rindex<n;rindex++){
                    long c = CombinatoricsUtils.binomialCoefficient(pos.size(),1);
                    int  r = random.nextInt((int)c);

                    int pp = Generator.combination(pos).simple(1).stream().collect(Collectors.toList()).get(r).get(0);

                    pep[pp] = aa;
                    //pp.forEach(k -> pep[k]=aa);

                    pos.remove(pp);
                }

                //System.out.println(aa+"\t"+n+"\t"+c+"\t"+r+ "\t"+pp.stream().map(k -> String.valueOf(k)).collect(Collectors.joining("_")));

            }
            String rpep = StringUtils.join(pep,"")+pepObj.getSequence().charAt(peptideLength);

            if(pSet.contains(rpep)){
                continue;
            }else{
                pSet.add(rpep);
                randomPeptides.add(rpep);
            }

            //System.out.println(rpep);

        }

        return(randomPeptides);
    }



    public void setRandomPeptides(ArrayList<String> rPeptides){
        this.randomPeptides = rPeptides;
    }


    /**
     * Generate peptide object
     * @param pepObj peptide object
     * @param peptideSequence peptide sequence
     * @param fixMods fixed modification
     * @return peptide object
     */
    public static Peptide getModificationPeptide(Peptide pepObj, String peptideSequence, String fixMods){

        // peptide object without any modification
        ArrayList<ModificationMatch> modificationMatches = new ArrayList<>();
        Peptide pep = new Peptide(peptideSequence, modificationMatches);

        // already modified site information
        HashSet<Integer> usedPos = new HashSet<>();

        // add fixed modification
        if (fixMods != null && !fixMods.isEmpty()) {
            String[] mods = fixMods.split(",");
            if (mods.length >= 1) {
                for (String mod : mods) {
                    PTM ptm = ModificationGear.getInstance().getPTM(Integer.valueOf(mod));

                    ArrayList<Integer> possibleSites = null;
                    try {
                        possibleSites = pep.getPotentialModificationSites(ptm, sequenceMatchingPreferences, ptmSequenceMatchingPreferences);
                    } catch (IOException e) {
                        e.printStackTrace();
                    } catch (InterruptedException e) {
                        e.printStackTrace();
                    } catch (ClassNotFoundException e) {
                        e.printStackTrace();
                    } catch (SQLException e) {
                        e.printStackTrace();
                    }

                    for (Integer k : possibleSites) {
                        // fixed modification
                        pep.addModificationMatch(new ModificationMatch(ptm.getName(), false, k));
                    }
                }
            }
        }

        // variable modification
        int n = pepObj.getModificationMatches().size();
        PTMFactory ptmFactory = PTMFactory.getInstance();
        if (n >= 1) {
            // modification data from base peptide object
            ArrayList<ModificationMatch> modMatches = pepObj.getModificationMatches();
            for (ModificationMatch modificationMatch : modMatches) {

                // variable modification. The type of modification here is important.
                if (modificationMatch.isVariable()) {
                    PTM ptm = ptmFactory.getInstance().getPTM(modificationMatch.getTheoreticPtm());
                    // get the possible modification sites.
                    ArrayList<Integer> possibleSites = null;
                    try {
                        possibleSites = pep.getPotentialModificationSites(ptm, sequenceMatchingPreferences, ptmSequenceMatchingPreferences);
                    } catch (IOException e) {
                        e.printStackTrace();
                    } catch (InterruptedException e) {
                        e.printStackTrace();
                    } catch (ClassNotFoundException e) {
                        e.printStackTrace();
                    } catch (SQLException e) {
                        e.printStackTrace();
                    }

                    // If the site has been considered
                    ArrayList<Integer> removeSites = new ArrayList<>();
                    for(int i : possibleSites){
                        if(usedPos.contains(i)){
                            removeSites.add(i);
                        }
                    }
                    if(removeSites.size()>=1){
                        possibleSites.removeAll(removeSites);
                    }

                    if(possibleSites.size()>=1) {

                        int index = ThreadLocalRandom.current().nextInt(possibleSites.size());
                        int ptmIndex = possibleSites.get(index);
                        // ptmIndex: the position of the modification in the sequence, 1 is the first residue
                        pep.addModificationMatch(new ModificationMatch(ptm.getName(), true, ptmIndex));

                        usedPos.add(ptmIndex);
                    }

                }

            }
        }
        return(pep);

    }

    public synchronized void addOutputLine(String line){
        this.outLines.add(line);
    }

    public ArrayList<String> getOutLines(){
        return(this.outLines);
    }

    public synchronized void addOutAdditionalLines(String line){
        this.outAdditionalLines.add(line);
    }

    public ArrayList<String> getOutAdditionalLines(){
        return(this.outAdditionalLines);
    }


    public synchronized void addOutPeakAnnotations(String line){
        this.outPeakAnnotations.add(line);
    }

    public ArrayList<String> getOutPeakAnnotations(){
        return(this.outPeakAnnotations);
    }




}
