package main.java.pg;


import com.compomics.util.experiment.biology.*;
import com.compomics.util.experiment.identification.protein_sequences.digestion.IteratorFactory;
import com.compomics.util.experiment.identification.protein_sequences.digestion.PeptideWithPosition;
import com.compomics.util.experiment.identification.protein_sequences.digestion.SequenceIterator;
import com.compomics.util.experiment.massspectrometry.MSnSpectrum;
import com.compomics.util.preferences.DigestionPreferences;
import com.compomics.util.pride.CvTerm;
import net.sf.jfasta.FASTAElement;
import net.sf.jfasta.FASTAFileReader;
import net.sf.jfasta.impl.FASTAElementIterator;
import net.sf.jfasta.impl.FASTAFileReaderImpl;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.sql.*;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;

/**
 * search an MS/MS spectrum against a reference protein database
 */
public final class DatabaseInput {

    public String db = null;

    /**
     * fixed modification list: ptm name
     */
    public ArrayList<String> fixedModifications = new ArrayList<>();


    /**
     * variable modification list: ptm name
     */
    public ArrayList<String> varModifications = new ArrayList<String>();


    public void addFixedModifications(int id) {
        if (!fixedModifications.contains(ModificationGear.getInstance().getPTMname(id))) {
            fixedModifications.add(ModificationGear.getInstance().getPTMname(id));
        }
    }


    /**
     * add variable modification according to the int id of ptm
     * @param id
     */
    public void addVarModifications(int id) {
        if (!varModifications.contains(ModificationGear.getInstance().getPTMname(id))) {
            varModifications.add(ModificationGear.getInstance().getPTMname(id));
        }
    }

    /**
     * saved the spectra that will need to map to peptides
     */
    public HashMap<String,MSnSpectrum> mSnSpectrums = new HashMap<>();

    /**
     * key: spectrum title, value: an ArrayList of Peptide objects
     */
    public ConcurrentHashMap<String,ArrayList<Peptide>> ms2peptide = new ConcurrentHashMap<>();



    public DatabaseInput(String file){
        this.db = file;

    }

    /**
     * Set the enzyme and missed cleavage
     * @param enzymeName Enzyme name
     * @param enzymeMissedCleavages The max missed cleavage
     * @return
     */
    public static DigestionPreferences getDigestionPreferences(String enzymeName, int enzymeMissedCleavages) {
        DigestionPreferences digestionPreferences = new DigestionPreferences();
        digestionPreferences.setCleavagePreference(DigestionPreferences.CleavagePreference.enzyme);
        Enzyme trypsin = EnzymeFactory.getInstance().getEnzyme(enzymeName);
        digestionPreferences.addEnzyme(trypsin);
        digestionPreferences.setnMissedCleavages(enzymeName, enzymeMissedCleavages);
        return digestionPreferences;
    }

    /**
     * In default, using trypsin as the enzyme with up to 2 missed cleavage.
     * @return
     */
    public static DigestionPreferences getDigestionPreferences(){
        return(getDigestionPreferences("Trypsin",2));
    }


    /**
     * Get the enzyme object according to enzyme index
     * @param ind enzyme index, 1-based index.
     * @return An enzyme object
     */
    public static Enzyme getEnzymeByIndex(int ind){

        ArrayList<Enzyme> enzymes = new ArrayList<>();

        // 1
        Enzyme enzyme = new Enzyme("Trypsin");
        enzyme.addAminoAcidBefore('R');
        enzyme.addAminoAcidBefore('K');
        enzyme.addRestrictionAfter('P');
        enzyme.setCvTerm(new CvTerm("PSI-MS", "MS:1001251", "Trypsin", null));
        enzymes.add(enzyme);

        // 2
        enzyme = new Enzyme("Trypsin (no P rule)");
        enzyme.addAminoAcidBefore('R');
        enzyme.addAminoAcidBefore('K');
        enzyme.setCvTerm(new CvTerm("PSI-MS", "MS:1001313", "Trypsin/P", null));
        enzymes.add(enzyme);

        // 3
        enzyme = new Enzyme("Arg-C");
        enzyme.addAminoAcidBefore('R');
        enzyme.addRestrictionAfter('P');
        enzyme.setCvTerm(new CvTerm("PSI-MS", "MS:1001303", "Arg-C", null));
        enzymes.add(enzyme);

        // 4
        enzyme = new Enzyme("Arg-C (no P rule)");
        enzyme.addAminoAcidBefore('R');
        enzymes.add(enzyme);

        // 5
        enzyme = new Enzyme("Arg-N");
        enzyme.addAminoAcidAfter('R');
        enzymes.add(enzyme);

        // 6
        enzyme = new Enzyme("Glu-C");
        enzyme.addAminoAcidBefore('E');
        enzymes.add(enzyme);

        // 7
        enzyme = new Enzyme("Lys-C");
        enzyme.addAminoAcidBefore('K');
        enzyme.addRestrictionAfter('P');
        enzyme.setCvTerm(new CvTerm("PSI-MS", "MS:1001309", "Lys-C", null));
        enzymes.add(enzyme);

        // 8
        enzyme = new Enzyme("Lys-C (no P rule)");
        enzyme.addAminoAcidBefore('K');
        enzyme.setCvTerm(new CvTerm("PSI-MS", "MS:1001310", "Lys-C/P", null));
        enzymes.add(enzyme);

        // 9
        enzyme = new Enzyme("Lys-N");
        enzyme.addAminoAcidAfter('K');
        enzymes.add(enzyme);

        // 10
        enzyme = new Enzyme("Asp-N");
        enzyme.addAminoAcidAfter('D');
        enzyme.setCvTerm(new CvTerm("PSI-MS", "MS:1001304", "Asp-N", null));
        enzymes.add(enzyme);

        // 11
        enzyme = new Enzyme("Asp-N (ambic)");
        enzyme.addAminoAcidAfter('D');
        enzyme.addAminoAcidAfter('E');
        enzyme.setCvTerm(new CvTerm("PSI-MS", "MS:1001305", "Asp-N_ambic", null));
        enzymes.add(enzyme);

        // 12
        enzyme = new Enzyme("Chymotrypsin");
        enzyme.addAminoAcidBefore('F');
        enzyme.addAminoAcidBefore('Y');
        enzyme.addAminoAcidBefore('W');
        enzyme.addAminoAcidBefore('L');
        enzyme.addRestrictionAfter('P');
        enzyme.setCvTerm(new CvTerm("PSI-MS", "MS:1001306", "Chymotrypsin", null));
        enzymes.add(enzyme);

        // 13
        enzyme = new Enzyme("Chymotrypsin (no P rule)");
        enzyme.addAminoAcidBefore('F');
        enzyme.addAminoAcidBefore('Y');
        enzyme.addAminoAcidBefore('W');
        enzyme.addAminoAcidBefore('L');
        enzymes.add(enzyme);

        // 14
        enzyme = new Enzyme("Pepsin A");
        enzyme.addAminoAcidBefore('F');
        enzyme.addAminoAcidBefore('L');
        enzyme.setCvTerm(new CvTerm("PSI-MS", "MS:1001311", "Pepsin A", null));
        enzymes.add(enzyme);

        // 15
        enzyme = new Enzyme("CNBr");
        enzyme.addAminoAcidBefore('M');
        enzyme.setCvTerm(new CvTerm("PSI-MS", "MS:1001307", "CNBr", null));
        enzymes.add(enzyme);

        // 16
        enzyme = new Enzyme("Thermolysin");
        enzyme.addAminoAcidAfter('A');
        enzyme.addAminoAcidAfter('F');
        enzyme.addAminoAcidAfter('I');
        enzyme.addAminoAcidAfter('L');
        enzyme.addAminoAcidAfter('M');
        enzyme.addAminoAcidAfter('V');
        enzymes.add(enzyme);

        // 17
        enzyme = new Enzyme("LysargiNase");
        enzyme.addAminoAcidAfter('R');
        enzyme.addAminoAcidAfter('K');
        enzymes.add(enzyme);

        if(ind<=0 || ind > enzymes.size()){
            System.err.println("Please provide a valid enzyme number:"+ind);
            System.exit(0);
        }
        System.out.println("Use enzyme:"+enzymes.get(ind-1).getName());

        return(enzymes.get(ind-1));
    }


    public void readDB() throws IOException, InterruptedException, SQLException, ClassNotFoundException {
        File dbFile = new File(this.db);
        FASTAFileReader reader = new FASTAFileReaderImpl(dbFile);
        FASTAElementIterator it = reader.getIterator();

        // variable modifications
        PTMFactory ptmFactory = PTMFactory.getInstance();
        ArrayList<PTM> varPTMs = new ArrayList<>();
        for(String ptmName: varModifications){
            varPTMs.add(ptmFactory.getPTM(ptmName));
        }

        // digest protein
        //DigestionPreferences digestionPreferences = DigestionPreferences.getDefaultPreferences();
        Enzyme enzyme = getEnzymeByIndex(CParameter.enzyme);
        DigestionPreferences digestionPreferences = getDigestionPreferences(enzyme.getName(),CParameter.maxMissedCleavages);
        IteratorFactory iteratorModifications = new IteratorFactory(fixedModifications);

        System.out.println("Enzyme: "+enzyme.getName()+", maxMissedCleavages: "+CParameter.maxMissedCleavages);

        // prepare spectrum index for fast peptide query
        HashMap<Integer,ArrayList<String>> spectraIndex = new HashMap<>();
        // step: 0.1
        for(String title : SpectraInput.spectraMap.keySet()){
            MSnSpectrum mSnSpectrum = SpectraInput.spectraMap.get(title);
            double mass = mSnSpectrum.getPrecursor().getMass(mSnSpectrum.getPrecursor().getPossibleCharges().get(0).value);
            int va = (int) Math.round(10.0*mass);
            if(spectraIndex.containsKey(va)){
                spectraIndex.get(va).add(mSnSpectrum.getSpectrumTitle());
            }else{
                ArrayList<String> spectra_titles = new ArrayList<>();
                spectra_titles.add(mSnSpectrum.getSpectrumTitle());
                spectraIndex.put(va,spectra_titles);
            }
        }

        // if a peptide is searched before, then it will be omitted.
        HashSet<String> searchedPeptides = new HashSet<>();

        int num = 0;
        PeptideWithPosition peptideWithPosition;

        MSnSpectrum spectrum;

        double peptideMass;
        int va;
        double mass;
        double del;

        while (it.hasNext()) {
            FASTAElement el = it.next();
            el.setLineLength(1);
            String headLine[] = el.getHeader().split("\\s+");
            String proID = headLine[0];
            num++;
            // System.out.println(proID);
            String proSeq = el.getSequence().toUpperCase();

            proSeq = proSeq.replaceAll("\\*","");

            //fixedThreadPool.execute(new DBdigestProteinWorker(spectraIndex, proID, proSeq, digestionPreferences, ms2peptide, digestedPeptides));
            //SequenceIterator sequenceIterator = iteratorModifications.getSequenceIterator(proSeq, digestionPreferences, 600.0, 5000.0);

            SequenceIterator sequenceIterator = iteratorModifications.getSequenceIterator(proSeq, digestionPreferences, CParameter.minPeptideMass, CParameter.maxPeptideMass);



            while ((peptideWithPosition = sequenceIterator.getNextPeptide()) != null) {

                Peptide peptide = peptideWithPosition.getPeptide();


                if (peptide.getSequence().length() < CParameter.minPeptideLength || peptide.getSequence().length() > CParameter.maxPeptideLength) {
                    continue;
                }

                if(searchedPeptides.contains(peptide.getSequence())){
                    continue;
                }else{
                    searchedPeptides.add(peptide.getSequence());
                }

                peptideMass = peptide.getMass();
                va = (int) Math.round(10.0*peptideMass);

                for(int i=(va-1);i<=(va+1);i++){
                    if(spectraIndex.containsKey(i)){
                        for(String title : spectraIndex.get(i)){
                            spectrum = SpectraInput.spectraMap.get(title);
                            mass = spectrum.getPrecursor().getMass(spectrum.getPrecursor().getPossibleCharges().get(0).value);
                            del = Math.abs(peptideMass - mass);
                            if (CParameter.tolu) {
                                del = (1.0e6) * 1.0 * del / peptideMass;
                            }
                            if (del <= CParameter.tol) {
                                if(ms2peptide.containsKey(title)){
                                    ms2peptide.get(title).add(peptide);
                                }else{
                                    ArrayList<Peptide> peps = new ArrayList<>();
                                    peps.add(peptide);
                                    ms2peptide.put(title,peps);
                                }
                            }

                        }
                    }
                }

                // Consider modification
                ArrayList<Peptide> peptideIsoforms = PeptideInput.calcPeptideIsoforms(peptide,varPTMs,CParameter.maxVarMods);
                peptideIsoforms.add(peptide);
                for(Peptide p: peptideIsoforms){
                    peptideMass = p.getMass();
                    va = (int) Math.round(10.0*peptideMass);

                    for(int i=(va-1);i<=(va+1);i++){
                        if(spectraIndex.containsKey(i)){
                            for(String title : spectraIndex.get(i)){
                                spectrum = SpectraInput.spectraMap.get(title);
                                mass = spectrum.getPrecursor().getMass(spectrum.getPrecursor().getPossibleCharges().get(0).value);
                                del = Math.abs(peptideMass - mass);
                                if (CParameter.tolu) {
                                    del = (1.0e6) * 1.0 * del / peptideMass;
                                }
                                if (del <= CParameter.tol) {
                                    if(ms2peptide.containsKey(title)){
                                        ms2peptide.get(title).add(p);
                                    }else{
                                        ArrayList<Peptide> peps = new ArrayList<>();
                                        peps.add(p);
                                        ms2peptide.put(title,peps);
                                    }
                                }

                            }
                        }
                    }
                }

            }


            if( (num%1000)==0){
                System.out.println("Finished proteins:"+num);

            }


        }
        reader.close();
        spectraIndex = null;
        System.out.println("Protein sequences:"+num);

    }


    /**
     * Calculate the left and right values for specified tol.
     * @param msMass Mass of MS/MS spectrum
     * @param tol tol
     * @param isPpm true if the unit of tol is ppm
     * @return
     */
    public double[] getRangeOfMass(double msMass, double tol, boolean isPpm){
        double [] massRange = new double [2];
        if(isPpm){
            massRange[0] = 1.0*msMass/(1+1.0*tol/(1.0e6));
            massRange[1] = 1.0*msMass/(1-1.0*tol/(1.0e6));

        }else{
            massRange[0] = msMass-tol;
            massRange[1] = msMass+tol;
        }
        return massRange;
    }


    public ArrayList<Peptide> getPeptideFromSQL(ResultSet rs) throws SQLException, IOException, ClassNotFoundException {
        ArrayList<Peptide> peptides = new ArrayList<>();
        HashSet<String> pepmod = new HashSet<>();
        String pmod;
        while(rs.next()){
            pmod = rs.getString("pep_mod");
            if(pepmod.contains(pmod)){
                continue;
            }else{
                pepmod.add(pmod);
            }
            byte buf[] = rs.getBytes("peptide");
            ObjectInputStream objectInputStream = new ObjectInputStream(new ByteArrayInputStream(buf));
            Peptide p = (Peptide) objectInputStream.readObject();
            peptides.add(p);

        }

        return peptides;

    }


    public void readDBSQL() throws IOException, SQLException, ClassNotFoundException {

        // read digested peptides from SQL database
        Connection connection = DriverManager.getConnection("jdbc:sqlite:" + this.db);
        PreparedStatement pstmt = connection.prepareStatement("select sequence,peptide,pep_mod from prodb where mass >= ? and mass <= ?");
        String psql;
        int npep = 0;
        MSnSpectrum spectrum;
        for(String title : SpectraInput.spectraMap.keySet()){
            spectrum = SpectraInput.spectraMap.get(title);
            double mass = spectrum.getPrecursor().getMass(spectrum.getPrecursor().getPossibleCharges().get(0).value);
            double massRange[] = getRangeOfMass(mass,CParameter.tol,CParameter.tolu);
            pstmt.setDouble(1,massRange[0]);
            pstmt.setDouble(2,massRange[1]);
            ResultSet rs    = pstmt.executeQuery();

            ArrayList<Peptide> rsPeptide = getPeptideFromSQL(rs);
            npep = npep + rsPeptide.size();
            ms2peptide.put(spectrum.getSpectrumTitle(),rsPeptide);
        }
        pstmt.close();
        connection.close();
        System.out.println("Peptides from reference database:"+npep);



    }


    public Comparator<MSnSpectrum> comparator = new Comparator<MSnSpectrum>() {
        public int compare(MSnSpectrum s1, MSnSpectrum s2) {
            if (s2.getPrecursor().getMass(s2.getPrecursor().getPossibleCharges().get(0).value) >
                    s1.getPrecursor().getMass(s1.getPrecursor().getPossibleCharges().get(0).value)) {
                return -1;
            } else if (s2.getPrecursor().getMass(s2.getPrecursor().getPossibleCharges().get(0).value) ==
                    s1.getPrecursor().getMass(s1.getPrecursor().getPossibleCharges().get(0).value)) {
                return 0;
            } else {
                return 1;
            }

        }
    };





}
