package main.java.OpenModificationSearch;

import com.compomics.util.experiment.biology.*;
import com.compomics.util.experiment.identification.matches.ModificationMatch;
import com.compomics.util.experiment.massspectrometry.MSnSpectrum;
import com.compomics.util.experiment.massspectrometry.SpectrumFactory;
import com.compomics.util.preferences.DigestionPreferences;
import com.compomics.util.preferences.SequenceMatchingPreferences;

import main.java.pg.CParameter;
import main.java.pg.DatabaseInput;
import main.java.pg.PeptideInput;
import main.java.pg.SpectraInput;
import main.java.PSMMatch.JPeptide;
import main.java.PSMMatch.ScoreResult;

import main.java.PSMMatch.HyperscoreMatch;
import net.sf.jfasta.FASTAElement;
import net.sf.jfasta.FASTAFileReader;
import net.sf.jfasta.impl.FASTAElementIterator;
import net.sf.jfasta.impl.FASTAFileReaderImpl;
import org.apache.commons.lang3.StringUtils;
import org.paukov.combinatorics3.Generator;
import uk.ac.ebi.jmzml.xml.io.MzMLUnmarshallerException;
import uk.ac.ebi.pride.utilities.pridemod.io.unimod.model.Specificity;
import uk.ac.ebi.pride.utilities.pridemod.io.unimod.model.Unimod;
import uk.ac.ebi.pride.utilities.pridemod.io.unimod.model.UnimodModification;
import uk.ac.ebi.pride.utilities.pridemod.io.unimod.xml.UnimodReader;

import javax.xml.bind.JAXBException;

import java.io.*;
import java.sql.*;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;



/**
 * 1. Build peptide database including peptide sequences and fixed modifications
 * 2. Build index for modifications
 *
 * Query method:
 * Define a windows (250 Da), get all the peptides with mass error less than the defined window compared with query
 * spectrum. Only consider the peptides with mass error less than pre-defined mass error (for example 10 ppm). For this
 * search, currently consider one variable modification for each PSM.
 */
public class ModificationDB {

    private static boolean debug = false;

    /**
     * Integer value of a mass
     */
    public static HashMap<Integer, Integer> mass2index = new HashMap<>();
    public static HashMap<Integer, Integer> mass2indexUp = new HashMap<>();



    private static SequenceMatchingPreferences sequenceMatchingPreferences = new SequenceMatchingPreferences();
    private static SequenceMatchingPreferences ptmSequenceMatchingPreferences = new SequenceMatchingPreferences();



    private static PTMFactory ptmFactory = PTMFactory.getInstance();
    private static HashMap<String, String> titleTFullName = new HashMap<>();

    /**
     * Store PTMs that will be considered in peptide matching
     */
    public static ArrayList<PTM> ptms = new ArrayList<>();
    public static boolean isPTMsort = false;


    /**
     * Index for modification searching
     * key=Integer value of a mass, value=all modifications to this key
     */
    private static HashMap<Long,ArrayList<PTM>> ptmIndex = new HashMap<>();


    private static int offset = 200;

    public static double leftMass = -250.0;
    public static double rightMass = 250.0;


    public static void main(String[] args) throws IOException, MzMLUnmarshallerException {

        long startTime=System.currentTimeMillis();

        ModificationDB.importPTMsFromUnimod();

        String mgf = args[0];
        String db = args[1];
        String title = args[2];
        int cpu = Integer.valueOf(args[3]);

        // Read spectra from mgf file
        File mgfFile = new File(mgf);
        SpectrumFactory spectrumFactory = SpectrumFactory.getInstance();
        spectrumFactory.addSpectra(mgfFile, null);

        MSnSpectrum spectrum = (MSnSpectrum) spectrumFactory.getSpectrum(mgfFile.getName(),title);

        double precursorMass = spectrum.getPrecursor().getMass(spectrum.getPrecursor().getPossibleCharges().get(0).value);

        if(debug) {
            System.out.println("precursorMass:" + precursorMass);
        }


        ArrayList<String> fixMods = new ArrayList<>();
        HashMap<Long,PeptideIndex> peptideIndexMap = buildPeptideDB(db,fixMods);

        ModPepQueryAndScoringWorker.fixMods = new ArrayList<>();
        ModificationDB.getCandidateRefPeptides(spectrum,peptideIndexMap,args[4],cpu);

        long endTime=System.currentTimeMillis();

        System.out.println("Used time: "+(endTime-startTime)/1000+"s");

    }


    private static ArrayList<PTM> getCandidatePTMs(double mass, double precursorMass){

        ArrayList<PTM> result = new ArrayList<>();

        long ptmmass = Math.round(mass);
        if(ptmIndex.containsKey(ptmmass) || ptmIndex.containsKey(ptmmass-1) || ptmIndex.containsKey(ptmmass+1)) {
            // left: 1Da, right: 1Da
            for(long i=ptmmass-1;i<=ptmmass+1;i++){
                if(ptmIndex.containsKey(i)){
                    for(PTM ptm:ptmIndex.get(i)){
                        double mdel = mass - ptm.getMass();
                        if (CParameter.tolu) {
                            mdel = mdel / precursorMass * 1000000.0;
                        }

                        if (Math.abs(mdel) <= CParameter.tol) {
                            result.add(ptm);
                        }
                    }
                }
            }
        }
        return result;

    }

    static ArrayList<Peptide> getCandidatePTMs(MSnSpectrum spectrum, Peptide peptide){

        ArrayList<Peptide> modPeptides = new ArrayList<>();

        double precursorMass = spectrum.getPrecursor().getMass(spectrum.getPrecursor().getPossibleCharges().get(0).value);
        double mass = peptide.getMass();
        double delmass = precursorMass - mass;
        ArrayList<PTM> rptms = getCandidatePTMs(delmass,precursorMass);
        Iterator<PTM> iterator = rptms.iterator();
        while(iterator.hasNext()){
            PTM ptm = iterator.next();
            if(ptm.getType() == PTM.MODN || ptm.getType() == PTM.MODNAA ||
                    ptm.getType() == PTM.MODC || ptm.getType() == PTM.MODCAA){

                iterator.remove();
            }else{
                ArrayList<Integer> possibleSites = null;
                //System.out.println(ptm.getName());
                try {
                    possibleSites = peptide.getPotentialModificationSites(ptm, sequenceMatchingPreferences, ptmSequenceMatchingPreferences);
                } catch (IOException e) {
                    e.printStackTrace();
                } catch (InterruptedException e) {
                    e.printStackTrace();
                } catch (ClassNotFoundException e) {
                    e.printStackTrace();
                } catch (SQLException e) {
                    e.printStackTrace();
                }

                if(possibleSites != null && possibleSites.size()>=1) {

                    HashMap<Integer, PTM> position2ptm = new HashMap<>();

                    for (Integer i : possibleSites) {
                        position2ptm.put(i, ptm);
                    }

                    if (position2ptm.size() >= 1) {
                        // >=1 possible modification site
                        Set<Integer> ptmPosition = position2ptm.keySet();
                        // the max number of modifications to consider
                        Iterator<List<Integer>> iter = Generator.combination(ptmPosition).simple(1).iterator();
                        while (iter.hasNext()) {
                            List<Integer> iCombination = iter.next();
                            Peptide modPeptide = new Peptide(peptide.getSequence(), peptide.getModificationMatches());
                            for (Integer pos : iCombination) {
                                modPeptide.addModificationMatch(new ModificationMatch(position2ptm.get(pos).getName(), true, pos));
                            }
                            modPeptides.add(modPeptide);

                        }
                    }
                }

            }



        }
        return(modPeptides);

    }




    /**
     * Import modification information from Unimod.
     */
    public static void importPTMsFromUnimod(){
        //ptmFactory.clearFactory();
        ArrayList<String > residues;

        ArrayList<String> testSite = new ArrayList<>();
        ArrayList<String> testPosition = new ArrayList<>();

        InputStream inputStream = ModificationDB.class.getResourceAsStream("/main/java/OpenModificationSearch/unimod.xml");

        UnimodReader unimodreader = null;
        try {
            unimodreader = new UnimodReader(inputStream);
        } catch (JAXBException e) {
            e.printStackTrace();
        }

        Unimod unimod = unimodreader.getUnimodObject();

        List<UnimodModification> unimodModificationList = unimod.getModifications().getMod();

        System.out.println("All modifications in unimod:"+unimodModificationList.size());

        HashMap<String,Integer> modificationClassMap = new HashMap<>();

        //Parsing and get each modification;
        int i=0;
        for (UnimodModification unimodModification: unimodModificationList){

            Double monoMass = unimodModification.getDelta().getMonoMass().doubleValue();//Get modification Mass

            if(monoMass < leftMass || monoMass > rightMass){
                continue;
            }

            String modificationTitle = unimodModification.getTitle();
            String modificationName = unimodModification.getFullName();
            titleTFullName.put(modificationTitle, modificationName);

            //Get amino acid modification detailed modification;
            List<Specificity> specificityList = unimodModification.getSpecificity();
            //String fullName = unimodModification.getFullName();

            for(Specificity specificity: specificityList){
                String site = specificity.getSite();
                String position = specificity.getPosition();
                String classification = specificity.getClassification();

                // Only importing these modifications when performing target protein (known) identification. But it's optional.
                // There is a global parameter in CParameter "addAAsubstitutionMods" to control this.
                // Only when CParameter.addAAsubstitutionMods is false, then it will be ignored.
                if(classification.equalsIgnoreCase("AA substitution") && !CParameter.addAAsubstitutionMods){
                    continue;
                }
                //System.out.println(classification);

                if(modificationClassMap.containsKey(classification)){
                    modificationClassMap.put(classification,modificationClassMap.get(classification)+1);
                }else{
                    modificationClassMap.put(classification,1);
                }

                PTM ptm = null;
                String ptmName;
                i++;

                if(site.equals("N-term")){

                    if(position.equals("Any N-term")){
                        ptmName = modificationTitle + " of " + site;
                        ptm = new PTM(PTM.MODNP, ptmName, monoMass, null);
                        ptm.setShortName(modificationTitle);
                    }else if(position.equals("Protein N-term")){
                        ptmName = modificationTitle + " of protein " + site;
                        ptm = new PTM(PTM.MODN, ptmName, monoMass, null);
                        ptm.setShortName(modificationTitle);
                    }else{

                        if(debug) {

                            System.out.println("position:" + position);
                            System.out.println("mass:" + monoMass);
                            System.out.println("site:" + site);
                        }
                    }
                }else if(site.equals("C-term")){
                    if(position.equals("Any C-term")){
                        ptmName = modificationTitle + " of " + site;
                        ptm = new PTM(PTM.MODCP, ptmName, monoMass, null);
                        ptm.setShortName(modificationTitle);
                    }else if(position.equals("Protein C-term")){
                        ptmName = modificationTitle + " of protein " + site;
                        ptm = new PTM(PTM.MODC, ptmName, monoMass, null);
                        ptm.setShortName(modificationTitle);
                    }
                }else {
                    residues = new ArrayList<>();
                    residues.add(site);
                    if(position.equals("Any N-term")){
                        ptmName = modificationTitle + " of " + site;
                        ptm = new PTM(PTM.MODNPAA, ptmName, monoMass, residues);
                        ptm.setShortName(modificationTitle);
                    }else if(position.equals("Protein N-term")){
                        ptmName = modificationTitle + " of protein " + site;
                        ptm = new PTM(PTM.MODNAA, ptmName, monoMass, residues);
                        ptm.setShortName(modificationTitle);
                    }else if (position.equals("Any C-term")){
                        ptmName = modificationTitle + " of " + site;
                        ptm = new PTM(PTM.MODCPAA, ptmName, monoMass, residues);
                        ptm.setShortName(modificationTitle);
                    }else if(position.equals("Protein C-term")){
                        ptmName = modificationTitle + " of protein " + site;
                        ptm = new PTM(PTM.MODCAA, ptmName, monoMass, residues);
                        ptm.setShortName(modificationTitle);
                    }else {
                        ptmName = modificationTitle + " of " + site;
                        ptm = new PTM(PTM.MODAA, ptmName, monoMass, residues);
                        ptm.setShortName(modificationTitle);
                    }
                }

                //System.out.println("The ptm name is "+ptmName+" and mass is "+ptm.getMass());

                if(ptm!=null) {
                    ptms.add(ptm);
                    ptmFactory.addUserPTM(ptm);

                    // Build index for modification search
                    long ptmmass = Math.round(monoMass);
                    if(ptmIndex.containsKey(ptmmass)){
                        ptmIndex.get(ptmmass).add(ptm);
                    }else{
                        ArrayList<PTM> pList = new ArrayList<>();
                        pList.add(ptm);
                        ptmIndex.put(ptmmass,pList);
                    }
                }
                //System.out.println(ptm.getMass());

                if(!testSite.contains(site)){
                    testSite.add(site);
                }
                if(!testPosition.contains(position)){
                    testPosition.add(position);
                }

            }
        }

        if(debug) {

            int total_mod = 0;
            for (String classification : modificationClassMap.keySet()) {
                System.out.println(classification + "\t" + modificationClassMap.get(classification));
                total_mod = total_mod + modificationClassMap.get(classification);
            }

            System.out.println("Total modifications imported:" + ModificationDB.ptms.size());
        }

    }


    /**
     *
     */
    private static Comparator<PTM> comparator = new Comparator<PTM>() {

        public int compare(PTM s1, PTM s2) {
            if (s1.getMass() < s2.getMass()) {
                return -1;
            } else if (s1.getMass() > s2.getMass()) {
                return 1;
            } else {
                return 0;
            }
        }

    };


    private static double getMinModificationMass(){
        double mass = 0.0;
        if(ModificationDB.ptms.size()>=1){
            if(ModificationDB.isPTMsort){
                mass = ModificationDB.ptms.get(0).getMass();
            }else{
                Collections.sort(ModificationDB.ptms, comparator);
                ModificationDB.isPTMsort = true;
                mass = ModificationDB.ptms.get(0).getMass();
            }
        }else{
            System.err.println("There is not modification!");
            System.exit(0);
        }
        return(mass);
    }


    private static HashMap<Long,PeptideIndex> buildPeptideDB(String db, ArrayList<String> fixedModifications){

        long startTime=System.currentTimeMillis();

        HashMap<Long,PeptideIndex> peptideIndexMap = new HashMap<>();

        String pepIndex_file = db+ ".pepIndex";
        File PEPInd = new File(pepIndex_file);
        boolean useIndex = false;
        if(PEPInd.isFile() && useIndex){

            System.out.println("Read peptide index file: "+pepIndex_file);

            // read index object
            try {
                FileInputStream pepindex_fi = new FileInputStream(new File(pepIndex_file));
                ObjectInputStream pepindex_oi = new ObjectInputStream(pepindex_fi);
                peptideIndexMap = (HashMap<Long,PeptideIndex>) pepindex_oi.readObject();

            } catch (FileNotFoundException e) {
                e.printStackTrace();
            } catch (IOException e) {
                e.printStackTrace();
            } catch (ClassNotFoundException e) {
                e.printStackTrace();
            }
        }else {

            ////////////////////////////////////////////////////////////////////////////////////////////////////////////////
            // read protein database
            File dbFile = new File(db);
            FASTAFileReader reader = null;
            try {
                reader = new FASTAFileReaderImpl(dbFile);
            } catch (IOException e) {
                e.printStackTrace();
            }
            FASTAElementIterator it = null;
            try {
                it = reader.getIterator();
            } catch (IOException e) {
                e.printStackTrace();
            }

            // digest protein
            Enzyme enzyme = DatabaseInput.getEnzymeByIndex(CParameter.enzyme);
            DigestionPreferences digestionPreferences = DatabaseInput.getDigestionPreferences(enzyme.getName(),CParameter.maxMissedCleavages);

            //DigestionPreferences digestionPreferences = DigestionPreferences.getDefaultPreferences();
            // if a peptide is searched before, then it will be omitted.
            HashSet<String> searchedPeptides = new HashSet<>();


            int cpu = Runtime.getRuntime().availableProcessors();
            ExecutorService fixedThreadPool = Executors.newFixedThreadPool(cpu);

            ConcurrentHashMap<String, HashMap<String, Double>> pro2pep = new ConcurrentHashMap<>();

            int num = 0;
            try {
                while (it.hasNext()) {
                    FASTAElement el = it.next();
                    el.setLineLength(1);
                    String headLine[] = el.getHeader().split("\\s+");
                    String proID = headLine[0];
                    num++;
                    String proSeq = el.getSequence();
                    // digest
                    fixedThreadPool.execute(new DigestProteinWorker(proID, proSeq, fixedModifications, digestionPreferences, pro2pep));

                }
            } catch (IOException e) {
                e.printStackTrace();
            }

            fixedThreadPool.shutdown();

            try {
                fixedThreadPool.awaitTermination(Long.MAX_VALUE, TimeUnit.HOURS);
            } catch (InterruptedException e) {
                e.printStackTrace();
            }

            reader.close();
            System.out.println("Protein sequences:" + num);


            for (String proID : pro2pep.keySet()) {
                for (String pep : pro2pep.get(proID).keySet()) { //at main.java.OpenModificationSearch.ModificationDB.buildPeptideDB(ModificationDB.java:874) Exception in thread "main" java.lang.NullPointerException
                    if (!searchedPeptides.contains(pep)) {
                        searchedPeptides.add(pep);
                        double mass = pro2pep.get(proID).get(pep);
                        long intValue = Math.round(mass);
                        if (peptideIndexMap.containsKey(intValue)) {
                            peptideIndexMap.get(intValue).peptideSequences.add(pep);
                            peptideIndexMap.get(intValue).peptideMasses.add(mass);
                        } else {
                            PeptideIndex peptideIndex = new PeptideIndex(pep, mass);
                            peptideIndexMap.put(intValue, peptideIndex);

                        }
                    }
                }
            }

            long bTime = System.currentTimeMillis();
            System.out.println("Total unique peptides: "+searchedPeptides.size());
            System.out.println("Time used for building peptide index:" + (bTime - startTime) / 1000 + " s.");


        }

        long endTime = System.currentTimeMillis();

        System.out.println("Time used for peptide indexing:" + (endTime - startTime) / 1000 + " s.");

        return(peptideIndexMap);
    }

    private static ConcurrentHashMap<Long, ArrayList<ScoreResult>> getCandidateRefPeptides(MSnSpectrum spectrum, HashMap<Long,PeptideIndex> peptideIndexMap, String targetPeptideSeq ,int cpu) {

        if(leftMass < getMinModificationMass()){
            leftMass = getMinModificationMass();
        }

        if(debug) {
            System.out.println("left mass:" + leftMass);
            System.out.println("left mass:" + rightMass);
        }
        double precursorMass = spectrum.getPrecursor().getMass(spectrum.getPrecursor().getPossibleCharges().get(0).value);
        if(debug) {
            System.out.println("precursorMass=" + precursorMass + ",leftMass=" + leftMass + ",rightMass=" + rightMass);
        }

        HyperscoreMatch.generateFactorialValues(60);


        long maxIntValue = Math.round(precursorMass-leftMass+2);
        long minIntValue = Math.round(precursorMass-rightMass-2);


        ConcurrentHashMap<Long,ArrayList<ScoreResult>> res = new ConcurrentHashMap<>();


        if(cpu!=1) {
            if (cpu == 0) {
                cpu = Runtime.getRuntime().availableProcessors();
            }
            //System.out.println("CPU:" + cpu);
            ExecutorService fixedThreadPool = Executors.newFixedThreadPool(cpu);

            //int ind = 1;

            for (long i = minIntValue; i <= maxIntValue; i++) {
                ArrayList<ScoreResult> sList = new ArrayList<>();
                res.put(i,sList);
                //
                fixedThreadPool.execute(new ModPepQueryAndScoringWorker(i,spectrum, peptideIndexMap,res,targetPeptideSeq));
            }

            fixedThreadPool.shutdown();

            try {
                fixedThreadPool.awaitTermination(Long.MAX_VALUE, TimeUnit.HOURS);
            } catch (InterruptedException e) {
                e.printStackTrace();
            }
        }

        return(res);

    }


    public static String doPTMValidation(String psmfile, ArrayList<PeptideInput> peptideInputs, String db, ArrayList<String> fixMods, String outdir){

        long startTime = System.currentTimeMillis();

        // read PSM file and only perform PTM validation for confident PSMs
        BufferedReader bReader = null;
        try {
            bReader = new BufferedReader(new FileReader(new File(psmfile)));
        } catch (FileNotFoundException e) {
            e.printStackTrace();
        }
        String line = null;
        try {
            line = bReader.readLine().trim();
        } catch (IOException e) {
            e.printStackTrace();
        }
        String headLine[] = line.split("\t");
        HashMap<String,Integer> headMap = new HashMap<>();
        for(int i=0;i<headLine.length;i++){
            headMap.put(headLine[i],i);
        }


        HashMap<String,MSnSpectrum> spectraMap = new HashMap<>();

        HashMap<String,String> spectra2pep = new HashMap<>();

        try {
            while((line = bReader.readLine())!=null){
                line = line.trim();
                String d[] = line.split("\t");
                String spectrum_title  = d[headMap.get("spectrum_title")];
                String peptideSequence = d[headMap.get("peptide")];

                double pvalue = Double.valueOf(d[headMap.get("pvalue")]);
                int rank = Integer.valueOf(d[headMap.get("rank")]);
                int n_db = Integer.valueOf(d[headMap.get("n_db")]);

                if(pvalue <= 0.01 && n_db == 0 && rank == 1){
                    spectraMap.put(spectrum_title,new MSnSpectrum());
                    spectra2pep.put(spectrum_title,peptideSequence);
                }

            }
        } catch (IOException e) {
            e.printStackTrace();
        }

        try {
            bReader.close();
        } catch (IOException e) {
            e.printStackTrace();
        }

        String outfile = outdir + "/ptm.txt";

        if(spectraMap.size()>=1) {

            for (PeptideInput peptideInput : peptideInputs) {
                if (peptideInput.getOutLines().size() >= 1) {
                    //String peptideSequence = peptideInput.jPeptide.peptide.getSequence();
                    //for (MSnSpectrum spectrum : peptideInput.jPeptide.mSnSpectrums) {
                    for (String title : peptideInput.jPeptide.spectraIndexs) {
                        MSnSpectrum spectrum = SpectraInput.spectraMap.get(title);
                        if (spectraMap.containsKey(spectrum.getSpectrumTitle())) {
                            spectraMap.put(spectrum.getSpectrumTitle(), spectrum);
                        }
                    }

                    for (JPeptide jPeptide : peptideInput.getPtmIsoforms()) {
                        //for (MSnSpectrum spectrum : jPeptide.mSnSpectrums) {
                        for (String title : jPeptide.spectraIndexs) {
                            MSnSpectrum spectrum = SpectraInput.spectraMap.get(title);
                            if (spectraMap.containsKey(spectrum.getSpectrumTitle())) {
                                spectraMap.put(spectrum.getSpectrumTitle(), spectrum);
                            }
                        }
                    }
                }
            }

            System.out.println("The number of spectra that need to be validated by PTM searching: " + spectraMap.size());

            ModificationDB.importPTMsFromUnimod();

            HashMap<Long, PeptideIndex> peptideIndexMap = buildPeptideDB(db, fixMods);




            BufferedWriter bWriter = null;
            try {
                bWriter = new BufferedWriter(new FileWriter(new File(outfile)));
            } catch (IOException e) {
                e.printStackTrace();
            }
            if(CParameter.scoreMethod==0) {
                try {
                    bWriter.write("spectrum_title\tpeptide\tcharge\texp_mass\tpep_mass\tmodification\tscore1\tscore2\n");
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }else{
                try {
                    bWriter.write("spectrum_title\tpeptide\tcharge\texp_mass\tpep_mass\tmodification\tscore\n");
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }

            // add fixed modification into peptide
            ModPepQueryAndScoringWorker.fixMods = fixMods;

            // Perform the search for each target spectrum.
            for (String title : spectraMap.keySet()) {

                ConcurrentHashMap<Long, ArrayList<ScoreResult>> res = ModificationDB.getCandidateRefPeptides(spectraMap.get(title), peptideIndexMap, spectra2pep.get(title), CParameter.cpu);

                for(long l : res.keySet()){
                    for(ScoreResult scoreResult: res.get(l)){
                        StringBuilder stringBuilder = new StringBuilder();
                        Peptide peptide = scoreResult.peptide;
                        double exp_mass = spectraMap.get(title).getPrecursor().getMass(spectraMap.get(title).getPrecursor().getPossibleCharges().get(0).value);
                        int charge = spectraMap.get(title).getPrecursor().getPossibleCharges().get(0).value;
                        double pep_mass = peptide.getMass();
                        if(CParameter.scoreMethod==0) {
                            stringBuilder.append(title).append("\t")
                                    .append(peptide.getSequence()).append("\t")
                                    .append(charge).append("\t")
                                    .append(getModificationString(peptide)).append("\t")
                                    .append(exp_mass).append("\t")
                                    .append(pep_mass).append("\t")
                                    .append(scoreResult.score).append("\t").append(scoreResult.score2).append("\n");
                        }else{
                            stringBuilder.append(title).append("\t")
                                    .append(peptide.getSequence()).append("\t")
                                    .append(charge).append("\t")
                                    .append(exp_mass).append("\t")
                                    .append(pep_mass).append("\t")
                                    .append(getModificationString(peptide)).append("\t").append(scoreResult.score).append("\n");
                        }

                        try {
                            bWriter.write(stringBuilder.toString());
                        } catch (IOException e) {
                            e.printStackTrace();
                        }
                    }
                }
            }

            try {
                bWriter.close();
            } catch (IOException e) {
                e.printStackTrace();
            }
        }

        long endTime = System.currentTimeMillis();

        System.out.println("Time used for PTM validation:" + (endTime - startTime) / 1000 + " s.");
        return(outfile);
    }

    public static String getModificationString(Peptide peptide){
        String mod = "-";
        ArrayList<ModificationMatch> modificationMatchs = peptide.getModificationMatches();
        if(modificationMatchs !=null && modificationMatchs.size()>=1){
            String d[] = new String[modificationMatchs.size()];
            for(int i=0;i<d.length;i++){
                PTM ptm = ptmFactory.getPTM(modificationMatchs.get(i).getTheoreticPtm());
                d[i] = ptm.getName()+"@"+modificationMatchs.get(i).getModificationSite()+"["+String.format("%.4f",ptm.getMass())+"]";
            }
            mod = StringUtils.join(d,';');
        }

        return(mod);
    }


    static void addFixedModification(ArrayList<String> fixMods, Peptide peptide){

        for (String mod : fixMods) {
            PTM ptm = ptmFactory.getPTM(mod);
            ArrayList<Integer> possibleSites = null;
            try {
                possibleSites = peptide.getPotentialModificationSites(ptm, sequenceMatchingPreferences, ptmSequenceMatchingPreferences);
            } catch (IOException e) {
                e.printStackTrace();
            } catch (InterruptedException e) {
                e.printStackTrace();
            } catch (ClassNotFoundException e) {
                e.printStackTrace();
            } catch (SQLException e) {
                e.printStackTrace();
            }
            for (Integer k : possibleSites) {
                peptide.addModificationMatch(new ModificationMatch(ptm.getName(),false,k));
            }
        }


    }

}
