function [Output_struct] = SpotOn_core(Params)
%SPOTON_CORE core SpotOn function 
%   The function takes the user-input and performs data processing and
%   function fitting

% This is Spot-On Matlab version 1.05. 
disp('--- Running Spot-On Matlab Version 1.05 ---');
disp('Please see <a href = "https://gitlab.com/tjian-darzacq-lab/spot-on-matlab">GitLab</a> for updates');
disp('Full documentation: <a href = "https://SpotOn.berkeley.edu/">Spot-On</a>');
disp('Please cite the <a href = "https://elifesciences.org/articles/33125">Spot-On paper</a>, if you use Spot-On in a publication');



% define neccesary global parameters:
global LocError dT HistVecJumps dZ HistVecJumpsCDF ModelFit FitLocError Z_corr_a Z_corr_b JumpsPerdT UseWeights
Output_struct = struct([]); % for saving outputs
figure1 = figure; colour = jet; close; % matlab is annoying with colormaps. So do this to avoid empty figures. 
% Get current screen size:
screen_size_vector = get(0,'ScreenSize');
% make sure the plots do not exceed the screen size:
plot_width = min([1200 screen_size_vector(3)]);
plot_height = min([800 screen_size_vector(4)]);

%%%%%%%%%%%%%%%%% Unpack the input structure Params: %%%%%%%%%%%%%%%%%%%%%%
TimeGap = Params.TimeGap; 
dZ = Params.dZ; 
dT = Params.TimeGap/1000; % convert to seconds
GapsAllowed = Params.GapsAllowed; 
TimePoints = Params.TimePoints; 
BinWidth = Params.BinWidth; 
UseEntireTraj = Params.UseEntireTraj;
JumpsToConsider = Params.JumpsToConsider; 
MaxJumpPlotPDF = Params.MaxJumpPlotPDF; 
MaxJumpPlotCDF = Params.MaxJumpPlotCDF; 
MaxJump = Params.MaxJump;
SavePlot = Params.SavePlot; 
ModelFit = Params.ModelFit;
DoSingleCellFit = Params.DoSingleCellFit; 
FitIterations = Params.FitIterations; 
FitLocErrorRange = Params.FitLocErrorRange; 
LocError = Params.LocError; 
FitLocError = Params.FitLocError;
NumberOfStates = Params.NumberOfStates;
D_Free_2State = Params.D_Free_2State; 
D_Bound_2State = Params.D_Bound_2State; 
D_Free1_3State = Params.D_Free1_3State; 
D_Free2_3State = Params.D_Free2_3State; 
D_Bound_3State = Params.D_Bound_3State;
curr_dir = Params.curr_dir; 
data_struct = Params.data_struct; 
SampleName = Params.SampleName;
UseWeights = Params.UseWeights;
DoPlots = Params.DoPlots;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%% If saving plots, make sure that the folder exists %%%%%%%%%%%%%
save_path = [curr_dir, filesep, 'SavedPlots', filesep];
if ~exist(save_path, 'dir')
    % the directory is not there, so make it
    mkdir(save_path)
end
if ModelFit ~=1 
    if ModelFit ~=2
        error(['ModelFit variable was set to ', num2str(ModelFit), '; Please choose either =1 for PDF-fitting or =2 for CDF-fitting']);
    end
end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%%%%%%%%% Define some additional variables %%%%%%%%%%%%%%%%%%%%%%%
HistVecJumps = 0:BinWidth:MaxJump; % histogram/PDF displacement bins in micrometers
HistVecJumpsCDF = 0:0.001:MaxJump; % CDF displacement bins in micrometers
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%%%%%%%%%%%%% PARAMETERS FOR Z-CORRECTION %%%%%%%%%%%%%%%%%%%%%%%%
% find the correct parameters for z-correction; these have been
% pre-computed through Monte Carlo simulations and will be inferred by
% matching dT and dZ to their closest values from a library.
[Z_corr_a, Z_corr_b] = MatchZ_corr_coeff(dT, dZ, GapsAllowed);
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%% PART 1: Analyze data from each single cell %%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if DoSingleCellFit == 1 
    %   Analyze each single cell individually
    %   1. Load data for each single cell
    %   2. Make histrogram of jump lengths
    %   3. Fit model to each single cell
    %   4. Plot the model fit and best fit parameters
    
    % initialize various counters
    last_integer = 0;
    PlotIndex = 0;
    CellNumb = 1;
    PlotNumber = 1;

    % loop over all replicates and all cells in each replicate
    for RepIter=1:length(data_struct)
        workspaces = data_struct(RepIter).workspaces;
        curr_path = data_struct(RepIter).path;
        for WorkIter=1:length(workspaces)
        
            %Can only plot 8 cells at a time, so create a new figure if
            %neccesary:
            PlotIndex = PlotIndex + 1;
            if ceil(CellNumb/8) > last_integer
                % save the single-cell plots:
                if SavePlot == 1 && PlotIndex > 1
                    set(figure1,'Units','Inches');
                    pos = get(figure1,'Position');
                    set(figure1,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
                    print(figure1,[save_path, SampleName, '_SingleCell_FitType',num2str(ModelFit), 'FitLocError', num2str(FitLocError), 'NumStates', num2str(NumberOfStates), 'Plot', num2str(PlotNumber-1),'.pdf'],'-dpdf','-r0')
                end
                %Plot everything on a big figure:
                figure1 = figure('position',[100 100 min([1600 screen_size_vector(3)]) min([1200 screen_size_vector(4)]);]); %[x y width height]
                set(gcf, 'Name',['Spot-On Matlab: Single-cell analysis #', num2str(PlotNumber)]);
                last_integer = ceil(CellNumb/8);
                PlotIndex = 1;
                PlotNumber = PlotNumber + 1;
            end
            
            
            %%%%% STEP 1: LOAD IN A CELL AND COMPILE HISTOGRAMS
            tic; disp(['Analysing cell ', num2str(WorkIter), ' of ', num2str(length(workspaces)), ' in replicate ', num2str(RepIter), ' of ', num2str(length(data_struct))]);
            % use function "compile_histograms_single_cell.m" to compile histograms
            disp('loading in trajectories and compiling histograms...');
            %full_path = [curr_path, char(workspaces(WorkIter)), '.mat'] % full_path of workspace to be loaded
            [JumpProb, JumpProbCDF, Min3Traj, CellLocs, CellJumps, CellJumps_used, CellFrames, TrajNumb, JumpsPerdT] = compile_histograms_single_cell([curr_path, char(workspaces(WorkIter))], UseEntireTraj, GapsAllowed, TimePoints, JumpsToConsider);
            
            
            %%%%% STEP 2: PERFORM MODEL-FITTING OF THE CURRENT CELL
            disp('performing model fitting of displacement histograms...');
            [model_params, ~, ~] = ModelFitting_main(JumpProb, JumpProbCDF, NumberOfStates, FitLocErrorRange, FitIterations, D_Free_2State, D_Bound_2State, D_Free1_3State, D_Free2_3State, D_Bound_3State);
            toc;

            %%%%% STEP 3: PLOT THE BEST FIT FOR THE CURRENT CELL
            % get a normalized PDF/CDF histogram for plotting using current
            % best-fit parameters:
            disp('proceeding to plotting the output of the model fitting...');
            [model_PDF, model_CDF] = GenerateModelFitforPlot(model_params, JumpProb, JumpProbCDF, NumberOfStates);

            % Generate a plot title with all the relevant info
            PlotTitle = GeneratePlotTitle(workspaces{WorkIter}, NumberOfStates, model_params, Min3Traj, CellLocs, CellJumps, CellJumps_used, CellFrames, TrajNumb);
            
            % Do actual plotting
            subplot(2,4,PlotIndex);
            histogram_spacer = 0.055;
            hold on;
            for i=size(JumpProb,1):-1:1
                new_level = (i-1)*histogram_spacer;%*y_max;
                colour_element = colour(round(i/size(JumpProb,1)*size(colour,1)),:);
                plot(HistVecJumps, new_level*ones(1,length(HistVecJumps)), 'k-', 'LineWidth', 1); 
                for j=2:size(JumpProb,2)
                    x1 = HistVecJumps(1,j-1); x2 = HistVecJumps(1,j);
                    y1 = new_level; y2 = JumpProb(i,j-1)+new_level;
                    patch([x1 x1 x2 x2], [y1 y2 y2 y1],colour_element);
                end
                plot(HistVecJumps, model_PDF(i,:)+new_level, 'k-', 'LineWidth', 2);
                text(0.6*MaxJumpPlotPDF, new_level+0.5*histogram_spacer, ['\Delta\tau: ', num2str(TimeGap*i), ' ms'], 'HorizontalAlignment','left', 'FontSize',9, 'FontName', 'Helvetica');
            end
            axis([0 MaxJumpPlotPDF 0 1.05*(max(JumpProb(end,:))+(size(JumpProb,1)-1)*histogram_spacer)]);
            title(PlotTitle, 'FontSize',8, 'FontName', 'Helvetica', 'Color', 'k', 'Interpreter', 'none');
            set(gca,'YColor','w')
            ylabel('Probability', 'FontSize',9, 'FontName', 'Helvetica', 'Color', 'k');
            xlabel('jump length \mu m', 'FontSize',9, 'FontName', 'Helvetica', 'Color', 'k');
            set(gca,'YTickLabel',''); set(gca, 'YTick', []);
            hold off;

            % save all the relevant info from fitting this cell:
            if CellNumb == 1
                single_cell_model_params = model_params;
                single_cell_model_PDF = model_PDF;
                single_cell_model_CDF = model_CDF;
                single_cell_data_PDF = JumpProb;
                single_cell_data_CDF = JumpProbCDF;
            else
                single_cell_model_params = vertcat(single_cell_model_params, model_params);
                single_cell_model_PDF = vertcat(model_PDF, single_cell_model_PDF);
                single_cell_model_CDF = vertcat(model_CDF, single_cell_model_CDF);
                single_cell_data_PDF = vertcat(single_cell_data_PDF, JumpProb);
                single_cell_data_CDF = vertcat(single_cell_data_CDF, JumpProbCDF);
            end
            
            % Increment CellNumb:
            CellNumb = CellNumb+1;
        end
    end
    % save the single-cell plots:
    if SavePlot == 1
        set(figure1,'Units','Inches');
        pos = get(figure1,'Position');
        set(figure1,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
        print(figure1,[save_path, SampleName, '_SingleCell_FitType',num2str(ModelFit), 'FitLocError', num2str(FitLocError), 'NumStates', num2str(NumberOfStates), 'Plot', num2str(PlotNumber-1),'.pdf'],'-dpdf','-r0')
    end
    Output_struct(1).single_cell_model_params = single_cell_model_params;
    Output_struct(1).single_cell_model_PDF = single_cell_model_PDF;
    Output_struct(1).single_cell_model_CDF = single_cell_model_CDF;
    Output_struct(1).single_cell_data_PDF = single_cell_data_PDF;
    Output_struct(1).single_cell_data_CDF = single_cell_data_CDF;
    clear JumpProb JumpProbCDF model_params 
end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%       


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%% PART 2: Analyze data from all cells %%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% OVERVIEW:
    %   1. Load data and merged data for all cells
    %   2. Make histrograms (PDF and CDF) of displacements
    %   3. Fit model to the merged dara
    %   4. Plot the model fit for both CDF and PDF and best fit parameters

% load in merged data and compile histograms:
disp('===========================================================');
disp('merging data from all cells and replicates...');
disp('loading in trajectories and compiling histograms...'); tic;
%%%%% STEP 1: LOAD IN MERGED DATA AND COMPILE HISTOGRAMS
[JumpProb, JumpProbCDF, Min3Traj, TotalLocs, TotalFrames, TotalJumps, TotalJumps_used, TrajNumb, DyeSurvivalProb, DyeHistVec, DyeMean, JumpsPerdT] = compile_histograms_many_cells( data_struct, UseEntireTraj, GapsAllowed, TimePoints, JumpsToConsider );
toc;
%%%%% STEP 2: PERFORM MODEL-FITTING OF THE MERGED DATA
disp('performing model fitting of displacement histograms...'); tic;
[model_params, residuals, list_of_model_parameters] = ModelFitting_main(JumpProb, JumpProbCDF, NumberOfStates, FitLocErrorRange, FitIterations, D_Free_2State, D_Bound_2State, D_Free1_3State, D_Free2_3State, D_Bound_3State);
toc;
%%%%% STEP 3: PLOT THE BEST FIT FOR THE MERGED DATA
% get a normalized PDF/CDF histogram for plotting using current
% best-fit parameters:
disp('proceeding to plotting the output of the model fitting...');
[model_PDF, model_CDF] = GenerateModelFitforPlot(model_params, JumpProb, JumpProbCDF, NumberOfStates);
% Generate a plot title with all the relevant info
PlotTitle = GeneratePlotTitle(SampleName, NumberOfStates, model_params, Min3Traj, TotalLocs, TotalJumps, TotalJumps_used, TotalFrames, TrajNumb);


if DoPlots == 1 % proceed to plotting only if DoPlots == 1
    
    % PLOT THE SURVIVAL PROBABILITY OF THE FLUOROPHORE
    figure2 = figure('position',[400 100 300 275]); %[x y width height]
    set(gcf, 'Name','Spot-On Matlab: distribution of trajectory lengths');
    hold on;
    plot(DyeHistVec, DyeSurvivalProb, 'ko', 'MarkerSize', 6, 'MarkerFaceColor', 'r');
    axis([1 51 0.001 1.01]);
    title(['1-CDF of trajectory lengths; mean = ', num2str(DyeMean), ' frames'], 'FontSize',9, 'FontName', 'Helvetica');
    ylabel('1-CDF', 'FontSize',9, 'FontName', 'Helvetica');
    xlabel('number of frames', 'FontSize',9, 'FontName', 'Helvetica');
    set(gca,'yscale','log');
    hold off;
    
    if SavePlot == 1
        set(figure2,'Units','Inches');
        pos = get(figure2,'Position');
        set(figure2,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)]);
        print(figure2,[save_path, SampleName, '_TrajLengthDist.pdf'],'-dpdf','-r0');
    end
    


    % PLOT THE RESIDUALS FROM THE FITTING
    % Plot residuals for the relevant fit: 
    %   so PDF residuals for PDF-fitting
    %   or CDF residuals for CDF-fitting
    figure3 = figure('position',[300 300 plot_width plot_height]); %[x y width height]
    set(gcf, 'Name','Spot-On Matlab: residuals');
    % find max_y for plot
    max_y = max([0.1 max(max(abs(residuals)))]); min_y = -max_y;
    for i=1:min([12 size(residuals,1)])
        colour_element = colour(round(i/size(residuals,1)*size(colour,1)),:);
        ax1(i) = subplot(3,4,i);
        hold on;
        if ModelFit == 1
            plot(HistVecJumps, residuals(i,:), '-', 'Color', colour_element, 'LineWidth', 2);
            max_x = MaxJumpPlotPDF;
        elseif ModelFit == 2
            plot(HistVecJumpsCDF, residuals(i,:), '-', 'Color', colour_element, 'LineWidth', 2);
            max_x = MaxJumpPlotCDF;
        end
        plot([0 max_x], [0 0], 'k--', 'LineWidth', 1);

        axis([0 max_x min_y max_y]);
        if ModelFit == 1
            title({SampleName; ['PDF residuals for \Delta \tau: ', num2str(TimeGap*i), ' ms']}, 'FontSize',8, 'FontName', 'Helvetica', 'Color', 'k');
        elseif ModelFit == 2
            title({SampleName; ['CDF residuals for \Delta \tau: ', num2str(TimeGap*i), ' ms']}, 'FontSize',8, 'FontName', 'Helvetica', 'Color', 'k');
        end
        ylabel('residuals', 'FontSize',9, 'FontName', 'Helvetica', 'Color', 'k');
        xlabel('displacements (\mu m)', 'FontSize',9, 'FontName', 'Helvetica', 'Color', 'k');
        hold off;
    end
    linkaxes(ax1,'xy'); % link all the axes to synchronize any zooming
    
    if SavePlot == 1
        set(figure3,'Units','Inches');
        pos = get(figure3,'Position');
        set(figure3,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)]);
        print(figure3,[save_path, SampleName, '_residuals_FitType',num2str(ModelFit), 'NumStates', num2str(NumberOfStates), '.pdf'],'-dpdf','-r0');
    end

    %PLOT CDFs of DISPLACEMENTS AND OF FIT
    figure4 = figure('position',[100 100 plot_width plot_height]); %[x y width height]
    set(gcf, 'Name','Spot-On Matlab: CDF of displacements vs. model-fit');
    for i=1:min([12 size(JumpProbCDF,1)])
        colour_element = colour(round(i/size(JumpProbCDF,1)*size(colour,1)),:);
        ax2(i) = subplot(3,4,i);
        hold on;
        plot(HistVecJumpsCDF, JumpProbCDF(i,:), '-', 'LineWidth', 2, 'Color', colour_element);
        plot(HistVecJumpsCDF, model_CDF(i,:), 'k-', 'LineWidth', 1);

        axis([0 MaxJumpPlotCDF 0 1.05]);
        title({SampleName; ['CDF for \Delta\tau: ', num2str(TimeGap*i), ' ms']}, 'FontSize',8, 'FontName', 'Helvetica', 'Color', 'k');
        ylabel('displacement CDF', 'FontSize',9, 'FontName', 'Helvetica', 'Color', 'k');
        xlabel('displacements (\mu m)', 'FontSize',9, 'FontName', 'Helvetica', 'Color', 'k');
        legend('raw data', 'Model fit', 'Location', 'SouthEast');
        legend boxoff
        hold off;
    end
    linkaxes(ax2,'xy'); % link all the axes to synchronize any zooming
    if SavePlot == 1
        set(figure4,'Units','Inches');
        pos = get(figure4,'Position');
        set(figure4,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)]);
        print(figure4,[save_path, SampleName, '_mergedCDFs_FitType',num2str(ModelFit), 'NumStates', num2str(NumberOfStates), '.pdf'],'-dpdf','-r0');
    end

    %PLOT THE HISTOGRAM OF TRANSLOCATIONS
    figure5 = figure('position',[200 200 300 400]); %[x y width height]
    set(gcf, 'Name','Spot-On Matlab: histograms of displacements');
    histogram_spacer = 0.055;
    hold on;
    for i=size(JumpProb,1):-1:1
        new_level = (i-1)*histogram_spacer;
        colour_element = colour(round(i/size(JumpProb,1)*size(colour,1)),:);
        plot(HistVecJumps, new_level*ones(1,length(HistVecJumps)), 'k-', 'LineWidth', 1); 
        for j=2:size(JumpProb,2)
            x1 = HistVecJumps(1,j-1); x2 = HistVecJumps(1,j);
            y1 = new_level; y2 = JumpProb(i,j-1)+new_level;
            patch([x1 x1 x2 x2], [y1 y2 y2 y1],colour_element);
        end
        plot(HistVecJumps, model_PDF(i,:)+new_level, 'k-', 'LineWidth', 2);
        text(0.6*MaxJumpPlotPDF, new_level+0.5*histogram_spacer, ['\Delta\tau: ', num2str(TimeGap*i), ' ms'], 'HorizontalAlignment','left', 'FontSize',9, 'FontName', 'Helvetica');
    end
    axis([0 MaxJumpPlotPDF 0 1.05*(max(JumpProb(end,:))+(size(JumpProb,1)-1)*histogram_spacer)]);
    title(PlotTitle, 'FontSize',9, 'FontName', 'Helvetica', 'Color', 'k', 'Interpreter', 'none');
    set(gca,'YColor','w')
    ylabel('Probability', 'FontSize',9, 'FontName', 'Helvetica', 'Color', 'k');
    xlabel('jump length \mu m', 'FontSize',9, 'FontName', 'Helvetica', 'Color', 'k');
    set(gca,'YTickLabel',''); set(gca, 'YTick', []);
    hold off;
    if SavePlot == 1
        set(figure5,'Units','Inches');
        pos = get(figure5,'Position');
        set(figure5,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)]);
        print(figure5,[save_path, SampleName, '_mergedPDFs_FitType',num2str(ModelFit), 'FitLocError', num2str(FitLocError), 'NumStates', num2str(NumberOfStates), '.pdf'],'-dpdf','-r0');
    end 
end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    

%%%%%%%%%%%%%%%%%%%%%%%% SAVE THE DESIRED OUTPUTS %%%%%%%%%%%%%%%%%%%%%%%%%
% save key model outputs for all merged cells:
Output_struct(1).merged_model_params = model_params;
Output_struct(1).merged_model_PDF = model_PDF;
Output_struct(1).merged_model_CDF = model_CDF;
Output_struct(1).merged_data_PDF = JumpProb;
Output_struct(1).merged_data_CDF = JumpProbCDF;
Output_struct(1).merged_model_residuals = residuals;
Output_struct(1).list_of_model_parameters = list_of_model_parameters;
Output_struct(1).r_bins_PDF = HistVecJumps;
Output_struct(1).r_bins_CDF = HistVecJumpsCDF;

end



