#' Plot gistic results as a bubble plot
#' @description Plots significantly altered cytobands as a function of number samples in which it is altered and number genes it contains. Size of each bubble is according to -log10 transformed q values.
#' @param gistic an object of class \code{GISTIC} generated by \code{readGistic}
#' @param fdrCutOff fdr cutoff to use. Default 0.1
#' @param markBands any cytobands to label.
#' @param color colors for Amp and Del events.
#' @param file if given saves plot as a pdf.
#' @param width width of the file to be saved.
#' @param height height of the file to be saved.
#' @param txtSize label size for bubbles.
#' @examples
#' all.lesions <- system.file("extdata", "all_lesions.conf_99.txt", package = "maftools")
#' amp.genes <- system.file("extdata", "amp_genes.conf_99.txt", package = "maftools")
#' del.genes <- system.file("extdata", "del_genes.conf_99.txt", package = "maftools")
#' scores.gistic <- system.file("extdata", "scores.gistic", package = "maftools")
#' laml.gistic = readGistic(gisticAllLesionsFile = all.lesions, gisticAmpGenesFile = amp.genes, gisticDelGenesFile = del.genes, gisticScoresFile = scores.gistic)
#' gisticBubblePlot(gistic = laml.gistic)
#' @return invisible ggplot2 object
#' @export
#'
gisticBubblePlot = function(gistic = NULL, color = NULL, markBands = NULL, fdrCutOff = 0.1, txtSize = 3, file = NULL, width = 6, height = 5){

  g = getCytobandSummary(gistic)
  g = g[qvalues < fdrCutOff]
  g[,Chromosome := sapply(strsplit(x = g$Wide_Peak_Limits, split = ':'), '[', 1)]
  g[,loc := sapply(strsplit(x = g$Wide_Peak_Limits, split = ':'), '[', 2)]
  g[,Start_Position := sapply(strsplit(x = g$loc, split = '-'), '[', 1)]
  g[,End_Position := sapply(strsplit(x = g$loc, split = '-'), '[', 2)]
  g.lin = transformSegments(segmentedData = g[,.(Chromosome, Start_Position, End_Position, qvalues, Cytoband, Variant_Classification)])

  if(is.null(color)){
    color = c('Amp' = 'red', 'Del' = 'blue')
  }

  g$lab = sapply(strsplit(x = g$Unique_Name, split = ':'), '[', 2)

  g$pos = ifelse(test = g$Variant_Classification %in% 'Amp', yes = g$nGenes, no = -1 * g$nGenes)

  gist.gg = ggplot(data = g, aes(x = nSamples, y = pos, size = -log10(qvalues), color = Variant_Classification, label = lab))+
    geom_point(alpha = 0.6)+xlab('nSamples')+ylab('nGenes')+
    scale_colour_manual(values = color, name = c('CNV', ''))+
    cowplot::theme_cowplot(font_size = 12, line_size = 1)+
    cowplot::background_grid(major = 'xy')+
    theme(legend.position = 'bottom', axis.title.x = element_text(face = "bold"), axis.text.x = element_text(face = "bold"),
          axis.title.y = element_text(face = "bold"), axis.text.y = element_text(face = "bold"),
          legend.text = element_text(face = "bold"), legend.title = element_text(face = "bold"))

  if(!is.null(markBands)){
    g.labs = g[Cytoband %in% markBands]
    if(nrow(g.labs) == 0){
      message("Nothing to label!")
    }else{
      gist.gg = gist.gg+ggrepel::geom_text_repel(data = g.labs, size = txtSize, fontface = 'bold', force = 10)
    }
  }else{
    gist.gg = gist.gg+ggrepel::geom_text_repel(size = txtSize, fontface = 'bold', force = 10)
  }

  if(!is.null(file)){
    cowplot::save_plot(filename = paste(file, 'pdf', sep='.'), plot = gist.gg, base_height = height, base_width = width)
  }

  gist.gg
}
