#' Plots wordcloud.
#'
#' @description Plots word cloud of mutated genes or altered cytobands with size proportional to the event frequency.
#' @param input an \code{\link{MAF}} or \code{\link{GISTIC}} object generated by \code{\link{read.maf}} or \code{\link{readGistic}}
#' @param minMut Minimum number of samples in which a gene is required to be mutated.
#' @param col vector of colors to choose from.
#' @param top Just plot these top n number of mutated genes.
#' @param genesToIgnore Ignore these genes.
#' @param ... Other options passed to \code{\link{wordcloud}}
#' @return nothing.
#' @examples
#' laml.input <- system.file("extdata", "tcga_laml.maf.gz", package = "maftools")
#' laml <- read.maf(maf = laml.input, useAll = FALSE)
#' geneCloud(input = laml, minMut = 5)
#' @importFrom wordcloud wordcloud
#' @export


geneCloud = function(input, minMut = 3, col = NULL, top = NULL, genesToIgnore = NULL, ... ){


  if(class(input)[1] == "GISTIC"){
    gs = getCytobandSummary(input)

    col = ifelse(test = gs$Variant_Classification %in% 'Amp', yes = 'red', no = 'blue')
    wordcloud::wordcloud(words = gs[,Cytoband], freq = -log10(gs[,qvalues]),
                         colors = col, ordered.colors = TRUE, rot.per = 0.2, ...)

  }else{

    gs = getGeneSummary(x = input)

    #Either choose top n genes or > minMut
    if(!is.null(top)){
      gs = gs[1:top]
    }else{
      gs = gs[MutatedSamples >= minMut]
    }

    #If any genes to ignore
    if(!is.null(genesToIgnore)){
      gs = gs[!Hugo_Symbol %in% genesToIgnore]
    }

    if(is.null(col)){
      col = c(RColorBrewer::brewer.pal(12,name = "Paired"),RColorBrewer::brewer.pal(9,name = "Set1"),'black')
    }

    wordcloud::wordcloud(words = gs[,Hugo_Symbol], freq = gs[,MutatedSamples],
                         colors = col, random.color = TRUE, rot.per = 0.2, ...)

  }

}
