#' callMotif
#'
#' callMotif calculates in parallel the motif enrichment scores for all k-mer motifs in native DNA, when only smrt data for native DNA is specified (either nat or split_nat);
#' it can also calculate the adjusted motif enrichment scores by comparing native DNA with WGA, when smrt data for both native DNA and WGA are specified (nat/split_nat and wga/split_wga).
#' 
#' @param nat a data frame of smrt data for native DNA, taking at least columns: refName, strand, tpl, ipdRatio/score
#' @param split_nat a list of smrt for native DNA, which is a split smrt generated by the function splitSmrt()
#' @param wga a data frame of smrt data for WGA, taking at least columns: refName, strand, tpl, ipdRatio/score
#' @param split_wga a list of smrt for WGA, which is a split smrt generated by the function splitSmrt()
#' @param genome a DNAStringSet value indicating the reference sequence. Refer to Biostrings:readDNAStringSet() 
#' @param kmer an integer value indicating the motif length of interest
#' @param meth_char a character value indicating the methylated base, by default "A"
#' @param cores an integer value indicating the number of cores for parallel computating
#' @param criterion a character value indicating the criterion for calling motif, either "ipdRatio" or "score", and by default, "ipdRatio" 
#' @param thres a numeric value indicating the threshold for the criterion, by default, 4 for ipdRatio
#'
#' @return a list comprising the following components
#' \itemize{
#'  \item {nat_motif_fdr} a data frame indicating the motif enrichment score for each k-mer motif in native DNA
#'  \item {wga_motif_fdr} a data frame indicating the motif enrichment score for each k-mer motif in WGA
#'  \item {adjust_motif_fdr} a data frame indicating the adjusted motif enrichment score by comparing native DNA with WGA
#' }
#' 
#' @export
#'
#' @examples
#' 
#' @seealso \code{\link{callFDR_eachMotif}}; \code{\link{enumerateMotif}}; 
#' 
callMotif = function(  nat=NULL , split_nat=NULL , wga=NULL , split_wga=NULL , genome , kmer=4 , meth_char = 'A' , cores=10 , criterion="ipdRatio" , thres=4 ) 
{

  if( is.null(wga) & is.null(split_wga) )
  {
    
    nat_motif_fdr = callFDR_eachMotif( smrt=nat , split_smrt=split_nat , genome=genome , kmer=kmer , meth_char=meth_char , cores=cores , criterion=criterion, thres=thres )
    drawMotifHeatmap( nat_motif_fdr , outputName="MotifCalling_Native.pdf" )
    res = list( nat_motif_fdr=nat_motif_fdr , wga_motif_fdr=NULL , adjust_motif_fdr=NULL )
  
  } else {
    
    nat_motif_fdr = callFDR_eachMotif( smrt=nat , split_smrt=split_nat , genome=genome , kmer=kmer , meth_char=meth_char , cores=cores , criterion=criterion, thres=thres )
    wga_motif_fdr = callFDR_eachMotif( smrt=wga , split_smrt=split_wga , genome=genome , kmer=kmer , meth_char=meth_char , cores=cores , criterion=criterion, thres=thres ) 
    adjust_motif_fdr = data.frame(nat_motif_fdr[,1:2] , score=nat_motif_fdr[,3]/wga_motif_fdr[,3] )
    
    drawMotifHeatmap( motif_fdr=nat_motif_fdr , outputName="MotifCalling_Native.pdf" )
    drawMotifHeatmap( motif_fdr=wga_motif_fdr , outputName="MotifCalling_WGA.pdf" )
    drawMotifHeatmap( motif_fdr=adjust_motif_fdr , outputName="MotifCalling_Adjusted.pdf" )
   
    res = list( nat_motif_fdr=nat_motif_fdr , wga_motif_fdr=wga_motif_fdr , adjust_motif_fdr=adjust_motif_fdr )
    
  }
  
  res
  
}
