#!/usr/bin/python
# SQANTI filtering: Filtering of Isoforms based on SQANTI attributes
# Authors: Lorena de la Fuente and Cecile Pereira

from collections import defaultdict
import getopt
import sys
import os.path, os
import timeit
import subprocess
import argparse
import psutil
import gc
import distutils.spawn
utilitiesPath =  os.path.dirname(os.path.realpath(__file__))+"/utilities/" 

sys.path.insert(0, utilitiesPath)

def main():

	#arguments
	parser = argparse.ArgumentParser(description="Filtering of Isoforms based on SQANTI attributes")
	parser.add_argument('sqanti_class', help='\t\tSQANTI classification at isoform level')
	parser.add_argument('-t',"-ml_threshold", default = 0.5, help='\tMachine learning probability threshold to classify posiive isoforms') 
	parser.add_argument('-a',"-intrapriming", default = 80, help='\t\tAdenine percentage at genomic 3\' end to flag an isoform as intra-priming') 
	parser.add_argument('-i','--isoforms', dest="fasta", help='\t\tFasta isoform file to be filtered by SQANTI', required=False)
	parser.add_argument('-d','--dir', dest="dir",help='\t\tOutput directory name. Default: "Filter_out" directory where the script was run', required=False)
	parser.add_argument('-p',"-TP_isoforms", dest="TP_isoforms",  help='\tFile with true positive set of isoforms') # one column file with isoform IDs
	parser.add_argument('-n',"-TN_isoforms", dest="TN_isoforms", help='\t\tFile with true negative set of isoforms') # one column file with isoform IDs
	parser.add_argument("-v", "--version", help="Display program version number.", action='version', version='SQANTI 1.2')


	args = parser.parse_args()
	
	args.sqanti_class = os.path.abspath(args.sqanti_class)
	if not os.path.isfile(args.sqanti_class):
		sys.stderr.write("ERROR: '%s' doesn't exist\n" %(args.sqanti_class))
		sys.exit()

	if args.fasta!=None:
		args.fasta = os.path.abspath(args.fasta)
		if not os.path.isfile(args.fasta):
			sys.stderr.write("ERROR: '%s' doesn't exist\n" %(args.fasta))
			sys.exit()

	if args.dir==None:   
		args.dir = os.getcwd() + "/Filter_out"
		if not os.path.exists(args.dir):
			os.makedirs(args.dir)
	else:
		args.dir = os.path.abspath(args.dir)
		if not os.path.exists(args.dir):
			#if os.path.exists(os.path.dirname(os.path.dirname(args.dir))):
			os.makedirs(args.dir)
			# else:
			# 	sys.stderr.write("ERROR: '%s' directory doesn't exist\n" %(os.path.dirname(os.path.dirname(args.dir))))
			# 	sys.exit()

	if args.TP_isoforms==None:
		args.TP_isoforms="NULL"
	else:
		args.TP_isoforms = os.path.abspath(args.TP_isoforms)
		if not os.path.isfile(args.TP_isoforms):
			sys.stderr.write("ERROR: '%s' doesn't exist\n" %(args.TP_isoforms))
			sys.exit()

	if args.TN_isoforms==None:
		args.TN_isoforms="NULL"
	else:
		args.TN_isoforms = os.path.abspath(args.TN_isoforms)
		if not os.path.isfile(args.TN_isoforms):
			sys.stderr.write("ERROR: '%s' doesn't exist\n" %(args.TN_isoforms))
			sys.exit()



	# Running functionality
	sys.stdout.write("\nRunning SQANTI filtering...\n")

	run(args)	



def fasta_parser(fastaFile):

	try:
		fasta = open(fastaFile, "r")
	except IOError:
		sys.stderr.write('ERROR: Unable to read %s file\n' % fastaFile)
		raise SystemExit(1)
	try:
		seqDicc = {}
		index = 0
		for line in fasta:
			if line.startswith(">"):
				if index > 0:
					seqDicc[name] = seq
				index+=1
				name = line[1:].rstrip()
				#name = line[1:].split()[0].rstrip()
				seq = ''
			elif line.strip()!="":
				seq += line.rstrip()
		if index>0:
			seqDicc[name] = seq

		fasta.close()

	except IOError:
		sys.stderr.write('File %s without fasta format' % fastaFile)
		raise SystemExit(1)

	return(seqDicc)

def run(args):

	## Generating classifier

	sys.stdout.write("\nGenerating isoform classifier...\n")

	rscriptPath = distutils.spawn.find_executable('Rscript')

	subprocess.call ([rscriptPath, utilitiesPath+"/SQANTI_filter.R", "-c", args.sqanti_class, "-j", str(args.t), "-i", str(args.a), "-d", args.dir, "-p" , args.TP_isoforms,"-n", args.TN_isoforms])

	filtIso_File = args.dir+"/"+os.path.basename(args.sqanti_class)+"_curatedTranscriptome.txt" #generated by Cecile

	if os.path.exists(filtIso_File):
		
		if args.fasta: # if user provides the fasta file

			sys.stdout.write("\nFiltering fasta file based on filtering results...\n")

			#read list of isoforms classified as true
			filtIso = open(filtIso_File, "r")
			trueIsolist = []
			[trueIsolist.append(line.rstrip()) for line in filtIso]
			filtIso.close()

			# reading fasta
			fastaFile = os.path.abspath(args.fasta)
		 	fastaDicc = fasta_parser(fastaFile)

		 	#reading and filtering fasta file
			fastaFiltFile = args.dir+"/"+os.path.basename(args.fasta)+"_curatedTranscriptome.fasta"
		 	fastaFilt = open(fastaFiltFile, "w")

			for ID in fastaDicc:
				if len(ID.split("|"))>2 and ID[0:2]=="PB": # PacBio fasta header (including chained format)
					ID_mod = ID.split("|")[0].split(" ")[0]
				elif len(ID.split("|"))>4: # Refseq fasta header
					ID_mod = ID.split("|")[3]
				else:
					ID_mod = ID.split()[0] # Ensembl fasta header

		 	 	if ID_mod in trueIsolist:
		 	 		fastaFilt.write(">"+ID+"\n"+fastaDicc[ID]+"\n")

		 	fastaFilt.close()
		

if __name__ == "__main__":
	main()
