package iptgxdb.utils;


import java.awt.Color;
import java.util.HashMap;
import java.util.List;

import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.ListMultimap;
import com.google.common.collect.Multimaps;

/**
 * GenomicsUtil provides genomics specific utility functions
 * 
 * @author Ulrich Omasits
 * @date 25.11.2011
 */
public class GenomicsUtil {

	// a translation table for bacterial genomes
	public final static HashMap<String, Character> translationTable11 = new HashMap<String, Character>() {
		private static final long serialVersionUID = 8562663901478718584L;
		{
			put("TTT",'F'); put("TCT",'S'); put("TAT",'Y'); put("TGT",'C'); put("TTC",'F'); put("TCC",'S'); put("TAC",'Y'); put("TGC",'C'); 
			put("TTA",'L'); put("TCA",'S'); put("TAA",'*'); put("TGA",'*'); put("TTG",'L'); put("TCG",'S'); put("TAG",'*'); put("TGG",'W'); 
			put("CTT",'L'); put("CCT",'P'); put("CAT",'H'); put("CGT",'R'); put("CTC",'L'); put("CCC",'P'); put("CAC",'H'); put("CGC",'R'); 
			put("CTA",'L'); put("CCA",'P'); put("CAA",'Q'); put("CGA",'R'); put("CTG",'L'); put("CCG",'P'); put("CAG",'Q'); put("CGG",'R'); 
			put("ATT",'I'); put("ACT",'T'); put("AAT",'N'); put("AGT",'S'); put("ATC",'I'); put("ACC",'T'); put("AAC",'N'); put("AGC",'S'); 
			put("ATA",'I'); put("ACA",'T'); put("AAA",'K'); put("AGA",'R'); put("ATG",'M'); put("ACG",'T'); put("AAG",'K'); put("AGG",'R'); 
			put("GTT",'V'); put("GCT",'A'); put("GAT",'D'); put("GGT",'G'); put("GTC",'V'); put("GCC",'A'); put("GAC",'D'); put("GGC",'G'); 
			put("GTA",'V'); put("GCA",'A'); put("GAA",'E'); put("GGA",'G'); put("GTG",'V'); put("GCG",'A'); put("GAG",'E'); put("GGG",'G'); 
		}
	};
	
	public static ListMultimap<Character, String> reverseTranslationTable = ArrayListMultimap.create();
	static {
		Multimaps.invertFrom(Multimaps.forMap(translationTable11), reverseTranslationTable);
		reverseTranslationTable.putAll('X', translationTable11.keySet());
		reverseTranslationTable.putAll('J', reverseTranslationTable.get('L'));
		reverseTranslationTable.putAll('J', reverseTranslationTable.get('I'));
		reverseTranslationTable.putAll('Z', reverseTranslationTable.get('Q'));
		reverseTranslationTable.putAll('Z', reverseTranslationTable.get('E'));
		reverseTranslationTable.putAll('B', reverseTranslationTable.get('N'));
		reverseTranslationTable.putAll('B', reverseTranslationTable.get('D'));
	}
	
	// returns the complement DNA sequence
	public static StringBuilder reverseNucleotides(String forward) {
		return new StringBuilder(
				forward.toUpperCase()
				.replace('A', 't')
				.replace('T', 'a')
				.replace('G', 'c')
				.replace('C', 'g')
				.toUpperCase()
		).reverse();
	}
	
	// returns a GFF format header
	public static String createGFFheader(String trackName, Color col) {
		String colString = "";
		if (col!=null)
			colString = " color="+col.getRed()+","+col.getGreen()+","+col.getBlue();
		return "##gff-version 3" + Utils.nl + "#track name=\""+trackName+"\""+colString;
	}

	// translates a nucleotide sequence to aminoacids
	public static StringBuilder translate(CharSequence nt) throws Exception {
		StringBuilder aa = new StringBuilder();
		for (int i=0; i<nt.length()-(nt.length()%3); i+=3) {
			String codon = nt.subSequence(i, i+3).toString().toUpperCase();
			if (translationTable11.containsKey(codon))
				aa.append(translationTable11.get(codon));
			else {
				System.err.println("WARNING: unkown codon "+codon);
				return new StringBuilder();
			}
		}
		return aa;
	}

	// translates a protein sequence to nucleotides
	public static StringBuilder reverseTranslate(CharSequence aa, boolean addStopCodon) throws Exception {
		StringBuilder nt = new StringBuilder();
		for (int i=0; i<aa.length(); i++) {
			List<String> codons = reverseTranslationTable.get(aa.charAt(i));
			nt.append(codons.get(Utils.randomIndex(codons.size())));
		}
		if (addStopCodon) {
			List<String> codons = reverseTranslationTable.get('*');
			nt.append(codons.get(Utils.randomIndex(codons.size())));
		}
		return nt;
	}
	
	public static String complement(String nt) {
		return nt.toUpperCase().replace('A', 't').replace('T', 'a').replace('G', 'c').replace('C', 'g').toUpperCase();
	}
}
