package iptgxdb.utils;


import java.util.EnumSet;

import org.apache.commons.lang3.StringUtils;

import com.google.common.base.Objects;
import com.google.common.collect.ComparisonChain;

/**
 * GenomeLocation defines a region on a genome. 
 * It consists of a two coordinates (from and to) and strand information (+ or -).
 * Given the strand information, the start and end positions can be computed.
 * Given the forward sequence of the genome, the sequence of the region can be computed.
 * 
 * @author Ulrich Omasits
 * @date 03.11.2011
 */
public class GenomeLocation implements Comparable<GenomeLocation> {
	
	/**
	 * Strand is either the PLUS or MINUS strand of a genome.
	 * This Enum comes with toString and fromString methods. 
	 * 
	 * @author Ulrich Omasits
	 */
	public enum Strand {
		PLUS ("+"),
		MINUS ("-");
		
		private final String str;
		Strand(String str) {
			this.str = str;
		}
		
		@Override
		public String toString() {
			return str;
		}
		
		public static Strand fromString(String value){
            Strand returnValue = null;
            for (final Strand element : EnumSet.allOf(Strand.class)) {
                if (element.toString().equals(value))
                    returnValue = element;
            }
            return returnValue;
        }
		
	}
	
	public String chromosome; // the chromosome this location resides on
	public int from; // the first position in the genome (i.e. lower coordinate)
	public int to; // the second position in the genome (i.e. upper coordinate)
	public Strand strand; // the strand of the genome this location resides on
	

	// constructor: creates a new GenomeLocation object given a start and end position
	// depening on the order, the strand will be determined
	public GenomeLocation(Integer start, Integer end, String chromosome) throws Exception {
		if (start<0 && end<0) {
			this.strand = Strand.MINUS;
			this.from = -end;
			this.to = -start;
		} else if (start>0 && end>0) {
			if (end < start) {
				this.from = end;
				this.to = start;
				this.strand = Strand.MINUS;
			} else {
				this.from = start;
				this.to = end;
				this.strand = Strand.PLUS;
			}
		} else {
			throw new Exception("either signed OR normal coordinates");
		}
		this.chromosome = chromosome;
	}
	
	// constructor: creates a new GenomeLocation object given coordinates and strand
	public GenomeLocation(Integer from, Integer to, Strand strand, String chromosome) throws Exception {
		if (to < from) throw new Exception("'to' coordinate cannot be before 'from' coordinate");
		this.from = from;
		this.to = to;
		this.strand = strand;
		this.chromosome = chromosome;
	}
	
	// constructor: creates a new GenomeLocation object given a GenBank formatted region, e.g. complement(1459863..1462346)
	public GenomeLocation(String genBankLocation, String chromosome) throws Exception {
		String location = genBankLocation.trim();
		strand = Strand.PLUS;
		if (location.startsWith("complement")) {
			strand = Strand.MINUS;
			location = location.substring(11, location.length()-1);
		}
		if (location.indexOf('<')!=-1 || location.indexOf('>')!=-1) {
			System.out.println("WARN: ignoring open location ends at '"+genBankLocation+"'");
			location = location.replace("<", "").replace(">", "");
			 
		}
		if (location.matches("\\d+\\.\\.>?\\d+")) {
			from = Integer.valueOf(location.split("\\.\\.")[0]);
			to = Integer.valueOf(location.split("\\.\\.>?")[1]);
//		} else if (location.matches("join\\(\\d+\\.\\.\\d+,\\d+\\.\\.\\d+\\)")) {
//			location = location.substring(5, location.length()-1);
//			from = Integer.valueOf(location.split(",")[0].split("\\.\\.")[0]);
//			to = Integer.valueOf(location.split(",")[1].split("\\.\\.")[1]);
		} else {
			throw new Exception("unknown location: "+genBankLocation);
		}
		this.chromosome = chromosome;
	}
	
	// returns the length of this GenomeLocation
	public int length() {
		return to-from+1;
	}
	
	// returns the length of this GenomeLocation in AA
	public Integer lengthAA() {
		if (length()%3==0) // length in nt is a multiple of 3
			return length()/3-1;
		else
			return null;
	}
	
	// check if this genome location is equal to another genome location but starting later on
	// i.e. same end and same strand but shorter
	public boolean isStartingInternalOf(GenomeLocation otherLoc) {
		if (Objects.equal(otherLoc.chromosome, this.chromosome) && otherLoc.strand == this.strand  &&  otherLoc.getEnd() == this.getEnd()) {
			return (otherLoc.length() > this.length());
		} else
			return false;
	}
	
	// extracts the sequence of this GenomeLocation given a forward sequence
	public String getSequence(StringBuilder forwardSequence) {
		int start = Math.max(1, from);
		int end = Math.min(forwardSequence.length(), to);
		String seq = StringUtils.repeat('-', start-from) + forwardSequence.substring(start-1, end) + StringUtils.repeat('-', to-end) ;
		if (strand==Strand.MINUS)
			return GenomicsUtil.reverseNucleotides(seq).toString();
		else
			return seq;
	}
	
	// calculates the frame this genome location resides on
	public int getFrame(int seqLength) {
		if (strand.equals(Strand.MINUS)) {
			return -(((seqLength-getStart()) % 3)+1);
		} else {
			return ((getStart()-1) % 3)+1;
		}
	}
	
	// computes the start coordinate
	public int getStart() {
		if (strand == Strand.PLUS)
			return from;
		else if (strand == Strand.MINUS)
			return to;
		else
			return 0;
	}
	public int getSignedStart() {
		if (strand == Strand.PLUS)
			return from;
		else if (strand == Strand.MINUS)
			return -to;
		else
			return 0;
	}
	
	// computes the end coordinate
	public int getEnd() {
		if (strand == Strand.PLUS)
			return to;
		else if (strand == Strand.MINUS)
			return from;
		else
			return 0;
	}
	public int getSignedEnd() {
		if (strand == Strand.PLUS)
			return to;
		else if (strand == Strand.MINUS)
			return -from;
		else
			return 0;
	}
	
	// converts the GenomeLocation to a nice string 
	@Override
	public String toString() {
//		return (chromosome!=null ? chromosome+":" : "") + from+".."+to+"("+strand+")";
		return from+".."+to+"("+strand+")";
	}
	
	// for comparison to another GenomeLocation
	@Override
	public boolean equals(Object o) { 
	  if (o == null) return false; 
	  if (o == this) return true; 
	  if (!o.getClass().equals(getClass())) return false; 
	  GenomeLocation that = (GenomeLocation) o; 
	  return	this.from == that.from 
	         && this.to == that.to
	         && this.strand == that.strand
	         && Objects.equal(this.chromosome, that.chromosome);
	}
	
	// producing a hash code of this object, needed for use in HashMaps
	@Override
	public int hashCode() {
		int hash = 7;
		hash = 31 * hash + from;
		hash = 31 * hash + to;
		hash = 31 * hash + strand.ordinal();
		if (chromosome!=null)
			hash = 31 * hash + chromosome.hashCode();
		return hash;
	}

	@Override
	public int compareTo(GenomeLocation that) {
		return ComparisonChain.start()
			.compare(this.chromosome, that.chromosome)
			.compare(this.strand, that.strand)
			.compare(this.to, that.to)
			.compare(this.from, that.from)
		.result();
	}
}