package iptgxdb.utils;


import java.awt.Color;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.util.Comparator;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;

import com.google.common.base.Splitter;
import com.google.common.collect.Lists;

import iptgxdb.utils.GenomeLocation.Strand;


/**
 * GenomeFeature describes a feature on a genome (i.e. an annotation).
 * It is closely related to an entry in an GFF3 file
 * (see http://www.sequenceontology.org/gff3.shtml).
 * 
 * @author Ulrich Omasits
 * @date 14.11.2011
 */
public class GenomeFeature {
	public String seqId;
	public String source;
	public String type;
	public GenomeLocation location;
	public Double score;
	public Integer phase;
	public AttributeMap atts;
	
	// constructor, builds an empty GenomeFeature
	public GenomeFeature() {
		this.atts = new AttributeMap();
	}
	
	// constructor, builds the specified GenomeFeature
	public GenomeFeature(String seqId, String source, String type, GenomeLocation location, String id, Boolean pseudo) {
		this.seqId = seqId;
		this.source = source;
		this.type = type;
		this.location = location;
		this.atts = new AttributeMap();
		if (id!=null)
			atts.put("ID", id);
		if (pseudo!=null)
			atts.put("pseudo", String.valueOf(pseudo));
	}
	
	// constructor, builds a GenomeFeature from a GFF3-type String
	public GenomeFeature(String gffLine) throws Exception {
		String arr[] = gffLine.split("\t");
		this.seqId = arr[0];
		this.source = arr[1];
		this.type = arr[2];
		int from = Integer.valueOf(arr[3]);
		int to = Integer.valueOf(arr[4]);
		Strand strand = Strand.fromString(arr[6]);
		this.location = new GenomeLocation(from, to, strand, seqId);
		this.score = arr[5].equals(".") ? null : Double.valueOf(arr[5]);
		this.phase = arr[7].equals(".") ? null : Integer.valueOf(arr[7]);
		this.atts = new AttributeMap(arr[8]);
	}
	
	public GenomeFeature(String bedLine, String source) throws Exception {
		List<String> line = Lists.newArrayList(Splitter.on('\t').split(bedLine));
		this.seqId = line.get(0);
		int from = Integer.parseInt( line.get(1) ) + 1; // zero based bed file start
		int to = Integer.parseInt( line.get(2) ); //  exclusive zero based end
		Strand strand = Strand.fromString(line.get(5));
		this.location = new GenomeLocation(from, to, strand, this.seqId);
		this.atts = new AttributeMap(line.get(3));
		this.score = Double.valueOf(line.get(4));
		this.source = source;
	}
	
	@Override
	public String toString() {
		return toGFFentry();
	}
	
	public String toGFFentry() {
		StringBuilder gff = new StringBuilder();
		gff.append(seqId!=null ? seqId : "seqId").append("\t");
		gff.append(source!=null ? source : "source").append("\t");
		gff.append(type).append("\t");
		gff.append(location.from).append("\t");
		gff.append(location.to).append("\t");
		gff.append(score!=null ? score : ".").append("\t");
		gff.append(location.strand).append("\t");
		gff.append(phase!=null ? phase : ".").append("\t"); // frame
		gff.append(atts).append("\t");
		return gff.toString();
	}
	
	public String toFastaEntry(StringBuilder forwardSequence, boolean setInitiatorMethionine, boolean inclHeader) throws Exception {
		StringBuilder aa = GenomicsUtil.translate(location.getSequence(forwardSequence));
		if (aa.length()==0) {
			System.out.println("WARN: no sequence generated for "+getID());
			return "";
		} else {
			if (setInitiatorMethionine)
				aa.setCharAt(0,'M'); // initiator methionine
			if (aa.charAt(aa.length()-1)=='*')
				aa.deleteCharAt(aa.length()-1); // remove stop codon
			else
				System.out.println("WARN: no stop codon for "+getID());
			if (aa.indexOf("*")>0)
				System.out.println("WARN: "+Utils.countChar(aa, '*')+" internal stop codon(s) for "+getID());
			StringBuilder res = new StringBuilder();
			if (inclHeader)
				res.append(">").append(getID()).append(Utils.nl);
			res.append(aa);
			return res.toString();
		}
	}
	
	// some fast access functions
	public boolean hasAtt(String att) {
		return atts.containsKey(att);
	}
	
	public String getAtt(String att, String defaultValue) {
		return atts.get(att, defaultValue);
	}

	public String getAtt(String att) {
		return getAtt(att, null);
	}
	
	public String setAtt(String att, String value) {
		return atts.put(att, value);
	}
	
	public String getID() {
		return getAtt("ID");
	}
	
	public String setID(String id) {
		if (id.startsWith("\"") && id.endsWith("\""))
			id = id.substring(1, id.length()-1);
		return setAtt("ID", id);
	}
	
	public String setColor(Color color) {
		return atts.setColor(color);
	}
	
	// a comparator useful for sorting/filtering genome features by their length
	public static Comparator<GenomeFeature> comparatorLength = new Comparator<GenomeFeature>() {
		@Override
		public int compare(GenomeFeature gff1, GenomeFeature gff2) {
			return Integer.valueOf(gff1.location.length()).compareTo(gff2.location.length());
		}
	};
	
	
	/**
	 * An IdManipulator serves only one function: generateId takes a 
	 * GenomeFeature as input and returns a (modified) identifier as string.
	 * 
	 * @author Ulrich Omasits
	 */
	public interface IdManipulator {
		public String generateId(GenomeFeature gff);
	}
	
	/**
	 * AttributeMap is an internal class of GenomeFeature and
	 * resembles a map of key-value pairs as used in the "attributes"
	 * column of the GFF3 file format.
	 * 
	 * @author Ulrich Omasits
	 */
	public static class AttributeMap extends LinkedHashMap<String,String> {
		private static final long serialVersionUID = 4402328163996824120L;
		
		public AttributeMap() {
			super();
		}
		
		public AttributeMap(String attributes) {
			super();
			if (attributes.startsWith("#"))
				return;
			String atts[] = attributes.split(";");
			for (String att : atts) {
				if (att.split("=").length>1) {
					String key = att.split("=")[0];
					String value = URLDecoder.decode(att.split("=")[1]);
					this.put(key, value);
				}
			}
		}
		
		@Override
		public String toString() {
			StringBuilder str = new StringBuilder();
			Iterator<String> keyIterator = keySet().iterator();
			while (keyIterator.hasNext()) {
				String key = keyIterator.next();
				str.append(key);
				str.append("=");
				if (key.equals("color"))
					str.append(get(key));
				else
					str.append(URLEncoder.encode(get(key)));
				if (keyIterator.hasNext())
					str.append(";");
			}
			return str.toString();
		}
		
		public String toString(boolean doNotEscape) {
			if (doNotEscape)
				return super.toString();
			else
				return toString();
		}
		
		@Override
		public String put(String key, String value) {
			if (key == null) {
				System.err.println("no null attribute key allowed ("+key+"="+value+")");
				return null;
			} else if (value == null) {
				System.err.println("no null attribute value allowed ("+key+"="+value+")");
				return super.put(key, "");
			} else
				return super.put(key, value);
		}
		
		public String setColor(Color col) {
			return put("color", col.getRed()+","+col.getGreen()+","+col.getBlue());
		}
		
		public String get(String key, String defaultValue) {
			if (containsKey(key))
				return get(key);
			else
				return defaultValue;
		}
	}
}
