package au.edu.wehi.idsv;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

import java.io.File;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Random;
import java.util.SortedSet;
import java.util.Spliterator;
import java.util.Spliterators;
import java.util.concurrent.ExecutorService;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;

import org.apache.commons.configuration.Configuration;
import org.apache.commons.configuration.ConfigurationException;
import org.apache.commons.lang3.StringUtils;

import com.google.common.base.Function;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterables;
import com.google.common.collect.Iterators;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;

import au.edu.wehi.idsv.configuration.GridssConfiguration;
import au.edu.wehi.idsv.debruijn.DeBruijnGraph;
import au.edu.wehi.idsv.debruijn.DeBruijnGraphBase;
import au.edu.wehi.idsv.debruijn.DeBruijnNodeBase;
import au.edu.wehi.idsv.debruijn.DeBruijnPathGraph;
import au.edu.wehi.idsv.debruijn.DeBruijnPathNode;
import au.edu.wehi.idsv.debruijn.DeBruijnPathNodeFactory;
import au.edu.wehi.idsv.debruijn.KmerEncodingHelper;
import au.edu.wehi.idsv.debruijn.PackedKmerList;
import au.edu.wehi.idsv.debruijn.ReadKmer;
import au.edu.wehi.idsv.debruijn.ReadKmerIterable;
import au.edu.wehi.idsv.debruijn.positional.AggregateNodeIterator;
import au.edu.wehi.idsv.debruijn.positional.ImmutableKmerNode;
import au.edu.wehi.idsv.debruijn.positional.KmerNode;
import au.edu.wehi.idsv.debruijn.positional.KmerNodeUtil;
import au.edu.wehi.idsv.debruijn.positional.KmerPathNode;
import au.edu.wehi.idsv.debruijn.positional.KmerPathSubnode;
import au.edu.wehi.idsv.debruijn.positional.KmerSupportNode;
import au.edu.wehi.idsv.debruijn.positional.PathNodeIterator;
import au.edu.wehi.idsv.debruijn.positional.PathNodeIteratorTest;
import au.edu.wehi.idsv.debruijn.positional.SupportNodeIterator;
import au.edu.wehi.idsv.graph.PathNode;
import au.edu.wehi.idsv.graph.PathNodeFactory;
import au.edu.wehi.idsv.metrics.IdsvSamFileMetrics;
import au.edu.wehi.idsv.metrics.IdsvSamFileMetricsCollector;
import au.edu.wehi.idsv.picard.BufferedReferenceSequenceFile;
import au.edu.wehi.idsv.picard.ReferenceLookup;
import au.edu.wehi.idsv.sam.ChimericAlignment;
import au.edu.wehi.idsv.sam.SAMRecordMateCoordinateComparator;
import au.edu.wehi.idsv.sam.SAMRecordUtil;
import au.edu.wehi.idsv.util.AutoClosingIterator;
import au.edu.wehi.idsv.visualisation.NontrackingSubgraphTracker;
import gridss.analysis.CigarDetailMetrics;
import gridss.analysis.CigarSizeDistribution;
import gridss.analysis.IdsvMetrics;
import gridss.analysis.InsertSizeDistribution;
import gridss.analysis.MapqMetrics;
import htsjdk.samtools.CigarOperator;
import htsjdk.samtools.DefaultSAMRecordFactory;
import htsjdk.samtools.QueryInterval;
import htsjdk.samtools.SAMFileHeader;
import htsjdk.samtools.SAMFileWriter;
import htsjdk.samtools.SAMLineParser;
import htsjdk.samtools.SAMReadGroupRecord;
import htsjdk.samtools.SAMRecord;
import htsjdk.samtools.SAMRecordCoordinateComparator;
import htsjdk.samtools.SAMSequenceDictionary;
import htsjdk.samtools.SAMTag;
import htsjdk.samtools.SamPairUtil;
import htsjdk.samtools.ValidationStringency;
import htsjdk.samtools.fastq.FastqRecord;
import htsjdk.samtools.metrics.Header;
import htsjdk.samtools.metrics.MetricsFile;
import htsjdk.samtools.reference.ReferenceSequenceFile;
import htsjdk.samtools.reference.ReferenceSequenceFileFactory;
import htsjdk.samtools.util.CloseableIterator;
import htsjdk.samtools.util.ProgressLoggerInterface;
import htsjdk.variant.variantcontext.VariantContext;
import htsjdk.variant.variantcontext.writer.VariantContextWriter;
import htsjdk.variant.vcf.VCFHeader;
import picard.analysis.InsertSizeMetrics;

public class TestHelper {
	public static final long LCCB = GenomicProcessingContext.LINEAR_COORDINATE_CHROMOSOME_BUFFER;
	public static final BreakendDirection FWD = BreakendDirection.Forward;
	public static final BreakendDirection BWD = BreakendDirection.Backward;

	public static byte[] B(String s) {
		if (s == null)
			return null;
		return s.getBytes(StandardCharsets.US_ASCII);
	}
	
	public static byte[] B(byte b, int repeats) {
		byte[] result = new byte[repeats];
		for (int i = 0; i < result.length; i++) {
			result[i] = b;
		}
		return result;
	}
	
	public static byte[] B(int b, int repeats) {
		return B((byte)b, repeats);
	}

	public static String S(byte[] b) {
		if (b == null)
			return null;
		return new String(b, StandardCharsets.US_ASCII);
	}
	
	public static String S(String s, int repeats) {
		return StringUtils.repeat(s, repeats);
	}

	@SafeVarargs
	public static <T> List<T> L(T[]... list) {
		List<T> result = Lists.newArrayList();
		for (int i = 0; i < list.length; i++) {
			if (list[i] == null)
				continue;
			for (int j = 0; j < list[i].length; j++) {
				if (list[i][j] == null)
					continue;
				result.add(list[i][j]);
			}
		}
		return result;
	}

	@SafeVarargs
	static public <T> List<T> L(T... data) {
		List<T> list = Lists.newArrayList(data);
		return list;
	}
	
	public static IdsvVariantContextBuilder minimalVariant() {
		return (IdsvVariantContextBuilder) new IdsvVariantContextBuilder(
				getContext()).chr("polyA").start(1).stop(1).alleles("A", "C");
	}

	public static IdsvVariantContextBuilder minimalBreakend() {
		IdsvVariantContextBuilder builder = new IdsvVariantContextBuilder(
				getContext());
		builder.chr("polyA").start(1).stop(1).alleles("A", "A.");
		return builder;
	}

	public static VariantContextDirectedBreakpoint BP(String id, BreakpointSummary bp) {
		return BP(id, bp, "");
	}
	
	public static VariantContextDirectedBreakpoint BP(String id, BreakpointSummary bp, String untemplatedSequence) {
		IdsvVariantContextBuilder builder = minimalBreakend()
				.breakpoint(bp, untemplatedSequence);
		builder.id(id);
		return (VariantContextDirectedBreakpoint)builder.make();
	}
	
	static public SAMFileHeader getHeader() {
		SAMFileHeader header = new SAMFileHeader();
		header.setSequenceDictionary(getSequenceDictionary());
		SAMReadGroupRecord rg = new SAMReadGroupRecord("RG");
		rg.setSample("sample");
		header.addReadGroup(rg);
		return header;
	}
	
	static public SAMLineParser getParser() {
		SAMLineParser parser = new SAMLineParser(new DefaultSAMRecordFactory(), ValidationStringency.LENIENT, getHeader(), null, null);
		return parser;
	}

	static public SAMSequenceDictionary getSequenceDictionary() {
		return SMALL_FA.getSequenceDictionary();
	}

	static public byte[] GetPolyA(int length) {
		byte[] b = new byte[length];
		for (int i = 0; i < length; i++)
			b[i] = 'A';
		return b;
	}

	static public List<SAMRecord> sorted(List<SAMRecord> list) {
		List<SAMRecord> out = Lists.newArrayList(list);
		Collections.sort(out, new SAMRecordCoordinateComparator());
		return out;
	}

	static public List<SAMRecord> mateSorted(List<SAMRecord> list) {
		List<SAMRecord> out = Lists.newArrayList(list);
		Collections.sort(out, new SAMRecordMateCoordinateComparator());
		return out;
	}

	static public SAMRecord[] OEA(int referenceIndex, int pos, String cigar,
			boolean forward) {
		SAMRecord mapped = new SAMRecord(getHeader());
		mapped.setReferenceIndex(referenceIndex);
		mapped.setAlignmentStart(pos);
		mapped.setCigarString(cigar);
		mapped.setReadNegativeStrandFlag(!forward);
		mapped.setReadName(String.format("%s-%d-%d-%s", cigar, referenceIndex,
				pos, forward));
		mapped.setMappingQuality(10);
		mapped.setFirstOfPairFlag(true);
		mapped.setReadUnmappedFlag(false);
		clean(mapped);
		SAMRecord unmapped = Unmapped(mapped.getReadLength());
		unmapped.setSecondOfPairFlag(true);
		mapped.setReadUnmappedFlag(true);
		clean(mapped, unmapped);
		return new SAMRecord[] { mapped, unmapped };
	}

	public static NonReferenceReadPair NRRP(SAMRecord... pair) {
		NonReferenceReadPair rp = NRRP(SES(), pair);
		return rp;
	}

	public static NonReferenceReadPair NRRP(SAMEvidenceSource source,
			SAMRecord... pair) {
		pair[1].setReadName(pair[0].getReadName());
		return NonReferenceReadPair.create(pair[0], pair[1], source);
	}
	public SplitReadEvidence A(BreakpointSummary bs) {
		// TODO handle more cases
		assert(bs.start == bs.end);
		assert(bs.start2 == bs.end2);
		assert(bs.direction2 == BWD);
		SAMRecord r = AssemblyFactory.createAnchoredBreakend(getContext(), AES(), new SequentialIdGenerator(String.format("asm(%d)-%d%s", bs.referenceIndex, bs.start, bs.direction.toChar())), bs.direction, null, bs.referenceIndex, bs.start, 1, B("TT"), B("TT"));
		SAMRecord ra = new SAMRecord(getContext().getBasicSamHeader());
		ra.setReferenceIndex(bs.referenceIndex2);
		ra.setAlignmentStart(bs.start2);
		ra.setReadUnmappedFlag(false);
		ra.setCigarString("1M");
		ra.setMappingQuality(30);
		if (bs.direction == bs.direction2) {
			ra.setReadNegativeStrandFlag(true);
		}
		SplitReadIdentificationHelper.convertToSplitRead(r, ImmutableList.of(r));
		return SplitReadEvidence.create(AES(), r).get(0);
	}

	public static DirectedEvidence E(int referenceIndex, int start,
			BreakendDirection direction) {
		return new MockDirectedEvidence(referenceIndex, direction, start);
	}
	public static DirectedEvidence E(int referenceIndex, int start, int end,
			BreakendDirection direction) {
		return new MockDirectedEvidence(referenceIndex, direction, start, end);
	}

	public static SoftClipEvidence SCE(BreakendDirection direction, SAMRecord... record) {

		return SCE(direction, SES(), record);
	}
	public static IndelEvidence IE(SAMRecord r) {
		return IE(SES(), r);
	}

	public static IndelEvidence IE(SAMEvidenceSource ses, SAMRecord r) {
		return IE(ses, r, 0);
	}
	
	public static IndelEvidence IE(SAMEvidenceSource ses, SAMRecord r, int indelIndex) {
		return IndelEvidence.create(ses, 0, r).get(indelIndex);
	}

	public static SoftClipEvidence SCE(BreakendDirection direction, SAMEvidenceSource source, SAMRecord... record) {
		assert(record.length == 1);
		return SoftClipEvidence.create(source, direction, record[0]);
	}

	public static SoftClipEvidence AE() {
		return SoftClipEvidence.create(AES(), FWD, AssemblyFactory.createAnchoredBreakend(getContext(), AES(), new SequentialIdGenerator("asm"), 
				FWD,
				null,
				0, 1,
				1, B("ATT"), new byte[] { 7, 7, 7 }));
	}
	public static class MockCigarSizeDistribution extends CigarSizeDistribution {
		public MockCigarSizeDistribution() {
			super(new ArrayList<CigarDetailMetrics>());
		}
		@Override
		public double getPhred(CigarOperator op, int softClipLength) {
			return softClipLength;
		}
	}

	public static class MockMetrics extends IdsvSamFileMetrics {
		public MockMetrics() {
			super(new InsertSizeMetrics() {{
				MEAN_INSERT_SIZE = 300;
				MEDIAN_INSERT_SIZE = 300;
				MIN_INSERT_SIZE = 300;
				MAX_INSERT_SIZE = 300;
				MEDIAN_ABSOLUTE_DEVIATION = 30;
			}}, new IdsvMetrics() {{
				MAX_READ_LENGTH = 100;
				MAX_PROPER_PAIR_FRAGMENT_LENGTH = 300;
				MIN_PROPER_PAIR_FRAGMENT_LENGTH = 300;
				READ_PAIRS = 1000;
				READ_PAIRS_ONE_MAPPED = 25;
				READ_PAIRS_ZERO_MAPPED = 50;
				READ_PAIRS_BOTH_MAPPED = READ_PAIRS - READ_PAIRS_ONE_MAPPED - READ_PAIRS_ZERO_MAPPED;
				READS = 2 * READ_PAIRS;
				MAPPED_READS = READS - READ_PAIRS_ONE_MAPPED - 2*READ_PAIRS_ZERO_MAPPED;
			}}, new MapqMetrics() {{
				ZERO_MAPQ = 10;
				MAPPED_READS = 1500;
				MIN_MAPQ = 0;
				MAX_MAPQ = 50;
			}}, new InsertSizeDistribution(
					new int[] { 1, 50, 100, 200, 300, 400, },
					new double[] { 1, 50, 500, 50, 20, 10, }),
			new ArrayList<CigarDetailMetrics>());
		}
		@Override
		public CigarSizeDistribution getCigarDistribution() {
			return new MockCigarSizeDistribution();
		}
	}

	public static FileSystemContext getFSContext() {
		return new FileSystemContext(new File(System.getProperty("java.io.tmpdir")), 500000);
	}
	private static Configuration defaultConfig;
	public static Configuration getDefaultConfig() {
		if (defaultConfig == null) {
			try {
				defaultConfig = GridssConfiguration.LoadConfiguration(null);
			} catch (ConfigurationException e) {
				throw new RuntimeException(e);
			}
		}
		return defaultConfig;
	}
	public static GridssConfiguration getConfig(File workingDirectory) {
		GridssConfiguration config;
		config = new GridssConfiguration(getDefaultConfig(), workingDirectory);
		config.getSoftClip().minAverageQual = 0;
		config.minAnchorShannonEntropy = 0;
		config.getAssembly().minReads = 2;
		config.getAssembly().positional.trimSelfIntersectingReads = false;
		config.getVariantCalling().breakendMargin = 3;
		config.getVisualisation().buffers = false;
		return config;
	}
	public static GridssConfiguration getConfig() {
		return getConfig(new File(System.getProperty("java.io.tmpdir")));
	}
	public static ProcessingContext getContext(ReferenceLookup ref) {
		ProcessingContext pc = new ProcessingContext(getFSContext(), SMALL_FA_FILE, ref, new ArrayList<Header>(),
				getConfig());
		pc.registerCategory("Normal");
		pc.registerCategory("Tumour");
		return pc;
	}
	public static ProcessingContext getContext() {
		return getContext(SMALL_FA);
	}
	static public SAMRecord[] applying(Consumer<SAMRecord> f, SAMRecord... data) {
		for (SAMRecord r : data) {
			f.accept(r);
		}
		return data;
	}
	static public SAMRecord[] withReadName(String name, SAMRecord... data) {
		for (SAMRecord r : data) {
			r.setReadName(name);
		}
		return data;
	}
	static public SAMRecord[] withSequence(String seq, SAMRecord... data) {
		byte[] qual = B(seq);
		for (int i = 0; i < qual.length; i++) {
			qual[i] = 30;
		}
		return withSequence(seq, qual, data);
	}
	static public SAMRecord[] withSequence(String seq, byte[] qual, SAMRecord... data) {
		for (SAMRecord r : data) {
			assert(r.getReadLength() == seq.length());
			r.setReadBases(B(seq));
			r.setBaseQualities(qual);
		}
		return data;
	}

	static public SAMRecord[] withSequence(byte[] seq, SAMRecord... data) {
		for (SAMRecord r : data) {
			r.setReadBases(seq);
		}
		return data;
	}

	static public SAMRecord[] withQual(byte[] seq, SAMRecord... data) {
		for (SAMRecord r : data) {
			r.setBaseQualities(seq);
		}
		return data;
	}

	static public SAMRecord[] withMapq(int mapq, SAMRecord... data) {
		for (SAMRecord r : data) {
			r.setMappingQuality(mapq);
		}
		return data;
	}
	
	static public SAMRecord[] withName(String readName, SAMRecord... data) {
		for (SAMRecord r : data) {
			r.setReadName(readName);
		}
		return data;
	}
	
	static public SAMRecord[] withAttr(String attr, Object value, SAMRecord... data) {
		for (SAMRecord r : data) {
			r.setAttribute(attr, value);
		}
		return data;
	}

	static public SAMRecord[] onNegative(SAMRecord... data) {
		for (SAMRecord r : data) {
			r.setReadNegativeStrandFlag(true);
		}
		return data;
	}

	public static SAMRecord[] withNM(SAMRecord... data) {
		for (SAMRecord r : data) {
			r.setAttribute("NM", null);
			SAMRecordUtil.ensureNmTag(SMALL_FA, r);
		}
		return data;
	}

	static public SAMRecord[] DP(int referenceIndex, int pos, String cigar,
			boolean forward, int referenceIndex2, int pos2, String cigar2,
			boolean forward2) {
		SAMRecord[] dp = new SAMRecord[] {
				OEA(referenceIndex, pos, cigar, forward)[0],
				OEA(referenceIndex2, pos2, cigar2, forward2)[0] };
		dp[0].setReadName(dp[0].getReadName() + "_" + dp[1].getReadName());
		dp[1].setReadName(dp[0].getReadName() + "_" + dp[1].getReadName());
		clean(dp[0], dp[1]);
		dp[0].setProperPairFlag(false);
		dp[1].setProperPairFlag(false);
		return dp;
	}
	static public SAMRecord Read(int referenceIndex, int pos, int length) {
		return Read(referenceIndex, pos, String.format("%dM", length));
	}

	static public SAMRecord Read(int referenceIndex, int pos, String cigar) {
		SAMRecord record = new SAMRecord(getHeader());
		record.setReadName(String
				.format("%s-%d-%d", cigar, referenceIndex, pos));
		record.setReferenceIndex(referenceIndex);
		record.setAlignmentStart(pos);
		record.setReadUnmappedFlag(false);
		record.setReadPairedFlag(false);
		record.setReadNegativeStrandFlag(false);
		record.setCigarString(cigar);
		record.setMappingQuality(10);
		clean(record);
		return record;
	}
	static public SAMRecord Read(String chimericAlignment) {
		ChimericAlignment ca = new ChimericAlignment(chimericAlignment);
		SAMRecord record = new SAMRecord(getHeader());
		record.setReferenceIndex(record.getHeader().getSequenceIndex(ca.rname));
		record.setAlignmentStart(ca.pos);
		record.setReadUnmappedFlag(false);
		record.setReadPairedFlag(false);
		record.setReadNegativeStrandFlag(ca.isNegativeStrand);
		record.setCigar(ca.cigar);
		record.setMappingQuality(ca.mapq);
		record.setReadName(chimericAlignment);
		clean(record);
		return record;
	}

	/**
	 * Read pair supporting the reference
	 * 
	 * @param referenceIndex
	 * @param pos
	 * @param readLength
	 * @return
	 */
	static public SAMRecord[] RP(int referenceIndex, int pos, int readLength) {
		return RP(referenceIndex, pos, pos + 2 * readLength, readLength);
	}

	/**
	 * Read pair supporting the reference
	 * 
	 * @param referenceIndex
	 * @param pos
	 *            start of first
	 * @param pos2
	 *            start of second
	 * @param readLength
	 *            read lengths
	 * @return read pair
	 */
	public static SAMRecord[] RP(int referenceIndex, int pos, int pos2,
			int readLength) {
		SAMRecord read1 = Read(referenceIndex, pos, readLength);
		SAMRecord read2 = Read(referenceIndex, pos2, readLength);
		read2.setReadNegativeStrandFlag(true);
		read1.setProperPairFlag(true);
		read2.setProperPairFlag(true);
		read1.setReadName(String.format("RP%d-%d:%d-%d", readLength, referenceIndex, pos, pos2));
		read1.setFirstOfPairFlag(true);
		read2.setFirstOfPairFlag(false);
		clean(read1, read2);
		return new SAMRecord[] { read1, read2 };
	}
	public static SplitReadEvidence SR(SAMRecord read, SAMRecord realigned) {
		return SR(SES(), read, realigned);
	}
	public static SplitReadEvidence SR(SAMEvidenceSource source, SAMRecord read, SAMRecord realigned) {
		List<FastqRecord> ralist = SplitReadIdentificationHelper.getSplitReadRealignments(read, false, source.getContext().getEvidenceIDGenerator());
		if (ralist.size() == 1) {
			realigned.setReadName(ralist.get(0).getReadHeader());
		}
		SplitReadIdentificationHelper.convertToSplitRead(read, ImmutableList.of(realigned));
		return SplitReadEvidence.create(source, read).get(0);
	}
	public static SplitReadEvidence SR(SAMEvidenceSource source, BreakendDirection dir, SAMRecord read, SAMRecord realigned) {
		List<FastqRecord> ralist = SplitReadIdentificationHelper.getSplitReadRealignments(read, false, source.getContext().getEvidenceIDGenerator());
		if (ralist.size() == 1) {
			realigned.setReadName(ralist.get(0).getReadHeader());
		}
		SplitReadIdentificationHelper.convertToSplitRead(read, ImmutableList.of(realigned));
		return SplitReadEvidence.create(SES(), read).stream()
				.filter(e -> e.getBreakendSummary().direction == dir)
				.findFirst()
				.get();
	}
	/**
	 * Fills in missing information from the given read
	 * 
	 * @param record
	 */
	public static void clean(SAMRecord record) {
		if (record.getHeader() == null) {
			record.setHeader(getHeader());
		}
		int cigarLength = record.getCigar() == null ? 0 : record.getCigar()
				.getReadLength();
		int readLength = record.getReadBases() == null ? 0 : record
				.getReadLength();
		int targetLength = cigarLength;
		if (targetLength == 0)
			targetLength = readLength;
		if (targetLength == 0)
			targetLength = 100;
		if (record.getReadBases() == null
				|| record.getReadBases().length != targetLength) {
			record.setReadBases(GetPolyA(targetLength));
		}
		if (record.getBaseQualities() == null
				|| record.getBaseQualities().length != targetLength) {
			byte[] quals = new byte[targetLength];
			Arrays.fill(quals, (byte) 10);
			record.setBaseQualities(quals);
		}
		if (record.getCigar() == null || record.getCigarLength() == 0) {
			record.setCigarString(String.format("%dM", targetLength));
		}
		if (!StringUtils.isNotBlank(record.getReadName())) {
			record.setReadName("placeholderReadName");
		}
		if (record.getReadGroup() == null) {
			record.setAttribute(SAMTag.RG.name(), getHeader().getReadGroups()
					.get(0).getId());
		}
		if (record.getReferenceIndex() != null
				&& record.getReferenceIndex() >= 0) {
			record.setReadUnmappedFlag(false);
		} else {
			record.setReadUnmappedFlag(true);
		}
		record.setAttribute("NM", 0);
		// if (record.getReadBases().length == targetLength)
		// SAMRecordUtil.ensureNmTag(SMALL_FA, record);
	}

	public static void clean(SAMRecord r1, SAMRecord r2) {
		clean(r1);
		clean(r2);
		r1.setFirstOfPairFlag(true);
		r2.setFirstOfPairFlag(false);
		r1.setSecondOfPairFlag(false);
		r2.setSecondOfPairFlag(true);
		r2.setReadName(r1.getReadName());
		r1.setReadPairedFlag(true);
		r2.setReadPairedFlag(true);
		boolean properpair = !r1.getReadUnmappedFlag()
				&& !r2.getReadUnmappedFlag()
				&& r1.getReferenceIndex().equals(r2.getReferenceIndex())
				&& r1.getReadNegativeStrandFlag() != r2
						.getReadNegativeStrandFlag();
		r1.setProperPairFlag(properpair);
		r2.setProperPairFlag(properpair);
		SamPairUtil.setMateInfo(r1, r2, true);
		SAMRecordUtil.calculateTemplateTags(ImmutableList.of(r1, r2), Sets.newHashSet(SAMTag.R2.name(), SAMTag.Q2.name(), SAMTag.MC.name(), SAMTag.MQ.name()), false, true, true);
	}
	public static final double DELTA = 1e-4;
	public static final byte[] POLY_A = B("AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA");
	public static final byte[] POLY_ACGT = B("ACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGT");
	public static final byte[] RANDOM = B("CATTAATCGCAAGAGCGGGTTGTATTCGACGCCAAGTCAGCTGAAGCACCATTACCCGATCAAAACATATCAGAAATGATTGACGTATCACAAGCCGGATTTTGTTTACAGCCTGTCTTATATCCTGAATAACGCACCGCCTATTCGAACGGGCGAATCTACCTAGGTCGCTCAGAACCGGCACCCTTAACCATCCATATCCTTCAGTTCCATAGGCCTCTGTGCGGGATTTGTGAACGTTCGGACCCCCCAGGACAGCGTGCATGAGTTTCTAAACGCCCGGCCCCAACGGTCGTCCGGATGAATACTAGGCTAGGCATTGCGTTTACCTAGCTGCGTAAAATTTCGCTGATGGCGGGGATTCACACCACGCATTGTCTGAACACTTGTCTCTTTAGGTTCACTTTTGAGATATCATCAATCGAGAACGTTAGCATAATATCGGTCCTTAACGCCCGTAGCCCATTACAAGCCTCGAATTATGCACCTCCCAGGTACATATATTAGGGTTGTAACGCAAGTCATAATGGAAGCATCGCCGAAAGCGCACGCTGCTCAGTTTAGCACCATCGTCGGGCAAAGACATCCTGGACACGAGTGATAGGTCTTCGGATTGGGAGTCTCATACGCGTGCTAATGCTAACCGCACTGCTAATAATGGGTCCCCTATATGTAAGAACCGTTACGATACCTTCTGTCACAATCACACTGTGTGCAGAAGGGACTGGCTTATTTCCGTGGTCGCCCGGGCACAGTCTTCCACCAGCCGCCACGCCCGCTTGGTTCTCGTTGTCGGCGGTACTCGGGCCATTAGATTAATTTCAAATAATCGGCGGAGCGGGTCTGTGGGCCCTATTTGAAACCATCCTAGATTGTGTCTACTTAGGGAAACGGCCTGGGACTTCGTGATAGTAATGGATCCCCGAAGAAATCTATTTGTACTTACTGGGACTTTCCTCAATCTTCTCAAGAATGACAGCGTGCGGGACTGCTCATCCGAAGGAGGGGCGAGGGAAAACGACCCATCTTTGCTGGGGACAGGGCAGCTGCCACTAATATTGGTTTATGAATAGATTTAACAACCTTTCCAGGGTTGATATTGGCCGCTATTAAGTCATCAATCGAGTAAATTTAACCGTGGGACTCATTTGACTGCTCTTGGAAGCTCGACTACCTCGACGGCTAATGGGTTATCTCCCGACGTAAGGTCAACGGGGCCGGGCAATTCATTCCTAGCAATAAGGGATTCAATTTCTATACGTAAATGTGAGGAAGTAGACGTTACAGCGGGCATTTTAAGGCGACTTCTGGTTGTGGCCTGGCGAGACGACCTAGTTCGATAGACTTCCTGTCAAGACCAGTAAGCGTACTGTTTGTGCACCGGAACTGCGGCGAACGTAGCCGTAGCGCGGTACTTCGAATCGGAGACGGGCCGCAGGTACGTTGTTCAAAGCTGACAATAGAATATCGGGTGGCTCACACACACCAGAGGTAAATGCTACGTTTGACCGTCTCCGTACCATCCGTGATGGCTTCCTCCATAACTCGCCCTCTGTATGTAGCTATGATCTACGACGTGTTAGCCCTCTGCCTAAATGGGTGCAACAAGAACGCGGATGTCGTCCGGATGTCGACATTCCGGCCGGTCACACAAAGTAAAGCCAATAAGGAGTCTAAATCCTATCCACTGGCCCCTAGTGAAGGCCTGCACTTGGAGTAAGCGGACTGGCTATAATTTACAGGATAAGGGAAGCAAATAGGACAACGTCGATTAGCTAGACGTCGATAGGGATTGACACTCAGGTAGGTGTATGAAACGCTGTGTGGTGTTTGCCGAGCAAAACAGATTGCGACGTATGCGTATTTGCACCACCAATCCCAAGAAAAGGTAATGCTACGGGAAGCTAATATGACTGCCGTCGGGGGTGCTATGGAAGGCTCATACGCCAGGGTTGTCAACCTCTTGGACGTCCGAGGCCGCACGCAACTGGCAAGAAGAGTTCAATCCCGGCCCAGCTTTCCGCCTGCGAGGTATGGCACAGTGCAGTGATCCTAGGTATTAACAGGGGAGGAGAGACGATAGGTCCACTGCTGGGCCTGAGTTCTGATGATTTGGGCTCGCACTGATAATAGGGCTATTGGCCCTTCGTACTGATGGCAAATGACGCCCAATCGAAACTCCACCACACCAACCGAACCTCCTAAGAATGATTACAGCCTTGGAGATAGTGCGAACTGTTGGTCGTGTTCTCATCAACGCGTAGGTTCATTTGGGCAACATGGTCAGAGCGACGTCCCTAATTCTAATTTATAGAGAGCCCGGGAAATAATCTACTGGTTGACGCAGATTTTAATATCAGAGGGTACGTATTCAACGGAGGCGATCGGAACTTGGATCCCCGCATCTCCATACCCCCAAAAACCAGAACGAGCGTAAAAGGACTAGGTCAGAGTGGCCTACCGGTACAGCAACAGTGCGTTCACCTTAGCAAACCTGGAAACGCATGGTTTGAATAAAAATAGCTTTACTGACTGCGATAGAAAGTCGGCTTCGTTCCAACTGTGAGACAGTCGTGTCTCAGAGCAGAGGAGCTGAAGTGGGGAACCGGCGGCCATCACCGTCAGAAGGCCTTCCTCCTGTTTCATTTGGACGTTACTGGCACATCAAGCAATCCCCTGGAGGAGACCCTCGAGCCTAGTTGTCAATCAAATAACCCTACCGACTTCTATGTCTCGTGCGACCGTCGAAAATATGATTGCTTGCCACTAGACAGTCTGGCTGTTCGGCGATGTAACCAACTGTTAGAGAGGCGACAAGCCCTGATTGACCGTAGTGGTATGTGTTTTACACCCGTCGGGAAAGGGTTACGACACTGCGATGTTAAGGGCGCATCGCGACATTATTCGCTAGAACAGGTAACAACCATTAAATTGACTCTACCGGATAGCGCCGTTTCCTGTAATATATGTGCAGGAATGTCACCATGGGTGGTTCTCTCTACCTCCTCGATAAAGTACTGAGTGCTGCGGTGAGGAAACTGTTGTCCGTTCCAATCGACTGCCTACTGGGACATGCGTCTGCGGATTCCGGCATTCTCGTACCAGAGACCGGCGCATAGGAGAAATCCGCGACGATTCTCCGGAAGGCATGGGCTAACGGGGTATTCCTCTCTATATGACTCATCGTGCTCCAGCGTACCTCTGAGCCGATTGGTGCATCGGAGGACTGTCGATCCAACTGCTGCAGGGACTGAACTCTGTTCTCAGACCGGTACGTCTTGTATAAGTAGTCTCCCAGACTGGCGTGGTGTGCCGAACCCGTCGATCTGTTCAAAATTTCGCCTACCAGTCGCACGGACGTCACATTAGCTTAGACGGACGAATCTCCCAGTTATGCGTGATAGGGAGATAGGGCAGGCCGTGCGTCGCGTTGGACCGAGATGGCACGCCGCAAGGCAGCAGAAAGGGAAAAAGCGCCCGTGCCAACAGATCGATCAGTCCGTTTCCGAGGGTGCTTATTTTTAAGATATCCCTGGTTCTATGCCCAGGTAAGCGCGAGTACTGTCTTGTAGAGGACGTACAGGCATCTCTTACATGTGGAATTCCTGAGCACTTCTTGACGTCGTGTAACAACCGAAAGTATATTTAGCTCGGTAATTCATGTTCGGGTTATAATCAATTATCCTATTTTCAACGTATAGCAGCAACGGGGCCGGTGGTCCGCGTAGTGGATCGAAGCAGCCCTCTTGCTGACTTTAAGTACACGGCTAACGGTGGCTCGCGCCAAAACACTTTGTACGAGCTCGCAATCACCGGTCGCTGTATTTTGGGCCTAGTTTCCAGTGACCGAGAAGCTACCGTATACTATTGCACAAAGCGAAAGCATTTTGGTAGGTCAGCGGCTGCGATATCATAATGACCCGTATCTGAACGTCTCTGGACTTATAAATCTGAGAATCCATCTGCTAGCGACTCGCCTGGGAACAGGCTTTTTGTGCCATGCCTGATGCCATACTCCGAACACATGCCTAATTCGGCACCTATTAGCAACCCGGGGGCGATTTATAGTCCCCGCAGCCAAGTGAAGCAGCAACTCGACATGGAACTCGAGAAATCGGTACGGTATCACGACAGGTATTGGTATTGTCCACGATATCGATAGCCTATTTCGAAACCTGGTCCTTATAGGACGTGCTTCATATGTGCTTGCTCTCACTCTATTCGAAGCTCCACATACCGACCAGCTGTTTACTAAGTCAACCGCCAAATAGTCGCGGCGCTAGACACAAACTCCCTATAAAGGCAGGGATATGGAGCCAGCGCTACGAAAGGATTGCGTAGTTATTGCTCGGCCTTGTGACTGGCCGGACACAGATATGTCGTTCGCCTACGATAAGTGCGAACCAGGCCGTTGCAACCTTACGTTAATGCGGATACTGTTAAGCGTAATTGGACGAAATCCGGCGTAGGGTGCAGTTATCGACGTCGAAAAGCTCAGGGCGCGTAGTCCGATCTCGCTTAGGCCATCGGCTGGATTCGTGACGCGAACCGACCGGAACGGATGGAACTAGTACTTGGCCGTGTCATACGCATCGTCAAATCGTCTGATGTCGGGTGATGATGTGCGAACTACAGTCCTCCAGCGAAAAACAGTCTGAGCAAGCTTATGATGCTGATGGCACTGGATCCAAGGAAGGGTATTGCCGAAATTAATGACAGTCTCTGTAAAGGCCGACGACCCATTGTAAATTGTAACGCGTTTTCACGCTCCTGCTGTGTCGGGCCCCGGTCGGCTGTCGCCATGCCGACTCCTGGGTGGGCTACAATTGTAACATGCGGATAGGATCGCACTTCATAGAACACATTATAGTAATTTAACTTCCTGTTCCTGGAGGTAAGCGCGTCGCCTGATCTTGAGAGTCCCGTTCGTGAGAACCACAATTCACAGATTTGACGACTCCTTTTAATATAGCGACGGAGCACACCGCTCGTAATTCAGCTCGTAGTCTTTGCGTCTATCCACACACTCCCGGGCATATAGCTAACCGGACGCACAACTTACGGATCCTTACAACGTACAGCATTGGCACACGGAAGTTCCAGATGCTTCAATGGTACTTTCTTTTCAATTGGACAGGCTACGTAGCTAGTGCAATTGATATCAATGGGACATATGCTGAGAAAGCGACGGGGGCCGTGGGTTTTTTTGAGAGATGTAGCCGTACCCGGCCTGGATAGGCAGGCGCAATAGGAACGCAGGCAGCAATGCTTCGCAAATTGACAATATGCAGTGTCGCACTCGCGAGCAGCTCCTGGCGAGCTACGGATTAATTGTCATCTGAATTCAGGACGCAAGCCCGCTAAACCACGGGGTTCGGCACTAAGGACCATATTTATGCGACCGTTAGCTAATCGCCAGACGTTCGTGTCTTGGCGCGAATATAATATTCCCAGGGACTGTCGTTATACGATCTGGCCGGCATGAGACTTTACAAACTGTAGTTATTGGCCCACCGTTTACGCCGTGGTAATTTTGGGCAGGCTTGGAGAAGATTGAGATATAGCAGCGAACCACAGCGCCGGTCGATCGGACAAATTCTCCCTGTAAAGTGTTTATACCGTTTAGGAGATTGCGTAGGCACAGCAGCTTACGCCGATATGTCGAGGACACCCTTTCGATTATGAGCTCGTTCTAAGTATGCAGTAAAAGACCGATTGTTTTTCAGTAGGTGCATACCATAAAGGAAAGGGCGATTTACGTACACTCTACCTTTGATCCTAAAGACCTGGGTTCGTTGCAATGTCTTGTACCACATCCCCCATGAGGCTAACGTTATAGTCGACTATCGTCATCATCATTCTCTACGTATTGTTGCAATGGGACGAACGTGGACGCGTGTAAGAAACGCCAGTGTTGAGGTGTCCAGGAACAGGGAAAACCATAGTCTCGGCCGCTCGGTGTGGTACTTGCAAAACAAAAAAGGTGTTAGTCGCGCAGTTACACCTGGCTGTATTCGTCAACCACGACATAAGGGCGTAGTGACGATTAAGTGTTAGGCTATGTGGTGAATCTGCGTGTGGTGGCACGGTATAAGCCGGATAGCAATAGACCCCACTATAGACTTAGTAGTTTTGCCCTGAATAACTCCGGGGAATGCAAGATCAGAAAAAACCTTCAGCGATCATATGAGGTTGCAGTATGGTACGGATGGGGTCCAGTCTTGAAAAAATAAGGTTATCCCCAATCTGCCGGTCAAATAGTTAACCGCCCTTCCCTATGCACCAACTTTTTTAGGATCGGCTCGGTTCCAATTAATATGTGTGGGGCTAGCCTCGCACGTCGGCCACAGCGAAAAGAATATCGAGATTTTGGAACGTGCCCAAGCATCCATAATAGTCCTTGATCTTCCAACCAGACAGCGATGCTCGAACCGGTCCCCAAATAGAATGACTAAACGGACGAAGAGAGGTTTCAATCACTCCTCTGTGGGCCGATCACGATTACAGTTAGCACCAGCGTTCGATCTCCTCTAGAGAACCCACACGCGATCCTCATATTGGTGGGGCCAATGATACTAAGCTCAGGTCCCTAATGATACCTCCATCGCTATACGCGTTTGGTGTAACCAGATTTTCCCGGCGGGACTACCTCACTGGTTTTCGGCGGTGCAGCCTTTGTGTTCGAGGTACGCGCCGACCGCAAGCCCCCGCGCCCGTACAAAACTTCGTCTGTTCAATAACAGCGGCGTTAAGCTAACGGCGAGAAATCTCTCACTTCCCGTCCAAAGGAAGACCTGAAGTCATAATCTTGCAACACCGAAGAACTGGCTAATGGGTTGTAAGATATGCTATTCATATGGTAGGAAGGTTCGCAAGAGATTTTAGAGGGTCTGTCTAGCCAGGAACCCCAGCGACTATGTTCTTCTCCGCATCGGCCCAACCACCTGGCAATTTCCCAATTCTAGTAACTCATCCAAACACCCTTCTCATAAGTGCAAGGTCTCAATACACAAGTCGAAATGTGGGGGGCTACGGCCCTACGTCTAGTTAGTTGGGCGAGTCTTCTATATGGACTCATAGTGTGGAACTATAGGCACATGGGTCCTATTCAAATAGCAGCTGATAGAGACATGTTACGGTGAGGTCCGCTGCATTAGCGGGAACCGCAATTATCATACTTGGCCTATTTGTCAGCTTGGCTAGCGTTTGCAGGCCATGTAAAGGATTGCATGTTCTACTCAGTTGGTCTGCCCCAATCGTTAGGCGCTGGGGCTGAAATTATTCCCGATAGTATTATTCGTGGTAGCGTGTTAGTAGCAAGATTGGGTAATACGATCTGAGCTCGGGTGATTCTGCTATCAAATGTGGGACTTAAGATTTTCTCAACGGCTGACGGCCGCCTGAGTGTGTCCCTGGTCCAAGATAGGGTCTCAGCAAACTCAAAGACGGGGAGCCTAGTTACGGTGGTATTAGAGATCGACAGAGAATGGAGGAGAACTATCGTTAACCTGTCGTGCGCAAGTTGGGACCGATAGAGTAGCTAGTAATAATATGGACCTTGCTTATTCATCTGCGTAGCATTATGCTGTAAATCTATACGTCCTTCTTCGGGAGTTACTTCCAGATCCAAATTGAACAATATTACCCGAGTTGGACTCGGTGCGCGGGTGACGGATAGACAGCAACGCAAATTAAATACCCAGTGTGTACGAGGTGCTCCTATGCGGCCCCTCGAGGGGCTAGGTGCTCGCGCCTTAACCCCACTTAAGGGGGACCCCAAGCCAGGCCTCTCGTCGTACTAACTTTATATTCCGCGTGCTTGCTCTTCCGTCTTTGCTTGAACCATCCGGGTGGATGAAGATTGCAAGGAGTTTATACTGTAAGGAGGCCTCAAATCTTCAAGTTTAGCATCACCCGCCCTACCAACTGCTCATATCAGCTCGCCGGAGGCCTCGAAGACCTCCTCTCAATGATAGGAGATCGCGTACCAGATGAACTCTACGCTAACAAGGGTCACCGACCAACGCCCTTAGGAGCAATCACATCACCTGTTCCATCATGCATCTAATGAACACGCGAGGCTGTCGAGAGGCACCACGAACGGCAAACTAGATAGAAGATTCAACAGCCGGTCTAACGCGCCAACGCACCGTTGTGACGTTGGGTTCGCATTGATAAGCGTTGACTTTCCAGTCTGGGGGCATAAATGAATATACGGATAAGGCGTTCAATGCAGGCAATTGGCTCCGCGAGTAGTCTGTGCCTCGTCAATCTTTTTCTGGTCTCACGTTGTCTCGATCCGATATACTCTTGAGCACACCATAGAACTACAACTAACGGGCGGCGTTATTTGAGTTAGAGAGCTTCGGGGATACTTATTCAGGTACGGTCTGTCTTCGGGAGTGTGACAGTACATGAATTAGCCTTGGTAGCAAGACAATTTTGTGAATATGCGGACAGTCAGGTAATCCCCCCATAATAGTTGCGAGATCCTCAGGCACTCGAGTGCGAGCCAGGGGAGAGGACCCTGGTGGAATATCCTCCAGTGTTCAGCTACTTCTCGCCCGCCGCGATAACACTGCGAGTGGGGCACCAGGTCTCACTCCCTGGGTATGCAGTTTCTAATGCATTAGAGGGTGCTGCGGTCCTACCCGCCAATAAGGCGTTAGGCTTAATGTTACGTGAGCTGCACCACACTGCCGTGCTCACAAAGGCCGTGATATTGGATCATTCCTGCTACGGCATCGATAGAAGAGTGAATACGGTATCCAAAAACATCTATTTATACGAGCATCCGAATGCCCCTGACCTGACATTGTCACTACGACAGCTCTTTCGGGTAATAACGCTTTTGCCTTTGCCTTTAACGTGCTGTATTTACGGCCCTTAGTGAGCCATCCACGGAAGGTCGCTTCCATATCAATGTCATAGAAAGCAAACATTGGGGAACAAGACGCTGATTCGCCTAGTGATTAGCGTGCTGCCGGCTGGGGCTTTTTTAATCTTCCTCGGCCACCCGATTATACGAAGTTAGGAATGGCGTGGTCGTACCCCGCCATCCTACCATAAAGCACAATATATCGTCTTTCGGAGTCAGGTCGAGGGGGTAGGGCCAGTTTTAGTTAGTGGATCCGTCACAACGTATCAGTGCCTGCGTCTACTGTGGTGTGGCTAAGATTAATAGCCTTGAGATCCCTTCTTCAGACACGCACAGCCAGTGGTCCTGTGTTCCATGACAGTGCTCGGGGGACCGTGGGCTAGTGCGGCGCGACCTGTATACGTTGACCTTGGCATCGTCTGAATGGCGGCCTGTTCAGTGTATGGGTGCCTTAACCATAGTTTGTAGGACTGTCCGGTCTGGGTAGGGTGCTCTGTTAGGCCATTTCCTCTTTGTTACATTCTGTCCTATCCACCCGCCAAAGGGCGTGAAAGAATCCTTTGTGAACCGGTTATCGCGAGCCCAGATCTCCATAGCGGGAGCGCGCTGAAGCACAACTCGTATCATGAGGAAGATCCGGGTAGAAAAAAGAACTCGAGTGAGCAAGTAGTATACTCAAGCCTCGTGGTATCAGCAATTAGGATGTTTATCATTGTCTGACACTAACCCCACAAGCAAACGTTTCGACATGTTGCCAGCAGGGCTAGCGCATACCTATTCATTACATGCACTTCTCGGAGTCTCCCCCTCCCAGTTCGTGTGCGCACCGGCCGATACATATCCATCGGGATTCGTAAAAGGGGACTAAAAGAACGCAAATTAGAGAACAAAATAACCCCCAGAGCGCTGTAAAACATAGTAAATTCTACTCGCTCAACTTGGGTTATACCCGTAGCTAAGTTCCCGTGATGTTGGGGACGCTTTGAAATAGTCCTCGAGGGTGAAACGTACCCTTAGGGCCTGTACACATTCTCTGTGTCGACTAGTTGCCACGTCTAAATTGGGCATACGATCACGGGTACAGGTGGCATAAGTATAACTGGGCTTCCGAGACCGCACGTG");
	public static final byte[] NPOWER2 = B("NNANAAANAAAAAAANAAAAAAAAAAAAAAANAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAANAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAANAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAANAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAANAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAANAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAANAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAANAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAANAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA");	
	public static final byte[] HOMOLOGY = B("CATTAATCGCAAGAGCGGGTTGTATTCGcCGCCAAGTCAGCTGAAGCACCATTACCCGAtCAAAACATATCAGAAATGATTGACGTATCACAAGCCGGAGTTTACAGCCTGTCTTATATCCTGAATAACGCACCGCCTATTCGAACGGGCGAATCTACCTAGGTCGCTCAGAACCGGCACCCTTAACCATCCATATCCTTCAGTTCCATAGGCCTCTGTGCGGGATTTGTGAACGTTCGGACCCCCCAGGACAGCGTGCATGAGTTTCTAAACGCCCGGCCCCAACGGTCGTCCGGATGAATACTAGGCTAGGCATTGCGTTTACCTAGCTGCGTAAAATTTCGCTGATGGCGGGGATTCACACCACGCATTGTCTGAACACTTGTCTCTTTAGGTTCACTTTTGAGATATCATCAATCGAGAACGTTAGCATAATATCGGTCCTTAACGCCCGTAGCCCATTACAAGCCTCGAATTATGCACCTCCCAGGTACATATATTAGGGTTGTAACGCAAGTCATAATGGAAGCATCGCCGAAAGCGCACGCTGCTCAGTTTAGCACCATCGTCGGGCAAAGACATCCTGGACACGAGTGATAGGTCTTCGGATTGGGAGTCTCATACGCGTGCTAATGCTAACCGCACTGCTAATAATGGGTCCCCTATATGTAAGAACCGTTACGATACCTTCTGTCACAATCACACTGTGTGCAGAAGGGACTGGCTTATTTCCGTGGTCGCCCGGGCACAGTCTTCCACCAGCCGCCACGCCCGCTTGGTTCTCGTTGTCGGCGGTACTCGGGCCATTAGATTAATTTCAAATAATCGGCGGAGCGGGTCTGTGGGCCCTATTTGAAACCATCCTAGATTGTGTCTACTTAGGGAAACGGCCTGGGACTTCGTGATAGTAATGGATCCCCGAAGAAATCTATTTGTACTTACTGGGACTTTCCTCAATCTTCTCAAGAATGACAGCGTGCGGGACTGCTCATCCGAAGGAGGGGCGAGGGAAAACGACCCATCTTTGCTGGGGACAGGGCAGCTGCCACTAATATTGGTTTATGAATAGATTTAACAACCTTTCCAGGGTTGATATTGGCCGCTATTAAGTCATCAATCGAGTAAATTTAACCGTGGGACTCATTTGACTGCTCTTGGAAGCTCGACTACCTCGACGGCTAATGGGTTATCTCCCGACGTAAGGTCAACGGGGCCGGGCAATTCATTCCTAGCAATAAGGGATTCAATTTCTATACGTAAATGTGAGGAAGTAGACGTTACAGCGGGCATTTTAAGGCGACTTCTGGTTGTGGCCTGGCGAGACGACCTAGTTCGATAGACTTCCTGTCAAGACCAGTAAGCGTACTGTTTGTGCACCGGAACTGCGGCGAACGTAGCCGTAGCGCGGTACTTCGAATCGGAGACGGGCCGCAGGTACGTTGTTCAAAGCTGACAATAGAATATCGGGTGGCTCACACACACCAGAGGTAAATGCTACGTTTGACCGTCTCCGTACCATCCGTGATGGCTTCCTCCATAACTCGCCCTCTGTATGTAGCTATGATCTACGACGTGTTAGCCCTCTGCCTAAATGGGTGCAACAAGAACGCGGATGTCGTCCGGATGTCGACATTCCGGCCGGTCACACAAAGTAAAGCCAATAAGGAGTCTAAATCCTATCCACTGGCCCCTAGTGAAGGCCTGCACTTGGAGTAAGCGGACTGGCTATAATTTACAGGATAAGGGAAGCAAATAGGACAACGTCGATTAGCTAGACGTCGATAGGGATTGACACTCAGGTAGGTGTATGAAACGCTGTGTGGTGTTTGCCGAGCAAAACAGATTGCGACGTATGCGTATTTGCACCACCAATCCCAAGAAAAGGTAATGCTACGGGAAGCTAATATGACTGCCGTCGGGGGTGCTATGGAAGGCTCATACGCCAGGGTTGTCAACCTCTTGGACGTCCGAGGCCGCACGCAACTGGCAAGAAGAGTTCAATCCCGGCCCAGCTTTCCGCCTGCGAGGTATGGCACAGTGCAGTGATCCTAGGTATTAACAGGGGAGGAGAGACGATAGGTCCACTGCTGGGCCTGAGTTCTGATGATTTGGGCTCGCACTGATAATAGGGCTATTGGCCCTTCGTACTGATGGCAAATGACGCCCAATCGAAACTCCACCACACCAACCGAACCTCCTAAGAATGATTACAGCCTTGGAGATAGTGCGAACTGTTGGTCGTGTTCTCATCAACGCGTAGGTTCATTTGGGCAACATGGTCAGAGCGACGTCCCTAATTCTAATTTATAGAGAGCCCGGGAAATAATCTACTGGTTGACGCAGATTTTAATATCAGAGGGTACGTATTCAACGGAGGCGATCGGAACTTGGATCCCCGCATCTCCATACCCCCAAAAACCAGAACGAGCGTAAAAGGACTAGGTCAGAGTGGCCTACCGGTACAGCAACAGTGCGTTCACCTTAGCAAACCTGGAAACGCATGGTTTGAATAAAAATAGCTTTACTGACTGCGATAGAAAGTCGGCTTCGTTCCAACTGTGAGACAGTCGTGTCTCAGAGCAGAGGAGCTGAAGTGGGGAACCGGCGGCCATCACCGTCAGAAGGCCTTCCTCCTGTTTCATTTGGACGTTACTGGCACATCAAGCAATCCCCTGGAGGAGACCCTCGAGCCTAGTTGTCAATCAAATAACCCTACCGACTTCTATGTCTCGTGCGACCGTCGAAAATATGATTGCTTGCCACTAGACAGTCTGGCTGTTCGGCGATGTAACCAACTGTTAGAGAGGCGACAAGCCCTGATTGACCGTAGTGGTATGTGTTTTACACCCGTCGGGAAAGGGTTACGACACTGCGATGTTAAGGGCGCATCGCGACATTATTCGCTAGAACAGGTAACAACCATTAAATTGACTCTACCGGATAGCGCCGTTTCCTGTAATATATGTGCAGGAATGTCACCATGGGTGGTTCTCTCTACCTCCTCGATAAAGTACTGAGTGCTGCGGTGAGGAAACTGTTGTCCGTTCCAATCGACTGCCTACTGGGACATGCGTCTGCGGATTCCGGCATTCTCGTACCAGAGACCGGCGCATAGGAGAAATCCGCGACGATTCTCCGGAAGGCATGGGCTAACGGGGTATTCCTCTCTATATGACTCATCGTGCTCCAGCGTACCTCTGAGCCGATTGGTGCATCGGAGGACTGTCGATCCAACTGCTGCAGGGACTGAACTCTGTTCTCAGACCGGTACGTCTTGTATAAGTAGTCTCCCAGACTGGCGTGGTGTGCCGAACCCGTCGATCTGTTCAAAATTTCGCCTACCAGTCGCACGGACGTCACATTAGCTTAGACGGACGAATCTCCCAGTTATGCGTGATAGGGAGATAGGGCAGGCCGTGCGTCGCGTTGGACCGAGATGGCACGCCGCAAGGCAGCAGAAAGGGAAAAAGCGCCCGTGCCAACAGATCGATCAGTCCGTTTCCGAGGGTGCTTATTTTTAAGATATCCCTGGTTCTATGCCCAGGTAAGCGCGAGTACTGTCTTGTAGAGGACGTACAGGCATCTCTTACATGTGGAATTCCTGAGCACTTCTTGACGTCGTGTAACAACCGAAAGTATATTTAGCTCGGTAATTCATGTTCGGGTTATAATCAATTATCCTATTTTCAACGTATAGCAGCAACGGGGCCGGTGGTCCGCGTAGTGGATCGAAGCAGCCCTCTTGCTGACTTTAAGTACACGGCTAACGGTGGCTCGCGCCAAAACACTTTGTACGAGCTCGCAATCACCGGTCGCTGTATTTTGGGCCTAGTTTCCAGTGACCGAGAAGCTACCGTATACTATTGCACAAAGCGAAAGCATTTTGGTAGGTCAGCGGCTGCGATATCATAATGACCCGTATCTGAACGTCTCTGGACTTATAAATCTGAGAATCCATCTGCTAGCGACTCGCCTGGGAACAGGCTTTTTGTGCCATGCCTGATGCCATACTCCGAACACATGCCTAATTCGGCACCTATTAGCAACCCGGGGGCGATTTATAGTCCCCGCAGCCAAGTGAAGCAGCAACTCGACATGGAACTCGAGAAATCGGTACGGTATCACGACAGGTATTGGTATTGTCCACGATATCGATAGCCTATTTCGAAACCTGGTCCTTATAGGACGTGCTTCATATGTGCTTGCTCTCACTCTATTCGAAGCTCCACATACCGACCAGCTGTTTACTAAGTCAACCGCCAAATAGTCGCGGCGCTAGACACAAACTCCCTATAAAGGCAGGGATATGGAGCCAGCGCTACGAAAGGATTGCGTAGTTATTGCTCGGCCTTGTGACTGGCCGGACACAGATATGTCGTTCGCCTACGATAAGTGCGAACCAGGCCGTTGCAACCTTACGTTAATGCGGATACTGTTAAGCGTAATTGGACGAAATCCGGCGTAGGGTGCAGTTATCGACGTCGAAAAGCTCAGGGCGCGTAGTCCGATCTCGCTTAGGCCATCGGCTGGATTCGTGACGCGAACCGACCGGAACGGATGGAACTAGTACTTGGCCGTGTCATACGCATCGTCAAATCGTCTGATGTCGGGTGATGATGTGCGAACTACAGTCCTCCAGCGAAAAACAGTCTGAGCAAGCTTATGATGCTGATGGCACTGGATCCAAGGAAGGGTATTGCCGAAATTAATGACAGTCTCTGTAAAGGCCGACGACCCATTGTAAATTGTAACGCGTTTTCACGCTCCTGCTGTGTCGGGCCCCGGTCGGCTGTCGCCATGCCGACTCCTGGGTGGGCTACAATTGTAACATGCGGATAGGATCGCACTTCATAGAACACATTATAGTAATTTAACTTCCTGTTCCTGGAGGTAAGCGCGTCGCCTGATCTTGAGAGTCCCGTTCGTGAGAACCACAATTCACAGATTTGACGACTCCTTTTAATATAGCGACGGAGCACACCGCTCGTAATTCAGCTCGTAGTCTTTGCGTCTATCCACACACTCCCGGGCATATAGCTAACCGGACGCACAACTTACGGATCCTTACAACGTACAGCATTGGCACACGGAAGTTCCAGATGCTTCAATGGTACTTTCTTTTCAATTGGACAGGCTACGTAGCTAGTGCAATTGATATCAATGGGACATATGCTGAGAAAGCGACGGGGGCCGTGGGTTTTTTTGAGAGATGTAGCCGTACCCGGCCTGGATAGGCAGGCGCAATAGGAACGCAGGCAGCAATGCTTCGCAAATTGACAATATGCAGTGTCGCACTCGCGAGCAGCTCCTGGCGAGCTACGGATTAATTGTCATCTGAATTCAGGACGCAAGCCCGCTAAACCACGGGGTTCGGCACTAAGGACCATATTTATGCGACCGTTAGCTAATCGCCAGACGTTCGTGTCTTGGCGCGAATATAATATTCCCAGGGACTGTCGTTATACGATCTGGCCGGCATGAGACTTTACAAACTGTAGTTATTGGCCCACCGTTTACGCCGTGGTAATTTTGGGCAGGCTTGGAGAAGATTGAGATATAGCAGCGAACCACAGCGCCGGTCGATCGGACAAATTCTCCCTGTAAAGTGTTTATACCGTTTAGGAGATTGCGTAGGCACAGCAGCTTACGCCGATATGTCGAGGACACCCTTTCGATTATGAGCTCGTTCTAAGTATGCAGTAAAAGACCGATTGTTTTTCAGTAGGTGCATACCATAAAGGAAAGGGCGATTTACGTACACTCTACCTTTGATCCTAAAGACCTGGGTTCGTTGCAATGTCTTGTACCACATCCCCCATGAGGCTAACGTTATAGTCGACTATCGTCATCATCATTCTCTACGTATTGTTGCAATGGGACGAACGTGGACGCGTGTAAGAAACGCCAGTGTTGAGGTGTCCAGGAACAGGGAAAACCATAGTCTCGGCCGCTCGGTGTGGTACTTGCAAAACAAAAAAGGTGTTAGTCGCGCAGTTACACCTGGCTGTATTCGTCAACCACGACATAAGGGCGTAGTGACGATTAAGTGTTAGGCTATGTGGTGAATCTGCGTGTGGTGGCACGGTATAAGCCGGATAGCAATAGACCCCACTATAGACTTAGTAGTTTTGCCCTGAATAACTCCGGGGAATGCAAGATCAGAAAAAACCTTCAGCGATCATATGAGGTTGCAGTATGGTACGGATGGGGTCCAGTCTTGAAAAAATAAGGTTATCCCCAATCTGCCGGTCAAATAGTTAACCGCCCTTCCCTATGCACCAACTTTTTTAGGATCGGCTCGGTTCCAATTAATATGTGTGGGGCTAGCCTCGCACGTCGGCCACAGCGAAAAGAATATCGAGATTTTGGAACGTGCCCAAGCATCCATAATAGTCCTTGATCTTCCAACCAGACAGCGATGCTCGAACCGGTCCCCAAATAGAATGACTAAACGGACGAAGAGAGGTTTCAATCACTCCTCTGTGGGCCGATCACGATTACAGTTAGCACCAGCGTTCGATCTCCTCTAGAGAACCCACACGCGATCCTCATATTGGTGGGGCCAATGATACTAAGCTCAGGTCCCTAATGATACCTCCATCGCTATACGCGTTTGGTGTAACCAGATTTTCCCGGCGGGACTACCTCACTGGTTTTCGGCGGTGCAGCCTTTGTGTTCGAGGTACGCGCCGACCGCAAGCCCCCGCGCCCGTACAAAACTTCGTCTGTTCAATAACAGCGGCGTTAAGCTAACGGCGAGAAATCTCTCACTTCCCGTCCAAAGGAAGACCTGAAGTCATAATCTTGCAACACCGAAGAACTGGCTAATGGGTTGTAAGATATGCTATTCATATGGTAGGAAGGTTCGCAAGAGATTTTAGAGGGTCTGTCTAGCCAGGAACCCCAGCGACTATGTTCTTCTCCGCATCGGCCCAACCACCTGGCAATTTCCCAATTCTAGTAACTCATCCAAACACCCTTCTCATAAGTGCAAGGTCTCAATACACAAGTCGAAATGTGGGGGGCTACGGCCCTACGTCTAGTTAGTTGGGCGAGTCTTCTATATGGACTCATAGTGTGGAACTATAGGCACATGGGTCCTATTCAAATAGCAGCTGATAGAGACATGTTACGGTGAGGTCCGCTGCATTAGCGGGAACCGCAATTATCATACTTGGCCTATTTGTCAGCTTGGCTAGCGTTTGCAGGCCATGTAAAGGATTGCATGTTCTACTCAGTTGGTCTGCCCCAATCGTTAGGCGCTGGGGCTGAAATTATTCCCGATAGTATTATTCGTGGTAGCGTGTTAGTAGCAAGATTGGGTAATACGATCTGAGCTCGGGTGATTCTGCTATCAAATGTGGGACTTAAGATTTTCTCAACGGCTGACGGCCGCCTGAGTGTGTCCCTGGTCCAAGATAGGGTCTCAGCAAACTCAAAGACGGGGAGCCTAGTTACGGTGGTATTAGAGATCGACAGAGAATGGAGGAGAACTATCGTTAACCTGTCGTGCGCAAGTTGGGACCGATAGAGTAGCTAGTAATAATATGGACCTTGCTTATTCATCTGCGTAGCATTATGCTGTAAATCTATACGTCCTTCTTCGGGAGTTACTTCCAGATCCAAATTGAACAATATTACCCGAGTTGGACTCGGTGCGCGGGTGACGGATAGACAGCAACGCAAATTAAATACCCAGTGTGTACGAGGTGCTCCTATGCGGCCCCTCGAGGGGCTAGGTGCTCGCGCCTTAACCCCACTTAAGGGGGACCCCAAGCCAGGCCTCTCGTCGTACTAACTTTATATTCCGCGTGCTTGCTCTTCCGTCTTTGCTTGAACCATCCGGGTGGATGAAGATTGCAAGGAGTTTATACTGTAAGGAGGCCTCAAATCTTCAAGTTTAGCATCACCCGCCCTACCAACTGCTCATATCAGCTCGCCGGAGGCCTCGAAGACCTCCTCTCAATGATAGGAGATCGCGTACCAGATGAACTCTACGCTAACAAGGGTCACCGACCAACGCCCTTAGGAGCAATCACATCACCTGTTCCATCATGCATCTAATGAACACGCGAGGCTGTCGAGAGGCACCACGAACGGCAAACTAGATAGAAGATTCAACAGCCGGTCTAACGCGCCAACGCACCGTTGTGACGTTGGGTTCGCATTGATAAGCGTTGACTTTCCAGTCTGGGGGCATAAATGAATATACGGATAAGGCGTTCAATGCAGGCAATTGGCTCCGCGAGTAGTCTGTGCCTCGTCAATCTTTTTCTGGTCTCACGTTGTCTCGATCCGATATACTCTTGAGCACACCATAGAACTACAACTAACGGGCGGCGTTATTTGAGTTAGAGAGCTTCGGGGATACTTATTCAGGTACGGTCTGTCTTCGGGAGTGTGACAGTACATGAATTAGCCTTGGTAGCAAGACAATTTTGTGAATATGCGGACAGTCAGGTAATCCCCCCATAATAGTTGCGAGATCCTCAGGCACTCGAGTGCGAGCCAGGGGAGAGGACCCTGGTGGAATATCCTCCAGTGTTCAGCTACTTCTCGCCCGCCGCGATAACACTGCGAGTGGGGCACCAGGTCTCACTCCCTGGGTATGCAGTTTCTAATGCATTAGAGGGTGCTGCGGTCCTACCCGCCAATAAGGCGTTAGGCTTAATGTTACGTGAGCTGCACCACACTGCCGTGCTCACAAAGGCCGTGATATTGGATCATTCCTGCTACGGCATCGATAGAAGAGTGAATACGGTATCCAAAAACATCTATTTATACGAGCATCCGAATGCCCCTGACCTGACATTGTCACTACGACAGCTCTTTCGGGTAATAACGCTTTTGCCTTTGCCTTTAACGTGCTGTATTTACGGCCCTTAGTGAGCCATCCACGGAAGGTCGCTTCCATATCAATGTCATAGAAAGCAAACATTGGGGAACAAGACGCTGATTCGCCTAGTGATTAGCGTGCTGCCGGCTGGGGCTTTTTTAATCTTCCTCGGCCACCCGATTATACGAAGTTAGGAATGGCGTGGTCGTACCCCGCCATCCTACCATAAAGCACAATATATCGTCTTTCGGAGTCAGGTCGAGGGGGTAGGGCCAGTTTTAGTTAGTGGATCCGTCACAACGTATCAGTGCCTGCGTCTACTGTGGTGTGGCTAAGATTAATAGCCTTGAGATCCCTTCTTCAGACACGCACAGCCAGTGGTCCTGTGTTCCATGACAGTGCTCGGGGGACCGTGGGCTAGTGCGGCGCGACCTGTATACGTTGACCTTGGCATCGTCTGAATGGCGGCCTGTTCAGTGTATGGGTGCCTTAACCATAGTTTGTAGGACTGTCCGGTCTGGGTAGGGTGCTCTGTTAGGCCATTTCCTCTTTGTTACATTCTGTCCTATCCACCCGCCAAAGGGCGTGAAAGAATCCTTTGTGAACCGGTTATCGCGAGCCCAGATCTCCATAGCGGGAGCGCGCTGAAGCACAACTCGTATCATGAGGAAGATCCGGGTAGAAAAAAGAACTCGAGTGAGCAAGTAGTATACTCAAGCCTCGTGGTATCAGCAATTAGGATGTTTATCATTGTCTGACACTAACCCCACAAGCAAACGTTTCGACATGTTGCCAGCAGGGCTAGCGCATACCTATTCATTACATGCACTTCTCGGAGTCTCCCCCTCCCAGTTCGTGTGCGCACCGGCCGATACATATCCATCGGGATTCGTAAAAGGGGACTAAAAGAACGCAAATTAGAGAACAAAATAACCCCCAGAGCGCTGTAAAACATAGTAAATTCTACTCGCTCAACTTGGGTTATACCCGTAGCTAAGTTCCCGTGATGTTGGGGACGCTTTGAAATAGTCCTCGAGGGTGAAACGTACCCTTAGGGCCTGTACACATTCTCTGTGTCGACTAGTTGCCACGTCTAAATTGGGCATACGATCACGGGTACAGGTGGCATAAGTATAACTGGGCTTCCGAGACCGCACGTG");
	public static File SMALL_FA_FILE = new File("src/test/resources/small.fa");
	public static ReferenceLookup SMALL_FA = new BufferedReferenceSequenceFile(ReferenceSequenceFileFactory.getReferenceSequenceFile(SMALL_FA_FILE));
	public static class StubSAMFileWriter implements SAMFileWriter {
		public StubSAMFileWriter() {
			this(new SAMFileHeader());
		}

		public StubSAMFileWriter(SAMFileHeader header) {
			this.header = header;
			this.header.addReadGroup(new SAMReadGroupRecord("RG"));
		}

		public StubSAMFileWriter(ReferenceSequenceFile reference) {
			this();
			this.header
					.setSequenceDictionary(reference.getSequenceDictionary());
		}

		public SAMFileHeader header;
		public ArrayList<SAMRecord> list = new ArrayList<SAMRecord>();

		@Override
		public void addAlignment(SAMRecord alignment) {
			list.add(alignment);
		}

		@Override
		public SAMFileHeader getFileHeader() {
			return header;
		}

		@Override
		public void close() {
			closed = true;
		}

		public boolean closed;

		@Override
		public void setProgressLogger(ProgressLoggerInterface progress) {
		}
	}

	public static class StubVariantContextWriter implements
			VariantContextWriter {
		public VCFHeader header = null;
		public ArrayList<VariantContext> data = new ArrayList<VariantContext>();

		@Override
		public void writeHeader(VCFHeader header) {
			this.header = header;
		}

		@Override
		public void close() {
		}

		@Override
		public void add(VariantContext vc) {
			this.data.add(vc);
		}
		@Override
		public boolean checkError() {
			return false;
		}
	}
	
	static public SAMRecord Unmapped(int readLength) {
		SAMRecord record = new SAMRecord(getHeader());
		record.setReadUnmappedFlag(true);
		record.setAlignmentStart(SAMRecord.NO_ALIGNMENT_START);
		record.setReferenceIndex(SAMRecord.NO_ALIGNMENT_REFERENCE_INDEX);
		record.setMappingQuality(SAMRecord.NO_MAPPING_QUALITY);
		record.setCigarString(SAMRecord.NO_ALIGNMENT_CIGAR);
		record.setReadBases(GetPolyA(readLength));
		return record;
	}

	// -------- Path graph helpers --------
	public static class BaseGraph extends DeBruijnGraphBase<DeBruijnNodeBase> {
		public BaseGraph(int k) {
			super(k);
		}

		@Override
		protected DeBruijnNodeBase merge(DeBruijnNodeBase node,
				DeBruijnNodeBase toAdd) {
			node.add(toAdd);
			return node;
		}

		@Override
		protected DeBruijnNodeBase remove(DeBruijnNodeBase node,
				DeBruijnNodeBase toRemove) {
			throw new RuntimeException("NYI");
		}

		public BaseGraph add(String sequence) {
			return add(sequence, 1);
		}

		public BaseGraph add(String sequence, int weight) {
			NonReferenceReadPair placeholderEvidence = NRRP(OEA(0, 1, "100M", true));
			int i = 1;
			for (long kmer : toKmer(this, sequence)) {
				add(new DeBruijnNodeBase(kmer, weight, i++, "evidence" + Integer.toString(i), false, 0, placeholderEvidence.getBreakendSummary()));
			}
			return this;
		}

		public String S(PathNode<DeBruijnNodeBase> node) {
			return new String(KmerEncodingHelper.baseCalls(DeBruijnNodeBase.asKmers(node.getPath()), getK()));
		}
	}

	public static class BasePathGraph extends
			DeBruijnPathGraph<DeBruijnNodeBase, DeBruijnPathNode<DeBruijnNodeBase>> {
		public BasePathGraph(
				BaseGraph graph,
				long seed,
				PathNodeFactory<DeBruijnNodeBase, DeBruijnPathNode<DeBruijnNodeBase>> factory) {
			super(graph, ImmutableList.of(graph.getKmer(seed)), factory, new NontrackingSubgraphTracker<DeBruijnNodeBase, DeBruijnPathNode<DeBruijnNodeBase>>());
		}

		public BasePathGraph(BaseGraph g) {
			super(g, new DeBruijnPathNodeFactory<DeBruijnNodeBase>(g), new NontrackingSubgraphTracker<DeBruijnNodeBase, DeBruijnPathNode<DeBruijnNodeBase>>());
		}
		public BaseGraph getGraph() { return (BaseGraph)super.getGraph(); }
		
		public DeBruijnPathNode<DeBruijnNodeBase> get(String kmer) {
			assert (kmer.length() == getGraph().getK());
			long state = KmerEncodingHelper.picardBaseToEncoded(getGraph()
					.getK(), B(kmer.substring(0, getGraph().getK())));
			for (DeBruijnPathNode<DeBruijnNodeBase> pn : getPaths()) {
				for (DeBruijnNodeBase n: Iterables.concat(pn.getPathAllNodes())) {
					if (state == n.kmer()) {
						return pn;
					}
				}
			}
			return null;
		}

		public DeBruijnPathNode<DeBruijnNodeBase> addNode(String sequence) {
			DeBruijnPathNode<DeBruijnNodeBase> n = new DeBruijnPathNode<DeBruijnNodeBase>(toNodes(getGraph(), sequence), getGraph());
			super.expectedWeight += n.weight();
			return n;
		}

		public String S(DeBruijnPathNode<DeBruijnNodeBase> node) {
			return new String(KmerEncodingHelper.baseCalls(DeBruijnNodeBase.asKmers(node.getPath()), getK()));
		}

		public String S(Iterable<DeBruijnPathNode<DeBruijnNodeBase>> nodes) {
			Iterator<DeBruijnNodeBase> it = PathNode.nodeIterator(nodes.iterator());
			ArrayList<DeBruijnNodeBase> list = Lists.newArrayList(it);
			return new String(KmerEncodingHelper.baseCalls(DeBruijnNodeBase.asKmers(list), getK()));
		}
		public List<DeBruijnPathNode<DeBruijnNodeBase>> splitPathToEnsureBreaksAt(Iterable<DeBruijnPathNode<DeBruijnNodeBase>> path, SortedSet<Integer> breaksAt) {
			return super.splitPathToEnsureBreaksAt(path, breaksAt);
		}
	}
	/**
	 * @param g
	 * @param path sequence of kmers or nodes
	 * @return
	 */
	@SuppressWarnings({ "unchecked", "rawtypes" })
	public static <T> String S(DeBruijnGraph<T> g, Iterable path) {
		List<Object> obj = Lists.newArrayList(path);
		if (obj.size() == 0) return "";
		if (obj.get(0) instanceof Long) {
			return new String(KmerEncodingHelper.baseCalls(Lists.<Long>newArrayList(path), g.getK()));
		} else {
			return toKmerString(g, path);
		}
	}
	public static <T> List<Long> toKmer(DeBruijnGraph<T> graph, String sequence) {
		assert (sequence.length() >= graph.getK());
		byte[] qual = new byte[sequence.length()];
		Arrays.fill(qual, (byte) 1);
		List<Long> result = Lists.newArrayList();
		for (ReadKmer rk : new ReadKmerIterable(graph.getK(), B(sequence),
				qual, false, false)) {
			result.add(rk.kmer);
		}
		return result;
	}
	public static long K(String seq) {
		return KmerEncodingHelper.picardBaseToEncoded(seq.length(), B(seq));
	}
	public static String K(int k, long kmer) {
		return S(KmerEncodingHelper.encodedToPicardBases(k, kmer));
	}
	public static <T, PN extends DeBruijnPathNode<T>> String toKmerString(DeBruijnPathGraph<T, PN> pg, PN node) {
		return toKmerString(pg.getGraph(), node.getPath());
	}
	public static <T> String toKmerString(final DeBruijnGraph<T> g, final Iterable<? extends T> node) {
		return new String(KmerEncodingHelper.baseCalls(
				Lists.<Long>newArrayList(Iterables.transform(node, new Function<T, Long>() {
					@Override
					public Long apply(T input) {
						return g.getKmer(input);
					}
				})), g.getK()));
	}
	public static <T> List<DeBruijnNodeBase> toNodes(final DeBruijnGraphBase<DeBruijnNodeBase> graph, final String sequence) {
		return Lists.transform(toKmer(graph, sequence), new Function<Long, DeBruijnNodeBase>() {
			@Override
			public DeBruijnNodeBase apply(Long input) {
				DeBruijnNodeBase n = graph.getKmer(input);
				if (n == null) throw new IllegalArgumentException("Test case setup error: " + KmerEncodingHelper.toString(graph.getK(), input) + " not in de Bruijn graph.");
				return n;
			}
		});
	}
	public static <T, PN extends DeBruijnPathNode<T>> PN PGN(DeBruijnPathGraph<T, PN> pg, String sequence) {
		assert (sequence.length() == pg.getGraph().getK());
		long state = KmerEncodingHelper.picardBaseToEncoded(pg.getGraph()
				.getK(), B(sequence));
		for (PN pn : pg.getPaths()) {
			for (T n : Iterables.concat(pn.getPathAllNodes())) {
				if (state == pg.getGraph().getKmer(n)) {
					return pn;
				}
			}
		}
		return null;
	}

	public static BaseGraph G(int k) {
		return new BaseGraph(k);
	}

	public static BasePathGraph PG(BaseGraph g, String seed) {
		return new BasePathGraph(g, KmerEncodingHelper.picardBaseToEncoded(
				g.getK(), B(seed)), new DeBruijnPathNodeFactory<DeBruijnNodeBase>(g));
	}

	public static BasePathGraph PG(BaseGraph g) {
		return new BasePathGraph(g);
	}

	public static class MockSAMEvidenceSource extends SAMEvidenceSource {
		public IdsvSamFileMetrics metrics = new MockMetrics();
		public int category = 0;
		public MockSAMEvidenceSource(ProcessingContext processContext) {
			super(processContext, new File("test.bam"), null, 0);
		}
		public MockSAMEvidenceSource(ProcessingContext processContext, int minFragmentSize, int maxFragmentSize) {
			super(processContext, new File("test.bam"), null, 0, minFragmentSize, maxFragmentSize);
			metrics.getIdsvMetrics().MAX_PROPER_PAIR_FRAGMENT_LENGTH = maxFragmentSize;
			metrics.getIdsvMetrics().MIN_PROPER_PAIR_FRAGMENT_LENGTH = minFragmentSize;
			metrics.getIdsvMetrics().MAX_READ_LENGTH = minFragmentSize == 0 ? maxFragmentSize : minFragmentSize;
			metrics.getInsertSizeMetrics().MAX_INSERT_SIZE = maxFragmentSize;
			metrics.getInsertSizeMetrics().MIN_INSERT_SIZE = minFragmentSize;
			metrics.getInsertSizeMetrics().MEDIAN_INSERT_SIZE = (minFragmentSize + maxFragmentSize) / 2;
			metrics.getInsertSizeMetrics().MEAN_INSERT_SIZE = (minFragmentSize + maxFragmentSize) / 2;
			metrics.getInsertSizeMetrics().MEDIAN_ABSOLUTE_DEVIATION = maxFragmentSize / 10 / 1.4;
			metrics.getInsertSizeMetrics().STANDARD_DEVIATION = maxFragmentSize / 10;
		}
		@Override
		public int getSourceCategory() {
			return category;
		}
		@Override
		public IdsvSamFileMetrics getMetrics() {
			assert(metrics != null);
			return metrics;
		}
	}

	public static MockSAMEvidenceSource SES() {
		return new MockSAMEvidenceSource(getContext());
	}
	public static MockSAMEvidenceSource SES(ProcessingContext context) {
		return new MockSAMEvidenceSource(context);
	}
	public static MockSAMEvidenceSource SES(boolean isTumour) {
		MockSAMEvidenceSource e = new MockSAMEvidenceSource(getContext());
		e.category = isTumour ? 1 : 0;
		return e;
	}
	public static MockSAMEvidenceSource SES(int maxFragmentSize) {
		return (SES(0, maxFragmentSize));
	}
	public static MockSAMEvidenceSource SES(int minFragmentSize, int maxFragmentSize) {
		return new MockSAMEvidenceSource(getContext(), minFragmentSize, maxFragmentSize);
	}
	public MockSAMEvidenceSource permissiveSES() {
		MockSAMEvidenceSource ses = SES();
		ses.getContext().getConfig().getSoftClip().minAnchorIdentity = 0;
		ses.getContext().getConfig().getSoftClip().minLength = 1;
		ses.getContext().getConfig().minMapq = 0;
		ses.getContext().getConfig().minAnchorShannonEntropy = 0;
		ses.getContext().getConfig().adapters = new AdapterHelper(new String[0]);
		return ses;
	}
	public static class StubSAMEvidenceSource extends SAMEvidenceSource {
		private final int maxFragmentSize;
		public int maxReadLength = 100;
		public IdsvSamFileMetrics metrics = new MockMetrics();
		public List<DirectedEvidence> evidence = new ArrayList<DirectedEvidence>();
		public StubSAMEvidenceSource(ProcessingContext processContext, File file, int category, int minFragmentSize, int maxFragmentSize) {
			super(processContext, file, null, category, minFragmentSize, maxFragmentSize);
			this.maxFragmentSize = maxFragmentSize;
		}
		@Override
		public int getMaxConcordantFragmentSize() { return maxFragmentSize; }
		@Override
		public int getMaxReadLength() { return maxReadLength; }
		@Override
		public int getMaxReadMappedLength() { return maxReadLength; }
		@Override
		public IdsvSamFileMetrics getMetrics() { return metrics; }
		@Override
		public CloseableIterator<DirectedEvidence> iterator() {
			return new AutoClosingIterator<DirectedEvidence>(evidence.iterator());
		}
		@Override
		public CloseableIterator<DirectedEvidence> iterator(QueryInterval[] qi) {
			return new AutoClosingIterator<DirectedEvidence>(
					Iterators.filter(evidence.iterator(), de -> 
						de != null &&
						de.getBreakendSummary() != null &&
						QueryIntervalUtil.overlaps(qi, de.getBreakendSummary()))
					);
		}
	}
	public static class StubAssemblyEvidenceSource extends AssemblyEvidenceSource {
		public int assemblyWindowSize = 10;
		public int fragSize = 300;
		public List<DirectedEvidence> assemblies = new ArrayList<DirectedEvidence>();
		public StubAssemblyEvidenceSource(ProcessingContext processContext) {
			super(processContext, ImmutableList.<SAMEvidenceSource>of(), null);
		}
		@Override
		public void assembleBreakends(ExecutorService threadpool) throws java.io.IOException {};
		@Override
		public int getMaxConcordantFragmentSize() { return fragSize; }
		@Override
		public CloseableIterator<DirectedEvidence> iterator() {
			return new AutoClosingIterator<>(assemblies.iterator());
		}
		@Override
		public CloseableIterator<DirectedEvidence> iterator(QueryInterval[] qi) {
			return new AutoClosingIterator<>(assemblies.stream().filter(
					e -> QueryIntervalUtil.overlaps(qi, e.getBreakendSummary())).iterator());
		}
	}
	public static AssemblyEvidenceSource AES() {
		return new AssemblyEvidenceSource(getContext(),
				ImmutableList.<SAMEvidenceSource>of(SES()), new File(
						"test.bam"));
	}
	public static AssemblyEvidenceSource AES(ProcessingContext context) {
		return new AssemblyEvidenceSource(context,
				ImmutableList.<SAMEvidenceSource>of(SES(context)), new File(
						"test.bam"));
	}
	public static AssemblyEvidenceSource AES(int maxFragmentSize) {
		return new AssemblyEvidenceSource(getContext(),
				ImmutableList.<SAMEvidenceSource>of(SES(maxFragmentSize)), new File(
						"test.bam"));
	}
	public static AssemblyEvidenceSource AES(SAMEvidenceSource ses) {
		return new AssemblyEvidenceSource(ses.getContext(),
				ImmutableList.<SAMEvidenceSource>of(ses), new File(
						"test.bam"));
	}
	public VariantContextDirectedEvidence CallSV(SAMRecord... evidence) {
		return CallSV(Stream.of(evidence)
			.flatMap(r -> SingleReadEvidence.createEvidence(AES(), 0, r).stream())
			.map(r -> (DirectedEvidence)r)
			.toArray( DirectedEvidence[]::new));
	}
	public VariantContextDirectedEvidence CallSV(DirectedEvidence... evidence) {
		IdsvVariantContextBuilder vcBuilder = new IdsvVariantContextBuilder(getContext());
		byte[] seq = evidence[0].getBreakendSequence();
		vcBuilder.breakend(evidence[0].getBreakendSummary(), seq == null ? "" : new String(seq, StandardCharsets.US_ASCII));
		StructuralVariationCallBuilder builder = new StructuralVariationCallBuilder(getContext(), (VariantContextDirectedEvidence)vcBuilder.make());
		for (DirectedEvidence e : evidence) {
			builder.addEvidence(e);
		}
		return builder.make();
	}
	public IdsvMetrics IDSV(Collection<SAMRecord> input) {
		IdsvSamFileMetricsCollector c = new IdsvSamFileMetricsCollector(null);
		for (SAMRecord r : input) {
			c.acceptRecord(r, null);
		}
		MetricsFile<IdsvMetrics, Integer> idsv = new MetricsFile<IdsvMetrics, Integer>();
		MetricsFile<InsertSizeMetrics, Integer> is = new MetricsFile<InsertSizeMetrics, Integer>();
		MetricsFile<CigarDetailMetrics, Integer> sc = new MetricsFile<CigarDetailMetrics, Integer>();
		MetricsFile<MapqMetrics, Integer> mqm = new MetricsFile<MapqMetrics, Integer>();
		c.finish(is, idsv, mqm, sc);
		IdsvMetrics metrics = idsv.getMetrics().get(0);
		return metrics;
	}
	/**
	 * Gets the reference sequence offset from the given breakend
	 * @param contig
	 * @param baseCount number of bases to return
	 * @param breakendPos breakend position
	 * @param direction direction of breakend
	 * @param localFragBaseCount number of fragment bases on this side of the breakend
	 * @return read bases
	 */
	public static byte[] getRef(byte[] contig, int baseCount, int breakendPos, BreakendDirection direction) {
		int genomicPositionOfFirstBase = getLeftmostBasePos(breakendPos, direction, baseCount, 0);
		int arrayOffsetOfFirstBase = genomicPositionOfFirstBase - 1;
		return B(S(contig).substring(arrayOffsetOfFirstBase, arrayOffsetOfFirstBase + baseCount));
	}
	/**
	 * Gets the position of the leftmost base of the sequence 
	 * @param be breakend
	 * @param baseCount length of sequence
	 * @param baseSkipCount bases between sequence and breakpoint
	 * @return
	 */
	public static int getLeftmostBasePos(BreakendSummary be, int baseCount, int baseSkipCount) {
		return getLeftmostBasePos(be.start, be.direction, baseCount, baseSkipCount);
	}
	public static int getLeftmostBasePos(int breakendPos, BreakendDirection direction, int baseCount, int baseSkipCount) {
		if (direction == BWD) return breakendPos + baseSkipCount;
		return breakendPos - baseCount - baseSkipCount + 1;
	}
	public static byte[] getRef(int referenceIndex, int baseCount, int breakendPos, BreakendDirection direction) {
		if (referenceIndex == 0) return getRef(POLY_A, baseCount, breakendPos, direction);
		if (referenceIndex == 1) return getRef(POLY_ACGT, baseCount, breakendPos, direction);
		if (referenceIndex == 2) return getRef(RANDOM, baseCount, breakendPos, direction);
		if (referenceIndex == 3) return getRef(NPOWER2, baseCount, breakendPos, direction);
		throw new IllegalArgumentException("Unknown contig");
	}
	public static int totalWeight(Iterable<? extends KmerNode> it) {
		return totalWeight(it.iterator());
	}
	public static int totalWeight(final Iterator<? extends KmerNode> it) {
		Stream<KmerNode> stream = StreamSupport.stream(Spliterators.spliteratorUnknownSize(it, Spliterator.ORDERED), false);
		return stream.mapToInt(n -> n.weight() * n.width()).sum();
	}
	public static List<KmerNode> split(List<? extends KmerNode> in) {
		List<KmerNode> list = ImmutableKmerNode.split(in).collect(Collectors.toList());
		list.sort(KmerNodeUtil.ByLastStartEndKmerReferenceWeight);
		return list;
	}
	public static void assertSameNodes(List<? extends KmerNode> expected, List<? extends KmerNode> actual) {
		List<KmerNode> splitExpected = split(expected);
		List<KmerNode> splitActual = split(actual);
		assertEquals(totalWeight(expected), totalWeight(splitExpected));
		assertEquals(totalWeight(actual), totalWeight(splitActual));
		assertEquals(splitExpected, splitActual);
	}
	/**
	 * Asserts that no nodes overlap the same interval
	 * @param nodes
	 */
	public static void assertDisjoint(List<? extends KmerNode> nodes) {
		List<KmerNode> split = split(nodes);
		for (int i = 0; i < split.size(); i++) {
			KmerNode ni = split.get(i);
			for (int j = i + 1; j < split.size(); j++) {
				KmerNode nj = split.get(j);
				assertTrue(ni.lastKmer() != nj.lastKmer() || ni.lastStart() != nj.lastStart());
			}
		}
	}
	public static KmerPathNode KPN(long[] kmers, int start, int end, boolean reference, int[] weights) {
		KmerPathNode pn = new KmerPathNode(kmers[0], start, end, reference, weights[0]);
		for (int i = 1; i < kmers.length; i++) {
			pn.append(new ImmutableKmerNode(kmers[i], start + i, end + i, reference, weights[i]));
		}
		return pn;
	}
	public static KmerPathNode KPN(long[] kmers, int start, int end, boolean reference, int weight) {
		int[] weights = new int[kmers.length];
		Arrays.fill(weights, weight);
		return KPN(kmers, start, end, reference, weights);
	}
	public static KmerPathNode KPN(long[] kmers, int start, int end, boolean reference) {
		return KPN(kmers, start, end, reference, 1);
	}
	public static KmerPathNode KPN(int k, String seq, int start, int end, boolean reference) {
		return KPN(k, seq, start, end, reference, 1);
	}
	public static KmerPathNode KPN(int k, String seq, int start, int end, boolean reference, int weight) {
		int[] w = new int[seq.length() - k + 1];
		Arrays.fill(w, weight);
		return KPN(k, seq, start, end, reference, w);
	}
	public static KmerPathNode KPN(int k, String seq, int start, int end, boolean reference, int[] weight) {
		PackedKmerList kmers = new PackedKmerList(k, B(seq), new byte[seq.length()], false, false);
		KmerPathNode pn = new KmerPathNode(kmers.kmer(0), start, end, reference, weight[0]);
		for (int i = 1; i < kmers.length(); i++) {
			pn.append(new ImmutableKmerNode(kmers.kmer(i), start + i, end + i, reference, weight[i]));
		}
		return pn;
	}
	public static Iterator<KmerPathNode> asKPN(int k, int maxPathLength, DirectedEvidence... input) {
		Arrays.sort(input, DirectedEvidence.ByStartEnd);
		//int maxReadLength = maxReadLength(input);
		int maxFrag = Arrays.stream(input).mapToInt(e -> e.getEvidenceSource().getMaxConcordantFragmentSize()).sum();
		SupportNodeIterator supportIt = new SupportNodeIterator(k, Arrays.stream(input).iterator(), maxFrag, null, false, 0);
		AggregateNodeIterator agIt = new AggregateNodeIterator(supportIt);
		Iterator<KmerPathNode> pnIt = new PathNodeIterator(agIt, maxPathLength, k);
		return pnIt;
	}
	public static List<KmerPathNode> asCheckedKPN(int k, int maxPathLength, DirectedEvidence... input) {
		return asCheckedKPN(k, maxPathLength, false, input);
	}
	public static List<KmerPathNode> asCheckedKPN(int k, int maxPathLength, boolean includePairAnchors, DirectedEvidence... input) {
		Arrays.sort(input, DirectedEvidence.ByStartEnd);
		//int maxReadLength = maxReadLength(input);
		int maxFrag = Arrays.stream(input).mapToInt(e -> e.getEvidenceSource().getMaxConcordantFragmentSize()).sum();
		List<KmerSupportNode> support = Lists.newArrayList(new SupportNodeIterator(k, Arrays.stream(input).iterator(), maxFrag, null, includePairAnchors, 0));
		int supportWeight = support.stream().mapToInt(n -> n.weight() * n.width()).sum();
		List<KmerNode> aggregate = Lists.newArrayList(new AggregateNodeIterator(support.iterator()));
		int aggregateWeight = aggregate.stream().mapToInt(n -> n.weight() * n.width()).sum();
		assertEquals(supportWeight, aggregateWeight);
		List<KmerPathNode> pnList = Lists.newArrayList(new PathNodeIterator(aggregate.iterator(), maxPathLength, k));
		int pathWeight = pnList.stream().flatMap(pn -> ImmutableKmerNode.splitKmers(pn)).mapToInt(n -> n.weight() * n.width()).sum();
		assertEquals(supportWeight, pathWeight);
		assertTrue(KmerNodeUtil.ByFirstStart.isOrdered(pnList));
		PathNodeIteratorTest.assertCompleteGraph(pnList, k);
		return pnList;
	}
	public static void assertSame(KmerPathNode n1, KmerPathNode n2) {
		assertEquals(n1.length(), n2.length());
		assertEquals(n1.weight(), n2.weight());
		assertEquals(n1.lastStart(), n2.lastStart());
		assertEquals(n1.lastEnd(), n2.lastEnd());
		assertEquals(n1.isReference(), n2.isReference());
		assertEquals(n1.isValid(), n2.isValid());
		for (int i = 0; i < n1.length(); i++) {
			assertEquals(n1.kmer(i), n2.kmer(i));
			assertEquals(n1.weight(i), n2.weight(i));
		}
	}
	public static void assertSame(KmerPathSubnode n1, KmerPathSubnode n2) {
		assertEquals(n1.firstStart() , n2.firstStart());
		assertEquals(n1.firstEnd(), n2.firstEnd());
		assertSame(n1.node(), n2.node());
		assertEquals(n1.length(), n2.length());
		assertEquals(n1.weight(), n2.weight());
	}
	public static int maxReadLength(DirectedEvidence... input) {
		return Arrays.stream(input).mapToInt(e -> {
			if (e instanceof NonReferenceReadPair) {
				return ((NonReferenceReadPair)e).getNonReferenceRead().getReadLength();
			} else {
				return ((SingleReadEvidence)e).getSAMRecord().getReadLength();
			}
		}).max().orElse(0);
	}
	public static class RandomSoftClipIterator implements Iterator<DirectedEvidence> {
		public int readLength = 100;
		public int stopAfter = 1000000;
		private int n = 0;
		private Random rng = new Random(0);
		private byte[] bases = new byte[] { 'A', 'C', 'G', 'T' };
		@Override
		public boolean hasNext() {
			return n < stopAfter;
		}
		@Override
		public DirectedEvidence next() {
			byte[] seq = new byte[readLength]; 
			byte[] qual = new byte[readLength];
			for (int i = 0; i < readLength; i++) {
				seq[i] = (byte)bases[rng.nextInt(4)];
				qual[i] = (byte)rng.nextInt(45);
			}
			SAMRecord r = Read(0, ++n, "50M50S");
			r.setReadBases(seq);
			r.setBaseQualities(qual);
			return SCE(FWD, r);
					
			//int sclen = rng.nextInt(readLength / 2) + 5;
			//boolean fwd = rng.nextBoolean();
			//SAMRecord r = Read(0, n++, fwd ? String.format("%dM%dS", readLength - sclen, sclen) : String.format("%dS%dM", sclen, readLength - sclen));
			//return SCE(fwd ? FWD : BWD, r);
		}
	}
	public static List<SingleReadEvidence> asAssemblyEvidence(SAMEvidenceSource aes, Iterable<SAMRecord> list) {
		return StreamSupport.stream(list.spliterator(), false)
				.flatMap(r -> SingleReadEvidence.createEvidence(aes, 0, r).stream())
				.collect(Collectors.toList());
	}
	public static SingleReadEvidence asAssemblyEvidence(SAMEvidenceSource aes, SAMRecord assembly) {
		List<SingleReadEvidence> list = SingleReadEvidence.createEvidence(aes, 0, assembly);
		assertEquals(1, list.size());
		return list.get(0);
	}
	public static SingleReadEvidence asAssemblyEvidence(SAMRecord assembly) {
		return asAssemblyEvidence(AES(), assembly);
	}
	public static SingleReadEvidence incorporateRealignment(SAMEvidenceSource aes, SingleReadEvidence assembly, List<SAMRecord> realignments) {
		return incorporateRealignment(aes, assembly.getSAMRecord(), realignments);
	}
	public static SingleReadEvidence incorporateRealignment(SAMEvidenceSource aes, SAMRecord assembly, List<SAMRecord> realignments) {
		SAMRecord out = assembly.deepCopy();
		realignments.stream().forEach(r -> r.setReadName(SplitReadIdentificationHelper.getSplitReadRealignments(assembly, false, aes.getContext().getEvidenceIDGenerator()).get(0).getReadHeader()));
		SplitReadIdentificationHelper.convertToSplitRead(out, realignments);
		List<SingleReadEvidence> list = SingleReadEvidence.createEvidence(aes, 0, out);
		return list.get(0);
	}
	public static SingleReadEvidence incorporateRealignment(SAMEvidenceSource aes, SingleReadEvidence assembly, SAMRecord... realignments) {
		return incorporateRealignment(aes, assembly, Lists.newArrayList(realignments));
	}
	public static SingleReadEvidence incorporateRealignment(SAMEvidenceSource aes, SAMRecord assembly, SAMRecord... realignments) {
		return incorporateRealignment(aes, assembly, Lists.newArrayList(realignments));
	}
}
