#' Generating distance decay curves of correlations
#'
#' @param mat1 Replicate 1 - a n*n intrachromesome Hi-C contact map.
#' @param mat2 Replicate 2 - a n*n intrachromesome Hi-C contact map.
#' @param resol An integer indicating the resolution of the Hi-C matrix.
#' @param lbr An integer indicating the minumum distance of interaction
#' that is considered. Default is 0.
#' @param ubr An integer indicating the maximum distance of interaction 
#' that is considered. Defalt is 5000000.
#' @return A graph of correlation vesus distance.
#' @references HiCRep: assessing the reproducibility of Hi-C data using 
#' a stratum-adjusted correlation coefficient. Tao Yang, Feipeng Zhang, 
#' Galip Gurkan Yardimci, Ross C Hardison, William Stafford Noble, Feng
#' Yue, Qunhua Li. bioRxiv 101386; doi: https://doi.org/10.1101/101386.
#' @export
#' @examples
#' data(HiCR1)
#' data(HiCR2)
#' plotSTF(HiCR1, HiCR2, 1000000, 1000000, 5000000)

plotSTF <- function(mat1, mat2, resol, lbr = 0, ubr = 5000000){
    
    options(scipen = 999)
  
    lb <- floor(lbr/resol)
    ub <- floor(ubr/resol)
    corr <- array(ub-lb+1)
  
    cor_dist = function(dist){
    
        ffd1 <- ffd2 <- NULL
  
        for (i in 1:(nrow(mat1)-dist)){
    
            ffd1 <- c(ffd1, mat1[i+dist, i])
            ffd2 <- c(ffd2, mat2[i+dist, i])
            filt <- which(ffd1 == 0 & ffd2 == 0)
    
            ffd <- cbind(ffd1[-filt], ffd2[-filt])
        }
  
        if (nrow(ffd) != 0){
            if (length(unique(ffd[,1])) != 1 & length(unique(ffd[,2])) != 1) {
                corr = cor(ffd[,1], ffd[,2])
            } else {
                corr = NA
            }
        } else {
            corr = NA 
        }
    }
    
    st = sapply(seq(lb,ub), cor_dist)
    corr0 = unlist(st)
    corr0[is.na(corr0)] = 0
  
    plot((lb:ub)*resol, corr0, pch=16, col="red", main = "correlation vs distance", 
         xlab = "distance", ylab = "correlations")
}
