#' INSPEcT package
#'
#' @description
#' INSPEcT (INference of Synthesis, Processing and dEgradation rates in Time-cousre experiments) 
#' is a package that analyse 4sU-seq and RNA-seq time-course data in order to evaluate synthesis, 
#' processing and degradation rates and asses via modeling the rates that determines changes in mature mRNA levels.
#'
#' It implements two classes (\code{\linkS4class{INSPEcT_model}} and 
#' \code{\linkS4class{INSPEcT}}) and their corresponding methods. To have a detailed 
#' description of how the two classes are structured and which methods apply on, type:
#'
#' \code{?'INSPEcT-class'} \cr
#' \code{?'INSPEcT_model-class'} \cr
#' \code{?'INSPEcT_diffsteady-class'}
#'
#' To see how the typical workflow of INSPEcT works, type: \cr
#' \code{vignette('INSPEcT')}
#' 
#' Last but not least, to obtain a citation, type: \cr
#' \code{citation('INSPEcT')}
#'
#'
#' @docType package
#' @name INSPEcT
NULL

#' An S4 class to represent models generated by INSPEcT
#'
#' @description 
#' INSPEcT_model is a class able to store all the results of the modeling of synthesis, processing and degradation rates made via the method \code{\link{modelRates}} (slot ratesSpecs).
#' It also stores the criteria (slot parameter) to choose between the many models tested for each gene the one that better describes the data and the results. 
#' The slot simple is a flag that distinguish wheter the model contains the information of the introns or not.
#' In case not, the flag simple is set to TRUE. 
#' Also the method \code{\link{makeSimModel}} of class \code{\link{INSPEcT-class}} creates an object of class INSPEcT_model. 
#' This object will be used by \code{\link{makeSimDataset}} to generate a complete simulated data-set, whose classification performance can be tested.
#' @slot params A list that defines thresholds and how to perform log likelihood ratio tests
#' @slot ratesSpecs A list containing the modeling output
#' @slot simple A logical that indicates whether the mode of INSPEcT is simple (no pre-mRNA and degradation rates) or not.
#' @details Methods that apply to INSPEcT_model class are
#'
#'	\code{\link{[}} \cr
#'	\code{\link{AIC}} \cr
#'	\code{\link{chisqmodel}} \cr
#'	\code{\link{chisqtest}} \cr
#'	\code{\link{geneClass}} \cr
#'	\code{\link{llrtests}} \cr
#'	\code{\link{logLik}} \cr
#'	\code{\link{makeModelRates}} \cr
#'	\code{\link{makeSimDataset}} \cr
#'	\code{\link{modelSelection}} \cr
#'	\code{\link{ratePvals}} \cr
#'	\code{\link{rocCurve}} \cr
#'	\code{\link{rocThresholds}} \cr
#'	\code{\link{thresholds}} \cr
setClass('INSPEcT_model', 
	slots=c(
		params='list'
		, ratesSpecs='list'
		, simple='logical'
		)
	, prototype=list(
		simple=FALSE
		, params=list(
			modelSelection=c('llr','aic')[1]
            , thresholds=list(
                    chisquare=2e-1
                    , brown=c(synthesis=.01, degradation=.01, processing=.01)
                    )
            , llrtests=list(
                    synthesis=list(c('0','a')
                    	,c('b','ab')
                    	,c('c','ac')
                    	,c('bc','abc'))
                    , degradation=list(c('0','b')
                    	,c('a','ab')
                    	,c('c','bc')
                    	,c('ac','abc'))
                    , processing=list(c('0','c')
                    	,c('a','ac')
                    	,c('b','bc')
                    	,c('ab','abc'))
                    )
				)
			)
		)

# class definition
#' An S4 class to contain all rates and models generated by INSPEcT
#'
#' @description 
#' INSPEcT is a class able to store all the estimated rates and concentrations (slot ratesFirstGuess), 
#' the modeled rates and concentrations (slot modelRates) and the model themselves (slot model).
#' Within the class INSPEcT other information regarding the experimental design are stored, such as the time
#' points where experiments were collected (slot tpts), the labeling time (slot tL) and the normalization scale
#' fators used for RNA- (totalSF) and 4sU-seq libraries (labeledSF). A list of parameters that will be used during the 
#' modeling process is stored within the slot params and can be accessed by \code{\link{modelingParams}}.
#' A new instance of the class INSPEcT can be generated by the constructor function \code{\link{newINSPEcT}}.
#' @slot params A list of parameters of the modeling part
#' @slot ratesFirstGuess An object of class ExpressionSet that contains all the rates and concentrations guessed from the first part of INSPEcT analysis (before modeling)
#' @slot precision A matrix that contains the estimated precision of the rates. Rows represent genes, Columns represent time points.
#' @slot model An object of class INSPEcT_model that contains the output of the mdoeling
#' @slot modelRates An object of class ExpressionSet that contains all modeled the rates and concentrations
#' @slot tpts A numeric vector of the time-points
#' @slot labeledSF A numeric vector of the scaling factor used for inter time-point normalization of 4sU-seq libraries
#' @slot totalSF A numeric vector of the scaling factor used for inter time-point normalization of RNA-seq libraries
#' @slot tL A numeric containing the length of the 4sU pulse
#' @details Methods that apply to INSPEcT class are
#'
#'	\code{\link[=Extract]{[}} \cr
#'	\code{\link{AIC}} \cr
#'	\code{\link{chisqmodel}} \cr
#'	\code{\link{chisqtest}} \cr
#'	\code{\link{combine}} \cr
#'	\code{\link{dim}} \cr
#'	\code{\link{featureNames}} \cr
#'	\code{\link{geneClass}} \cr
#'	\code{\link{getModel<-}} \cr
#'	\code{\link{getModel}} \cr
#'	\code{\link{inHeatmap}} \cr
#'	\code{\link{labeledSF}}
#'	\code{\link{llrtests}} \cr
#'	\code{\link{logLik}} \cr
#'	\code{\link{makeModelRates}} \cr
#'	\code{\link{makeSimModel}} \cr
#'	\code{\link{modelingParams<-}} \cr
#'	\code{\link{modelingParams}} \cr
#'	\code{\link{modelRates}} \cr
#'	\code{\link{modelSelection}} \cr
## #'	\code{\link{nCores<-}} \cr
## #'	\code{\link{nCores}} \cr
#'	\code{\link{nGenes}} \cr
#'	\code{\link{nTpts}} \cr
#'	\code{\link{plotGene}} \cr
#'	\code{\link{ratePvals}} \cr
#'	\code{\link{ratesFirstGuessVar}} \cr
#'	\code{\link{ratesFirstGuess}} \cr
#'	\code{\link{removeModel}} \cr
#'	\code{\link{sfPlot}} \cr
#'	\code{\link{thresholds}} \cr
#'	\code{\link{totalSF}} \cr
#'	\code{\link{tpts}} \cr
#'	\code{\link{viewModelRates}} \cr
setClass('INSPEcT', 
	slots=c(
		params='list'
		, ratesFirstGuess='ExpressionSet'
		, precision='matrix'
		, model='INSPEcT_model'
		, modelRates='ExpressionSet'
		, tpts='numeric'
		, labeledSF='numeric'
		, totalSF='numeric'
		, tL='numeric'
		)
	, prototype=list(
		params=list(
			nInit=10
			, nIter=300
			, na.rm=TRUE
			# , nCores=2L
			, verbose=TRUE
			, estimateRatesWith=c('int', 'der')[1]
			, useSigmoidFun=TRUE
			, testOnSmooth=TRUE
			)
		)
	)


#' An S4 class to represent comparisons between two steady-state conditions
#'
#' @description 
#' INSPEcT_diffsteady is a class able to store the results of the comparisons between two steady states.
#' An object of class INSPEcT_diffsteady is created with the method "compareSteady" applied on 
#' two "INSPEcT" objects (see \code{\link{compareSteady}}).
#' @slot synthesis A data.frame which contains both input data and comparisons results regarding
#' synthesis rates
#' @slot degradation A data.frame which contains both input data and comparisons results regarding
#' degradation rates
#' @slot processing A data.frame which contains both input data and comparisons results regarding
#' processing rates
#' @details
#' Methods associated to the class INSPEcT_diffsteady are:
#' \itemize{
#'   \item synthesis: Accessor to the synthesis rates and their comparisons.
#'   \item degradation: Accessor to the degradation rates and their comparisons.
#'   \item processing: Accessor to the processing rates and their comparisons.
#'	 \item plotMA: visualization fuction for rates comparisons, see \code{\link{plotMA}}
#' }
setClass('INSPEcT_diffsteady', 
	slots=c(
		synthesis='data.frame'
		, degradation='data.frame'
		, processing='data.frame'
		)
	)
