# Copyright 2017 by Steven Rozen and Alvin Wei Tian Ng

# Released under the GPL-3 license

# Modified from
#
# Renaud Gaujoux (2015). Algorithms and Framework for Nonnegative
# Matrix Factorization (NMF) CRAN. R package version
# 0.20.6. [http://cran.r-project.org/package=NMF] Cathal Seoighe
# (2015). Algorithms and Framework for Nonnegative Matrix
# Factorization (NMF) CRAN. R package version
# 0.20.6. [http://cran.r-project.org/package=NMF]

# Non-negative factorization with a fixed W matrix and a Pascual and
# Montano smoothing matrix

require(NMF)

# Function my.algorithm.supervised.smooth will be provided to NMF as
# an alternative to the Lee and Seung update algorithm.
my.algorithm.supervised.smooth <-
      function(i,
                            v,              # The input spectra
                            seed,           # An NMF object
                            rescale=TRUE,   # Not used, but is is used by caller?
                            w=NULL,         # The signature matrix
                            th=NULL,        # Smoothing parameter, theta
                            eps=10^-9,      # epsilon
                            ...){

                v <- as.matrix(v)

                        ## makes it supervised by using a known W matrix
                        if (is.null(w)){
                                      w <-  .basis(seed) / colSums(.basis(seed))
                                    } else {
                                                  w <-  w / colSums(w)
                                                }

                        h <- .coef(seed)
                        wh <- w%*%h

                        r <- nrow(h)

                        # Create the smoothing matrix, s
                        s <- (1-th) * diag(r) + th/r * matrix(1, nrow=r, ncol=r)

                        # w.s is smoothed version of w
                        w.s = w %*% s

                        # Standard update of the h matrix
                        h <- pmax(h * (t(w.s) %*% v),eps) /
                                      ((t(w.s) %*% w.s) %*% h + eps)

                        # Save the unmodified w back to the model
                        .basis(seed) <- w

                        # Save the updated h in the model
                        .coef(seed) <- h;

                        # Standard return value for the NMF package
                        return(seed)
              }

# Register the update in the R NMF package
nmf.lee <- setNMFMethod('supervised_smooth',
                                                'lee',
                                                Update=my.algorithm.supervised.smooth,
                                                overwrite=TRUE,
                                                objective='KL')

# Example usage
# nmf(v, rank= ncol(w), 'supervised_smooth', w=w, th=th, ...)

