from copy import deepcopy
from math import ceil

import numpy as np

def remove_primer_primer_pairs(counts_superdict, primermap, count_threshold=10,
                               num_reps=3, max_reps=None,
                               percentage_reps=None, all_reps=False):
    """
    Removes primer-primer pairs that express low counts across a few or all replicates
    by wiping with np.nan

    Parameters
    ----------
    counts_superdict : Dict[str,Dict[str,np.ndarray]]
        The outer keys are the replicate name, the inner keys are the
        region name and the values are the raw counts matrices
    primermap : Dict[str, List[Dict[str, Any]]]
        The keys of the outer dict are region names. The values are lists, where
        the :math:`i` th entry represents the :math:`i` th primer in that
        region. Primers are represented as dicts with the following structure::

            {
                'chrom' : str,
                'start' : int,
                'end'   : int
            }
    count_threshold : int
        Primer-primer pairs that express less counts than this number will
        be considered low enough for wiping
    num_reps : int
        Primer-primer pairs that have low counts across this number
        of replicates will be classified as "failed" and wiped
    max_reps : int
        Keep a primer-primer pair if it is low in more than this
        many replicates.
    percentage_reps : float
        Specifies a percentage of total replicates that must contain the
        low count primer-primer pair of interest for that primer-primer pair to be wiped.
    all_reps : bool
        If set to True, the primer-primer pair must have low
        counts across all replicates to be wiped.If set to False,
        the user must specify num_reps or percentage_reps.

    Returns
    -------
    Dict[str,Dict[str,np.ndarray]]
    The outer keys are the replicate name, the inner keys are the
        region name and the values are the trimmed counts matrices
    """
    new_counts_superdict = deepcopy(counts_superdict)

    # resolve num_reps vs percentage_reps
    if not num_reps and percentage_reps:
        num_reps = int(ceil(len(new_counts_superdict) * percentage_reps / float(100)))

    # fall back to all_reps
    if not num_reps and not percentage_reps:
        all_reps = True

    # set nan's
    for region in primermap:
        for i in range(len(primermap[region])):
            for j in range(i + 1):
                # flag indicating whether or not this primer-primer combination is considered to be failed
                failed = False

                # counts for this primer-primer combination across all replicates
                replicate_values = [new_counts_superdict[rep][region][i, j]
                                    for rep in new_counts_superdict
                                    if np.isfinite(new_counts_superdict[rep][region][i, j])]

                # check for failure
                if all_reps:
                    replicate_sum = sum(replicate_values)
                    if replicate_sum < count_threshold:
                        failed = True
                else:
                    num_failed = len(filter(lambda x: x < count_threshold, replicate_values))
                    if max_reps:
                        if num_failed >= num_reps and num_failed < max_reps:
                            failed = True
                    else:
                        if num_failed >= num_reps:
                            failed = True

                # if failed, set this primer-primer combination to nan for all replicates
                if failed:
                    for rep in new_counts_superdict:
                        new_counts_superdict[rep][region][i, j] = np.nan
                        new_counts_superdict[rep][region][j, i] = np.nan

    return new_counts_superdict
