import numpy as np
import scipy.stats as stats

from function_util import parallelize_regions

@parallelize_regions
def queried_counts_to_pvalues(queried_counts):
    """
    Convert a queried counts matrix to a matrix of equivalent right-tail
    p-values using the emprical CDF.

    Parameters
    ----------
    queried_counts : np.ndarray
        The matrix of queried counts for this region. See
        ``lib5c.util.counts.extract_queried_counts()``.

    Returns
    -------
    np.ndarray
        The empirical p-value queried counts matrix for this region.
    """
    flattened_counts = queried_counts.flatten()
    flat_pvalues = 1.0 - (stats.rankdata(flattened_counts, 'min') - 1.0) / \
                         np.isfinite(flattened_counts).sum()
    flat_pvalues[flat_pvalues <= 0] = np.nan
    return np.reshape(flat_pvalues, queried_counts.shape)
