import numpy as np

def polyfit_model_fragments(counts, distances, degree=1, pseudocount=1,
                            logging_function=np.log,
                            exponentiating_function=np.exp,
                            is_global=False):
    """
    Make a one-dimensional fragment-level expected model by fitting a
    polynomial in log-log space.

    Parameters
    ----------
    counts : Dict[str, np.ndarray] or np.ndarray
        The observed counts dict or matrix to fit the model to.
    distances : Dict[str, np.ndarray]
        A dict of pairwise distance matrices describing the genomic distances
        between the elements of the matrices in ``counts``. The keys and array
        dimensions should match the keys and array dimensions of ``counts``.
    degree : int
        The degree of the polynomial to fit.
    pseudocount : int
        The pseudocount to add to the counts before logging.
    is_global : bool
        Sets the scale of the expected model. If False, the algorithm will
        determine the regional expected model
    Returns
    -------
    Dict[int, float]
        A mapping from interaction distances in units of base pairs to the
        expected value at that distance.
    """
    if is_global:
        # log transform
        log_counts = {region: logging_function(counts[region] + pseudocount)
                      for region in counts.keys()}
        log_distances = {region: logging_function(distances[region] + pseudocount)
                         for region in distances.keys()}

        # make data of the form [log_distance, log_count], ignoring nans
        data = np.asarray([[log_distances[region][i, j], log_counts[region][i, j]]
                            for region in log_counts.keys()
                            for i in range(len(log_counts[region]))
                            for j in range(i + 1)
                            if np.isfinite(log_counts[region][i, j])])
    else:
        log_counts = logging_function(counts + pseudocount)
        log_distances = logging_function(distances + pseudocount)

        # make data of the form [log_distance, log_count], ignoring nans
        data = np.asarray([[log_distances[i, j], log_counts[i, j]]
                            for i in range(len(log_counts))
                            for j in range(i + 1)
                            if np.isfinite(log_counts[i, j])])

    # do the linear fit
    fit = np.poly1d(np.polyfit(data[:, 0], data[:, 1], degree))

    if is_global:
        # repackage
        distance_expected = {dist: exponentiating_function(fit(logging_function(dist + pseudocount))) - pseudocount
                            for dist in np.unique(np.concatenate([distances[region].flatten()
                            for region in distances.keys()]))}
    else:
        # repackage
        distance_expected = {dist: exponentiating_function(fit(logging_function(dist + pseudocount))) - pseudocount
                            for dist in np.unique(distances.flatten())}

    return distance_expected

