import numpy as np

def polyfit_model_bins(counts, degree=1, pseudocount=1,is_global=True):
    """
    Make a one-dimensional bin-level expected model by fitting a
    polynomial in log-log space.

    Parameters
    ----------
    counts : Dict[str, np.ndarray]
        The observed counts dict to fit the model to.
    degree : int
        The degree of the polynomial to fit.
    pseudocount : int
        The pseudocount to add to the counts before logging.
    is_global : bool
        Sets the scale of the expected model. If False, the algorithm
        will calculate the regional model

    Returns
    -------
    List[float]
        The one-dimensional expected model. The ``i`` th element of the list
        corresponds to the expected value for interactions between loci
        separated by ``i`` bins. The length of this list will match the size of
        the largest region in the input counts dict.
    """
    if is_global:
        # log transform
        log_counts = {region: logging_function(counts[region] + pseudocount)
                      for region in counts.keys()}

        # make data of the form [log_distance, log_count], ignoring nans
        data = np.asarray([[logging_function(i - j + pseudocount), log_counts[region][i, j]]
                            for region in log_counts.keys()
                            for i in range(len(log_counts[region]))
                            for j in range(i + 1)
                            if np.isfinite(log_counts[region][i, j])])
    else:
        log_counts = logging_function(counts + pseudocount)

        # make data of the form [log_distance, log_count], ignoring nans
        data = np.asarray([[logging_function(i - j + pseudocount), log_counts[i, j]]
                            for i in range(len(log_counts))
                            for j in range(i + 1)
                            if np.isfinite(log_counts[i, j])])

    # do the linear fit
    fit = np.poly1d(np.polyfit(data[:, 0], data[:, 1], degree))

    if is_global:
        # repackage
        distance_expected = [exponentiating_function(fit(logging_function(i + pseudocount))) - pseudocount
                            for i in range(max([len(counts[region])
                            for region in counts.keys()]))]
    else:
        # repackage
        distance_expected = [exponentiating_function(fit(logging_function(i + pseudocount))) - pseudocount
                            for i in range(len(counts))]


    return distance_expected

