import numpy as np

from statsmodels.nonparametric.smoothers_lowess import lowess

def lowess_model_fragments(counts, distances, pseudocount=1, frac=0.8,
                           logging_function=np.log,
                           exponentiating_function=np.exp,is_global=True):
    """
    Make a one-dimensional fragment-level expected model by performing lowess
    regression in log-log space.

    Parameters
    ----------
    counts : Dict[str, np.ndarray] or np.ndarray
        The observed counts dict or matrix to fit the model to.
    pseudocount : int
        The pseudocount to add to the counts before logging.
    distances : Dict[str, np.ndarray]
        A dict of pairwise distance matrices describing the genomic distances
        between the elements of the matrices in ``counts``. The keys and array
        dimensions should match the keys and array dimensions of ``counts``.
    frac : float
        The lowess smoothing fraction parameter to use.
    is_global : bool
        Sets the scale of the expected model. If False, the algorithm
        will determine the regional expected model

    Returns
    -------
    List[float]
        The one-dimensional expected model. The ``i`` th element of the list
        corresponds to the expected value for interactions between loci
        separated by ``i`` bins. If the model is global, the length of this list 
        will match the size of the largest region in the input counts dict.
    """
    if is_global:
        # log transform
        log_counts = {region: logging_function(counts[region] + pseudocount)
                      for region in counts.keys()}
        log_distances = {region: logging_function(distances[region] + pseudocount)
                        for region in distances.keys()}

        # make data of the form [log_distance, log_count], ignoring nans
        data = np.asarray([[log_distances[region][i, j], log_counts[region][i, j]]
                            for region in log_counts.keys()
                            for i in range(len(log_counts[region]))
                            for j in range(i + 1)
                            if np.isfinite(log_counts[region][i, j])])
        pass
    else:
        log_counts = logging_function(counts + pseudocount)
        log_distances = logging_function(distances + pseudocount)

        # make data of the form [distance, count], ignoring nans
        data = np.asarray([[log_distances[i, j], log_counts[i, j]]
                            for i in range(len(log_counts))
                            for j in range(i + 1)
                            if np.isfinite(log_counts[i, j])])

    # do the lowess fit
    fit = lowess(data[:, 1], data[:, 0], frac=frac, it=3)

    # unlog
    fit_dists = np.rint(exponentiating_function(fit[:, 0]) - pseudocount).astype(int)
    fit_counts = exponentiating_function(fit[:, 1]) - pseudocount

    # repackage
    distance_expected = {fit_dists[i]: fit_counts[i] for i in range(len(fit))}
    
    if is_global:
        # fill nans
        for dist in np.unique(np.concatenate([distances[region].flatten() for region in distances.keys()])):
            if dist not in distance_expected:
                distance_expected[dist] = np.nan
        pass
    else:
        # fill nans
        for dist in np.unique(distances.flatten()):
            if dist not in distance_expected:
                distance_expected[dist] = np.nan

    return distance_expected
