import numpy as np

from statsmodels.nonparametric.smoothers_lowess import lowess

def lowess_model_bins(counts, pseudocount=1, frac=0.8,is_global=True,
                     logging_function=np.log,exponentiating_function=np.exp):
    """
    Make a one-dimensional bin-level expected model by performing lowess
    regression in log-log space.

    Parameters
    ----------
    counts : Dict[str, np.ndarray] or np.ndarray
        The observed counts dict or matrix to fit the model to.
    pseudocount : int
        The pseudocount to add to the counts before logging.
    frac : float
        The lowess smoothing fraction parameter to use.
    is_global : bool
        Sets the scale of the expected model. If False, the algorithm
        will determine the regional expected model

    Returns
    -------
    List[float]
        The one-dimensional expected model. The ``i`` th element of the list
        corresponds to the expected value for interactions between loci
        separated by ``i`` bins. If the model is global, the length of this list 
        will match the size of the largest region in the input counts dict.
    """
    if is_global:
        # log transform
        log_counts = {region: logging_function(counts[region] + pseudocount)
                      for region in counts.keys()}

        # make data of the form [distance, count], ignoring nans
        data = np.asarray([[logging_function(i - j + pseudocount), log_counts[region][i, j]]
                            for region in log_counts.keys()
                            for i in range(len(log_counts[region]))
                            for j in range(i + 1)
                            if np.isfinite(log_counts[region][i, j])])
        pass
    else:
        log_counts = logging_function(counts + pseudocount)

        # make data of the form [distance, count], ignoring nans
        data = np.asarray([[logging_function(i - j + pseudocount), log_counts[i, j]]
                            for i in range(len(log_counts))
                            for j in range(i + 1)
                            if np.isfinite(log_counts[i, j])])

    # do the lowess fit
    fit = lowess(data[:, 1], data[:, 0], frac=frac, it=3)

    # unlog
    fit_dists = np.rint(exponentiating_function(fit[:, 0]) - pseudocount).astype(int)
    fit_counts = exponentiating_function(fit[:, 1]) - pseudocount

    # repackage
    distance_expected = {fit_dists[i]: fit_counts[i] for i in range(len(fit))}
    
    if is_global:
        distance_expected = [distance_expected[i]
                            if i in distance_expected
                            else np.nan
                            for i in range(max([len(counts[region])
                            for region in counts.keys()]))]
        pass
    else:
        distance_expected = [distance_expected[i]
                         if i in distance_expected
                         else np.nan
                         for i in range(len(counts))]

    return distance_expected
