#!/bin/bash

# Script to filter out peaks mainly overlapping promoter regions.
# This script relies on 'Supplemental_File_S1.py' and bedtools intersect.
# Usage: ./Supplemental_File_S2.sh peaks.bed normalized.bw promoter_regions.bed Supplemental_File_S1.py output_file.bed
# 	peaks.bed = BED file containing the coordinates of the significant peaks identified by a peak caller (e.g. SICER).
# 	normalized.bw = bw file generated using a normalized WIG file (i.e. the WIG file generated by SICER using the reads kept after filtering out redundant reads).
# 	promoter_regions.bed = BED file with the coordinates of promoter regions to be considered (i.e. +-2kb from TSSs).
# 	Supplemental_File_S1.py = path to the 'Supplemental_File_S1.py' script.
#	output_file.bed = name for the output file.

peaks=$1
bw_f=$2
promoters=$3
summit_p=$4
output_f=$5

# Generate temporal files to store the summits and filtered peaks.
tmpf_s=$(mktemp tmp_file.XXXXXX)
tmp_c=$(mktemp tmp_file.XXXXXX)

# Run program to identify summits.
python $summit_p -BigWigFile $bw_f -enhancers $peaks > $tmpf_s

# Get coordinates of the peaks with summits outside promoter regions.
bedtools intersect -v -a $tmpf_s -b $promoters | bedtools intersect -wa -a $peaks -b - > $tmp_c

# Identify peaks for which the percentage of overlap with promoter regions is less than 50% of their total length.
bedtools intersect -a $tmpf_s -b $promoters | sort | uniq | bedtools intersect -wa -a $peaks -b - | bedtools intersect -f 0.5 -v -a - -b $promoters >> $tmp_c

# Format file with the filtered peaks.
awk 'BEGIN{c=1};{print$0"\t"c;c=c+1}' $tmp_c > $output_f

rm "$tmpf_s"
rm "$tmp_c"

