/* Copyright 2012 Tobias Marschall
 *
 * This file is part of CLEVER.
 *
 * Adapted by Stefan Canzar for use in SpliceHunter
 *
 * You should have received a copy of the GNU General Public License
 * along with CLEVER.  If not, see <http://www.gnu.org/licenses/>.
 */


#include <cassert>

#include "NamedDnaSequence.h"
#include <iostream>

using std::string;


NamedDnaSequence::NamedDnaSequence(const string& name) : case_sensitive(false), name(name), length(0) {
}

NamedDnaSequence::NamedDnaSequence(const string& name, const string& sequence) : case_sensitive(false), name(name), length(0) {
	append(sequence);
}

NamedDnaSequence::NamedDnaSequence(const string& name, bool case_sensitive) : case_sensitive(case_sensitive), name(name), length(0) {
}

NamedDnaSequence::NamedDnaSequence(const string& name, const string& sequence, bool case_sensitive) : case_sensitive(case_sensitive), name(name), length(0) {
	append(sequence);
}

char NamedDnaSequence::decode(unsigned char c) const {
	switch (c) {
	case 0: return 'A';
	case 1: return 'C';
	case 2: return 'G';
	case 3: return 'T';
	case 4: return 'a';
	case 5: return 'c';
	case 6: return 'g';
	case 7: return 't';
	case 8: return 'n';
	default: return 'N';
	}
}

unsigned char NamedDnaSequence::encode(char c) const {
	if (case_sensitive) {
		switch (c) {
		case 'A': return 0;
		case 'C': return 1;
		case 'G': return 2;
		case 'T': return 3;
		case 'a': return 4;
		case 'c': return 5;
		case 'g': return 6;
		case 't': return 7;
		case 'n': return 8;
		default: return 9;
		}
	} else {
		switch (c) {
		case 'A': return 0;
		case 'C': return 1;
		case 'G': return 2;
		case 'T': return 3;
		case 'a': return 0;
		case 'c': return 1;
		case 'g': return 2;
		case 't': return 3;
		case 'n': return 9;
		default: return 9;
		}
	}
}

void NamedDnaSequence::append(const string& s) {
	string::const_iterator it = s.begin();
	for (;it!=s.end(); ++it) {
		if (length % 2 == 0) {
			sequence.push_back(encode(*it));
		} else {
			sequence[length/2] |= encode(*it)<<4;
		}
		length += 1;
	}
}

char NamedDnaSequence::operator[](size_t pos) const {
	assert(pos < length);   
	if (pos % 2 == 0) {
		return decode(sequence[pos>>1] & 0x0f);
	} else {
		return decode(sequence[pos>>1] >>4);
	}
}

string NamedDnaSequence::substr(size_t start, size_t end) const {
    string result = "";
    for (size_t i = start; i < end; ++i) {
        result += (*this)[i];
    }
    return result;
}
