/**
 * Mutinack mutation detection program.
 * Copyright (C) 2014-2016 Olivier Cinquin
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, version 3.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package uk.org.cinquin.mutinack.misc_util;

import java.io.BufferedWriter;
import java.io.FileWriter;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.Iterator;
import java.util.function.Consumer;
import java.util.stream.Stream;

import uk.org.cinquin.mutinack.features.ParseRTException;
import uk.org.cinquin.mutinack.misc_util.exceptions.AssertionFailedException;

/**
 * Utility class to load paired reads from fastq files, and merge barcode and barcode
 * quality comments (generated by ad hoc Mutinack perl script) to the read name. That
 * way the information is preserved when the reads are run through an aligner that
 * does not allow for the comments to be stored as BAM read attributes.
 * This class is meant to be used directly from the command line.
 * @author olivier
 *
 */
public class MergeBarcodesToFastQReadName {
	
	private static final Consumer<String> checkStartsWithAt = 
			s -> {if (!s.startsWith("@")) throw new ParseRTException(s + " does not start with @" );};

	private static final Consumer<String> checkIsPlus = 
			s -> {if (!"+".equals(s)) throw new ParseRTException(s + " is not equal to +" );};

	/**
	 * Arguments are four file paths: two input paths (fastq formatted files) and 
	 * two output files. Reads must appear in the same order in input files. The output
	 * files are overwritten, and deleted if any failure is encountered.
	 * @param args
	 * @throws IOException, ParseRTException
	 */
	public static void main(String[] args) throws IOException, ParseRTException {
		final String inFile1 = args[0], inFile2 = args[1], outFile1 = args[2], outFile2 = args[3];

		try (BufferedWriter fw1 = new BufferedWriter(new FileWriter(outFile1));
			 BufferedWriter fw2 = new BufferedWriter(new FileWriter(outFile2));
			 Stream<String> lines1 = Files.lines(Paths.get(inFile1));
			 Stream<String> lines2 = Files.lines(Paths.get(inFile2))) {
			
			final Iterator<String> l1it = lines1.iterator();
			final Iterator<String> l2it = lines2.iterator();
			
			int phase = -1;

			while (l1it.hasNext()) {
				final String l1 = l1it.next();
				if (!l2it.hasNext()) {
					throw new ParseRTException("File 2 has fewer lines than file 1");
				}
				final String l2 = l2it.next();
				phase++;
				if (phase == 4) {
					phase = 0;	
				}
				final Consumer<String> formatCheck;
				switch (phase) {
					case 0:
						String[] l1s = l1.split(" ");
						String[] l2s = l2.split(" ");
						if (!l1s[0].equals(l2s[0])) {
							throw new ParseRTException("Read names " + l1s[0] + " and " + l2s[0] + " do not match");
						}
						String bc1 = l1s[1] + "_" + l1s[2];
						String bc2 = l2s[1] + "_" + l2s[2];
						String newName = l1s[0] + "_" + bc1 + "_" + bc2;
						StringBuilder restOfLine1 = new StringBuilder(newName), restOfLine2 = new StringBuilder(newName);
						for (int i = 1; i < l1s.length; i++) {
							restOfLine1.append(" ").append(l1s[i]);
						}
						for (int i = 1; i < l2s.length; i++) {
							restOfLine2.append(" ").append(l2s[i]);
						}
						fw1.append(restOfLine1.append("\n"));
						fw2.append(restOfLine2.append("\n"));
						formatCheck = checkStartsWithAt;
						break;
					case 1:
						fw1.append(l1).append("\n");
						fw2.append(l2).append("\n");
						formatCheck = null;
						break;
					case 2:
						fw1.append(l1).append("\n");
						fw2.append(l2).append("\n");
						formatCheck = checkIsPlus;
						break;
					case 3:
						fw1.append(l1).append("\n");
						fw2.append(l2).append("\n");
						formatCheck = null;
						break;
					default: throw new AssertionFailedException();
				}
				if (formatCheck != null) {
					formatCheck.accept(l1);
					formatCheck.accept(l2);
				}
			}
			if (l2it.hasNext()) {
				throw new ParseRTException("File 2 has more lines than file 1");
			}
		} catch (Throwable t) {
			t.printStackTrace();
			try {
				Files.delete(Paths.get(outFile1));
				Files.delete(Paths.get(outFile2));
			} catch (Exception e) {	
			}
			System.exit(1);
		}
	}
}
