/*
 * The MIT License
 *
 * Copyright (c) 2009 The Broad Institute
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package contrib.net.sf.picard.sam;

import java.io.File;
import java.io.IOException;
import java.io.PrintStream;

import contrib.net.sf.picard.PicardException;
import contrib.net.sf.picard.cmdline.CommandLineProgram;
import contrib.net.sf.picard.cmdline.Option;
import contrib.net.sf.picard.cmdline.StandardOptionDefinitions;
import contrib.net.sf.picard.cmdline.Usage;
import contrib.net.sf.picard.io.IoUtil;
import contrib.net.sf.samtools.*;
import contrib.net.sf.samtools.util.AsciiWriter;

/**
 * Very simple command that just reads a SAM or BAM file and writes out the header
 * and each records to standard out.
 *
 * @author tfennell@broad.mit.edu
 */
public class ViewSam extends CommandLineProgram {
    public enum AlignmentStatus { Aligned, Unaligned, All }
    public enum PfStatus { PF, NonPF, All }

    @Usage public final String USAGE = getStandardUsagePreamble() + "Prints a SAM or BAM file to the screen.";
    @Option(shortName=StandardOptionDefinitions.INPUT_SHORT_NAME, doc="The SAM or BAM file to view.")
    public File INPUT;

    @Option(doc="Print out all reads, just the aligned reads or just the unaligned reads.")
    public AlignmentStatus ALIGNMENT_STATUS = AlignmentStatus.All;

    @Option(doc="Print out all reads, just the PF reads or just the non-PF reads.")
    public PfStatus PF_STATUS = PfStatus.All;

    public static void main(final String[] args) {
        new ViewSam().instanceMain(args);
    }

    @Override
    protected int doWork() {
        return writeSamText(System.out);
    }

    /**
     * This is factored out of doWork only for unit testing.
     */
    int writeSamText(PrintStream printStream) {
        try {
            IoUtil.assertFileIsReadable(INPUT);
            final SAMFileReader in = new SAMFileReader(INPUT);
            final AsciiWriter writer = new AsciiWriter(printStream);
            final SAMFileHeader header = in.getFileHeader();
            if (header.getTextHeader() != null) {
                writer.write(header.getTextHeader());
            } else {
                // Headers that are too large are not retained as text, so need to regenerate text
                new SAMTextHeaderCodec().encode(writer, header, true);
            }

            for (final SAMRecord rec : in) {
                if (printStream.checkError()) {
                    return 1;
                }

                if (this.ALIGNMENT_STATUS == AlignmentStatus.Aligned   && rec.getReadUnmappedFlag()) continue;
                if (this.ALIGNMENT_STATUS == AlignmentStatus.Unaligned && !rec.getReadUnmappedFlag()) continue;

                if (this.PF_STATUS == PfStatus.PF    && rec.getReadFailsVendorQualityCheckFlag()) continue;
                if (this.PF_STATUS == PfStatus.NonPF && !rec.getReadFailsVendorQualityCheckFlag()) continue;

                writer.write(rec.getSAMString());
            }
            writer.flush();
            if (printStream.checkError()) {
                return 1;
            }
            return 0;
        } catch (IOException e) {
            throw new PicardException("Exception writing SAM text", e);
        }
    }
}
