/*
 * The MIT License
 *
 * Copyright (c) 2010 The Broad Institute
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package contrib.net.sf.picard.sam;

import java.io.File;

import contrib.net.sf.picard.cmdline.CommandLineProgram;
import contrib.net.sf.picard.cmdline.Option;
import contrib.net.sf.picard.cmdline.StandardOptionDefinitions;
import contrib.net.sf.picard.cmdline.Usage;
import contrib.net.sf.picard.io.IoUtil;
import contrib.net.sf.picard.util.Log;
import contrib.net.sf.samtools.BAMIndex;
import contrib.net.sf.samtools.BAMIndexMetaData;

/**
 * Command line program to print statistics from BAM index (.bai) file
 * Statistics include count of aligned and unaligned reads for each reference sequence
 * and a count of all records with no start coordinate.
 * Similar to the 'samtools idxstats' command.
 *
 * @author Martha Borkan
 */
public class BamIndexStats extends CommandLineProgram {

    private static final Log log = Log.getInstance(BuildBamIndex.class);

    @Usage
    public String USAGE = getStandardUsagePreamble() + "Generates BAM index statistics. " +
            "Input BAM file must have a corresponding index file.\n";

    @Option(shortName= StandardOptionDefinitions.INPUT_SHORT_NAME,
            doc="A BAM file to process.")
    public File INPUT;


    /** Stock main method for a command line program. */
    public static void main(final String[] argv) {
        System.exit(new BamIndexStats().instanceMain(argv));
    }

    /**
     * Main method for the program.  Checks that input file is present and
     * readable, then iterates through the index printing meta data to stdout.
     */
    @Override
		protected int doWork() {

        if (INPUT.getName().endsWith(BAMIndex.BAMIndexSuffix))
               log.warn("INPUT should be BAM file not index file");
        IoUtil.assertFileIsReadable(INPUT);
        BAMIndexMetaData.printIndexStats(INPUT);

        return 0;
    }
}
